/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
// import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.ColumnActionListener;			// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.io.TableReader;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
// import org.opengion.fukurou.util.FileUtil;
// import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.FileInfo;						// 6.2.3.0 (2015/05/01)
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.model.ArrayDataModel;

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.util.HybsConst.BR;			// 6.1.0.0 (2014/12/26) refactoring

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import java.io.File;
// import java.io.BufferedReader;
// import java.io.IOException;

/**
 * 指定のファイルを直接データベースに登録するデータ入力タグです。
 *
 * 通常の readTable などは、DBTableModelオブジェクトを介して全件メモリに
 * ロードしてから表示させる為、大量データ処理ができません。
 * このタグでは、直接ファイルを読み取りながらデータベース登録するので
 * 大量データをバッチ的に登録する場合に使用します。
 *
 * 読み取るファイルは、先頭(または実データが現れるまでに) #NAME 行が必要です。
 * これは、ファイルデータのカラム名を指定しています。また、columns 属性を使用すれば、
 * ファイルの#NAME 行より優先して(つまり存在していなくても良い)データのカラム名を
 * 指定することが出来ます。
 * この#NAME 行は、ファイルのセパレータと無関係に必ずタブ区切りで用意されています。
 * タグのBODY部に、実行するSQL文を記述します。
 * このSQL文は、
 * INSERT INTO GE41 (CLM,NAME_JA,SYSTEM_ID,FGJ,DYSET)
 * VALUES ([CLM],[NAME_JA],[SYSTEM_ID],'1','{&#064;USER.YMDH}')
 * と、いう感じで、ファイルから読み込んだ値は、[カラム名]に割り当てられます。
 * もちろん、通常の固定値(FGJに'1'をセット)や、リクエスト変数(DYSETの{&#064;USER.YMDH})
 * なども使用できます。
 *
 * ※ 6.2.3.0 (2015/05/01)
 *    BODY部にSQL文を記述しない場合は、table 属性に、INSERTするテーブルIDを指定します。
 * ※ 6.2.4.0 (2015/05/15)
 *    omitNames に、WRITABLE と ROWID を、強制的に含めます(無条件)。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：&lt;og:directTableInsert filename="[･･･]" ･･･ &gt;INSERT INTO ･･･ &lt;/og:directTableInsert &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:directTableInsert
 *       readerClass        【TAG】実際に読み出すクラス名の略称(TableReader_**** の ****)をセットします
 *                                  (初期値:TABLE_READER_DEFAULT_CLASS[={@og.value org.opengion.hayabusa.common.SystemData#TABLE_READER_DEFAULT_CLASS}])
 *  専   commitBatch        【TAG】指定数毎にコミットを発行します(初期値:0 終了までコミットしません)
 *  専   dbid               【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)
 *  専   table              【TAG】BODYのSQL文を指定しない場合に使用するテーブルIDを指定します
 *       command            【TAG】コマンド (NEW,RENEW)をセットします(初期値:NEW)
 *       fileURL            【TAG】読取元ディレクトリ名を指定します(初期値:FILE_URL)
 *       filename           【TAG】ファイルを作成するときのファイル名をセットします (初期値:FILE_FILENAME[=file.xls])
 *       encode             【TAG】ファイルを作成するときのファイルエンコーディング名をセットします(初期値:FILE_ENCODE)
 *       skipRowCount       【TAG】(通常は使いません)データの読み飛ばし件数を設定します
 *       maxRowCount        【TAG】読取時の最大取り込み件数をセットします (初期値:0:[無制限])
 *       errRowCount        【TAG】読取時の最大エラー件数をセットします (初期値:{@og.value org.opengion.hayabusa.taglib.ReadTableTag#ERROR_ROW_COUNT})(0:[無制限])
 *       separator          【TAG】可変長ファイルを作成するときの項目区切り文字をセットします
 *       columns            【TAG】読取元ファイルのカラム列を、外部(タグ)よりCSV形式で指定します
 *       omitNames          【TAG】読取対象外のカラム列を、外部(タグ)よりCSV形式で指定します
 *       modifyType         【TAG】ファイル取り込み時の モディファイタイプ(A(追加),C(更新),D(削除))を指定します
 *       displayMsg         【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:VIEW_DISPLAY_MSG[={@og.value org.opengion.hayabusa.common.SystemData#VIEW_DISPLAY_MSG}]))
 *       overflowMsg        【TAG】読取データが最大検索数をオーバーした場合に表示するメッセージリソースIDを指定します (初期値:MSG0007[検索結果が、制限行数を超えましたので、残りはカットされました])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *  ※   sheetName          【TAG】EXCELファイルを読み込むときのシート名を設定します(初期値:指定なし)
 *  ※   sheetNos           【TAG】EXCELファイルを読み込むときのシート番号を複数設定できます(初期値:0)
 *  ※   sheetConstKeys     【TAG】EXCELファイルを読み込むときの固定値となるカラム名(CSV形式)
 *  ※   sheetConstAdrs     【TAG】EXCELファイルを読み込むときの固定値となるアドレス(行-列,行-列,・・・)
 *       nullBreakClm       【TAG】カラム列に NULL が現れた時点で読取を中止します(複数Sheetの場合は、次のSheetを読みます)。
 *       nullSkipClm        【TAG】カラム列に NULL が現れたレコードは読み飛ばします。
 *       useNumber          【TAG】行番号情報を、使用している/していない[true/false]を指定します(初期値:true)
 *       useRenderer        【TAG】読取処理でKEY:VAL形式のコードリソースから、KEYを取り出す処理を行うかどうかを指定します(初期値:USE_TABLE_READER_RENDERER[=false])
 *       adjustColumns      【TAG】読取元ファイルのデータ変換を行うカラム列をカンマ指定します
 *       checkColumns       【TAG】読取元ファイルの整合性チェックを行うカラム列をカンマ指定します
 *       nullCheck          【TAG】NULL チェックすべきカラム列をCSV形式(CVS形式)で指定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       stopZero           【TAG】読込件数が０件のとき処理を続行するかどうか[true/false]を指定します(初期値:false[続行する])
 *       mainTrans          【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true) 5.7.7.2 (2014/06/20)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:directTableInsert&gt;
 *
 * ●使用例
 *     &lt;og:directTableInsert
 *         dbid         = "ORCL"                接続データベースID(初期値:DEFAULT)
 *         separator    = ","                   ファイルの区切り文字(初期値:タブ)
 *         fileURL      = "{&#064;USER.ID}"     読み取り元ディレクトリ名
 *         filename     = "{&#064;filename}"    読み取り元ファイル名
 *         encode       = "Shift_JIS"           読み取り元ファイルエンコード名
 *         displayMsg   = "MSG0040"             登録完了後のメッセージ
 *         columns      = "CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG"
 *                                              #NAME の代わりに使用するカラム列名
 *         commitBatch  = "100"                 この件数ずつコミットを発行(初期値:無制限)
 *         useColumnCheck  = "true"             カラムチェックを行うかどうか(初期値:false)
 *         useColumnAdjust = "true"             カラム変換を行うかどうか(初期値:false)
 *         nullCheck       = "CLM,SYSTEM_ID"    NULLチェックを実行します。
 *     &gt;
 *          INSERT INTO GE41
 *              (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *                 FGJ,DYSET,DYUPD,USRSET,USRUPD,PGUPD)
 *          VALUES
 *              ([CLM],[NAME_JA],[LABEL_NAME],[KBSAKU],[SYSTEM_ID],[LANG],
 *                '1','{&#064;USER.YMDH}','{&#064;USER.YMDH}','{&#064;USER.ID}','{&#064;USER.ID}','{&#064;GUI.KEY}')
 *     &lt;/og:directTableInsert &gt;
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectTableInsertTag extends ReadTableTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.4.0 (2015/05/15)" ;

	private static final long serialVersionUID = 624020150515L ;

	// 6.2.4.0 (2015/05/15) 無条件でOMITする名称を指定します(WRITABLE,ROWID)
	private static final String DEFAULT_OMIT = "WRITABLE,ROWID" ;

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	private String		dbid		;
	private int			commitBatch	;				// コミットするまとめ件数
//	private String[]	clmKeys		;				// SQL文の[カラム名]配列
	private String		sql			;
	private String		table		;				// 6.2.3.0 (2015/05/01)
	private long		dyStart		;				// 実行時間測定用のDIV要素を出力します。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
	 * @og.rev 6.2.3.0 (2015/05/01) table属性追加。BODYのSQL文が無くても、table属性で自動生成します。
	 * @og.rev 6.2.4.0 (2015/05/15) 無条件でOMITする名称を指定します。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();

		// 6.2.4.0 (2015/05/15) 無条件でOMITする名称を指定します。
		addOmitNames( DEFAULT_OMIT );

		// 6.2.3.0 (2015/05/01) table属性追加
		if( table == null || table.isEmpty() ) {
			return EVAL_BODY_BUFFERED ;		// Body を評価する。( extends BodyTagSupport 時)
		}
		else {
			return SKIP_BODY ;				// Body を評価しない
		}
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) SQL文の [カラム] 対応とパーサー機能追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 6.2.3.0 (2015/05/01) table属性追加。BODYのSQL文が無くても、table属性で自動生成します。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		sql = getBodyString();
		if( sql == null || sql.isEmpty() || sql.trim().isEmpty() ) {
//			final String errMsg = "BODY 部の登録用 Insert/Update文は、必須です。";
			final String errMsg = "table属性を指定しない場合は、BODY 部の登録用 Insert/Update文は、必須です。";
			throw new HybsSystemException( errMsg );
		}

		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * #doEndTag() の後続処理を記述します。
	 * 
	 * これは、サブクラスで、DBTableModel以外の処理を行う場合に、
	 * 処理内容を分けるために用意します。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) #afterEnd() メソッド 新規作成。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	protected int afterEnd() {
		// 実行件数の表示
		// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
		if( displayMsg != null && displayMsg.length() > 0 ) {
			final String status = executeCount + getResource().getLabel( displayMsg ) ;
			jspPrint( status + BR );
		}

		// 5.7.6.2 (2014/05/16) 検索結果の件数を、"DB.COUNT" キーでリクエストにセットする。
		setRequestAttribute( "DB.COUNT" , String.valueOf( executeCount ) );

		// 5.7.6.2 (2014/05/16) 件数０件かつ stopZero = true
		if( executeCount == 0 && stopZero ) { return SKIP_PAGE; }

		// 時間測定用の DIV 要素を出力
		final long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

		// 4.0.0 (2005/01/31) セキュリティチェック(データアクセス件数登録)
		final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addWriteCount( executeCount,dyTime,sql ); }

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) useColumnCheck,useColumnAdjust 属性追加
	 * @og.rev 3.8.0.2 (2005/06/30) nullCheck 属性追加
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.5.7.1 (2012/10/05) skipRowCount追加
	 * @og.rev 5.7.6.2 (2014/05/16) stopZero属性追加
	 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
	 * @og.rev 6.2.3.0 (2015/05/01) table属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		dbid			= null;
		commitBatch		= 0;					// コミットするまとめ件数
		table			= null;					// 6.2.3.0 (2015/05/01) 
	}

	/**
	 * ファイルオブジェクト より読み込み、データベースに書き込みます。
	 *
	 * @og.rev 3.6.0.2 (2004/10/04) カラムオブジェクトのDBType属性の整合性チェック
	 * @og.rev 3.8.0.2 (2005/06/30) nullチェック確認
	 * @og.rev 3.8.5.1 (2006/05/08) 取込データが name 列より少ない場合の対応を追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2005/01/31) CheckColumnDataクラス static 化、引数にResourceManager追加
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.2.2.0 (2010/11/01)) ""で囲われているデータに改行が入っていた場合の対応
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) pstmt.setObject で、useParamMetaData の判定を避けるため、pstmt.setString で代用(PostgreSQL対応)
	 * @og.rev 5.5.7.1 (2012/10/05) omitFirstLine対応
	 * @og.rev 5.7.0.3 (2013/11/22) BufferedReaderのclose処理をこのメソッド内のfinallyで行う
	 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
	 * @og.rev 6.2.3.0 (2015/05/01) 行読み飛ばし nullSkipClm追加
	 *
	 * @param   file ファイルオブジェクト
	 */
//	private int create( final BufferedReader reader )  {
	@Override
	protected void create( final File file )  {
		final String className = HybsSystem.sys( "TableReader_" + readerClass ) ;		// 4.0.0 (2005/01/31)

		// NullPointerException が発生するので、事前にチェックします。
		if( className == null ) {
			final String errMsg = "TableReader クラスが見つかりません。class=" + "TableReader_" + readerClass;
			throw new RuntimeException( errMsg );
		}

		final TableReader reader = (TableReader)HybsSystem.newInstance( className );	// 3.5.5.3 (2004/04/09)

		/**
		 * ColumnActionListenerインターフェースの内部無名クラス
		 *
		 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
		 *
		 * @param names カラム名配列
		 */
		final ColumnActionListener listener = new ColumnActionListener() {
			private DBColumn[]	dbClms ;
//			private String[]	nms ;
			private String[]	clmKeys		;				// SQL文の[カラム名]配列
			private int			commitCount ;

			int[] clmNos 	;
			int   clmNosLen ;

//			Transaction       tran   ;
			PreparedStatement pstmt  ;
			Connection        conn   ;

			/**
			 * 初期値を設定します。
			 *
			 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
			 *
			 * @param obj Object配列(可変長引数)
			 */
			public void init( final Object... obj ) {
//				pstmt = (PreparedStatement)obj[0];
//				tran  = (Transaction)obj[1];
				conn  = (Connection)obj[0];
			}

			/**
			 * 一連の作業終了時に呼ばれます。
			 *
			 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
			 *
			 * @param okFlag 正常終了時 true をセットします。
			 */
			public void end( final boolean okFlag ) {
				Closer.stmtClose( pstmt );
			}

			/**
			 * カラム名の配列が設定された場合に、呼び出されます。
			 *
			 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
			 *
			 * @param names カラム名配列
			 */
			@Override
			public void columnNames( final String[] names ) {
				final String[] nms = clmAct.makeNames( names );

				if( table != null && !table.isEmpty() ) {
					if( "FILE.NAME".equals( table ) ) {
						table = FileInfo.getNAME( file );
					}

					clmNosLen = nms.length;
					clmNos  = new int[clmNosLen];
					clmKeys = nms;

					final StringBuilder buf = new StringBuilder();
					buf.append( "INSERT INTO " ).append( table ).append( " ( " )
						.append( String.join( ",",nms ) )
						.append( " ) VALUES ( ?" );
					clmNos[0] = 0;
					for( int i=1; i<clmNosLen; i++ ) {
						clmNos[i] = i;
						buf.append( ",?" );
					}
					buf.append( ')' );
					sql = buf.toString();
				}
				else if( sql != null && !sql.isEmpty() ) {
					final ArrayDataModel nmdata = new ArrayDataModel( nms );
					final Formatter format = new Formatter( nmdata );
					format.setFormat( sql.trim() );
					clmNos		= format.getClmNos();
					clmNosLen	= clmNos.length ;
					clmKeys		= format.getClmKeys();
					sql = format.getQueryFormatString();
				}

				try {
					pstmt = conn.prepareStatement( sql );
				}
				catch( SQLException ex ) {
					final String errMsg = "prepareStatementの構築に失敗しました。" + CR
										+ " sql=[" + sql + "]"  + CR
										+ " ErrorCode=[" + ex.getErrorCode() + "]"
										+ " State=["     + ex.getSQLState()  + "]"  + CR ;
					jspPrint( errMsg );
					throw new HybsSystemException( errMsg );
				}

				dbClms = new DBColumn[nms.length];
				for( int no=0; no<nms.length; no++ ) {
					dbClms[no] = getDBColumn( nms[no] );
				}
			}

			/**
			 * １行分のデータが設定された場合に、呼び出されます。
			 *
			 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
			 *
			 * @param   vals    文字列値の１行分の配列
			 * @param   rowNo   行番号(0～)
			 */
			@Override
			public void values( final String[] vals, final int rowNo ) {
				if( maxRowCount == 0 || executeCount <= maxRowCount ) {		// 読み取り件数オーバーフロー
					try {
						String[] newVals = clmAct.clmAction( vals , dbClms , rowNo );

						for( int i=0; i<clmNosLen; i++ ) {
							pstmt.setString( i+1,newVals[clmNos[i]] );
						}
						pstmt.execute();
						if( commitBatch > 0 && ( executeCount%commitBatch == 0 ) ) {
//							tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
							conn.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
							commitCount = executeCount;
						}
						executeCount++ ;
					}
					catch( SQLException ex ) {
						final String errMsg = CR + "sql=[" + sql + "]" + CR
									+ " keys =[" + StringUtil.array2csv( clmKeys ) + "]" + CR
									+ " vals =[" + StringUtil.array2csv( vals )    + "]" + CR
									+ " 行番号=["    + (rowNo+1)         + "]"
									+ " 登録件数=["  + commitCount       + "]"  + CR
									+ " ErrorCode=[" + ex.getErrorCode() + "]"
									+ " State=["     + ex.getSQLState()  + "]"  + CR ;
						jspPrint( errMsg );
						throw new HybsSystemException( errMsg,ex );
					}
				}
			}
		};

//		reader.setResourceManager( getResource() );		// 4.0.0 (2005/01/31)
		reader.setSeparator( separator );
//		reader.setEncode( encode );						// 3.5.4.5 (2004/01/23)
		reader.setColumns( columns );					// 3.5.4.5 (2004/01/23) 、6.2.0.0 (2015/02/27) 削除
//		reader.setOmitNames( omitNames );				// 6.1.0.0 (2014/12/26)
		reader.setUseNumber( useNumber );				// 3.7.0.5 (2005/04/11)
//		reader.setMaxRowCount( maxRowCount );
		reader.setSkipRowCount( skipRowCount );			// 5.1.6.0 (2010/05/01)
//		reader.setUseRenderer( useRenderer );			// 5.2.1.0 (2010/10/01)
//		reader.setDBTableModel( table );				// 6.2.1.0 (2015/03/13)
		reader.setColumnActionListener( listener );		// 6.2.2.0 (2015/03/27)
		reader.setDebug( isDebug() );					// 5.5.7.2 (2012/10/09) デバッグ情報を出力するかどうかを指定
		// 6.2.0.0 (2015/02/27) EXCELでない場合でも、メソッドは呼び出す。(空振りします)
//		if( reader.isExcel() ) {						// 3.5.4.3 (2004/01/05)
//			reader.setFilename( HybsSystem.url2dir( StringUtil.urlAppend( fileURL,filename ))); // 6.2.0.0 (2015/02/27) 削除
			reader.setSheetName( sheetName );			// 3.5.4.2 (2003/12/15)
			reader.setSheetNos( sheetNos );				// 5.5.7.2 (2012/10/09) 複数シートを指定できるようにシート番号を指定できるようにする。
			reader.setSheetConstData( sheetConstKeys,sheetConstAdrs ) ;		// 5.5.8.2 (2012/11/09) 固定値となるカラム名、アドレスの指定
			reader.setNullBreakClm( nullBreakClm ) ;	// 5.5.8.2 (2012/11/09) 取込み条件/Sheet BREAK条件
			reader.setNullSkipClm( nullSkipClm ) ;		// 6.2.3.0 (2015/05/01) 行読み飛ばし
//			reader.readDBTable();
//			final String directory = HybsSystem.url2dir( fileURL );
//			final File file = new File( directory,filename );
	//		reader.readDBTable( file,encode );			// 6.2.0.0 (2015/02/27) 追加
//		}
//		else {
//			reader.readDBTable( out );
//		}
//		table = reader.getDBTableModel();				// 6.2.1.0 (2015/03/13) 先に作成しておく。

		boolean errFlag  = true;

		Transaction       tran  = null ;	// 5.1.9.0 (2010/08/01) Transaction 対応
	//	PreparedStatement pstmt = null ;
		try {
			// 5.1.9.0 (2010/08/01) Transaction 対応
			final TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
			if( tranTag == null ) {
				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
			}
			else {
				tran = tranTag.getTransaction();
			}

			final Connection conn = tran.getConnection( dbid );		// 5.1.9.0 (2010/08/01) Transaction 対応

//			pstmt = conn.prepareStatement( sql );

//			listener.init( pstmt,tran );
			listener.init( conn );
			reader.readDBTable( file,encode );						// 6.2.0.0 (2015/02/27) 追加

			tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
			errFlag = false;		// エラーではない
		}
//		catch (SQLException ex) {
//			final String errMsg = "sql=[" + sql + "]" + CR
//				//		+	"names=[" + StringUtil.array2csv( clmAct.getNames() ) + "]" + CR
//				//		+	"vals =[" + StringUtil.array2csv( clmAct.getVals()  ) + "]" + CR
//						+ " 行番号=[" + executeCount +"]"
//				//		+ " 登録件数=[" + commitCount + "]"  + CR
//						+ " errorCode=[" + ex.getErrorCode() + "] State=[" +
//						ex.getSQLState() + "]" + CR ;
//			throw new HybsSystemException( errMsg,ex );
//		}
		finally {
//			Closer.stmtClose( pstmt );
			listener.end( !errFlag );			// 6.2.2.0 (2015/03/27)
			if( tran != null ) {				// 5.5.2.6 (2012/05/25) findbugs対応
				tran.close( errFlag );			// 5.1.9.0 (2010/08/01) Transaction 対応
			}
		}
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】指定数毎にコミットを発行します(初期値:0 終了までコミットしません)。
	 *
	 * @og.tag
	 * 通常は、全ての処理が正常に終了するか、なにもしないか(トランザクション)
	 * を判断すべきで、途中でのコミットはしません。
	 * しかし、場合によって、件数が異常に多い場合や、再実行可能な場合は、
	 * 途中でコミットして、都度、処理できるものだけを処理してしまうという方法があります。
	 * また、ロールバックエリアの関係などで、データ量が多い場合に、処理時間が異常に
	 * 長くなる事があり、指定件数ごとのコミット機能を用意しています。
	 * 0 に設定すると、終了までコミットしません。初期値は、0 です。
	 *
	 * @param   cmtBat コミットを発行する行数 (初期値:0)
	 */
	public void setCommitBatch( final String cmtBat ) {
		commitBatch = nval( getRequestParameter( cmtBat ),commitBatch );
	}

	/**
	 * 【TAG】BODYのSQL文を指定しない場合に使用するテーブルIDを指定します。
	 *
	 * @og.tag
	 * 通常は、BODYに記述したSQL文を実行しますが、テーブルIDを指定すると、
	 * INSERT用のSQL文を自動作成します。
	 * その場合は、BODYのSQL文は設定不要です。
	 * また、FILE.NAME という文字列を指定した場合は、file1 に指定した
	 * ファイル名から、拡張子を取り除いた名称をテーブル名として使用します。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) table属性追加
	 *
	 * @param   tbl テーブルID
	 */
	public void setTable( final String tbl ) {
		table = nval( getRequestParameter( tbl ),table );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
	//			.println( "clmKeys"			,clmKeys		)
				.println( "sql"				,sql			)
				.println( "commitBatch"		,commitBatch	)
				.println( "dyStart"			,dyStart		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
