/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.resource.LDAPSearch;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.util.HybsConst.BR;		// 6.1.0.0 (2014/12/26) refactoring

import java.util.List;
import java.util.Locale;

/**
 * LDAPの内容を検索するための、ldapQueryタグです。
 *
 * 検索した結果は、DBTableModel にセットされるため、
 * JDBCQuery と同様に、viewタグで表示させることが可能です。
 *
 * 下記の項目については、src/resource/システムパラメータ に、予め
 * 設定しておくことで、タグごとに指定する必要がなくなります。
 * ・LDAP_INITIAL_CONTEXT_FACTORY
 * ・LDAP_PROVIDER_URL
 * ・LDAP_ENTRYDN
 * ・LDAP_SEARCH_BASE
 * ・LDAP_SEARCH_SCOPE
 *
 * このタグリブでは、検索のみサポートしています。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:ldapQuery  debug=&quot;false&quot;
 *         command     = &quot;NEW&quot;                                       NEW,RENEW が使用可能です。
 *         initctx     = &quot;com.sun.jndi.ldap.LdapCtxFactory&quot;          初期コンテキストファクトリ
 *         providerURL = &quot;ldap://ldap.opengion.org:389&quot;              サービスプロバイダの構成情報
 *         entrydn     = &quot;cn=inquiry-sys,o=opengion,c=JP&quot;            属性の取得元のオブジェクトの名前
 *         password    = &quot;xxxxxx&quot;                                    属性の取得元のオブジェクトのパスワード
 *         searchbase  = &quot;soouid=employeeuser,o=opengion,c=JP&quot;       検索するコンテキストまたはオブジェクトのベース名
 *         filter      = &quot;(objectClass=person)&quot;                      検索に使用するフィルタ式。null は不可
 *         attributes  = &quot;uid,cn,officeName,ou,mail,belongOUID&quot;      属性を識別する属性 IDをCSV形式で指定
 *         searchScope = &quot;SUBTREE&quot;                                   検索範囲を設定(『OBJECT』『ONELEVEL』『SUBTREE』のどれか)
 *         maxRowCount = &quot;0&quot;                                         検索時の最大検索数(0は、無制限：デフォルト)
 *     /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:ldapQuery
 *       initctx            【TAG】初期コンテキストファクトリを指定します (初期値:LDAP_INITIAL_CONTEXT_FACTORY[=com.sun.jndi.ldap.LdapCtxFactory])
 *       providerURL        【TAG】サービスプロバイダの構成情報を指定します (初期値:LDAP_PROVIDER_URL[=ldap://ldap.opengion.org:389])
 *       entrydn            【TAG】属性の取得元のオブジェクトの名前を設定します (初期値:LDAP_ENTRYDN[=cn=inquiry-sys,o=opengion,c=JP])
 *       password           【TAG】属性の取得元のオブジェクトのパスワードを設定します (初期値:LDAP_PASSWORD[=password])
 *       searchbase         【TAG】検索するコンテキストまたはオブジェクトの名前を設定します (初期値:LDAP_SEARCH_BASE[=soouid=employeeuser,o=opengion,c=JP])
 *       filter           ○【TAG】検索に使用するフィルタ式(例：belongOUID=61200)(必須)。
 *       searchScope        【TAG】検索範囲(OBJECT/ONELEVEL/SUBTREE)を設定します (初期値:LDAP_SEARCH_SCOPE[=SUBTREE])
 *       timeLimit          【TAG】これらの SearchControls の時間制限をミリ秒単位で設定します(初期値:0[無制限])
 *       derefLinkFlag      【TAG】検索中のリンクへの間接参照を有効または無効[true/false]にします(初期値:false)
 *       maxRowCount        【TAG】レジストリの最大検索件数をセットします(初期値:0[無制限])
 *       returningObjFlag   【TAG】結果の一部としてオブジェクトを返すことを有効または無効[true/false]にします(初期値:false)
 *       attributes         【TAG】検索の一部として返される属性をCSVで複数指定します
 *       columns            【TAG】作成する DBTableModel のカラム名をCSV形式で指定します
 *       orderBy            【TAG】検索した結果を表示する表示順をファイル属性名で指定します
 *       tableId            【TAG】(通常は使いません)結果をDBTableModelに書き込んで、sessionに登録するときのキーを指定します
 *       command            【TAG】コマンド (NEW,RENEW)をセットします(初期値:NEW)
 *       displayMsg         【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します(初期値:VIEW_DISPLAY_MSG[={@og.value org.opengion.hayabusa.common.SystemData#VIEW_DISPLAY_MSG}])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *       overflowMsg        【TAG】オーバーフロー時に画面上に表示するメッセージIDを指定します(初期値:MSG0007)
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       mainTrans          【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     システムパラメータ に、必要なLDAP定数を定義しておいた場合は、下記のタグで検索できます。
 *     例では、filter にあいまい検索(*)とAND条件(&amp;(条件A)(条件B)) を指定した例を挙げておきます。
 *     filter の形式と解釈は、RFC 2254 に準拠しています。
 *     (RFC 2254 で定義されている演算子のすべてをすべての属性に適用できるわけではありません。)
 *
 *      &lt;og:ldapQuery  debug=&quot;false&quot;
 *          command     = &quot;NEW&quot;
 *          filter      = &quot;(&amp;(cnphoneticlangjp=OPENGION*)(businessName=*COMMON*))&quot;
 *          attributes  = &quot;uid,cn,officeName,ou,mail,belongOUID&quot;
 *      /&gt;
 *
 *    ・queryOption タグの内部に記述すれば、プルダウンメニューを作成することが可能です。
 *    &lt;og:select name="UID"&gt;
 *        &lt;og:queryOption&gt;
 *            &lt;og:ldapQuery
 *                filter      = "(&amp;amp;(|(belongOUID=60000)(belongOUID=70000))(uid=C*))"
 *                attributes  = "uid,uid,sn"
 *                orderBy     = "uid"
 *            /&gt;
 *        &lt;/og:queryOption&gt;
 *    &lt;/og:select&gt;
 *
 * @og.rev 3.1.0.0 (2003/03/20) ＬＤＡＰにアクセスできる、LDAPQueryTag.java を新規に作成。
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class LDAPQueryTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.6.0 (2010/05/01)" ;

	private static final long serialVersionUID = 516020100501L ;

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = new String[] { CMD_NEW , CMD_RENEW };

	private static final long	COUNTLIMIT		= 0;	// 返すエントリの最大数。0 の場合、フィルタを満たすエントリをすべて返す

	private String			initctx 			= HybsSystem.sys( "LDAP_INITIAL_CONTEXT_FACTORY" );
	private String			providerURL 		= HybsSystem.sys( "LDAP_PROVIDER_URL" );
	private String			entrydn 			= HybsSystem.sys( "LDAP_ENTRYDN" );
	private String			password 			= HybsSystem.sys( "LDAP_PASSWORD" );		// 4.2.2.0 (2008/05/10)
	private String			searchbase			= HybsSystem.sys( "LDAP_SEARCH_BASE" );
	private String			filter 				;			// "employeeNumber=87019";

	// 検索範囲。OBJECT_SCOPE、ONELEVEL_SCOPE、SUBTREE_SCOPE のどれか 1 つ
	private String			searchScope			= HybsSystem.sys( "LDAP_SEARCH_SCOPE" );
	private int				timeLimit			;			// 結果が返されるまでのミリ秒数。0 の場合、無制限
	private String[]		attrs				;			// エントリと一緒に返される属性の識別子。null の場合、すべての属性を返す。空の場合、属性を返さない
	private String[]		columns				;			// DBTableModel を作成する場合のカラム名になります。null の場合は、attributes 属性
	private boolean			returningObjFlag	;			// true の場合、エントリの名前にバインドされたオブジェクトを返す。false 場合、オブジェクトを返さない
	private boolean			derefLinkFlag		;			// true の場合、検索中にリンクを間接参照する

	private String			tableId 			= HybsSystem.TBL_MDL_KEY;	// sessionｷｰ
	private String			orderBy				;							// ｿｰﾄ項目(csv)
	private String			command				= CMD_NEW;					// ｺﾏﾝﾄﾞ
	private int				executeCount		;							// 検索/実行件数
	private int				errCode			 	= ErrorMessage.OK;			// Ver4.3.1.1 (2008/09/05) 検索エラーコード
	private int 		 	maxRowCount			;							// 最大検索数(0は無制限)
	private String			displayMsg			= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String			overflowMsg  		= "MSG0007";	// 検索結果が、制限行数を超えましたので、残りはカットされました。
	private String			notfoundMsg			= "MSG0077";	// 対象データはありませんでした。
	private boolean			isMainTrans			= true;			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 4.3.1.1 (2008/09/03) 検索結果の件数、状況コードをリクエスト変数 "DB.COUNT"、"DB.ERR_CODE" にセットする。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// QueryOptionTag の内部で記述された場合の処理。
		final QueryOptionTag queryOption = (QueryOptionTag)findAncestorWithClass( this,QueryOptionTag.class );
		if( queryOption != null ) {
			queryOption.setTableModel( makeDBTable() );
			return EVAL_PAGE ;
		}

		// 通常の LDAPQuery 処理
		if( check( command, COMMAND_LIST ) ) {
			useMainTrans( isMainTrans );			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
			startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)

			final DBTableModel table = makeDBTable();

			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "LDAPQueryTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return SKIP_PAGE ;
			}
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

			// 実行件数の表示 command="NEW" のときのみ、displayMsg を表示させます。
			// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
			if( CMD_NEW.equals( command ) ) {
				if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
					buf.append( executeCount );
					buf.append( getResource().getLabel( displayMsg ) );
					buf.append( BR );
				}
				else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
					buf.append( getResource().getLabel( notfoundMsg ) );
					buf.append( BR );
				}
			}

			if( maxRowCount > 0 && maxRowCount <= executeCount ) {
				buf.append( getResource().getLabel( overflowMsg ) );
				buf.append( BR ).append( CR );
			}
			// 4.3.1.1 (2008/09/03) 検索結果の件数、エラーコードをセットする。
			setRequestAttribute( "DB.COUNT" , String.valueOf( executeCount ) );
			setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

			jspPrint( buf.toString() );
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 4.2.2.0 (2008/05/10) password 追加
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();

		searchScope			= HybsSystem.sys( "LDAP_SEARCH_SCOPE" );	// 検索範囲。OBJECT_SCOPE、ONELEVEL_SCOPE、SUBTREE_SCOPE のどれか 1 つ
		timeLimit			= 0;			// 結果が返されるまでのミリ秒数。0 の場合、無制限
		derefLinkFlag		= false;		// true の場合、検索中にリンクを間接参照する
		returningObjFlag	= false;		// true の場合、エントリの名前にバインドされたオブジェクトを返す。false 場合、オブジェクトを返さない
		attrs				= null;			// エントリと一緒に返される属性の識別子。null の場合、すべての属性を返す。空の場合、属性を返さない
		columns				= null;			// DBTableModel を作成する場合のカラム名になります。null の場合は、attributes 属性

		initctx 			= HybsSystem.sys( "LDAP_INITIAL_CONTEXT_FACTORY" );
		providerURL 		= HybsSystem.sys( "LDAP_PROVIDER_URL" );
		entrydn 			= HybsSystem.sys( "LDAP_ENTRYDN" );
		password 			= HybsSystem.sys( "LDAP_PASSWORD" );		// 4.2.2.0 (2008/05/10)
		searchbase			= HybsSystem.sys( "LDAP_SEARCH_BASE" );
		filter 				= null;		// "employeeNumber=87019";

		tableId 			= HybsSystem.TBL_MDL_KEY;	// sessionｷｰ
		orderBy				= null;			// ｿｰﾄ項目(csv)
		command				= CMD_NEW;		// ｺﾏﾝﾄﾞ
		executeCount		= 0;			// 検索/実行件数
		errCode			 	= ErrorMessage.OK; // Ver4.3.1.1 (2008/09/05) 検索エラーコード
		maxRowCount			= 0;			// 最大検索数(0は無制限)
		displayMsg			= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		overflowMsg  		= "MSG0007";	// 検索結果が、制限行数を超えましたので、残りはカットされました。		*/
		notfoundMsg			= "MSG0077";	// 対象データはありませんでした。
		isMainTrans			= true;			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
	}

	/**
	 * LDPA から、値を取り出し、DBTableModel を作成します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) entrydn と、password 処理の追加
	 *
	 * @return	テーブルモデル
	 */
	private DBTableModel makeDBTable() {
		if( filter == null ) {
			final String errMsg = "S検索フィルターに null が指定されました。" ;
			throw new HybsSystemException( errMsg );
		}

		final LDAPSearch serch = new LDAPSearch();
		serch.setSearchScope( searchScope ) ;
		serch.setTimeLimit( timeLimit ) ;
		serch.setDerefLinkFlag( derefLinkFlag ) ;
		serch.setReturningObjFlag( returningObjFlag ) ;
		serch.setMaxRowCount( maxRowCount ) ;
		serch.setAttributes( attrs ) ;
		serch.setInitctx( initctx ) ;
		serch.setProviderURL( providerURL ) ;
		serch.setSearchbase( searchbase ) ;
		if( entrydn != null  ) { serch.setEntrydn( entrydn ) ; }	//4 .2.2.0 (2008/05/10)
		if( password != null ) { serch.setPassword( password ) ; }	//4 .2.2.0 (2008/05/10)
		serch.setOrderBy( orderBy ) ;
		serch.init();

		final List<String[]> list = serch.search( filter );
		executeCount = list.size();

		String[] ids = serch.getAttributes();	// attrs が null の場合、全件。それ以外は、attrs

		// DBTableModel に使用するカラム名の登録
		if( columns != null && columns.length > 0 ) {
			if( ids.length != columns.length ) {
				final String errMsg = "attributes 属性とcolumns属性の個数が異なります。"
							+ "attrs=[" + attrs.length + "]:" + StringUtil.array2csv( attrs )
							+ CR
							+ " columns=[" + columns.length + "]:" + StringUtil.array2csv( columns )
							+ CR ;
				throw new HybsSystemException( errMsg );
			}
			for( int i=0; i<columns.length; i++ ) {
				// columns にカラム名が設定されている場合のみ、使用する。
				if( columns[i] != null && columns[i].length() > 0 ) {
					ids[i] = columns[i];
				}
			}
		}

		DBTableModel table = null;
		if( ids.length > 0 ) {
			table = DBTableModelUtil.newDBTable();
			table.init( ids.length );
			for( int clm=0; clm<ids.length; clm++ ) {
				final DBColumn dbColumn = getResource().makeDBColumn( ids[clm] );
				table.setDBColumn( clm,dbColumn );
			}

			for( int row=0; row<executeCount; row++ ) {
				table.addColumnValues( list.get( row ) );
			}
		}
		return table ;
	}

	/**
	 * 【TAG】検索範囲(OBJECT/ONELEVEL/SUBTREE)を設定します
	 *		(初期値:LDAP_SEARCH_SCOPE[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_SEARCH_SCOPE}])。
	 *
	 * @og.tag
	 * 検索範囲を OBJECT_SCOPE、ONELEVEL_SCOPE、SUBTREE_SCOPE のどれか 1 つです。
	 * 指定文字列は、それぞれ『OBJECT』『ONELEVEL』『SUBTREE』です。
	 * (初期値:システム定数のLDAP_SEARCH_SCOPE[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_SEARCH_SCOPE}])。
	 *
	 * @param	scope SearchControlsの検索範囲 [OBJECT/ONELEVEL/SUBTREE]
	 * @see		org.opengion.hayabusa.common.SystemData#LDAP_SEARCH_SCOPE
	 */
	public void setSearchScope( final String scope ) {
		searchScope = nval( getRequestParameter( scope ), searchScope );
	}

	/**
	 * 【TAG】これらの SearchControls の時間制限をミリ秒単位で設定します(初期値:0[無制限])。
	 *
	 * @og.tag
	 * 値が 0 の場合、無制限に待つことを意味します。
	 *
	 * @param	limit ミリ秒単位の時間制限(初期値:無制限)
	 */
	public void setTimeLimit( final String limit ) {
		timeLimit = nval( getRequestParameter( limit ), timeLimit );
	}

	/**
	 * 【TAG】検索中のリンクへの間接参照を有効または無効[true/false]にします(初期値:false)。
	 *
	 * @og.tag 検索中のリンクへの間接参照を有効または無効にします。
	 * true:有効で、リンクを間接参照可能になります。
	 * 初期値は、false:無効です。
	 *
	 * @param	deref 間接参照 [true:有効/false:無効]
	 */
	public void setDerefLinkFlag( final String deref ) {
		derefLinkFlag = nval( getRequestParameter( deref ), derefLinkFlag );
	}

	/**
	 * 【TAG】結果の一部としてオブジェクトを返す[true:有効/false:無効]かどうか設定します(初期値:false)。
	 *
	 * @og.tag
	 * true:有効にした場合、オブジェクトが返されます。
	 * false:無効にした場合、オブジェクトの名前およびクラスだけが返されます。
	 * 初期値は、false:無効 です。
	 *
	 * @param	pbjflag オブジェクト返し [true:有効/false:無効]
	 */
	public void setReturningObjFlag( final String pbjflag ) {
		returningObjFlag = nval( getRequestParameter( pbjflag ), returningObjFlag );
	}

	/**
	 * 【TAG】LDAPデータの最大検索件数をセットします(初期値:0[無制限])。
	 *
	 * @og.tag
	 * DBTableModelのデータとして登録する最大検索件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 * 初期値は、0 は、無制限です。
	 *
	 * @param	count 最大件数
	 */
	public void setMaxRowCount( final String count ) {
		maxRowCount = nval( getRequestParameter( count ),maxRowCount );
	}

	/**
	 * 【TAG】検索の一部として返される属性をCSVで複数指定します。
	 *
	 * @og.tag
	 * null は属性が何も返されないことを示します。
	 * このメソッドからは、空の配列をセットすることは出来ません。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.7.0.4 (2005/03/18) リクエスト変数の値を取得後、分解します。
	 *
	 * @param	val 返される属性を識別する属性ID(CSV形式)
	 */
	public void setAttributes( final String val ) {
		final String att = nval( getRequestParameter( val ),null );
		if( att != null ) {
			attrs = StringUtil.csv2Array( att );
		}
	}

	/**
	 * 【TAG】作成する DBTableModel のカラム名をCSV形式で指定します。
	 *
	 * @og.tag
	 * なにも指定しない場合は、attributes 属性が使用されます。
	 * LDAP検索結果に、別名をつけるイメージで使用します。
	 *
	 * @og.rev 3.7.1.0 (2005/04/26) 新規追加
	 *
	 * @param	clm カラム名(CSV形式)
	 */
	public void setColumns( final String clm ) {
		final String cl = nval( getRequestParameter( clm ),null );
		if( cl != null ) {
			columns = StringUtil.csv2Array( cl );
		}
	}

	/**
	 * 【TAG】初期コンテキストファクトリを指定します
	 *		(初期値:LDAP_INITIAL_CONTEXT_FACTORY[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_INITIAL_CONTEXT_FACTORY}])。
	 *
	 * @og.tag
	 * 例)com.sun.jndi.ldap.LdapCtxFactory
	 * (初期値:システム定数のLDAP_INITIAL_CONTEXT_FACTORY[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_INITIAL_CONTEXT_FACTORY}])。
	 *
	 * @param	ctx 初期コンテキストファクトリ
	 * @see		org.opengion.hayabusa.common.SystemData#LDAP_INITIAL_CONTEXT_FACTORY
	 */
	public void setInitctx( final String ctx ) {
		initctx = nval( getRequestParameter( ctx ), initctx );
	}

	/**
	 * 【TAG】サービスプロバイダの構成情報を指定します
	 *		(初期値:LDAP_PROVIDER_URL[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_PROVIDER_URL}])。
	 *
	 * @og.tag
	 * プロトコルとサーバーとポートを指定します。
	 * 例)『ldap://ldap.opengion.org:389』
	 * (初期値:システム定数のLDAP_PROVIDER_URL[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_PROVIDER_URL}])。
	 *
	 * @param	url サービスプロバイダの構成情報
	 * @see		org.opengion.hayabusa.common.SystemData#LDAP_PROVIDER_URL
	 */
	public void setProviderURL( final String url ) {
		providerURL = nval( getRequestParameter( url ), providerURL );
	}

	/**
	 * 【TAG】検索するコンテキストまたはオブジェクトの名前を設定します
	 *		(初期値:LDAP_SEARCH_BASE[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_SEARCH_BASE}])。
	 *
	 * @og.tag
	 * 例)『soOUID=employeeuser,o=opengion,c=JP』
	 * (初期値:システム定数のLDAP_SEARCH_BASE[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_SEARCH_BASE}])。
	 *
	 * @param	base 検索ベース
	 * @see		org.opengion.hayabusa.common.SystemData#LDAP_SEARCH_BASE
	 */
	public void setSearchbase( final String base ) {
		searchbase = nval( getRequestParameter( base ), searchbase );
	}

	/**
	 * 【TAG】検索に使用するフィルタ式(例：belongOUID=61200)。
	 *
	 * @og.tag
	 * 例)『belongOUID=61200』
	 *
	 * @param	siki フィルタ式
	 */
	public void setFilter( final String siki ) {
		filter = nval( getRequestParameter( siki ), filter );
	}

	/**
	 * 【TAG】属性の取得元のオブジェクトの名前を設定します
	 *		(初期値:LDAP_ENTRYDN[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_ENTRYDN}])。
	 *
	 * @og.tag
	 * 例)『soOUID=opengion-sys,o=opengion,c=JP』
	 * (初期値:システム定数のLDAP_ENTRYDN[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_ENTRYDN}])。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 初期値変更
	 *
	 * @param	dn 取得元のオブジェクトの名前
	 * @see		org.opengion.hayabusa.common.SystemData#LDAP_ENTRYDN
	 */
	public void setEntrydn( final String dn ) {
		entrydn = nval( getRequestParameter( dn ), entrydn );
	}

	/**
	 * 【TAG】属性の取得元のオブジェクトのパスワードを設定します
	 *		(初期値:LDAP_PASSWORD[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_PASSWORD}])。
	 *
	 * @og.tag
	 * (初期値:システム定数のLDAP_PASSWORD[={@og.value org.opengion.hayabusa.common.SystemData#LDAP_PASSWORD}])。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 新規追加
	 *
	 * @param	pwd パスワード
	 * @see		org.opengion.hayabusa.common.SystemData#LDAP_PASSWORD
	 */
	public void setPassword( final String pwd ) {
		password = nval( getRequestParameter( pwd ), password );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】検索した結果を表示する表示順をファイル属性名で指定します。
	 *
	 * @og.tag
	 * attributes 属性で指定するキー、または、LDAPから返されたキーについて
	 * その属性でソートします。逆順を行う場合は、DESC を指定のカラム名の後ろに
	 * 付けて下さい。
	 *
	 * @param	ordr	ソートキー
	 */
	public void setOrderBy( final String ordr ) {
		orderBy = nval( getRequestParameter( ordr ), orderBy );
		if( orderBy != null ) { orderBy = orderBy.toUpperCase(Locale.JAPAN) ; }
	}

	/**
	 * 【TAG】コマンド (NEW,RENEW)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"NEW" が初期値にセットされます。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.LDAPQueryTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します
	 *		(初期値:VIEW_DISPLAY_MSG[={@og.value org.opengion.hayabusa.common.SystemData#VIEW_DISPLAY_MSG}])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 件数を表示させる場合は、displayMsg = "MSG0033"[　件検索しました] をセットしてください。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * (初期値:システム定数のVIEW_DISPLAY_MSG[={@og.value org.opengion.hayabusa.common.SystemData#VIEW_DISPLAY_MSG}])。
	 *
	 * @param	id 処理結果表示メッセージID
	 */
	public void setDisplayMsg( final String id ) {
		displayMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ゼロ件時表示メッセージID
	 */
	public void setNotfoundMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】検索データが最大検索数をオーバーした場合に表示するメッセージリソースIDを指定します
	 *		(初期値:MSG0007[検索結果が、制限行数を超えましたので、残りはカットされました])。
	 *
	 * @og.tag
	 * 表示させたくない場合は, overflowMsg = "" をセットしてください。
	 *
	 * @param	id 最大検索数オーバー時メッセージID
	 */
	public void setOverflowMsg( final String id ) {
		overflowMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 *
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクションかどうか [true:メイン/false:その他]
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ),isMainTrans );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"				,VERSION		)
				.println( "initctx" 			,initctx 		)
				.println( "providerURL" 		,providerURL 	)
				.println( "entrydn" 			,entrydn 		)
				.println( "searchbase"			,searchbase		)
				.println( "filter" 				,filter 		)
				.println( "searchScope"			,searchScope	)
				.println( "COUNTLIMIT"			,COUNTLIMIT		)
				.println( "timeLimit"			,timeLimit		)
				.println( "attrs"				,attrs			)
				.println( "columns"				,columns		)
				.println( "returningObjFlag"	,returningObjFlag)
				.println( "derefLinkFlag"		,derefLinkFlag	)
				.println( "tableId" 			,tableId 		)
				.println( "orderBy"				,orderBy		)
				.println( "command"				,command		)
				.println( "executeCount"		,executeCount	)
				.println( "errCode"				,errCode		)
				.println( "maxRowCount"			,maxRowCount	)
				.println( "displayMsg"   		,displayMsg   	)
				.println( "overflowMsg"  		,overflowMsg  	)
				.println( "Other..."			,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
