/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.db.DBEditConfig;					// 6.2.0.1 (2015/03/06)
// import org.opengion.fukurou.util.Attributes;					// 6.1.1.0 (2015/01/17) TagBuffer でよい。
import org.opengion.fukurou.util.TagBuffer;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.XHTMLTag ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.http.HttpServletRequest ;

/**
 * サブミットボタンの表示や、画面遷移先情報(forward.jsp)を制御するタグです。
 *
 * ユーザー情報と画面情報を元に、ロール/グループの権限と
 * RW(読取/書込)モードを比較して、条件が成立しない場合は,そのBODY部を表示しないように
 * できます。
 * また、編集設定を行っている場合は、強制的に書き込み禁止に設定しています。
 * (ただし、編集設定が、カラムの並べ替えのみの場合か、または、useEdit="true" に
 * 設定した場合は、書き込み許可されます)
 * このタグは、form 内に一つだけ記述してください。
 *
 * @og.formSample
 * ●形式：&lt;og:writeCheck /&gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:writeCheck
 *       useGamenId         【TAG】GAMENID,BACK_GAMENID 属性を付加するかどうか[true/false]を設定します(初期値:true)
 *       roles              【TAG】画面と別にロールズを与えます
 *       writable           【TAG】書き込み可能かどうか[true/false]を設定します(初期値:true)
 *       useEdit            【TAG】EDITNAME利用時の編集を可能にするか設定します
 *                                   (初期値:USE_EDIT_WITH_EDITNAME[={@og.value org.opengion.hayabusa.common.SystemData#USE_EDIT_WITH_EDITNAME}])。
 *       outputEditName     【TAG】editNameをhiddenタグで出力するか設定します
 *                                   (初期値:OUTPUT_EDITNAME[={@og.value org.opengion.hayabusa.common.SystemData#OUTPUT_EDITNAME}])。
 *       useDummy           【TAG】IE用ダミーinputを出力するかどうか
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:writeCheck&gt;
 *
 * ●使用例
 *      &lt;og:writeCheck&gt;
 *          &lt;og:submit  value=&quot;copy&quot;     lbl=&quot;COPY&quot;    accesskey=&quot;C&quot; /&gt;
 *          &lt;og:submit  value=&quot;modify&quot;   lbl=&quot;MODIFY&quot;  accesskey=&quot;M&quot; /&gt;
 *          &lt;og:submit  value=&quot;delete&quot;   lbl=&quot;DELETE&quot;  accesskey=&quot;D&quot; /&gt;
 *          &lt;jsp:directive.include file=&quot;{&#064;SYS.JSP}/common/Excel_direct.jsp&quot;  /&gt;
 *          &lt;br /&gt;
 *      &lt;/og:writeCheck&gt;
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class WriteCheckTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.4.0 (2015/05/15)" ;

	private static final long serialVersionUID = 624020150515L ;

	// 6.1.1.0 (2015/01/17) DUMMY_INPUT_TAG は、システム共通で固定値でよい。
	// 5.6.1.0 (2013/02/01) display:noneのダミー出力(IEのEnterボタンのための、隠しinput出力)
//	private static final String DUMMY_INPUT	= "HYBS_DUMMY_NPUT";	// 5.6.1.0 (2013/02/01)
	private static final String DUMMY_INPUT_TAG =
		new TagBuffer( "input" )
			.add( "name"	, "HYBS_DUMMY_NPUT" )
			.add( "style"	, "display:none;" )
			.makeTag() ;

	// 3.5.4.0 (2003/11/25) BACK_GAMENID を使用する/しないを指定します。
	private boolean		useGamenId	= true;
	// 3.5.4.0 (2003/11/25) 画面と別にロールズを与えられます。
	private String 		roles		;

	// 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	private boolean		writable 	= true;

	// 5.5.4.2 (2012/06/13) EDITNAME利用時の編集
	private boolean useEdit			= HybsSystem.sysBool( "USE_EDIT_WITH_EDITNAME" );
	private boolean outputEditName	= HybsSystem.sysBool( "OUTPUT_EDITNAME" );			// EditNameをhidden出力するか

	// 5.6.1.0 (2013/02/01) display:noneのダミー出力
	private boolean useDummy		= HybsSystem.sysBool( "USE_DUMMY_INPUT" );

	// 6.2.4.0 (2015/05/15)
	private String logMsg			;		// 途中の状況表示用(ログメッセージ)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.0.1.2 (2003/03/07) forward.jsp の代替用 CommonForwardTag を新規作成
	 * @og.rev 3.5.4.0 (2003/11/25) useGamenId で、BACK_GAMENID を使用する/しないを指定します。
	 * @og.rev 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	 * @og.rev 5.6.1.0 (2013/02/01) display:noneのdummyインプットを出力する
	 * @og.rev 5.7.8.0 (2014/07/04) caseKey,caseVal,caseNN,caseNull 属性を追加
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		if( writable && isWrite() && useTag() ) {			// 5.7.8.0 (2014/07/04) 追加
//		if( writable && isWrite() ) {
//			if( useDummy ){ jspPrint( getDummyInput() ); }	// 5.6.1.0 (2013/02/01)
			if( useDummy ){ jspPrint( DUMMY_INPUT_TAG ); }	// 6.1.1.0 (2015/01/17) 固定値でよい
			if( useGamenId ) { jspPrint( getHiddenTag() ); }
			return EVAL_BODY_INCLUDE ;
		}
//		else {
			return SKIP_BODY ;
//		}
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規追加。
	 * @og.rev 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	 * @og.rev 5.5.4.2 (2012/07/13) EDITNAME利用時の編集
	 * @og.rev 5.6.1.0 (2013/02/01) useDummy 追加
	 * @og.rev 6.2.4.0 (2015/05/15) logMsg 追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		useGamenId	= true;
		roles		= null;
		writable 	= true;													// 3.8.5.1 (2006/05/08)
		useEdit			= HybsSystem.sysBool( "USE_EDIT_WITH_EDITNAME" );	// 5.5.4.2 (2012/06/13) 
		outputEditName	= HybsSystem.sysBool( "OUTPUT_EDITNAME" );			// 5.5.4.2 (2012/06/13) 
		useDummy		= HybsSystem.sysBool( "USE_DUMMY_INPUT" );			// 5.6.1.0 (2013/02/01)
		logMsg		= null ;		// 6.2.4.0 (2015/05/15) 途中の状況表示用(ログメッセージ)
	}

	/**
	 * guiInfo.isWrite() の結果をチェック[true/false]して返します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 画面と別にロールズを与えられます。
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 * @og.rev 5.3.6.0 (2011/06/01) 編集設定時もfalse(書き込み禁止)とする。
	 * @og.rev 5.5.4.2 (2012/07/13) EDITNAME利用時の登録
	 * @og.rev 6.2.0.1 (2015/03/06) EDITNAME利用時に、カラムの非表示が無ければ、書き込み許可する。
	 * @og.rev 6.2.4.0 (2015/05/15) 途中の状況表示用(ログメッセージ)
	 *
	 * @return	書き込み許可があるかどうか[true/false]
	 */
	private boolean isWrite() {
		boolean rtnFlag = false;

		// 4.0.0 (2005/11/30) 指定のロールが、アクセス許可の場合は、出力します。
		if( roles != null ) {
			rtnFlag = getUser().isAccess( roles ) ;
		}
		else {
			final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			rtnFlag = guiInfo.isWrite() ;		// 4.0.0 (2005/01/31)
		}

		// 5.3.6.0 (2011/06/01) 編集設定時もfalse(書き込み禁止)とする。
		if( rtnFlag ) {
			final String editName = getRequestValue( "editName" );
			if( editName != null && editName.length() > 0 ) {
				// rtnFlag = false;
				rtnFlag = useEdit;			// 5.5.4.2 (2012/07/13)

				// 6.2.0.1 (2015/03/06) 編集設定時に、カラムの非表示が無ければ、書き込み許可する。
				final String       gamenId = getGUIInfoAttri( "KEY" );
				final DBEditConfig config  = getUser().getEditConfig( gamenId, editName );
				if( config != null ) {
					final String viewClms = config.getViewClms();
					// カラム列があり、非表示カラムがない。
					if( viewClms != null && viewClms.indexOf( '!' ) < 0 ) {
						rtnFlag = true;		// 書き込み許可する。
					}
					// 6.2.4.0 (2015/05/15) 途中の状況表示用(ログメッセージ)
					else {
						logMsg = "gamenId=" + gamenId + " , editName=" + editName + "=" + viewClms ;
					}
				}
			}
		}

		return rtnFlag;
	}

	/**
	 * 隠し属性タグを取得します。
	 *
	 * ※内部コール用
	 *
	 * @og.rev 3.0.1.2 (2003/03/07) forward.jsp の代替用 CommonForwardTag を新規作成
	 * @og.rev 3.1.7.0 (2003/05/02) 画面ＩＤのセットで、自画面を、BACK_GAMENID 、飛び先を、GAMENID にする。
	 * @og.rev 3.1.8.0 (2003/05/16) BACK_GAMENID を、リクエスト情報ではなく、GUI.KEY を利用するように修正。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.8.8.2 (2007/01/26) hidden の出力に、GAMENID を使用します。
	 * @og.rev 5.5.4.2 (2012/07/13) editNameの出力を追加
	 *
	 * @return	隠し属性タグ
	 * @og.rtnNotNull
	 */
	private final String getHiddenTag() {

		final String requestURI = ((HttpServletRequest)getRequest()).getRequestURI();
		if( requestURI == null ) {
			return "";
		}

		final int index2 = requestURI.lastIndexOf( '/' );
		final String jspID   = requestURI.substring( index2 + 1 );
		final String gamenId = getGUIInfoAttri( "KEY" );			// 4.0.0 (2005/01/31)

		final StringBuilder strRet = new StringBuilder( BUFFER_MIDDLE )
			.append( XHTMLTag.hidden( "BACK_GAMENID",gamenId ) )
			.append( XHTMLTag.hidden( "BACK_JSPID"  ,jspID   ) )
			.append( XHTMLTag.hidden( "GAMENID"     ,gamenId ) );		// 3.8.8.2
		if( useEdit & outputEditName ){ // 5.5.4.2 (2012/07/13)
			strRet.append( XHTMLTag.hidden( "editName"     ,getRequestValue( "editName" ) ) );
		}

		return strRet.toString();
	}

//	/**
//	 * IEのEnterボタンのための、隠しinput出力。
//	 *
//	 * ※内部コール用
//	 *
//	 * @og.rev 5.6.1.0 (2013/02/01) 新規作成
//	 * @og.rev 6.1.1.0 (2015/01/17) static final String DUMMY_INPUT_TAG で定義するので、廃止
//	 *
//	 * @return	隠し属性タグ
//	 * @og.rtnNotNull
//	 */
//	private final String getDummyInput() {
//
//		final StringBuilder strRet = new StringBuilder( BUFFER_MIDDLE );
//
//		final Attributes attributes = new Attributes();
//		attributes.set("name",DUMMY_INPUT);
//		attributes.set("value",null);
//		attributes.set("optionAttributes","style='display:none;'");
//		strRet.append( XHTMLTag.input( attributes ) );
//
//		return strRet.toString();
//	}

	/**
	 * 【TAG】GAMENID,BACK_GAMENID 属性を付加するかどうか[true/false]を設定します(初期値:true)。
	 *
	 * @og.tag
	 * "true"の場合は、BACK_GAMENID 、BACK_JSPID 属性を hidden で出力します。
	 * "false"の場合は、付加しません。
	 * これは、戻るリンクを作成する場合に利用されます。
	 * 初期値は、 "true"です。
	 * ボタンで遷移先がポップアップ画面の場合に、戻るリンクを出しても戻れない為、
	 * そのようなケースでは false にします。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) useGamenId のために、新規追加。
	 *
	 * @param	flag  付加するかどうか [true:付加する/false:付加しない]
	 */
	public void setUseGamenId( final String flag ) {
		useGamenId = nval( getRequestParameter( flag ),useGamenId );
	}

	/**
	 * 【TAG】画面と別にロールズを与えます。
	 *
	 * @og.tag
	 * このロールズが与えられた場合は、画面のロールズではなく、
	 * 個別のロールズを使用して、表示の制御を行います。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規追加。
	 *
	 * @param	rols  ロールズ
	 */
	public void setRoles( final String rols ) {
		roles = nval( getRequestParameter( rols ),roles );
	}

	/**
	 * 【TAG】書き込み可能かどうか[true/false]を設定します(初期値:true)。
	 *
	 * @og.tag
	 * 書込み許可する場合は, true、許可しない場合は, true 以外を設定します。
	 * デフォルト値は、互換性のため、true です。
	 * ここで、false を設定するとBODY部の内容は表示されません。true を設定した場合、
	 * 通常通り BODY 部は表示されます。
	 * これは、roles の簡易版として、外部から ON/OFF を制御するのに使用できます。
	 *
	 * @og.rev 3.8.5.1 (2006/05/08) 書き込み許可属性 writable 追加
	 *
	 * @param	flag 書き込み許可 [true:許可/false:不可]
	 */
	public void setWritable( final String flag ) {
		writable = nval( getRequestParameter( flag ),writable );
	}

	/**
	 * 【TAG】EDITNAME利用時の編集を可能にするか設定します
	 *		(初期値:USE_EDIT_WITH_EDITNAME[={@og.value org.opengion.hayabusa.common.SystemData#USE_EDIT_WITH_EDITNAME}])。
	 *
	 * @og.tag
	 * カラムの並び替えや表示等を設定した場合に編集可能にするかを設定します。
	 * (つまり、ボタンを出すかどうかをコントロールします)
	 * (初期値:USE_EDIT_WITH_EDITNAME[={@og.value org.opengion.hayabusa.common.SystemData#USE_EDIT_WITH_EDITNAME}])。
	 *
	 * @og.rev 5.5.4.2 (2012/07/13) 新規追加
	 *
	 * @param   flag 編集許可 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_EDIT_WITH_EDITNAME
	 */
	public void setUseEdit( final String flag ) {
		useEdit = nval( getRequestParameter( flag ), useEdit );
	}

	/**
	 * 【TAG】editNameをhiddenタグで出力するか設定します
	 *		(初期値:OUTPUT_EDITNAME[={@og.value org.opengion.hayabusa.common.SystemData#OUTPUT_EDITNAME}])。
	 *
	 * @og.tag
	 * 選択したeditNameを次の画面に送信するためのhiddenタグを出力します。
	 * editNameを次の画面に送信した場合は送信先画面のviewタグのviewFormTypeが
	 * 内部的に自動セットされるため注意が必要です。
	 * useEditがfalseの場合は出力しません。
	 * (初期値:OUTPUT_EDITNAME[={@og.value org.opengion.hayabusa.common.SystemData#OUTPUT_EDITNAME}])。
	 *
	 * @og.rev 5.5.4.2 (2012/07/13) 新規追加
	 *
	 * @param   flag 編集名出力 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#OUTPUT_EDITNAME
	 */
	public void setOutputEditName( final String flag ) {
		outputEditName = nval( getRequestParameter( flag ), outputEditName );
	}

	/**
	 * 【TAG】ダミーのinputタグ(display:none)を出力するか設定します
	 *		(初期値:USE_DUMMY_INPUT[={@og.value org.opengion.hayabusa.common.SystemData#USE_DUMMY_INPUT}])。
	 *
	 * @og.tag
	 * ダミーのinputタグ(display:none)を出力するか設定します。
	 * IEではinputが１つだけの場合にエンターでのsubmitが上手くいかないため、ダミーを配置する必要があります。
	 * trueに設定するとWriteCheckタグがダミーを（常に）出力します。
	 * (初期値:USE_DUMMY_INPUT[={@og.value org.opengion.hayabusa.common.SystemData#USE_DUMMY_INPUT}])。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) 新規追加
	 *
	 * @param   flag ダミー出力 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_DUMMY_INPUT
	 */
	public void setUseDummy( final String flag ) {
		useDummy = nval( getRequestParameter( flag ), useDummy );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "useGamenId"	,useGamenId	)
				.println( "writable"	,writable	)		// 6.2.3.0 (2015/05/01)
				.println( "isWrite"		,isWrite()	)		// 6.2.3.0 (2015/05/01)
				.println( "useTag"		,useTag()	)		// 6.2.3.0 (2015/05/01)
				.println( "roles"		,roles		)
				.println( "logMsg"		,logMsg		)		// 6.2.4.0 (2015/05/15)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
