/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.LogWriter;
import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * 開始、終了、ステップの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したりします。
 * 
 * ここでは、数字（連番）の自動生成を行います。パラメータで、開始、終了、ステップを指定します。
 * パラメータの初期値は、開始(1)、終了(10)、ステップ(1) です。
 * 
 * 例：1,10,1    → 1,2,3,4,5,6,7,8,9,10 のプルダウン
 * 例：10,100,10 → 10,20,30,40,50,60,70,80,90,100 のプルダウン
 * 例：-5,5,1    → -5,-4,-3,-2,-1,0,1,2,3,4,5 のプルダウン
 * 例：5,-5,-2   → 5,3,1,-1,-3,-5 のプルダウン
 *
 * @og.group 選択データ制御
 * @og.rev 5.6.1.1 (2013/02/08) 新規追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_NUM extends Selection_NULL {
	private final String   CACHE ;
	private final String   ST_ED_STEP ;

	/**
	 * コンストラクター
	 *
	 * 引数は、開始、終了、ステップです。
	 * パラメータの初期値は、開始(1)、終了(10)、ステップ(1) です。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) type別Selectionの場合、ラベルリソースを使用する為、言語を引数で渡す。
	 * @og.rev 6.3.4.0 (2015/08/01) Selection_NUM の引数から、lang 属性を削除します。
	 *
	 * @param	editPrm	開始、終了、[ステップ]を表す引数(例：1,10,1)
//	 * @param	lang  言語(今は未使用)
	 */
//	public Selection_NUM( final String editPrm,final String lang ) {
	public Selection_NUM( final String editPrm ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
	//	if( param.length < 2 ) {
	//	final String errMsg = "引数は、開始、終了、[ステップ] です。最低でも２個必要です。";
	//		throw new IllegalArgumentException( errMsg );
	//	}

		final String[] param = editPrm == null ? new String[0] : editPrm.split( "," ) ;

		final int start = param.length > 0 ? Integer.parseInt( param[0].trim() ) : 1;
		final int end   = param.length > 1 ? Integer.parseInt( param[1].trim() ) : 10 ;
		final int step  = param.length > 2 ? Integer.parseInt( param[2].trim() ) : 1;

		if( step == 0 ) {
			final String errMsg = "ステップ に ０ は指定できません。無限ループします。";
			throw new IllegalArgumentException( errMsg );
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// ステップの正負による判定の違い。while( Math.signum( end-start ) * step >= 0.0 ) で、判る？
		// 終了条件は、含む(val<=end)
		int val  = start;
		final int sign = step > 0 ? 1 : -1 ;	// ステップの符号。
		while( (end - val) * sign >= 0 ) {
			buf.append( "<option value=\"" ).append( val )
				.append( "\">" ).append( val ).append( "</option>" );		// 6.0.2.5 (2014/10/31) char を append する。
			val += step;
		}

		CACHE = buf.toString();
		ST_ED_STEP = "Start=" + start + " , End=" + end + " , Step=" + step ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。(未使用)
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		// マッチするアドレスを探す。
		final int selected = CACHE.indexOf( "\"" + selectValue + "\"" );

		if( selected < 0 ) {
			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "数字範囲に存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ CR + ST_ED_STEP ;
				LogWriter.log( errMsg );
			}
			return CACHE;
		}
		else {
			// "値" 文字列の位置が、selected なので、値の文字数＋２までが、前半部分になる。
			final int indx = selected + selectValue.length() + 2 ;

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( selectValue ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( CACHE.substring( 0,indx ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( CACHE.substring( indx ) );
			return buf.toString() ;
		}
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを [true:使用する/false:しない] (未使用)
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		// あろうがなかろうが、選択肢そのものを返します。
		return selectValue;
	}

}
