/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.sql.Connection;

/**
 * コネクションを共有して、トランザクションを実現するインターフェースです。
 *
 * 基本的には、TransactionTag で利用されますが、一部、このオブジェクトを
 * 渡して、直接、利用するケースもあります。
 *
 * トランザクション の実クラスには、close()時に、commit,rollback を行う、
 * TransactionReal クラスと、内部にキャッシュされた コネクションを、終了時点で
 * 一括処理を行う、TransactionImpl があります。
 * TransactionTag で利用するのが、TransactionImpl で、Connectionのラッパーとして
 * 利用するのが、TransactionReal クラスになります。
 *
 * 6.3.6.1 (2015/08/28)
 *   selectを実行した後で明示的にcommit,rollbackを行わないのはOracle位
 *   らしいので、検索終了時でも、commit か、rollback を行うようにします。
 *   つまり、commit されない(=途中で処理が打ち切られた)場合は、
 *   rollback するように仕様変更しますので、Transactionオブジェクトを
 *   呼び出した処理の最後には、検索であろうとなかろうと、commit()を入れてください。
 *   ただし、Transaction オブジェクトは、DBアクセス以外にも適用可能に
 *   作成しているため、Connection がある場合のみ、実際の commit/rollback が
 *   実行されます。
 *
 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
// public interface Transaction {
public interface Transaction extends AutoCloseable {

	/**
	 * 指定のDBID に対応した、Connection オブジェクトを返します。
	 * 内部Mapに存在していれば、そのコネクションを、存在しなければ、
	 * 新しく作成します。
	 *
	 * Connection をキャッシュに戻すなどは、Transaction実装クラスで行います。
	 *
	 * @param	dbid  接続先ID
	 *
	 * @return	指定のDBID に対応した、コネクションオブジェクト
	 */
	Connection getConnection( final String dbid ) ;

	/**
	 * コミット処理が行われた場合に、内部フラグ(isCommit)を true にセットします。
	 * １回でもコミットが行われており、ロールバックが行われていなければ、
	 * コミットされます。
	 *
//	 * 検索処理のみで、エラーが発生していない場合は、コミットも行われないケースがあります。
	 * 検索処理時でも、最後に commit() を実行してください。実行されていない場合は、
	 * 自動的に、rollback() が、実行されます。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。return 不要。
//	 * @return 正常:true/異常:false
	 */
//	boolean commit() ;
	void commit() ;

	/**
	 * ロールバック処理が行われた場合に、内部フラグ(isRollback)を true にセットします。
	 * １回でもロールバックが行われていれば、最終的にはロールバックされます。
	 *
	 * 一度も、ロールバックが行われていない場合でも、コミットが行われていない場合は、
	 * rollback()を実行します。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。return 不要。
//	 * @return 正常:true/異常:false
	 */
//	boolean rollback() ;
	void rollback() ;

	/**
	 * トランザクションの、終了時処理を行います。
	 *
	 * それまでの処理は、すべて正常に処理できた場合に、使用します。
//	 * close( false ) と同じです。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。return 不要。
	 *
//	 * @return 正常:true/異常:false
	 * @see AutoCloseable#close()
	 */
//	boolean close() ;
	@Override
	void close();

	/**
	 * 最終的なコミットが行われた場合に、内部フラグ(isEndCommit)を true にセットします。
	 * 通常は、この処理は、１値度だけ実行されます。
	 * 初期値が、false なので、途中で一度でも実行されると、true にセットされ、最後まで
	 * 処理されたとみなされてしまうので、注意してください。
	 *
	 * 通常は、タグリブの、doEndTag() が実行された場合に、呼びます。
	 * このフラグが、true でないと（つまり、一度でも呼ばれないと）最終的に、commit されません。
	 *
	 * なお、endCommit() が呼ばれると、自動的に、commit() も呼んでおきます。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) 一般的なタグで、SKIP_PAGE された場合、rollback するようにします。
	 */
	void endCommit() ;

//	/**
//	 * トランザクションの、終了時処理を行います。
//	 *
//	 * 引数は、正常かどうかを判定するフラグです。異常の場合は、true をセットします。
//	 * ここでは、実際には何もしませんが、内部的にエラーフラグをセットします。
//	 * (エラーの場合のみセット。リセットはされません)
//	 * 一度でも、エラーが発生したコネクションは、破棄します。それ以外は、プールに戻します。
//	 *
//	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。廃止
//	 *
//	 * @param	errFlag 	エラー状態 [true:/false:通常]
//	 *
//	 * @return 正常:true/異常:false
//	 */
//	boolean close( final boolean errFlag ) ;
}
