/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.system.Closer ;
import org.opengion.fukurou.system.LogWriter;

import java.util.Map ;
// import java.util.HashMap ;
import java.util.concurrent.ConcurrentMap;					// 6.4.3.4 (2016/03/11)
import java.util.concurrent.ConcurrentHashMap;				// 6.4.3.1 (2016/02/12) refactoring
import java.util.LinkedHashMap ;
import java.nio.charset.CharacterCodingException;			// 6.3.1.0 (2015/06/28)

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Process_TableDiffは、ファイルから読み取った内容を、LineModel に設定後、
 * 下流に渡す、FirstProcess インターフェースの実装クラスです。
 *
 * DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、
 * 下流(プロセスチェインのデータは上流から下流に渡されます。)に渡します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_TableDiff -infile1=INFILE -infile2=INFILE2 -action=DIFF1 -encode=UTF-8 -columns=AA,BB,CC
 *
 *    -infile1=入力ファイル名1    ：入力ファイル名1
 *    -infile2=入力ファイル名2    ：入力ファイル名2
 *    -action=比較結果の方法      ：ONLY,DIFF,INTERSEC
 *   [-sep1=セパレータ文字      ] ：区切り文字1(初期値:タブ)
 *   [-sep2=セパレータ文字      ] ：区切り文字2(初期値:タブ)
 *   [-encode1=文字エンコード   ] ：入力ファイルのエンコードタイプ1
 *   [-encode2=文字エンコード   ] ：入力ファイルのエンコードタイプ2
 *   [-columns=読み取りカラム名 ] ：入力カラム名(CSV形式)
 *   [-keyClms=比較するカラム名 ] ：比較する列の基準カラム名(CSV形式)
 *   [-diffClms=比較するカラム名] ：比較するカラム名(CSV形式)
 *   [-display=[false/true]     ] ：結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *   [-debug=[false/true]       ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @og.rev 4.2.3.0 (2008/05/26) 新規作成
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_TableDiff extends AbstractProcess implements FirstProcess {
	private static final String ENCODE = System.getProperty("file.encoding");

	private char		  	separator1	= TAB;	// 6.0.2.5 (2014/10/31) TAB を char 化
	private char		  	separator2	= TAB;	// 6.0.2.5 (2014/10/31) TAB を char 化
	private String			infile1		;
	private String			encode1		;		// 6.3.1.0 (2015/06/28) エラー時のメッセージに使う。
	private String			infile2		;
	private BufferedReader	reader1		;
	private LineModel		model		;
	private String			line		;
	private int[]			clmNos		;		// ファイルのヘッダーのカラム番号
	private int[]			keyClmNos	;		// 比較する列の基準カラム名のカラム番号
	private int[]			diffClmNos	;		// 比較するカラム名のカラム番号
	private String			actCmnd		;		// action から名称変更
	private boolean			display		;		// 表示しない
	private boolean			debug		;		// 表示しない
	private boolean			nameNull	;		// ０件データ時 true

	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
//	private final Map<String,String> file2Map = new HashMap<>();			// 4.3.1.1 (2008/08/23) final化
	private final ConcurrentMap<String,String> file2Map = new ConcurrentHashMap<>();	// 4.3.1.1 (2008/08/23) final化

	private int				inCount1	;
	private int				inCount2	;
	private int				outCount	;

	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "infile1",	"入力ファイル名1 (必須)" );
		MUST_PROPARTY.put( "infile2",	"入力ファイル名2 (必須)" );
		MUST_PROPARTY.put( "action",		"(必須)ONLY,DIFF,INTERSEC" );
		MUST_PROPARTY.put( "keyClms",	"比較する列の基準カラム名(必須)(CSV形式)" );
		MUST_PROPARTY.put( "diffClms",	"比較するカラム名(必須)(CSV形式)" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "sep1",			"区切り文字1 (初期値:タブ)" );
		USABLE_PROPARTY.put( "sep2",			"区切り文字2 (初期値:タブ)" );
		USABLE_PROPARTY.put( "encode1",		"入力ファイルのエンコードタイプ1" );
		USABLE_PROPARTY.put( "encode2",		"入力ファイルのエンコードタイプ2" );
		USABLE_PROPARTY.put( "columns",		"入力カラム名(CSV形式)" );
		USABLE_PROPARTY.put( "display",		"結果を標準出力に表示する(true)かしない(false)か" +
											CR + " (初期値:false:表示しない)" );
		USABLE_PROPARTY.put( "debug",		"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
											CR + " (初期値:false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_TableDiff() {
		super( "org.opengion.fukurou.process.Process_TableDiff",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		infile1				= arg.getProparty( "infile1" );
		encode1				= arg.getProparty( "encode1",ENCODE );
//		final String  encode1	= arg.getProparty( "encode1",ENCODE );
		infile2				= arg.getProparty( "infile2" );
		actCmnd				= arg.getProparty( "action"  );
//		final String  encode2	= arg.getProparty( "encode2",ENCODE );
//		final String  clms		= arg.getProparty( "columns"  );
//		final String  keyClms	= arg.getProparty( "keyClms"  );
//		final String  diffClms	= arg.getProparty( "diffClms" );
		display				= arg.getProparty( "display",display );
		debug				= arg.getProparty( "debug"  ,debug );

		// 6.0.2.5 (2014/10/31) TAB を char 化
		final String sep1 = arg.getProparty( "sep1",null );
		final String sep2 = arg.getProparty( "sep2",null );
		if( sep1 != null ) { separator1 = sep1.charAt(0); }
		if( sep2 != null ) { separator2 = sep2.charAt(0); }

		if( infile1 == null || infile2 == null ) {
			final String errMsg = "ファイル名が指定されていません。"
						+ "File1=[" + infile1 + "] , File2=[" + infile2 + "]" ;
			throw new OgRuntimeException( errMsg );
		}

		final File file1 = new File( infile1 );
		final File file2 = new File( infile2 );

		if( ! file1.exists() || ! file2.exists() ) {
			// 4.3.1.1 (2008/08/23) Avoid if(x != y) ..; else ..;
			final String errMsg = "ファイルが存在しません。"
						+ ( file1.exists() ? "" : "File1=[" + file1 + "] " )
						+ ( file2.exists() ? "" : "File2=[" + file2 + "]" );
			throw new OgRuntimeException( errMsg );
		}

		if( ! file1.isFile() || ! file2.isFile() ) {
			// 4.3.1.1 (2008/08/23) Avoid if(x != y) ..; else ..;
			final String errMsg = "フォルダは指定できません。ファイル名を指定してください。"
						+ ( file1.isFile() ? "" : "File1=[" + file1 + "] " )
						+ ( file2.isFile() ? "" : "File2=[" + file2 + "]" );
			throw new OgRuntimeException( errMsg );
		}

		reader1 = FileUtil.getBufferedReader( file1,encode1 );

		final String[] names ;
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String  clms = arg.getProparty( "columns" );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( clms == null ) {
			final String[] clmNames = readName( reader1 );		// ファイルのカラム名配列
			if( clmNames == null || clmNames.length == 0 ) { nameNull = true; return ; }
			names = clmNames;
		}
		else {
			names = StringUtil.csv2Array( clms );	// 指定のカラム名配列
		}

//		if( clms != null ) {
//			names = StringUtil.csv2Array( clms );	// 指定のカラム名配列
//		}
//		else {
//			final String[] clmNames = readName( reader1 );		// ファイルのカラム名配列
//			if( clmNames == null || clmNames.length == 0 ) { nameNull = true; return ; }
//			names = clmNames;
//		}

		model = new LineModel();
		model.init( names );

		if( display ) { println( model.nameLine() ); }

		// 入力カラム名のカラム番号
		clmNos = new int[names.length];
		for( int i=0; i<names.length; i++ ) {
			clmNos[i] = i+1;						// 行番号分を＋１しておく。
		}

		// 比較する列の基準カラム名
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String  keyClms = arg.getProparty( "keyClms" );
		if( debug ) { println( "DEBUG:\tkeyClms=" + keyClms ); }
		final String[] keyClmNms = StringUtil.csv2Array( keyClms );
		keyClmNos = new int[keyClmNms.length];
		for( int i=0; i<keyClmNms.length; i++ ) {
			keyClmNos[i] = model.getColumnNo( keyClmNms[i] );
	//		if( debug ) { println( "DEBUG:" + keyClmNms[i] + ":[" + keyClmNos[i] + "]" ); }
	//		int no = model.getColumnNo( keyClmNms[i] );
	//		if( no >= 0 ) { keyClmNos[no] = i+1; }		// 行番号分を＋１しておく。
		}

		// 比較するカラム名
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String  diffClms = arg.getProparty( "diffClms" );
		if( debug ) { println( "DEBUG:\tdiffClms=" + diffClms ); }
		final String[] diffClmNms = StringUtil.csv2Array( diffClms );
		diffClmNos = new int[diffClmNms.length];
		for( int i=0; i<diffClmNms.length; i++ ) {
			diffClmNos[i] = model.getColumnNo( diffClmNms[i] );
	//		if( debug ) { println( "DEBUG:" + diffClmNms[i] + ":[" + diffClmNos[i] + "]" ); }
	//		int no = model.getColumnNo( diffClmNms[i] );
	//		if( no >= 0 ) { diffClmNos[no] = i+1; }		// 行番号分を＋１しておく。
		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String  encode2 = arg.getProparty( "encode2",ENCODE );
		readF2Data( file2,encode2 );
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		Closer.ioClose( reader1 );
		reader1 = null;
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 *
	 * @return	処理できる:true / 処理できない:false
	 */
	public boolean next() {
		if( nameNull ) { return false; }

		boolean flag = false;
		try {
			while((line = reader1.readLine()) != null) {
				inCount1++ ;
				if( line.isEmpty() || line.charAt(0) == '#' ) { continue; }
				else {
					flag = true;
					break;
				}
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch ( CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + infile1 + "] , Encode=[" + encode1 + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
		catch (IOException ex) {
			final String errMsg = "ファイル読込みエラー[" + infile1 + "]:(" + inCount1 + ")"  ;
			throw new OgRuntimeException( errMsg,ex );
		}
		return flag;
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * ファイルより読み込んだ１行のデータを テーブルモデルに
	 * セットするように分割します
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 *
	 * @param	rowNo	処理中の行番号
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		outCount++ ;
		final String[] vals = StringUtil.csv2Array( line ,separator1 );		// 6.0.2.5 (2014/10/31) TAB を char 化

		final int len = vals.length;
		for( int clmNo=0; clmNo<model.size(); clmNo++ ) {
			final int no = clmNos[clmNo];
			if( len > no ) {
				model.setValue( clmNo,vals[no] );
			}
			else {
				// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
				model.setValue( clmNo,"" );
			}
		}
		model.setRowNo( rowNo ) ;

	//	if( display ) { println( model.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更

		return action( model );
	}

	/**
	 * キーと、DIFF設定値を比較し、action に応じた LineModel を返します。
	 * action には、ONLY,DIFF,INTERSEC が指定できます。
	 *   ONLY      inFile1 のみに存在する行の場合、inFile1 のレコードを返します。
	 *   DIFF      inFile1 と inFile2 に存在し、かつ、DIFF値が異なる、inFile1 のレコードを返します。
	 *   INTERSEC  inFile1 と inFile2 に存在し、かつ、DIFF値も同じ、inFile1 のレコードを返します。
	 * inFile2 側をキャッシュしますので、inFile2 側のデータ量が少ない様に選んでください。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param 	model LineModelオブジェクト
	 *
	 * @return	実行後のLineModel
	 */
	private LineModel action( final LineModel model ) {
		LineModel rtn = null;
		final Object[] obj = model.getValues();

		// キーのカラムを合成します。
		final StringBuilder keys = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<keyClmNos.length; i++ ) {
			keys.append( obj[keyClmNos[i]] ).append( ',' );				// 6.0.2.5 (2014/10/31) char を append する。
		}

		final String data = file2Map.get( keys.toString() );
	//	if( debug ) { println( "DEBUG:" + keys.toString() + ":" + data ); }

		if( "ONLY".equalsIgnoreCase( actCmnd ) && data == null ) {
			if( debug ) { println( "DEBUG:ONLY\t" + keys.toString() ); }
			rtn = model;
		}
		else {
			// DIFF値のカラムを合成します。
			final StringBuilder vals = new StringBuilder( BUFFER_MIDDLE );
			for( int i=0; i<diffClmNos.length; i++ ) {
				vals.append( obj[diffClmNos[i]] ).append( ',' );		// 6.0.2.5 (2014/10/31) char を append する。
			}

			final boolean eq = vals.toString().equals( data );

			if( "DIFF".equalsIgnoreCase( actCmnd ) && ! eq ) {
				if( debug ) { println( "DEBUG:DIFF\t" + keys.toString() + TAB + data + TAB + vals.toString() ); }
				rtn = model;
			}
			else if( "INTERSEC".equalsIgnoreCase( actCmnd ) && eq ) {
				if( debug ) { println( "DEBUG:INTERSEC\t" + keys.toString() + TAB + data ); }
				rtn = model;
			}
		}
		if( display && rtn != null ) { println( rtn.dataLine() ); }
		return rtn;
	}

	/**
	 * BufferedReader より、#NAME 行の項目名情報を読み取ります。
	 * データカラムより前に、項目名情報を示す "#Name" が存在する仮定で取り込みます。
	 * この行は、ファイルの形式に無関係に、TAB で区切られています。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) #NAME 行の区切り文字は、指定の区切り文字を優先して利用する。
	 * @og.rev 6.0.4.0 (2014/11/28) #NAME 判定で、桁数不足のエラーが発生する箇所を修正。
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 *
	 * @param 	reader PrintWriterオブジェクト
	 *
	 * @return	カラム名配列(存在しない場合は、サイズ０の配列)
	 * @og.rtnNotNull
	 */
	private String[] readName( final BufferedReader reader ) {
		try {
			// 4.0.0 (2005/01/31) line 変数名変更
			String line1;
			while((line1 = reader.readLine()) != null) {
				inCount1++ ;
				if( line1.isEmpty() ) { continue; }
				if( line1.charAt(0) == '#' ) {
					// 6.0.4.0 (2014/11/28) #NAME 判定で、桁数不足のエラーが発生する箇所を修正。
					if( line1.length() >= 5 && "#NAME".equalsIgnoreCase( line1.substring( 0,5 ) ) ) {
						// 6.0.4.0 (2014/11/28) #NAME 行の区切り文字は、指定の区切り文字を優先して利用する。
						final char sep ;
						if( TAB != separator1 && line1.indexOf( separator1 ) >= 0 ) {
							sep = separator1;
						}
						else {
							sep = TAB;
						}
						// 超イレギュラー処理。#NAME をカラム列に入れない(#NAME+区切り文字 の 6文字分、飛ばす)。
						return StringUtil.csv2Array( line1.substring( 6 ) ,sep );
					}
					else  { continue; }
				}
				else {
					final String errMsg = "#NAME が見つかる前にデータが見つかりました。";
					throw new OgRuntimeException( errMsg );
				}
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch ( CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + infile1 + "] , Encode=[" + encode1 + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
		catch (IOException ex) {
			final String errMsg = "ファイル読込みエラー[" + infile1 + "]:(" + inCount1 + ")"  ;
			throw new OgRuntimeException( errMsg,ex );
		}
		return new String[0];
	}

	/**
	 * ファイル属性を読取り、キー情報を作成し、内部メモリマップにキャッシュします。
	 * このマップをもとに、inFile1 のデータを逐次読み取って、処理を進めます。
	 *
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 *
	 * @param 	file2 読取り元のファイル
	 * @param 	encode2 ファイルのエンコード
	 */
	private void readF2Data( final File file2, final String encode2 ) {
		BufferedReader reader2 = null;
		try {
			if( debug ) { println( "DEBUG:\tFile2="+ file2 + " 初期処理" ); }
			reader2 = FileUtil.getBufferedReader( file2,encode2 );
			// 4.0.0 (2005/01/31) line 変数名変更
			String line1;
			final StringBuilder keys = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
			final StringBuilder vals = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
			while((line1 = reader2.readLine()) != null) {
				inCount2++ ;
				if( line1.isEmpty() || line1.charAt(0) == '#' ) { continue; }
//				if( line1.isEmpty() ) { continue; }
//				if( line1.charAt(0) == '#' ) { continue; }
				else {
					// 超イレギュラー処理 最初の TAB 以前の文字は無視する。
					final String line2 = line1.substring( line1.indexOf( separator2 )+1 );
					final Object[] obj = StringUtil.csv2Array( line2 , separator2 );		// 6.0.2.5 (2014/10/31) TAB を char 化

					// キーのカラムを合成します。
					keys.setLength(0);												// 6.1.0.0 (2014/12/26) refactoring
					for( int i=0; i<keyClmNos.length; i++ ) {
						keys.append( obj[keyClmNos[i]] ).append( ',' );				// 6.0.2.5 (2014/10/31) char を append する。
					}

					// DIFF値のカラムを合成します。
					vals.setLength(0);												// 6.1.0.0 (2014/12/26) refactoring
					for( int i=0; i<diffClmNos.length; i++ ) {
						vals.append( obj[diffClmNos[i]] ).append( ',' );			// 6.0.2.5 (2014/10/31) char を append する。
					}

					if( debug ) { println( "DEBUG:\t" + keys.toString() + "\t" + vals.toString() ); }

					file2Map.put( keys.toString(), vals.toString() );
				}
			}
			if( debug ) { println( "DEBUG:\t======初期処理終了======" ); }
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch ( CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + file2.getPath() + "] , Encode=[" + encode2 + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
		catch (IOException ex) {
//			final String errMsg = "ファイル読込みエラー[" + infile2 + "]:(" + inCount2 + ")"  ;
			final String errMsg = "ファイル読込みエラー[" + file2.getPath() + "]:(" + inCount2 + ")"  ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader2 );
		}
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Input  File1  : " + infile1	+ CR
				+ TAB + "Input  File2  : " + infile2	+ CR
				+ TAB + "Input  Count1 : " + inCount1	+ CR
				+ TAB + "Input  Count2 : " + inCount2	+ CR
				+ TAB + "Output Count  : " + outCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_TableDiffは、ファイルから読み取った内容を、LineModel に設定後、" 		).append( CR )
			.append( "下流に渡す、FirstProcess インターフェースの実装クラスです。"					).append( CR )
			.append( CR )
			.append( "DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、"		).append( CR )
			.append( "下流(プロセスチェインのデータは上流から下流に渡されます。)に渡します。"		).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_TableDiff().usage() );
	}
}
