/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.XHTMLTag;

/**
 * インラインフレームを作成するHTML拡張タグで、引数の受け渡しが可能です。
 *
 * @og.formSample
 * ●形式：&lt;og:iframe src="…" name="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:iframe
 *       src              ○【HTML】フレームに表示するソースファイルを指定します(必須)。
 *       name             ○【HTML】フレーム名を指定します(例：QUERY,RESULT,CONTENTS など)(必須)。
 *       changeGamen        【TAG】ソース指定の画面を変更します
 *       keys               【TAG】引数を指定します
 *       dbTable            【TAG】前ページで選択したデータ列の情報を次のページに渡すかどうか[true/false]を指定します(初期値:false)
 *  ※   width              【HTML】フレームの幅を指定します
 *  ※   height             【HTML】フレームの高さを指定します
 *  ※   srcdoc             【HTML】フレームに表示する内容を指定します
 *  ※   seamless           【HTML】シームレス化の指定をします
 *  ※   sandbox            【HTML】サンドボックスの指定をします
 * <del>      marginwidth        【廃止】フレームの左右余白サイズ(marginwidth)を指定します(HTML5廃止属性)</del>
 * <del>      marginheight       【廃止】フレームの上下余白サイズ(marginheight)を指定します(HTML5廃止属性)</del>
 * <del>      noresize           【廃止】フレームサイズを変更できないよう(noresize)に指定します(HTML5廃止属性)</del>
 * <del>      scrolling          【廃止】スクロールバー(scrolling)の表示/非表示[auto/yes/no]を指定します(初期値:auto)(HTML5廃止属性)</del>
 * <del>      frameborder        【廃止】フレームの境界線(frameborder)の[0:非表示/1:表示]を指定します(初期値:1)(HTML5廃止属性)</del>
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 *  ※ は、iframe専用属性。それ以外は、frameタグからの属性。
 *
 * ●使用例：
 *    ・一般的な例：インラインフレームの構成はHTML準拠。リクエスト変数は各フレームまで転送されます。
 *        &lt;og:iframe src="query.jsp"   name="QUERY"  /&gt;
 *    &lt;/frameset&gt;
 *
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Takahashi Masakazu
 * @since    JDK5.0,
 */
public class IframeTag extends FrameTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public IframeTag() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

//	/**
//	 * タグリブオブジェクトをリリースします。
//	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
//	 *
//	 * ※ 独自のインスタンス変数を持たないので、release2() は、親クラスのcallだけでよい。
//	 */
//	@Override
//	protected void release2() {
//		super.release2();
//	}

//	/**
//	 * リンクを作成します。
//	 *
//	 * @return	リンクタグ文字列
//	 */
//	@Override
//	protected String makeTag() {
//		if( changeGmn != null ) { set( "src",changeGmn ); }
//
//		String cmd = getRequest().getParameter( "command" );
//		if( "RENEW".equals( cmd ) ) { dbTable = false; }
//
//		setIdName();
//		String urlEnc = getUrlEncode();
//		return XHTMLTag.iframe( getAttributes(),urlEnc );
//	}

	/**
	 * リンクを作成します。
	 *
	 * @og.rev 5.9.1.2 (2015/10/23) iframeとの継承関係で、共通的に呼び出すメソッドを追加します。
	 *
	 * @param	urlEnc エンコードされたURL
	 * @return	リンクタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTagFrame( final String urlEnc ) {
		return XHTMLTag.iframe( getAttributes(),urlEnc );
	}

	/**
	 * 【HTML】フレームに表示するコンテンツ(srcdoc)を指定します。
	 *
	 * @og.tag srcdocを指定します。
	 *
	 * @param	srcdoc 表示コンテンツ
	 */
	public void setSrcdoc( final String srcdoc ) {
		set( "srcdoc",getRequestParameter( srcdoc ) );
	}

	/**
	 * 【HTML】シームレス表示(seamless)を指定します。
	 *  HTML上は属性値が省略可能なため、seamlessやtrue等を指定してください。
	 *
	 * @og.tag seamlessを指定します。
	 *
	 * @param	seamless シームレス表示指定
	 */
	public void setSeamless( final String seamless ) {
		set( "seamless",getRequestParameter( seamless ) );
	}

	/**
	 * 【HTML】フレームに表示するコンテンツのセキュリティ上の制限(sandbox)を指定します。
	 * スペース等で空指定して属性を付加する事でサンドボックス化され、セキュリティが強化されます。
	 * 属性値を指定すると、対応した制限が解除されます。
	 * 標準は属性無し（サンドボックス化無し）です。
	 *
	 * @og.tag sandboxを指定します。
	 *
	 * @param	sandbox 表示コンテンツ
	 */
	public void setSandbox( final String sandbox ) {
		set( "sandbox",getRequestParameter( sandbox ) );
	}

	/**
	 * 【HTML】フレームの幅を指定します。
	 *
	 * @og.tag widthを指定します。
	 *
	 * @param	width 幅
	 */
	public void setWidth( final String width ) {
		set( "width",getRequestParameter( width ) );
	}

	/**
	 * 【HTML】フレームの高さを指定します。
	 *
	 * @og.tag heightを指定します。
	 *
	 * @param	height 高さ
	 */
	public void setHeight( final String height ) {
		set( "height",getRequestParameter( height ) );
	}

//	/**
//	 * 【廃止】フレームの左右余白サイズ(marginwidth)を指定します。
//	 * HTML5では廃止されている属性のため、可能であればCSSを利用してください。
//	 *
//	 * @og.tag フレームの左右余白サイズを指定します。
//	 *
//	 * @param	marginwidth 左右余白サイズ
//	 */
//	@Override
//	public void setMarginwidth( final String marginwidth ) {
//		set( "marginwidth",getRequestParameter( marginwidth ) );
//	}

//	/**
//	 * 【廃止】フレームの上下余白サイズ(marginheight)を指定します。
//	 * HTML5では廃止されている属性のため、可能であればCSSを利用してください。
//	 *
//	 * @og.tag フレームの上下余白サイズを指定します。
//	 *
//	 * @param	marginheight 上下余白サイズ
//	 */
//	@Override
//	public void setMarginheight( final String marginheight ) {
//		set( "marginheight",getRequestParameter( marginheight ) );
//	}

//	/**
//	 * 【HTML】スクロールバー(scrolling)の表示/非表示[auto/yes/no]を指定します(初期値:auto)。
//	 *  HTML5では廃止されている属性のため、可能であればCSSを利用してください。
//	 *
//	 * @og.tag
//	 * auto:必要に応じてスクロールバーを表示(初期値)
//	 * yes:常にスクロールバーを表示
//	 * no:常にスクロールバーを表示しない
//	 *
//	 * @param	scrolling  スクロールバーの表示/非表示[auto:自動/yes:常時表示/no:非表示]
//	 */
//	@Override
//	public void setScrolling( final String scrolling ) {
//		set( "scrolling",getRequestParameter( scrolling ) );
//	}

//	/**
//	 * 【HTML】フレームの境界線(frameborder)の[0:非表示/1:表示]を指定します(初期値:1)。
//	 *  HTML5では廃止されている属性のため、可能であればCSSを利用してください。
//	 *
//	 * @og.tag
//	 * 0:枠を表示しない
//	 * 1:枠を表示する。
//	 * 初期値は、1:枠を表示する です。
//	 *
//	 * @param	frameborder	フレームの境界線[0:枠非表示/1:枠表示]
//	 */
//	@Override
//	public void setFrameborder( final String frameborder ) {
//		set( "frameborder",getRequestParameter( frameborder ) );
//	}
}
