/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.Calendar;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.LogWriter;

/**
 * データのコード情報を取り扱うクラスです。
 *
 * 開始、終了、ステップの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したりします。
 * 
 * ここでは、時間(時:分）の自動生成を行います。パラメータで、開始、終了、ステップ、開始前設定値、終了後設定値
 * を指定できます。
 * キーは、４文字の HHMM 形式で与えられます。ラベルは、HH:MM になります。
 * ステップは、分単位です。つまり、１時間の場合は、"60" と指定します。"0100"ではありません。
 * 開始前設定値、終了後設定値はそれぞれ、開始の前と終了の後ろに特別に値を設定できます。
 *
 * 開始、または、終了に、現在時刻からの相対値を指定する事ができます。
 * H1 ～ HXXX とすれば、現在時刻の時に数字部分を＋－します。分は０に初期化されます。
 *
 * パラメータの初期値は、開始(0700)、終了(1900)、ステップ(30)、開始前設定値(null)、終了後設定値(null) です。
 * 
 * 例：0800,2000,30  → 0800,0830,0900,0930,1000,・・・1900,1930,2000 のプルダウン
 * 例：0800,2000,30,0000:△,2400:▽  → 0000,0800,0830,0900,0930,1000,・・・1900,1930,2000,2400 のプルダウン
 *
 * @og.group 選択データ制御
 * @og.rev 5.6.1.1 (2013/02/08) 新規追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class Selection_HM implements Selection {
public class Selection_HM extends Selection_NULL {
	private final String   CACHE ;
	private final String   ST_ED_STEP ;

	private final long	maxCacheTime ;		// キャッシュの破棄タイミングを計るための最大有効時間

	/**
	 * コンストラクター
	 *
	 * 引数は、開始、終了、ステップ、開始前設定値、終了後設定値 です。
	 * パラメータの初期値は、開始(0700)、終了(1900)、ステップ(30)、開始前設定値(null)、終了後設定値(null) です。
	 *
	 * @param	editPrm	開始、終了、ステップ、開始前設定値、終了後設定値 を表す引数(例：0800,2000,30)
	 */
	public Selection_HM( final String editPrm ) {
	//	if( param.length < 3 ) {
	//		String errMsg = "引数は、開始、終了、ステップ、[開始前設定値]、[終了後設定値]です。最低でも３個必要です。";
	//		throw new IllegalArgumentException( errMsg );
	//	}

		String[] param = (editPrm == null) ? new String[0] : editPrm.split( "," ) ;

		String start = (param.length > 0) ? param[0].trim() : "0700" ;
		String end   = (param.length > 1) ? param[1].trim() : "1900" ;
		String step  = (param.length > 2) ? param[2].trim() : "30" ;

		String stOp  = (param.length > 3) ? param[3].trim() : null ;
		String enOp  = (param.length > 4) ? param[4].trim() : null ;

		int stepTime	= Integer.parseInt( step );
		if( stepTime == 0 ) {
			String errMsg = "ステップ に ０ は指定できません。無限ループします。";
			throw new IllegalArgumentException( errMsg );
		}

		Calendar cal = Calendar.getInstance();
		calendarCalc( cal, start );

		Calendar endCal = Calendar.getInstance();
		calendarCalc( endCal, end );

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		addOption( buf,stOp );		// 開始前設定値 の追加

		// cal.before( endCal ) では、同一時刻の場合に false になる為、ここの判定では使えません。
		// sign を掛け算しているのは、逆順対応
		int sign = ( stepTime > 0 ) ? 1 : -1 ;	// ステップの符号。
		while( endCal.compareTo( cal ) * sign >= 0 ) {
			int hh = cal.get( Calendar.HOUR_OF_DAY );	// 時
			int mm = cal.get( Calendar.MINUTE );		// 分

			String hVal = (hh < 10) ? "0" + hh : String.valueOf( hh ) ;
			String mVal = (mm < 10) ? "0" + mm : String.valueOf( mm ) ;

			buf.append( "<option value=\"" ).append( hVal ).append( mVal ).append( "\"" );
			buf.append( ">" ).append( hVal ).append( ":" ).append( mVal ).append( "</option>" );

			cal.add( Calendar.MINUTE,stepTime );		// 時刻に加えるのは、分
		}

		addOption( buf,enOp );		// 終了後設定値 の追加

		CACHE = buf.toString();
		ST_ED_STEP = "Start=" + start + " , End=" + end + " , Step=" + step + " , StartBefore=" + stOp + " , EndAfter=" + enOp ;


		// キャシュの有効期間を求めるための時刻を作成します。キャッシュは、時間指定があれば、同一時間内のみ有効です。
		Calendar now   = Calendar.getInstance();
		boolean nowBase = start.charAt(0) == 'H' || end.charAt(0) == 'H' ;
		if( nowBase ) {
			now.add( Calendar.HOUR   , 1 );				// １時間進めます。
			now.set( Calendar.MINUTE , 0 );				// 分、秒 をリセットします。
			now.set( Calendar.SECOND , 0 );
		}
		else {
			now.add( Calendar.YEAR   , 1 );				// １年間進めます。(現在時刻をベースに指定ない為、無制限キャッシュの意味)
		}

		maxCacheTime = now.getTimeInMillis() ;
	}

	/**
	 * 開始、または 終了の文字列から、カレンダオブジェクトを作成します。
	 * 基準となる日付に計算した結果を反映させます。
	 *
	 * prmB は、日付についての加減算処理を行うためのコマンドを指定します。
	 * ・数字       ：HHMM 形式の時分です。
	 * ・H1 ～ HXXX ：現在時刻に数字部分を＋－します。分は０に初期化されます。
	 *
	 * @param   cal     基準となる日付（Calendarオブジェクト）
	 * @param   prmB	処理コマンド
	 *
	 */
	private void calendarCalc( final Calendar cal,final String prmB ) {
		boolean nowBase = prmB.charAt(0) == 'H' ;

		if( nowBase ) {
			int hour	= Integer.parseInt( prmB.substring( 1 ) );
			cal.add( Calendar.HOUR_OF_DAY,hour );
			cal.set( Calendar.MINUTE     ,0 );
			cal.set( Calendar.SECOND     ,0 );
		}
		else {
			int hour	= Integer.parseInt( prmB.substring( 0,2 ) );
			int minute	= Integer.parseInt( prmB.substring( 2,4 ) );
			cal.set( Calendar.HOUR_OF_DAY,hour );
			cal.set( Calendar.MINUTE     ,minute );
			cal.set( Calendar.SECOND     ,0 );
		}
	}

	/**
	 * 開始前設定値、または 終了後設定値の文字列から、オプション文字列を合成します。
	 * このオプションは、引数のStringBuilder に、オプションタグを追加して返します。
	 * optVal が null の場合は、処理しません。
	 *
	 * @param   buf     文字列連結する StringBuilderオブジェクト。このオブジェクトに追加します。
	 * @param   optVal  開始前設定値、または 終了後設定値 文字列（"0000:△" 形式）
	 *
	 */
	private void addOption( final StringBuilder buf,final String optVal ) {
		if( optVal != null ) {
			int adrs = optVal.indexOf( ':' );
			if( adrs > 0 ) {
				buf.append( "<option value=\"" ).append( optVal.substring( 0,adrs ) ).append( "\"" );
				buf.append( ">" ).append( optVal.substring( adrs+1 ) ).append( "</option>" );
			}
			// 開始前設定値 が存在する場合、"0000:△" 形式必須
			else {
				String errMsg = "引数は、0000:△ 形式です。";
				throw new IllegalArgumentException( errMsg );
			}
		}
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、ラベル(短)が設定されている場合でも、これを使用せずに必ずラベル(長)を使用します。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 *
	 * @return  オプションタグ
	 * @see     #getOption( String, boolean, boolean )
	 */
//	public String getOption( final String selectValue,final boolean seqFlag ) {
//		return getOption( selectValue, seqFlag, false );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。(未使用)
	 *
	 * @return  オプションタグ
	 * @see     #getOption( String, boolean )
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		// マッチするアドレスを探す。
		int selected = CACHE.indexOf( "\"" + selectValue + "\"" );

		if( selected < 0 ) {
			if( selectValue != null && selectValue.length() > 0 ) {
				String errMsg = "時分範囲に存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ HybsSystem.CR + ST_ED_STEP ;
				LogWriter.log( errMsg );
			}
			return CACHE;
		}
		else {
			// "時分" 文字列の位置が、selected なので、時分の文字数＋２までが、前半部分になる。(時分の文字数は４固定のはず)
			int indx = selected + selectValue.length() + 2 ;

			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( selectValue ).append( "\"" );
			}
			else {
				buf.append( CACHE.substring( 0,indx ) );
			}
			buf.append( " selected=\"selected\"" );
			buf.append( CACHE.substring( indx ) );
			return buf.toString() ;
		}
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 */
//	public String getRadio( final String name,final String selectValue,final boolean useLabel ) {
//		String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue  選択されている値
	 *
	 * @return  オプションタグ
	 */
//	public String getRadioLabel( final String selectValue ) {
//		String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String,boolean )
	 */
//	public String getValueLabel( final String selectValue ) {
//		return getValueLabel( selectValue,false );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを [true:使用する/false:しない](未使用)
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// あろうがなかろうが、選択肢そのものを返します。
		return selectValue;
	}

	/**
	 * マルチ・キーセレクトを使用するかどうかを返します。
	 * true：使用する。false:使用しない です。
	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
	 * true を返します。
	 *
	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する)(false固定)
	 */
	@Override
	public boolean useMultiSelect() {
		return true;
	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	@Override
	public boolean isTimeOver() {
		return System.currentTimeMillis() > maxCacheTime ;
	}
}
