/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * 積上ガント（内部積上方式）でガント全体（ページ全体一つ設定）の指定を行います。
 *
 * iTask タグは、積上ガントのうち、内部積上方式の場合に使用します。
 * 内部積上げとは、フレームワーク側で積上げ計算を行う方式の事です。
 * 処理速度は増しますが、積上ガント(JavaScript版)は、積上と通常ガントの混在が
 * 可能ですが、こちらは、積上のみの表示になります。
 * それぞれの使い分けを考慮してください。
 * このタグは、viewFormType = "HTMLStackedGanttTable" 、innerStack="true" を指定した view の後に記述します。
 *
 * @og.formSample
 * ●形式：&lt;og:iTask  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Task" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iTask
 *       fixedCols          【TAG】左右２分割で、固定したいカラム数(必須)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *   &lt;og:view
 *       viewFormType = "HTMLStackedGanttTable"
 *       command      = "{&#064;command}"
 *       writable     = "false"
 *       useScrollBar = "false"
 *       useParam     = "true"
 *       numberType   = "none"
 *   &gt;
 *     &lt;!-- stackParamでstackColumnsを指定しなかった場合はcustomTableとほぼ同じ処理となる --&gt;
 *     &lt;og:stackParam
 *         stackColumns = "NOORDER"
 *         innerStack   = "true"
 *         costColumns  = "COSTCLM,DYJYU,DYNOKI"
 *         stackHoliday = "{&#064;stackHoliday}"
 *         capacityColumn = "CAPACITY"
 *     /&gt;
 *     &lt;og:thead rowspan="2"&gt;
 *       &lt;tr&gt;
 *           &lt;td&gt;[NOORDER]&lt;/td&gt;
 *           &lt;td rowspan="2" class="gantt zoom{&#064;SZOOM}" style="text-align:left;background-image: url('../image/stackBG{&#064;SZOOM}.gif');" &gt;
 *               &lt;div style="position:relative;left:0px;z-index:101; " height="100%" width="100%" &gt;
 *                   &lt;og:stackHeader
 *                       startDate   = "{&#064;MIN_DATE}"
 *                       endDate     = "{&#064;MAX_DATE}"
 *                       zoom        = "{&#064;SZOOM}"
 *                       calDB       = "GE13"
 *                       arg1        = "A"
 *                   /&gt;
 *               &lt;/div&gt;
 *           &lt;/td&gt;
 *       &lt;/tr&gt;
 *     &lt;/og:thead&gt;
 *     &lt;og:tbody rowspan="2"&gt;
 *       &lt;tr&gt;
 *         &lt;td&gt;[NOORDER]&lt;/td&gt;
 *       &lt;/tr&gt;
 *     &lt;/og:tbody&gt;
 *   &lt;/og:view&gt;
 *
 *   &lt;og:iTask
 *       fixedCols  = "4"
 *   /&gt;
 *  
 *   @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewITaskTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.3.2 (2013/04/12)" ;

	private static final long serialVersionUID = 563220130412L ;

	private TagBuffer tag = new TagBuffer( "iTask" ) ;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.8.1.0 (2014/11/07) HTML5対応。javaScriptで、BODYがないと入れ子になってしまう。
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		tag.setBody( "<!-- -->" );		// 5.8.1.0 (2014/11/07) HTML5対応。
		jspPrint( tag.makeTag() );

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iTask" );
	}

	/**
	 * 【TAG】左右２分割で、固定したいカラム数を指定します(必須)。
	 *
	 * @og.tag
	 * １段組でも２段組でも、固定したいカラム数を指定します。
	 *
	 * @param   fixedCols 固定したいカラム数
	 */
	public void setFixedCols( final String fixedCols ) {
		tag.add( "fixedCols",StringUtil.nval( getRequestParameter( fixedCols ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "iTask" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
