/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.opengion.fukurou.model.EventReader_XLS;
import org.opengion.fukurou.model.EventReader_XLSX;
import org.opengion.fukurou.model.TableModelHelper;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.FileInfo;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModelUtil;

/**
 * POI による、EXCELバイナリファイルを読み取る実装クラスです。
 *
 * ファイル名、シート名を指定して、データを読み取ることが可能です。
 * 第一カラムが # で始まる行は、コメント行なので、読み飛ばします。
 * カラム名の指定行で、カラム名が null の場合は、その列は読み飛ばします。
 *
 * 入力形式は、openXML形式にも対応しています。
 * ファイルの内容に応じて、.xlsと.xlsxのどちらで読み取るかは、内部的に
 * 自動判定されます。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規作成
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * @og.rev 5.9.0.0 (2015/09/04) EventReaderを利用する対応
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableReader_Excel extends TableReader_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.8.2 (2012/11/09)" ;

	private String  filename		= null;		// 3.5.4.3 (2004/01/05)
	private String  sheetName		= null;		// 3.5.4.2 (2003/12/15)
	private String  sheetNos		= null;		// 5.5.7.2 (2012/10/09)

	private String  constKeys		= null;		// 5.5.8.2 (2012/11/09) 固定値となるカラム名(CSV形式)
	private String  constAdrs		= null;		// 5.5.8.2 (2012/11/09) 固定値となるアドレス(行-列,行-列,・・・)
	private String	nullBreakClm	= null;		// 5.5.8.2 (2012/11/09) 取込み条件/Sheet BREAK条件

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.6.0 (2010/05/01) columns 処理 追加
	 * @og.rev 5.1.6.0 (2010/05/01) skipRowCountの追加
	 * @og.rev 5.1.8.0 (2010/07/01) Exception をきちっと記述(InvalidFormatException)
	 * @og.rev 5.2.1.0 (2010/10/01) setTableColumnValues メソッドを経由して、テーブルにデータをセットする。
	 * @og.rev 5.5.1.2 (2012/04/06) HeaderData を try の上にだし、エラーメッセージを取得できるようにする。
	 * @og.rev 5.5.7.2 (2012/10/09) sheetNos 追加による複数シートのマージ読み取りサポート
	 * @og.rev 5.5.8.2 (2012/11/09) HeaderData に デバッグフラグを渡します。
	 * @og.rev 5.9.0.0 (2015/09/04) EventReader利用のため、ロジックをV6風に書き換えます
	 *
	 * @see #isExcel()
	 */
	@Override
/*
  	public void readDBTable() {
		InputStream  in = null;
		HeaderData data = null;		// 5.5.1.2 (2012/04/06)
		try {
			boolean isDebug = isDebug();			// 5.5.7.2 (2012/10/09) デバッグ情報

			if( isDebug ) { System.out.println( " Filename=" + filename ) ; }

			in = new FileInputStream(filename);

			Workbook wb = WorkbookFactory.create(in);
			Sheet[] sheets ;									// 5.5.7.2 (2012/10/09) 配列に変更

			if( isDebug ) { wb = ExcelUtil.activeWorkbook( wb ); }		// デバッグモード時には、エクセルのアクティブセル領域のみにシュリンクを行う

			// 5.5.7.2 (2012/10/09) 複数シートのマージ読み取り。 sheetNos の指定が優先される。
			if( sheetNos != null && sheetNos.length() > 0 ) {
				String[] sheetList = StringUtil.csv2ArrayExt( sheetNos , wb.getNumberOfSheets()-1 );	// 最大シート番号は、シート数-1
				sheets = new Sheet[sheetList.length];
				for( int i=0; i<sheetList.length; i++ ) {
					sheets[i] = wb.getSheetAt( Integer.parseInt( sheetList[i] ) );
				}
			}
			else if( sheetName != null && sheetName.length() > 0 ) {
				Sheet sheet = wb.getSheet( sheetName );
				if( sheet == null ) {
					String errMsg = "対応するシートが存在しません。 Sheet=[" + sheetName + "]" ;
					throw new HybsSystemException( errMsg );
				}
				sheets = new Sheet[] { sheet };
			}
			else {
				Sheet sheet = wb.getSheetAt(0);
				sheets = new Sheet[] { sheet };
			}

			boolean  nameNoSet = true;
			table = DBTableModelUtil.newDBTable();

			int numberOfRows = 0;
			data = new HeaderData();				// 5.5.1.2 (2012/04/06)

			data.setDebug( isDebug );				// 5.5.8.2 (2012/11/09)

			// 5.1.6.0 (2010/05/01) columns 処理
			data.setUseNumber( isUseNumber() );

			// 5.5.8.2 (2012/11/09) 固定値となるカラム名(CSV形式)とアドレス(行-列,行-列,・・・)を設定
			data.setSheetConstData( constKeys,constAdrs );

			int nullBreakClmAdrs = -1;						// 5.5.8.2 (2012/11/09) nullBreakClm の DBTableModel上のアドレス。-1 は、未使用
			if( data.setColumns( columns ) ) {
				nameNoSet = false;
				table.init( data.getColumnSize() );
				setTableDBColumn( data.getNames() ) ;
				nullBreakClmAdrs = table.getColumnNo( nullBreakClm, false );	// 5.5.8.2 (2012/11/09) カラム番号取得。存在しなければ -1 を返す。
			}

			int skip = getSkipRowCount();					// 5.1.6.0 (2010/05/01)
			// 5.5.7.2 (2012/10/09) 複数シートのマージ読み取り。 
			for( int i=0; i<sheets.length; i++ ) {			// 5.5.7.2 (2012/10/09) シート配列を処理します。
				Sheet sheet = sheets[i] ;					// 5.5.7.2 (2012/10/09)

				data.setSheetConstValues( sheet );				// 5.5.8.2 (2012/11/09) シート単位に固定カラムの値をキャッシュする。

				int nFirstRow = sheet.getFirstRowNum();
				if( nFirstRow < skip ) { nFirstRow = skip; }	// 5.1.6.0 (2010/05/01)
				int nLastRow  = sheet.getLastRowNum();
				if( isDebug ) {		// 5.5.7.2 (2012/10/09) デバッグ情報
					System.out.println( " Debug: 行連番=" + numberOfRows + " : Sheet= " + sheet.getSheetName() + " , 開始=" + nFirstRow + " , 終了=" + nLastRow );
				}
				for( int nIndexRow = nFirstRow; nIndexRow <= nLastRow; nIndexRow++) {
	//				HSSFRow oRow = sheet.getRow(nIndexRow);
					Row oRow = sheet.getRow(nIndexRow);
					if( data.isSkip( oRow ) ) { continue; }
					if( nameNoSet ) {
						nameNoSet = false;
						table.init( data.getColumnSize() );
						setTableDBColumn( data.getNames() ) ;
						nullBreakClmAdrs = table.getColumnNo( nullBreakClm, false );	// 5.5.8.2 (2012/11/09) カラム番号取得。存在しなければ -1 を返す。
					}

					if( numberOfRows < getMaxRowCount() ) {
						String[] tblData = data.row2Array( oRow );			// 5.5.8.2 (2012/11/09) nullBreakClm の判定のため、一旦配列に受ける。
						if( nullBreakClmAdrs >= 0 && ( tblData[nullBreakClmAdrs] == null || tblData[nullBreakClmAdrs].isEmpty() ) ) {
							break;		// nullBreakClm が null の場合は、そのSheet処理を中止する。
						}
						setTableColumnValues( tblData );					// 5.5.8.2 (2012/11/09)
						numberOfRows ++ ;
					}
					else {
						table.setOverflow( true );
					}
				}

				// 最後まで、#NAME が見つから無かった場合
				if( nameNoSet ) {
					String errMsg = "最後まで、#NAME が見つかりませんでした。"
									+ HybsSystem.CR
									+ "ファイルが空か、もしくは損傷している可能性があります。"
									+ HybsSystem.CR ;
					throw new HybsSystemException( errMsg );
				}
			}
		}
		catch ( IOException ex ) {
			String errMsg = "ファイル読込みエラー[" + filename + "]"  ;
			if( data != null ) { errMsg = errMsg + data.getLastCellMsg(); }		// 5.5.1.2 (2012/04/06)
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		// 5.1.8.0 (2010/07/01) Exception をきちっと記述
		catch (InvalidFormatException ex) {
			String errMsg = "ファイル形式エラー[" + filename + "]"  ;
			if( data != null ) { errMsg = errMsg + data.getLastCellMsg(); }		// 5.5.1.2 (2012/04/06)
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			Closer.ioClose( in );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}
	}
*/

	public void readDBTable() {
		boolean isDebug = isDebug();			// 5.5.7.2 (2012/10/09) デバッグ情報
		table = DBTableModelUtil.newDBTable();
		
		final TableModelHelper helper = new TableModelHelper() {
			private boolean[] useShtNo;			// 6.1.0.0 (2014/12/26) 読み取り対象のシート管理

			/**
			 * シートの数のイベントが発生します。
			 *
			 * 処理の開始前に、シートの数のイベントが発生します。
			 * これを元に、処理するシート番号の選別が可能です。
			 * 初期実装は、されていません。
			 *
			 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
			 * @og.rev 6.2.6.0 (2015/06/19) #csv2ArrayExt(String,int)の戻り値を、文字列配列から数字配列に変更。
			 *
			 * @param   size  シートの数
			 */
			@Override
			public void sheetSize( final int size ) {
				if( isDebug() ) { System.out.println( " sheetSize=" + size ) ; }
				// 5.5.7.2 (2012/10/09) 複数シートのマージ読み取り。 sheetNos の指定が優先される。
				useShtNo = new boolean[size];			// シート数だけ、配列を作成する。
				if( sheetNos != null && sheetNos.length() > 0 ) {
					Integer[] sheetList = StringUtil.csv2ArrayExt2( sheetNos , size-1 );	// 最大シート番号は、シート数-1
					for( int i=0; i<sheetList.length; i++ ) {
						useShtNo[sheetList[i]] = true;	// 読み取り対象のシート番号のみ、ture にセット
					}
				}
				else {
					useShtNo[0] = true;					// 一番目のシート
				}
			}

			/**
			 * シートの読み取り開始時にイベントが発生します。
			 *
			 * 新しいシートの読み取り開始毎に、１回呼ばれます。
			 * 戻り値が、true の場合は、そのシートの読み取りを継続します。
			 * false の場合は、そのシートの読み取りは行わず、次のシートまで
			 * イベントは発行されません。
			 *
			 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
			 *
			 * @param   shtNm  シート名
			 * @param   shtNo  シート番号(0～)
			 * @return  true:シートの読み取り処理を継続します/false:このシートは読み取りません。
			 */
			@Override
			public boolean startSheet( final String shtNm,final int shtNo ) {
	//			if( isDebug ) { System.out.println( " Sheet[" + shtNo + "]=" + shtNm ) ; }
				super.startSheet( shtNm , shtNo );		// cnstData の呼び出しの為。無しで動くようにしなければ…

				return  ( useShtNo != null && useShtNo[shtNo] ) ||
						( sheetName != null && sheetName.equalsIgnoreCase( shtNm ) ) ;
			}

			/**
			 * カラム名配列がそろった段階で、イベントが発生します。
			 *
			 * openGion での標準的な処理は、colNo==0 の時に、val の先頭が、#NAME
			 * で始まるレコードを、名前配列として認識します。
			 * #value( String,int,int ) で、この #NAME だけは、継続処理されます。
			 * その上で、#NAME レコードが終了した時点で、カラム名配列が完成するので
			 * そこで初めて、このメソッドが呼ばれます。
			 *
			 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
			 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
			 *
			 * @param   names  カラム名配列(可変長引数)
			 * @see		#value( String,int,int )
			 */
			@Override
			public void columnNames( final String[] names ) {
				setTableDBColumn( names ) ;
			}

			/**
			 * row にあるセルのオブジェクト値がそろった段階で、イベントが発生します。
			 *
			 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
			 * @og.rev 6.2.1.0 (2015/03/13) setTableColumnValuesに、行番号を引数に追加
			 *
			 * @param   vals    文字列値の１行分の配列
			 * @param   rowNo   行番号(0～)
			 */
			@Override
			public void values( final String[] vals,final int rowNo ) {
	//			if( isDebug && rowNo % 100 == 0 ) { System.out.println( "   rowNo=" + rowNo ) ; }
				setTableColumnValues( vals );		// 6.2.1.0 (2015/03/13)
			}
		};
		
		helper.setDebug( isDebug );							
		helper.setConstData( constKeys , constAdrs );		// 外部から固定値情報を指定。
		helper.setNames( columns , isUseNumber() );			// 外部からカラム名配列を指定。
		helper.setSkipRowCount( getSkipRowCount() );		// 外部からスキップ行数を指定。
		helper.setNullBreakClm( nullBreakClm );				// 外部からnullBreakClmを指定。
//		helper.setNullSkipClm( nullSkipClm );				// 外部からnullSkipClmを指定。 V5未対応
		
		File file = new File(filename);
		
		// 6.2.4.2 (2015/05/29) POIUtil を使わず、EventReader_XLS、EventReader_XLSX を直接呼び出します。
		final String SUFIX = FileInfo.getSUFIX( file );
		if( "xls".equalsIgnoreCase( SUFIX ) ) {
			new EventReader_XLS().eventReader( file,helper );
		}
		else if( "xlsx".equalsIgnoreCase( SUFIX ) || "xlsm".equalsIgnoreCase( SUFIX ) ) {
			new EventReader_XLSX().eventReader( file,helper );
		}
		else {
			final String errMsg = "拡張子は、xls,xlsx,xlsm にしてください。[" + file + "]" ;
			throw new RuntimeException( errMsg );
		}
		
		// 最後まで、#NAME が見つから無かった場合
		if( !helper.isNameSet() ) {
			final String errMsg = "最後まで、#NAME が見つかりませんでした。"
							+ "ファイル形式が異なるか、もしくは損傷している可能性があります。"
							+ "Class=[Excel], File=[" + file + "]";
			throw new HybsSystemException( errMsg );
		}

		if( isDebug ) { System.out.println( "  TableReader End." ) ; }
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取る要に変更します。
	 * @og.rev 4.0.0.0 (2006/09/31) UnsupportedOperationException を発行します。
	 *
	 * @param   reader 各形式のデータ(使用していません)
	 */
	@Override
	public void readDBTable( final BufferedReader reader ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModelのデータとしてEXCELファイルを読み込むときのシート名を設定します。
	 * これにより、複数の形式の異なるデータを順次読み込むことや、シートを指定して
	 * 読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * のでご注意ください。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName シート名
	 */
	@Override
	public void setSheetName( final String sheetName ) {
		this.sheetName = sheetName;
	}

	/**
	 * EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、カンマ区切りで、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 *
	 * @param   sheetNos EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	@Override
	public void setSheetNos( final String sheetNos ) {
		this.sheetNos = sheetNos;
	}

	/**
	 * EXCELファイルを読み込むときのシート単位の固定値を設定するためのカラム名とアドレスを指定します。
	 * カラム名は、カンマ区切りで指定します。
	 * 対応するアドレスを、EXCEL上の行-列を０から始まる整数でカンマ区切りで指定します。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 *
	 * @param   constKeys 固定値となるカラム名(CSV形式)
	 * @param   constAdrs 固定値となるアドレス(行-列,行-列,・・・)
	 */
	@Override
	public void setSheetConstData( final String constKeys,final String constAdrs ) {
		this.constKeys = constKeys;
		this.constAdrs = constAdrs;
	}

	/**
	 * ここに指定されたカラム列に NULL が現れた時点で読み取りを中止します。
	 *
	 * これは、指定のカラムは必須という事を条件に、そのレコードだけを読み取る処理を行います。
	 * 複数Sheetの場合は、次のSheetを読みます。
	 * 現時点では、Excel の場合のみ有効です。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 *
	 * @param   clm カラム列
	 */
	@Override
	public void setNullBreakClm( final String clm ) {
		nullBreakClm = clm;
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか(常にtrue)
	 */
	@Override
	public boolean isExcel() {
		return true;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	@Override
	public void setFilename( final String filename ) {
		this.filename = filename;
		if( filename == null ) {
			String errMsg = "ファイル名が指定されていません。" ;
			throw new HybsSystemException( errMsg );
		}
	}
}

/**
 * EXCEL ネイティブのデータを処理する ローカルクラスです。
 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
 * 行情報(Row)から、カラムの配列の取得などを行います。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規追加
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   儲
 * @since    JDK5.0,
 */
class HeaderData {
	private String[] names ;
	private int[]    index; // 4.3.4.0 (2008/12/01) POI3.2対応
	private int		 columnSize = 0;
	private boolean  nameNoSet = true;
	private boolean  useNumber = true;
	private boolean  isDebug   = false;		// 5.5.8.2 (2012/11/09)

	private String[] orgNames ;			// 5.5.1.2 (2012/04/06) オリジナルのカラム名
	private Cell     lastCell = null;	// 5.5.1.2 (2012/04/06) 最後に実行しているセルを保持(エラー時に使用する。)

	// 5.5.8.2 (2012/11/09) 固定値のカラム名、DBTableModelのアドレス、Sheetの行-列番号
	private int		 cnstLen = 0;		// 初期値=0 の場合は、固定値を使わないという事。
	private String[] cnstKeys ;
	private int[]	 cnstIndx ;
	private int[]    cnstRowNo;
	private int[]    cnstClmNo;
	private String[] cnstVals ;			// Sheet単位の固定値のキャッシュ(シートの最初に値を取得して保持しておく)

	/**
	 * デバッグ情報を、出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * 初期値は、false(出力しない) です。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規作成
	 *
	 * @param	isDebug	デバッグ情報 [true:出力する/false:出力しない]
	 */
	void setDebug( final boolean isDebug ) {
		this.isDebug = isDebug ;
	}

	/**
	 * 行番号情報を、使用しているかどうか[true/false]を指定します(初期値:true)。
	 *
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	useNumber	行番号情報 [true:使用している/false:していない]
	 */
	void setUseNumber( final boolean useNumber ) {
		this.useNumber = useNumber ;
	}

	/**
	 * 固定値となるカラム名(CSV形式)と、constAdrs 固定値となるアドレス(行-列,行-列,・・・)を設定します。
	 *
	 * アドレスは、EXCEL上の行-列をカンマ区切りで指定します。
	 * 行列は、EXCELオブジェクトに準拠するため、０から始まる整数です。
	 * 0-0 ⇒ A1 , 1-0 ⇒ A2 , 0-1 ⇒ B1 になります。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * 5.7.6.3 (2014/05/23) より、
	 *   ①EXCEL表記に準拠した、A1,A2,B1 の記述も処理できるように対応します。
	 *     なお、A1,A2,B1 の記述は、必ず、英字1文字＋数字 にしてください。(A～Zまで)
	 *   ②処理中のEXCELシート名をカラムに割り当てるために、"SHEET" という記号に対応します。
	 * 例えば、sheetConstKeys="CLM,LANG,NAME" とし、sheetConstAdrs="0-0,A2,SHEET" とすると、
	 * NAMEカラムには、シート名を読み込むことができます。
	 * これは、内部処理の簡素化のためです。
	 *
	 * ちなみに、EXCELのセルに、シート名を表示させる場合の関数は、下記の様になります。
	 * =RIGHT(CELL("filename",$A$1),LEN(CELL("filename",$A$1))-FIND("]",CELL("filename",$A$1)))
	 *
	 * @param	constKeys	固定値となるカラム名(CSV形式)
	 * @param	constAdrs	固定値となるアドレス(行-列,行-列,・・・)
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 * @og.rev 5.7.6.3 (2014/05/23) EXCEL表記(A2,B1等)の対応と、特殊記号(SHEET)の対応
	 */
	void setSheetConstData( final String constKeys,final String constAdrs ) {
		if( constKeys == null || constKeys.isEmpty() ) {
			return ;
		}

		cnstKeys  = constKeys.split( "," );
		cnstLen   = cnstKeys.length;
		cnstIndx  = new int[cnstLen];
		cnstRowNo = new int[cnstLen];
		cnstClmNo = new int[cnstLen];

		String[] row_col = constAdrs.split( "," ) ;
		cnstRowNo = new int[cnstLen];
		cnstClmNo = new int[cnstLen];
		for( int j=0; j<cnstLen; j++ ) {
			cnstKeys[j] = cnstKeys[j].trim();		// 前後の不要なスペースを削除
			String rowcol = row_col[j].trim();		// 前後の不要なスペースを削除

			// 5.7.6.3 (2014/05/23) EXCEL表記(A2,B1等)の対応と、特殊記号(SHEET)の対応
			int sep = rowcol.indexOf( '-' );
			if( sep > 0 ) {
				cnstRowNo[j] = Integer.parseInt( rowcol.substring( 0,sep ) );
				cnstClmNo[j] = Integer.parseInt( rowcol.substring( sep+1 ) );
			}
			else {
				if( "SHEET".equalsIgnoreCase( rowcol ) ) {		// "SHEET" 時は、cnstRowNo をマイナスにしておきます。
					cnstRowNo[j] = -1 ;
					cnstClmNo[j] = -1 ;
				}
				else if( rowcol.length() >= 2 ) {
					cnstRowNo[j] = Integer.parseInt( rowcol.substring( 1 ) ) -1;	// C6 の場合、RowNoは、6-1=5
					cnstClmNo[j] = rowcol.charAt(0) - 'A' ;							// C6 の場合、'C'-'A'=2
				}
			}

			if( isDebug ) {
				System.out.println( " Debug: constKey=" + cnstKeys[j] + " : RowNo=" + cnstRowNo[j] + " , ClmNo=" + cnstClmNo[j] );
			}
		}
	}

	/**
	 * カラム名を外部から指定します。
	 * カラム名が、NULL でなければ、＃NAME より、こちらが優先されます。
	 * カラム名は、順番に、指定する必要があります。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 * @og.rev 5.5.8.2 (2012/11/09) 固定値取得用の cnstIndx の設定を行う。
	 *
	 * @param	columns	EXCELのカラム列(CSV形式)
	 *
	 * @return true:処理実施/false:無処理
	 */
	boolean setColumns( final String columns ) {
		if( columns != null && columns.length() > 0 ) {
			names = StringUtil.csv2Array( columns );
			columnSize = names.length ;
			index = new int[columnSize];
			int adrs = useNumber ? 1:0 ;	// useNumber =true の場合は、１件目(No)は読み飛ばす。
			// 5.5.8.2 (2012/11/09) 固定値取得用の cnstIndx の設定を行う。
			for( int i=0; i<columnSize; i++ ) {
				index[i] = adrs++;
				for( int j=0; j<cnstLen; j++ ) {
					if( names[i].equalsIgnoreCase( cnstKeys[j] ) ) {
						cnstIndx[j] = index[i];
					}
				}
			}
			nameNoSet = false;

			return true;
		}
		return false;
	}

	/**
	 * EXCEL ネイティブのデータを処理する ローカルクラスです。
	 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
	 * 行情報(Row)から、カラムの配列の取得などを行います。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 *
	 * @return true:コメント行/false:通常行
	 */
	boolean isSkip( Row oRow ) {
		if( oRow == null ) { return true; }

		int nFirstCell = oRow.getFirstCellNum();
		Cell oCell = oRow.getCell(nFirstCell);
		String strText =  getValue( oCell );
		if( strText != null && strText.length() > 0 ) {
			if( nameNoSet ) {
				if( "#Name".equalsIgnoreCase( strText ) ) {
					makeNames( oRow );
					nameNoSet = false;
					return true;
				}
				else if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。"
									+ HybsSystem.CR
									+ "可能性として、ファイルが、ネイティブExcelでない事が考えられます。"
									+ HybsSystem.CR ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
			}
		}

		return nameNoSet ;
	}

	/**
	 * EXCEL ネイティブの行情報(Row)からカラム名情報を取得します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 * @og.rev 5.1.6.0 (2010/05/01) useNumber(行番号情報を、使用している(true)/していない(false)を指定)
	 * @og.rev 5.1.6.0 (2010/05/01) useNumber(行番号情報を、使用している(true)/していない(false)を指定)
	 * @og.rev 5.5.1.2 (2012/04/06) オリジナルのカラム名を取得
	 * @og.rev 5.5.8.2 (2012/11/09) 固定値取得用の cnstIndx の設定を行う。
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 */
	private void makeNames( final Row oRow ) {
		// 先頭カラムは、#NAME 属性行であるかどうかを、useNumber で判定しておく。
		short nFirstCell = (short)( useNumber ? 1:0 );
		short nLastCell  = oRow.getLastCellNum();

		orgNames = new String[nLastCell+1];		// 5.5.1.2 (2012/04/06) オリジナルのカラム名を取得

		int maxCnt = nLastCell - nFirstCell;
		String[] names2 = new String[maxCnt];
		int[]    index2 = new int[maxCnt];

		// 先頭カラムは、#NAME 属性行である。++ で、一つ進めている。
		// 先頭カラムは、#NAME 属性行であるかどうかを、useNumber で判定しておく。
		for( int nIndexCell = nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
			Cell oCell = oRow.getCell(nIndexCell);
			String strText = getValue( oCell );

			orgNames[nIndexCell] = strText;		// 5.5.1.2 (2012/04/06) オリジナルのカラム名を取得

			// #NAME 行が、ゼロ文字列の場合は、読み飛ばす。
			if( strText != null && strText.length() > 0 ) {
				names2[columnSize] = strText;
				index2[columnSize] = nIndexCell;
				columnSize++;
			}
		}

		// #NAME を使用しない場合：no欄が存在しないケース
		if( maxCnt == columnSize ) {
			names = names2;
			index = index2;
		}
		else {
			names = new String[columnSize];
			index = new int[columnSize];
			System.arraycopy(names2, 0, names, 0, columnSize);
			System.arraycopy(index2, 0, index, 0, columnSize);
		}

		// 5.5.8.2 (2012/11/09) 固定値取得用の cnstIndx の設定を行う。
		if( cnstLen > 0 ) {
			for( int i=0; i<columnSize; i++ ) {
				for( int j=0; j<cnstLen; j++ ) {
					if( names[i].equalsIgnoreCase( cnstKeys[j] ) ) {
						cnstIndx[j] = index[i];
					}
				}
			}
		}
	}

	/**
	 * カラム名情報を返します。
	 * ここでは、内部配列をそのまま返します。
	 *
	 * @return String[] カラム列配列情報
	 */
	String[] getNames() {
		return names;
	}

	/**
	 * カラムサイズを返します。
	 *
	 * @return	カラムサイズ
	 */
	int getColumnSize() {
		return columnSize;
	}

	/**
	 * Sheet単位の固定値のキャッシュ(シートの最初に値を取得して保持しておく)を設定します。
	 * これは、シートチェンジの最初に一度呼び出しておくことで、それ以降の列取得時に
	 * 固定値を利用することで処理速度向上を目指します。
	 *
	 * "SHEET" が指定された場合は、cnstRowNo[j]=-1 が設定されている。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規作成
	 * @og.rev 5.7.6.3 (2014/05/23) 特殊記号(SHEET)の対応
	 *
	 * @param sheet Sheet EXCELのSheetオブジェクト
	 */
	void setSheetConstValues( final Sheet sheet ) {
		cnstVals = new String[cnstLen];
		for( int j=0; j<cnstLen; j++ ) {
			// 5.7.6.3 (2014/05/23) 特殊記号(SHEET)の対応
			if( cnstRowNo[j] < 0 ) {
				cnstVals[j] = sheet.getSheetName() ;
			}
			else {
				Row  oRow  = sheet.getRow( cnstRowNo[j] );
				Cell oCell = oRow.getCell( cnstClmNo[j] );
				cnstVals[j] = getValue( oCell );
			}

			if( isDebug ) {
				System.out.println( " Debug: Sheet=" + sheet.getSheetName() + " : RowNo=" + cnstRowNo[j] + " , ClmNo=" + cnstClmNo[j] + " , " + cnstKeys[j] + "=" + cnstVals[j] );
			}
		}
	}

	/**
	 * カラム名情報を返します。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 固定値の設定を行う。
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 *
	 * @return String[] カラム列配列情報
	 */
	String[] row2Array( final Row oRow ) {
		if( nameNoSet ) {
			String errMsg = "#NAME が見つかる前にデータが見つかりました。";
			throw new HybsSystemException( errMsg );
		}

		String[] data = new String[columnSize];
		for( int i=0;i<columnSize; i++ ) {
			Cell oCell = oRow.getCell( index[i] );
			data[i] = getValue( oCell );
		}

		// 5.5.8.2 (2012/11/09) 固定値の設定を行う。
		for( int j=0; j<cnstLen; j++ ) {
			data[cnstIndx[j]] = cnstVals[j];
		}
		return data;
	}

	/**
	 * セルオブジェクト(Cell)から値を取り出します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 取り出し方法を少し修正
	 * @og.rev 5.5.1.2 (2012/04/06) フォーマットセルを実行して、その結果を再帰的に処理する。
	 *
	 * @param oCell Cell EXCELのセルオブジェクト
	 *
	 * @return	セルの値
	 */
	private String getValue( final Cell oCell ) {
		lastCell = oCell;	// 5.5.1.2 (2012/04/06) 今から実行するセルを取得しておきます。

		if( oCell == null ) { return null; }

		String strText = "";
		RichTextString richText;
		int nCellType = oCell.getCellType();
		switch(nCellType) {
			case Cell.CELL_TYPE_NUMERIC:
					strText = getNumericTypeString( oCell );
					break;
			case Cell.CELL_TYPE_STRING:
	// POI3.0		strText = oCell.getStringCellValue();
					richText = oCell.getRichStringCellValue();
					if( richText != null ) {
						strText = richText.getString();
					}
					break;
			case Cell.CELL_TYPE_FORMULA:
	// POI3.0		strText = oCell.getStringCellValue();
					// 5.5.1.2 (2012/04/06) フォーマットセルを実行して、その結果を再帰的に処理する。
					Workbook wb = oCell.getSheet().getWorkbook();
					CreationHelper crateHelper = wb.getCreationHelper();
					FormulaEvaluator evaluator = crateHelper.createFormulaEvaluator();

					try {
						strText = getValue(evaluator.evaluateInCell(oCell));
					}
					catch ( Throwable th ) {
						String errMsg = "セルフォーマットが解析できません。[" + oCell.getCellFormula() + "]"
									+ getLastCellMsg();
						throw new HybsSystemException( errMsg,th );
					}
					break;
			case Cell.CELL_TYPE_BOOLEAN:
					strText = String.valueOf(oCell.getBooleanCellValue());
					break;
			case Cell.CELL_TYPE_BLANK :
			case Cell.CELL_TYPE_ERROR:
					break;
			default :
				break;
		}
		return strText.trim();
	}

	/**
	 * セル値が数字の場合に、数字か日付かを判断して、対応する文字列を返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param oCell Cell
	 *
	 * @return	数字の場合は、文字列に変換した結果を、日付の場合は、"yyyyMMddHHmmss" 形式で返します。
	 */
	private String getNumericTypeString( final Cell oCell ) {
		final String strText ;

		double dd = oCell.getNumericCellValue() ;
		if( DateUtil.isCellDateFormatted( oCell ) ) {
			strText = HybsDateUtil.getDate( DateUtil.getJavaDate( dd ).getTime() , "yyyyMMddHHmmss" );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
		}
		else {
			NumberFormat numFormat = NumberFormat.getInstance();
			if( numFormat instanceof DecimalFormat ) {
				((DecimalFormat)numFormat).applyPattern( "#.####" );
			}
			strText = numFormat.format( dd );
		}
		return strText ;
	}

	/**
	 * 最後に実行しているセル情報を返します。
	 *
	 * エラー発生時に、どのセルでエラーが発生したかの情報を取得できるようにします。
	 *
	 * @og.rev 5.5.1.2 (2012/04/06) 新規追加
	 * @og.rev 5.5.8.2 (2012/11/09) エラー情報に、シート名も追加
	 *
	 * @return	最後に実行しているセル情報の文字列
	 */
	String getLastCellMsg() {
		String lastMsg = null;

		if( lastCell != null ) {
			int rowNo = lastCell.getRowIndex();
			int celNo = lastCell.getColumnIndex();
			int no = lastCell.getColumnIndex();
			String shtNm = lastCell.getSheet().getSheetName();


			lastMsg = "Sheet=" + shtNm + ", Row=" + rowNo + ", Cel=" + celNo ;
			if( orgNames != null && orgNames.length < no ) {
				lastMsg = lastMsg + ", NAME=" + orgNames[no] ;
			}
		}
		return lastMsg;
	}
}
