/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import com.sun.tools.doclets.Taglet;

/**
 * ソースコメントから、Javadoc を作成する場合のカスタムタグ情報を作成する
 * Taglet インターフェースのAbstract 実装クラスを作成します。
 * デフォルトの タグレットとしての初期値を設定します。
 * つまり、isInlineTag() のみ false を返すメソッドを定義してあります。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public abstract class AbstractTaglet implements Taglet {

	/**
	 * このタグレットがフィールドドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return フィールドドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inField() {
		return true;
	}

	/**
	 * このタグレットがコンストラクタドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return コンストラクタドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inConstructor() {
		return true;
	}

	/**
	 * このタグレットがメソッドドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return メソッドドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inMethod() {
		return true;
	}

	/**
	 * このタグレットが概要ドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return 概要ドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inOverview() {
		return true;
	}

	/**
	 * このタグレットがパッケージドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return パッケージドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inPackage() {
		return true;
	}

	/**
	 * このタグレットがタイプドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return タイプドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inType() {
		return true;
	}

	/**
	 * このタグレットがインラインタグで
	 * 使用されている場合、true を返します。
	 * そうでない場合については false が設定されます。
	 *
	 * @return インラインタグの場合は true、そうでない場合は false を返します
	 */
	public boolean isInlineTag() {
		return false;
	}

	/**
	 * このタグレットがインラインタグで {&#064;link XXXX YYYY} を処理するように
	 * 用意された、カスタムメソッドです。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 *
	 * @param	in	オリジナルの文字列
	 *
	 * @return	インラインタグの link を処理した結果の文字列
	 */
	protected String link( final String in ) {
		if( in == null ) { return ""; }
		int index = in.indexOf( "{@link" );
		if( index < 0 ) { return in; }

		StringBuilder rtn = new StringBuilder();
		int start = 0;
		while( index >= 0 ) {
			int end = in.indexOf( '}',index );

			// {@ より前方の文字列を追加
			if( index > 0 ) { rtn.append( in.substring( start,index ) ); }

			// {@link XXXX YY} の XXXX YY部分を処理
			String val = in.substring( index+7,end );
			if( val != null ) {
				int sp = val.indexOf( ' ' );
				String xxx = val.substring( 0,sp ).trim();	// 前半：アドレス変換
				String yyy = val.substring( sp ).trim();	// 後半：ラベル
				String zzz = xxx.replace( '.','/' );
				rtn.append( "<a href=\"../../../../" ).append( zzz ).append( ".html\" " )
					.append( "title=\"" ).append( xxx ).append( "\">" )
					.append( yyy ).append( "</a>" );
			}

			start = end+1 ;
			index = in.indexOf( "{@",start );
		}
		rtn.append( in.substring( start ) );

		return rtn.toString();
	}
}
