/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

// import java.text.DateFormat;
// import java.text.SimpleDateFormat;
// import java.util.Locale;
// import java.util.Calendar;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;

/**
 * SystemParameter は、{&#064;XXXX} 文字列を処理するクラスです。
 * このクラスでは、{&#064;XXXX} 文字列を別の文字列と置き換えることや、
 * 予め予約されている予約語 {&#064;DATE.XXXX} 文字列を置き換えます。
 * 通常の {&#064;XXXX} 文字列の置き換えは、キーと値のペアを、HybsEntry オブジェクトに
 * セットして、その配列を受け取って処理します。
 *
 * 以下の値はあらかじめ、動的に作成されます。
 * ・DATE.YMD       ８byte の今日のシステム日付(yyyyMMdd)
 * ・DATE.YMDH    １４byte の今日のシステム日時(yyyyMMddHHmmss)
 * ・DATE.HMS       ６byte の今日のシステム時間(HHmmss)
 *
 * @og.group ユーティリティ
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class SystemParameter {

//	private final Map<String,String> sysMap = new HashMap<String,String>();
	/** 改行コード */
	public static final String CR = System.getProperty("line.separator");	// 5.1.9.0 (2010/08/01) 追加

	// 	5.5.7.2 (2012/10/09) HybsDateUtil を利用するため、削除します。
// 	private static final Map<String,String>	DATE_FORMAT = new HashMap<String,String>();	// 5.3.4.0 (2011/04/01)
// 	static {
// 		DATE_FORMAT.put( "SYS.YMD"		,"yyyyMMdd"				);
// 		DATE_FORMAT.put( "SYS.YMDH" 	,"yyyyMMddHHmmss"		);
// 		DATE_FORMAT.put( "SYS.HMS"		,"HHmmss"				);
// 		DATE_FORMAT.put( "DATE.YMD" 	,"yyyyMMdd" 			);
// 		DATE_FORMAT.put( "DATE.Y2MD"	,"yyMMdd"				);
// 		DATE_FORMAT.put( "DATE.YM"		,"yyyyMM"				);
// 		DATE_FORMAT.put( "DATE.HMS" 	,"HHmmss"				);
// 		DATE_FORMAT.put( "DATE.YMDHMS"	,"yyyyMMddHHmmss"		);
// 		DATE_FORMAT.put( "DATE.YMDF"	,"yyyy/MM/dd"			);
// 		DATE_FORMAT.put( "DATE.Y2MDF"	,"yy/MM/dd" 			);
// 		DATE_FORMAT.put( "DATE.YMF" 	,"yyyy/MM"				);
// 		DATE_FORMAT.put( "DATE.HMSF"	,"HH:mm:ss" 			);
// 		DATE_FORMAT.put( "DATE.YMDHMSF" ,"yyyy/MM/dd/ HH:mm:ss" );
// 		DATE_FORMAT.put( "DATE.EEE" 	,"EEE"					);
// 	}

	private final String	original ;
	// 5.5.7.2 (2012/10/09) Calendarオブジェクトから、String時刻に変更。
//	private final Calendar	rightNow;										// 5.3.4.0 (2011/04/01)
//	private final String	RIGHT_NOW = HybsDateUtil.getDate( "yyyyMMdd" );	// 5.7.4.1 (2014/03/14) 廃止

	private final String[] clms;
	private final String[] formats;

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) パース方法見直し(StringTokenizerでは、{&#064;XXXX}が連続してある場合に対応できない)
	 * @og.rev 5.3.2.0 (2011/02/01) original データを、パース結果を利用するように修正する。
	 * @og.rev 5.3.4.0 (2011/04/01) {&#064;DATE.XXXX} を処理できるように機能追加
	 * @og.rev 5.3.5.0 (2011/05/01) {&#064;SYS.XXXX} は、廃止
	 * @og.rev 5.5.7.2 (2012/10/09) rightNow をCalendarオブジェクト ではなく、String時刻とします。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	orig	変換する文字列(オリジナル)
	 */
	public SystemParameter( final String orig ) {
//		rightNow = Calendar.getInstance();

		if( orig == null || orig.length() == 0 || orig.indexOf( "{@" ) < 0 ) {
			clms     = null;
			formats  = null;
			original = orig;		// 5.3.2.0 (2011/02/01)
		}
		else {
			StringBuilder buf = new StringBuilder(orig.length());		// 5.3.2.0 (2011/02/01)

			ArrayList<String> fmtList = new ArrayList<String>();
			ArrayList<String> clmList = new ArrayList<String>();

			// 5.1.8.0 (2010/07/01) パース方法見直し
			int start = 0;
			int index = orig.indexOf( "{@" );
			String val ;
			while( index >= 0 ) {
//				buf.append(  orig.substring( start, index ) );		// 5.3.2.0 (2011/02/01)
//				fmtList.add( orig.substring( start, index ) );
				val = orig.substring( start, index );					// 5.3.4.0 (2011/04/01)
				buf.append(  val );
				fmtList.add( val );
				int end = orig.indexOf( '}',index );
				if( end < 0 ) {
					String errMsg = "{@ と } との対応関係がずれています。" + CR
								+ "str=[" + orig + "] : index=" + index ;
					throw new RuntimeException( errMsg );
				}
				String param = orig.substring( index+2,end );
//				if( param.startsWith( "SYS." ) || param.startsWith( "DATE." ) ) {
				if( param.startsWith( "DATE." ) ) {		// 5.3.5.0 (2011/05/01) {&#064;SYS.XXXX} は、廃止
//					val = getDateFormat( param );
					val = getDateFormat( param.substring( 5 ) );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用時に "DATE." は不要
					clmList.add( null );		// パースした場合は、clmList は、使用しない。
					buf.append( val );
				}
				else {
					clmList.add( param );
					buf.append( "{@" ).append( param ).append( "}" );		// 元のままの文字列を生成
				}
				start = end+1;
				index = orig.indexOf( "{@",start );
			}
//			buf.append(  orig.substring( start, orig.length() ) );		// 5.3.2.0 (2011/02/01)
//			fmtList.add( orig.substring( start, orig.length() ) );
			val = orig.substring( start, orig.length() );				// 5.3.4.0 (2011/04/01)
			buf.append(  val );
			fmtList.add( val );

			original = buf.toString();		// 5.3.2.0 (2011/02/01)
			if( original.indexOf( "{@" ) < 0 ) {
				clms     = null;
				formats  = null;
			}
			else {
				clms	= clmList.toArray( new String[clmList.size()] );
				formats = fmtList.toArray( new String[fmtList.size()] );
			}
		}
	}

	/**
	 * 日付関係の情報を簡易的に取り出す処理を行います。
	 *
	 * 引数は、"XXXX AA BB CC" という状態で受け取ります。
	 *
	 * 処理の詳細は、{@link org.opengion.fukurou.util.HybsDateUtil#getDateFormat( String,String,String,int ) }
	 * または、{@link org.opengion.hayabusa.taglib.CommonTagSupport#getDateFormat( String ) } を
	 * 参照してください。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 5.5.8.2 (2012/11/09) prmA の判定に、null と ゼロ文字列を判定する。
	 * @og.rev 5.7.4.1 (2014/03/14) AA 引数の@解析後のコマンド判定方法を、８ケタ以下から先頭が数字以外に変更します。
	 * @og.rev 5.7.4.1 (2014/03/14) taglib.CommonTagSupport#getDateFormat( String ) と処理を合わせます。
	 * @og.rev 5.7.4.2 (2014/03/20) リクエストパラメータ(@で始まる引数)は使えません。
	 *
	 * @param   value パラメータ(引数は、"DATE.XXXX AA BB" などという状態)
	 *
	 * @return   メッセージ情報
	 * @see		org.opengion.fukurou.util.HybsDateUtil#getDateFormat( String,String,String,int )
	 * @see		org.opengion.hayabusa.taglib.CommonTagSupport#getDateFormat( String )
	 */
	private String getDateFormat( final String value ) {
		// 5.7.4.1 (2014/03/14) taglib.CommonTagSupport#getDateFormat( String ) と処理を合わせます。
		String[] vals = StringUtil.csv2Array( value,' ' );

		String key = vals[0] ;

		String prmA = (vals.length >= 2) ? vals[1] : null ;
		String prmB = (vals.length >= 3) ? vals[2] : null ;
		String prmC = (vals.length >= 4) ? vals[vals.length-1] : null ;		// 互換性。最後の値が、CC引数

		// 5.7.4.2 (2014/03/20) 現時点では、SystemParameter 処理にはリクエスト変数は使えないので、@変数も使えない。
		if( prmA != null && prmA.startsWith( "@" ) ) {
//			prmA = getDateFormat( prmA.substring(1) );		// getRequestValue ではなく、自身を再呼出し
			String errMsg = "AA引数に、リクエストパラメータ(@で始まる引数)は使えません。value=[" + value + "]" ;
			throw new RuntimeException( errMsg );
		}

		if( prmB != null && prmB.startsWith( "@" ) ) {
//			prmB = getDateFormat( prmB.substring(1) );		// getRequestValue ではなく、自身を再呼出し
			String errMsg = "BB引数に、リクエストパラメータ(@で始まる引数)は使えません。value=[" + value + "]" ;
			throw new RuntimeException( errMsg );
		}

		if( prmC != null && prmC.startsWith( "@" ) ) {
//			prmC = getDateFormat( prmC.substring(1) );		// getRequestValue ではなく、自身を再呼出し
			String errMsg = "CC引数に、リクエストパラメータ(@で始まる引数)は使えません。value=[" + value + "]" ;
			throw new RuntimeException( errMsg );
		}

		// 5.7.4.1 (2014/03/14) AA 引数の@解析後のコマンド判定方法を、８ケタ以下から先頭が数字以外に変更します。
		if( prmA != null && prmA.length() > 0 ) {
			char chA = prmA.charAt(0);
			if( chA < '0' || chA > '9' ) {		// 先頭が、数字以外の場合は、コマンドなので、一つずつずらす。
				prmC = prmB;
				prmB = prmA;
				prmA = null;
			}
		}

		// 5.7.4.1 (2014/03/14) CC 引数を、"H" , "D" , "M" 以外でも使用できるように拡張します。
		int intC = 0;
		if( prmC != null && prmC.length() > 0 ) {
			try { 
				intC = Integer.parseInt( prmC );
			}
			catch( NumberFormatException ex ) {
				String errMsg = "CC引数が数字ではありません。value=[" + value + "]" 
								+ ex.getMessage() ;
				System.err.println( errMsg );
			}
		}

		// prmA が null か、isEmpty() の場合は、現在時刻が使用される。
		return HybsDateUtil.getDateFormat( key,prmA,prmB,intC );	// 5.7.4.1 (2014/03/14) CC 引数を拡張します。

// 5.7.4.1 (2014/03/14) taglib.CommonTagSupport#getDateFormat( String ) と処理を合わせます。
//		String[] vals = StringUtil.csv2Array( value,' ' );
//
//		String key = vals[0].trim() ;
//
//		String prmA = null;				// 5.5.7.2 (2012/10/09) 引数として渡すので上位で初期化しておく。
//		String prmB = null;
//		if( vals.length == 3 ) { prmB = vals[2].trim(); }
//
//		if( vals.length > 1 ) {
//			prmA = vals[1].trim();
//			if( prmA.startsWith( "@" ) ) {
//				prmA = getDateFormat( prmA.substring(1) );
//			}
//
//			// prmA の@解析後、８ケタ以下の場合は、コマンドとみなし、prmB にセットし、自身は、null をセットする。
//			if( prmA != null && prmA.length() < 8 ) {
//				prmB = prmA;
//				prmA = null;
//			}
//		}
//
//		if( prmA == null || prmA.isEmpty() ) { prmA = RIGHT_NOW; }		// 5.5.8.2 (2012/11/09) null と ゼロ文字列を判定する。
//		return HybsDateUtil.getDateFormat( key,prmA,prmB );				// 5.5.7.2 (2012/10/09) HybsDateUtil を利用する
	}

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 判定方法 修正
	 *
	 * @param	entry	置換文字列のキーと値のペアを管理しているEntryオブジェクトの配列
	 *
	 * @return	置換後の文字列
	 */
	public String replace( final HybsEntry[] entry ) {
//		if( orginal == null ) { return null; }
//		int index = orginal.indexOf( "{@" );
//		if( index < 0 ) { return orginal; }
//		if( formats == null || clms == null ) { return original; }
		if( formats == null ) { return original; }		// 5.3.4.0 (2011/04/01) 判定方法 修正
		if( entry == null || entry.length == 0 ) { return original; }

		// HybsEntry[] データを、Mapにマッピングします。
		Map<String, String> sysMap = new HashMap<String, String>();
//		if( entry != null ) {
			int size = entry.length;
			for( int i=0; i<size; i++ ) {
				sysMap.put( entry[i].getKey(),entry[i].getValue() );
			}
//		}
		return replace( sysMap );
	}

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @param  map  置換文字列のキーと値のペアを管理しているMapオブジェクト
	 *
	 * @return	置換後の文字列
	 */
	public String replace( final Map<String,String> map ) {
//		if( formats == null || clms == null ) { return original; }
		if( formats == null ) { return original; }		// 5.3.4.0 (2011/04/01) 判定方法 修正
//		if( map == null || map.size() == 0 ) { return original; }
		if( map == null || map.isEmpty() ) { return original; }

		StringBuilder sb = new StringBuilder();
		for( int i=0; i<formats.length; i++ ) {
			sb.append( formats[i] );
			if( i < clms.length && clms[i] != null ) {		// 5.3.4.0 (2011/04/01) nullチェック追加
				sb.append(  StringUtil.nval( map.get( clms[i] ), "" ) );
			}
		}

		return sb.toString();
	}

	/**
	 * フォーマットをパースした結果から、カラム一覧を配列形式で返します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 *
	 * @return カラム配列
	 */
	public String[] getColumns() {
		if( clms == null ) { return new String[0]; }
		return clms.clone();
	}

	/**
	 * フォーマットをパースした結果から、フォーマット一覧を配列形式で返します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 *
	 * @return フォーマット配列
	 */
	public String[] getFormats() {
		if( formats == null ) { return new String[0]; }
		return formats.clone();
	}
}
