/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.system.OgRuntimeException ;			// 6.4.2.0 (2016/01/29)
// import org.opengion.fukurou.db.ApplicationInfo;
import static org.opengion.fukurou.system.HybsConst.CR ;		// 6.3.6.1 (2015/08/28)

import java.sql.Connection;

import java.util.Locale;
// import java.util.Map;
// import java.util.HashMap;
// import java.util.Collections;									// 6.3.9.0 (2015/11/06)
import java.util.concurrent.ConcurrentMap;							// 6.4.3.4 (2016/03/11)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.4 (2016/03/11)

/**
 * コネクションを共有して、トランザクションを実現します。
 *
 * 基本的には、TransactionTag で利用されますが、一部、このオブジェクトを
 * 渡して、直接、利用するケースもあります。
 *
 * トランザクションがすべて完了した後で、realClose() メソッドを呼び出します。
 * 一度でも、rollback が指定されていれば、ロールバックを行い、コネクションを
 * 破棄します。それ以外で、commit が指定されていれば、コミットを行い、
 * コネクションを、プールに戻します。どちらも指定されていなければ、
 * コネクションプールに戻すだけになります。
 *
 * 6.3.6.1 (2015/08/28)
 *   selectを実行した後で明示的にcommit,rollbackを行わないのはOracle位
 *   らしいので、検索終了時でも、commit か、rollback を行うようにします。
 *   つまり、commit されない(=途中で処理が打ち切られた)場合は、
 *   rollback するように仕様変更しますので、Transactionオブジェクトを
 *   呼び出した処理の最後には、検索であろうとなかろうと、commit()を入れてください。
 *   ただし、Transaction オブジェクトは、DBアクセス以外にも適用可能に
 *   作成しているため、Connection がある場合のみ、実際の commit/rollback が
 *   実行されます。
 *
 * 6.3.6.1 (2015/08/28)
 *   一度、finish() を実行すると、次回実行時にエラーにします。
 * 6.3.9.0 (2015/11/06)
 *   synchronized メソッドをsynchronizedブロックに変更。
 *
 * 考え方として、下記のような流れになります。
 * <pre>
 *   Transaction tran = new TransactionImpl/Real( appInfo ) ;
 *   try {
 *      ・・・・・
 *      tran.commit();
 *   }
 *   catch( final Exception ex ) {
 *      tran.rollback();
 *   }
 *   finally {
 *      tran.close();				// TransactionReal の場合
 *      tran.finish();				// TransactionImpl の場合
 *   }
 * </pre>
 *
 * 6.3.6.1 (2015/08/28)
 *    AutoCloseableを使用したtry-with-resources 構文を使用した場合。close/finish 不要。
 * <pre>
 *   try( final Transaction tran = new TransactionImpl/Real( appInfo ) ) {
 *      ・・・・・
 *      tran.commit();
 *   }
 *   ただし、処理自体がアベンドしないケースでは、rollback() を、自分で呼ぶか、commit() しない(=rollback()される)ようにします。
 * </pre>
 *
 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class TransactionImpl implements Transaction {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.3.4 (2016/03/11)" ;
	private static final long serialVersionUID = 643420160311L ;

	private static final String  DBID = "DEFAULT";
	private final ApplicationInfo appInfo ;

	private Connection defconn ;			// 最も利用率の高いDEFAULTだけ、別に変数を用意。

//	private final Map<String,Connection> dbidMap = new HashMap<>();
//	/** 6.4.3.1 (2016/02/12) Collections.synchronizedMap で同期処理を行います。  */						// ※ もう少し考えてから対応します。
//	private final Map<String,Connection> dbidMap = Collections.synchronizedMap( new HashMap<>() );		// 6.3.9.0 (2015/11/06)
	/** 6.4.3.4 (2016/03/11) ConcurrentHashMap で同期処理を行います。  */
	private final ConcurrentMap<String,Connection> dbidMap = new ConcurrentHashMap<>();		// 6.4.3.4 (2016/03/11)
	private boolean isCommit	;			// commit() されたかどうか。
	private boolean isEndCommit	;			// 6.4.3.3 (2016/03/04) doEndTag() が実行されたかどうか。
	private boolean isRollback	;			// rollback() されたかどうか。
//	private boolean isError		;			// エラー発生時にセットします。
	private boolean isFinish	;			// 処理の最後にセットします。

//	public static final Map<Integer,String> trace = new HashMap<>();				// 6.3.9.0 (2015/11/06) 意味不明な変数なので、削除しておきます。

	/**
	 * ApplicationInfo を指定して作成する、コンストラクター
	 *
	 * このクラスは、基本的には、TransactionTag クラスから作成されます。
	 *
	 * @param	appInfo	内部統制用のアクセス情報
	 */
	public TransactionImpl( final ApplicationInfo appInfo ) {
		this.appInfo = appInfo ;
	}

	/**
	 * 指定のDBID に対応した、Connection オブジェクトを返します。
	 * 内部Mapに存在していれば、そのコネクションを、存在しなければ、
	 * 新しく作成します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 * @og.rev 6.4.3.4 (2016/03/11) Map#computeIfAbsent で対応する。
	 *
	 * @param	dbid  接続先ID
	 *
	 * @return	指定のDBID に対応した、Connectionオブジェクト
	 */
	@Override
//	public synchronized Connection getConnection( final String dbid ) {
	public Connection getConnection( final String dbid ) {
		if( dbid == null || dbid.isEmpty() || DBID.equalsIgnoreCase( dbid ) ) {
			if( defconn == null ) {
				defconn = ConnectionFactory.connection( DBID,appInfo );
			}
			return defconn;
		}

		final String udbid = dbid.toUpperCase( Locale.JAPAN );	// 大文字化

		// Map#computeIfAbsent ： 戻り値は、既存の、または計算された値。追加有り、置換なし、削除なし
		return dbidMap.computeIfAbsent( udbid , k -> ConnectionFactory.connection( udbid,appInfo ) );

//		synchronized( dbidMap ) {								// 6.3.9.0 (2015/11/06)
//			Connection conn = dbidMap.get( udbid );
//			if( conn == null ) {
//				conn = ConnectionFactory.connection( udbid,appInfo );
//				dbidMap.put( udbid,conn );
//			}
//
//			return conn;
//		}
	}

	/**
	 * コミット処理が行われた場合に、内部フラグ(isCommit)を true にセットします。
	 * １回でもコミットが行われており、ロールバックが行われていなければ、
	 * コミットされます。
	 *
//	 * 検索処理のみで、エラーが発生していない場合は、コミットも行われないケースがあります。
	 * 検索処理時でも、最後に commit() を実行してください。実行されていない場合は、
	 * 自動的に、rollback() が、実行されます。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。return 不要。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
//	 * @return 正常:true/異常:false
	 */
//	public boolean commit() {
	@Override
//	public synchronized void commit() {
	public void commit() {
		isCommit = true;
//		return true;
	}

	/**
	 * ロールバック処理が行われた場合に、内部フラグ(isRollback)を true にセットします。
	 * １回でもロールバックが行われていれば、最終的にはロールバックされます。
	 *
	 * 一度も、ロールバックが行われていない場合でも、コミットが行われていない場合は、
	 * rollback()を実行します。
	 *
//	 * ロールバック指定の場合は、isError フラグを true(エラー有)にセットします。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。return 不要。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
//	 * @return 正常:true/異常:false
	 */
//	public boolean rollback() {
	@Override
//	public synchronized void rollback() {
	public void rollback() {
		isRollback = true;
//		isError    = true;
//		return true;
	}

	/**
	 * トランザクションの、終了時処理を行います。
	 *
	 * それまでの処理は、すべて正常に処理できた場合に、使用します。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。return 不要。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @see AutoCloseable#close()
	 */
	@Override
//	public synchronized void close() {
	public void close() {
		// Impl は、finish() で、実質的な完了処理を行う。
	}

	/**
	 * 最終的なコミットが行われた場合に、内部フラグ(isEndCommit)を true にセットします。
	 * 通常は、この処理は、１値度だけ実行されます。
	 * 初期値が、false なので、途中で一度でも実行されると、true にセットされ、最後まで
	 * 処理されたとみなされてしまうので、注意してください。
	 *
	 * 通常は、タグリブの、doEndTag() が実行された場合に、呼びます。
	 * このフラグが、true でないと（つまり、一度でも呼ばれないと）最終的に、commit されません。
	 *
	 * なお、endCommit() が呼ばれると、自動的に、commit() も呼んでおきます。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) 一般的なタグで、SKIP_PAGE された場合、rollback するようにします。
	 */
	public void endCommit() {
		isEndCommit = true ;
		isCommit    = true;
	}

//	/**
//	 * トランザクションの、終了時処理を行います。
//	 *
//	 * 実質的には、なにもしません。
//	 *
//	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。
//	 * @see #close( boolean )
//	 *
//	 * @return 正常:true/異常:false
//	 */
//	public boolean close() {
//		return close( false );			// 正常で登録
//	}

//	/**
//	 * トランザクションの、終了時処理を行います。
//	 *
//	 * 引数は、正常かどうかを判定するフラグです。異常の場合は、true をセットします。
//	 * ここでは、実際には何もしませんが、内部的にエラーフラグをセットします。
//	 * (エラーの場合のみセット。リセットはされません)
//	 * 一度でも、エラーが発生したコネクションは、破棄します。それ以外は、プールに戻します。
//	 *
//	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。
//	 * @param	errFlag 	[true:エラー状態/false:通常]
//	 *
//	 * @return 正常:true/異常:false
//	 */
//	public boolean close( final boolean errFlag ) {
//		if( errFlag ) { isError = true; }
//		return true;
//	}

//	/**
//	 * トランザクションとして、正常終了時に処理を行います。
//	 *
//	 * 実質的には、内部のisFinishフラグをセットするだけです。
//	 * ただし、このフラグがセットされていない場合は、処理が途中で止まった
//	 * 可能性があるため、トランザクションとしては、正常終了させることができません。
//	 *
//	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。実装を変更。
//	 * @see #realClose()
//	 */
//	public void finish() {
//		isFinish = true;
//	}

	/**
	 * トランザクションとして、終了時処理を行います。
	 *
	 * トランザクションがすべて完了した後で、呼び出します。
	 * 一度でも、Rollback が指定されていれば、ロールバックを行い、コネクションを
	 * 破棄します。それ以外で、Commit が指定されていれば、コミットを行い、
	 * コネクションを、プールに戻します。どちらも指定されていなければ、
	 * コネクションプールに戻すだけになります。
	 *
//	 * @og.rev 5.3.8.0 (2011/08/01) 内部変数を初期化し、このオブジェクトが再利用できるようにする。
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。メソッド名変更。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 * @og.rev 6.4.3.4 (2016/03/11) Map#computeIfAbsent で対応する。
	 */
//	public void realClose() {
//	public synchronized void finish() {
	public void finish() {
		if( isFinish ) {
			final String errMsg = "すでに、finish() 実行済みです。"		+ CR
						+ "  新規に Transaction オブジェクトの作成から行ってください。"		+ CR ;
			throw new OgRuntimeException( errMsg );
		}

		if( defconn != null ) {
			connClose( defconn,DBID );
			defconn = null;			// 内部変数を初期化します。
		}

		// Map#computeIfAbsent ： 戻り値は、既存の、または計算された値。追加有り、置換なし、削除なし
		dbidMap.forEach( (dbid,conn) -> connClose( conn,dbid ) );
		dbidMap.clear();

//		synchronized( dbidMap ) {	// 6.3.9.0 (2015/11/06)
//			for( final Map.Entry<String,Connection> entry : dbidMap.entrySet() ) {
//				final String     dbid = entry.getKey();
//				final Connection conn = entry.getValue();
//
//				connClose( conn,dbid );
//			}
//
////			// 内部変数を初期化します。
////			defconn    = null;
//
//			// 5.3.8.0 (2011/08/01) 内部変数を初期化し、このオブジェクトが再利用できるようにする。
//			// 6.3.6.1 (2015/08/28) 内部変数を初期化します。
//			dbidMap.clear();
//		}

		isFinish   = true;		// 次回実行時にエラーにします。

//		isCommit   = false;
//		isRollback = false;
//		isError    = false;
//		isFinish   = false;
	}

	/**
	 * Connection オブジェクトをクローズします。
	 *
//	 * 実際にはクローズせず、commit または、rollback を行った後、
//	 * エラー状況に応じて、ConnectionFactory に返却するか、削除します。
//	 * なお、commit または、rollback 時にエラーが発生した場合でも、無視して
//	 * そのまま、処理を継続します。
	 *
	 * commit されており、rollback されていない場合のみ、commit します。
	 * それ以外は、rollback します。
	 * rollback された場合は、コネクションプールに戻さずに、破棄します。
	 * 本来は、その、Connection に問題はないかもしれませんが、あえて、
	 * キャッシュを継続させる必要もないと考えます。
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) AutoCloseable の close() メソッドに対応。内部処理見直し。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 * @og.rev 6.4.3.3 (2016/03/04) 一般的なタグで、SKIP_PAGE された場合、rollback するようにします。
	 *
	 * @param  conn クローズ処理を行う、Connectionオブジェクト
	 * @param  dbid 接続先ID
	 */
//	private synchronized void connClose( final Connection conn, final String dbid ) {
	private void connClose( final Connection conn, final String dbid ) {
		// commit されており、rollback されていない場合
		final boolean isOK ;
//		if( isCommit && !isRollback ) {
		if( isCommit && isEndCommit && !isRollback ) {		// 6.4.3.3 (2016/03/04)
			isOK = Closer.commit( conn );
		}
		else {
			isOK = Closer.rollback( conn );
		}

		// rollback されているか、commit/rollback でエラーが発生した場合は、削除
		if( isRollback || !isOK ) {	ConnectionFactory.remove( conn,dbid ); }	// 削除
		else {						ConnectionFactory.close( conn,dbid );  }	// 返却

//		// まず、コミットかロールバックされた場合は、どちらかの処理が必要
//		if( isCommit || isRollback ) {
//			// commit できる条件：コミットされ、フィニッシュされ、エラーでなく、ロールバックでない
//			if( isCommit && isFinish && !isError && !isRollback ) {
//				Closer.commit( conn );
//			}
//			// それ以外は、ロールバックする。
//			else {
//				Closer.rollback( conn );
//			}
//		}

//		// 残りは、コミットもロールバックもされていないため、単にキャッシュへの返し方のみ判定する。
//		// isFinish されていないが、エラーにもなっていない場合は、接続要因以外の問題なので、返却でよい。
//		if( isError ) { ConnectionFactory.remove( conn,dbid ); }		// 削除
//		else {			ConnectionFactory.close( conn,dbid );  }	// 返却
	}
}
