/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

import java.util.Locale ;

/**
 * 半角文字＋半角カタカナの大文字のみに制限された文字列を扱う為の、カラム属性を定義します。
 *
 * ホスト送信用なので、半角カタカナ小文字を半角カタカナ大文字に変換します。
 * 通常の半角文字や半角カタカナを入力すると、大文字に変換後に、チェックを行います。
 * 半角カタカナの小文字とは、「ｧ,ｨ,ｩ,ｪ,ｫ,ｬ,ｭ,ｮ,ｯ」のことで、これらの文字列を、
 * 「ｱ,ｲ,ｳ,ｴ,ｵ,ﾔ,ﾕ,ﾖ,ｼ」に置き換えます。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角文字＋半角カタカナチェック
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_XHU extends AbstractDBType {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBType_XHU() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 3.3.3.1 (2003/07/18) 後ろスペースを取り除く。(StringUtil#rTrim)
	 * @og.rev 3.4.0.0 (2003/09/01) 半角カナ文字化け対策(ｱ,ｴ,ｶ,ｰ,ﾗ)の変換処理の廃止
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null ) { return null; }
		String rtnVal = StringUtil.rTrim( value );
		char[] ch = rtnVal.toCharArray();		// 3.3.3.1 (2003/07/18)

		boolean flag = false;
		for( int i=0; i<ch.length; i++ ) {
			switch( ch[i] ) {
	// 3.4.0.0 (2003/09/01) 半角カナ文字化け対策(ｱ,ｴ,ｶ,ｰ,ﾗ)の変換処理の廃止
	//			case '±' : ch[i] = 'ｱ'; flag = true; break;
	//			case '´' : ch[i] = 'ｴ'; flag = true; break;
	//			case '¶' : ch[i] = 'ｶ'; flag = true; break;
	//			case '°' : ch[i] = 'ｰ'; flag = true; break;
	//			case '×' : ch[i] = 'ﾗ'; flag = true; break;
				case 'ｧ'  : ch[i] = 'ｱ'; flag = true; break;
				case 'ｨ'  : ch[i] = 'ｲ'; flag = true; break;
				case 'ｩ'  : ch[i] = 'ｳ'; flag = true; break;
				case 'ｪ'  : ch[i] = 'ｴ'; flag = true; break;
				case 'ｫ'  : ch[i] = 'ｵ'; flag = true; break;
				case 'ｬ'  : ch[i] = 'ﾔ'; flag = true; break;
				case 'ｭ'  : ch[i] = 'ﾕ'; flag = true; break;
				case 'ｮ'  : ch[i] = 'ﾖ'; flag = true; break;
				case 'ｯ'  : ch[i] = 'ｼ'; flag = true; break;
				default   : break;
			}
		}
		if( flag ) {
			rtnVal = new String( ch );
		}
		return rtnVal.toUpperCase(Locale.JAPAN);
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) DBTypeCheckUtilクラスを利用するように修正
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータ(文字パラメータ)を引数に追加
	 * @og.rev 3.6.1.0 (2005/01/05) 半角カタカナに、『､』を含めます。(0xff65 以上 → 0xff64以上)
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ(文字パラメータ)
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//		ErrorMessage msg = new ErrorMessage();
		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		String check = DBTypeCheckUtil.byteLengthCheck( value,len );
		if( check != null ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,check,String.valueOf( len ) );
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			final char ch = value.charAt( i );
			// 半角カタカナの小文字(ｧ ｨ ｩ ｪ ｫ ｬ ｭ ｮ ｯ)も、NG
			if( ch < 0x20 || ( ch > 0x7e && ch < 0xff64 ) ||
				( ch > 0xff66 && ch < 0xff70 ) || ( ch >= 0xffa0 ) ) {
					buf.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
					isError = true;
			}
			else {
				buf.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,buf.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータ(文字パラメータ)を使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// クロスサイトスクリプティング対策：'<', '>' は登録させない。
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//		msg = xssCheck( key ,value, msg );
//		return msg;
		return xssCheck( key ,value, msg );
	}
}
