/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.ApplicationInfo;

/**
 * 伝送要求に対するの読取方法を定義するインターフェースです。
 *
 * 各実装クラスでは、読取処理{@link #read(TransferConfig, Transaction)}
 * 、完了処理{@link #complete(TransferConfig, Transaction)}、エラー処理{@link #error(TransferConfig, ApplicationInfo)}を実装する必要があります。
 * サブクラス内部で、DB関連の処理を行う場合は、引数のTransactionオブジェクトを利用して下さい。
 * また、接続に対するCommit/Rollbackは、呼び出し元で行いますので、実行処理の中では行わないで下さい。
 *
 * また、HTTP経由での実行などで、読取処理とその後の完了/エラー処理でオブジェクトが同一出ない場合に、
 * 外部から読取キーの取得及び設定を行うことができます。
 * 具体的には、{@link #read(TransferConfig, Transaction)}を実行後、{@link #getKeys()}を呼び出しすることで
 * 終了処理で処理対象となるキー情報を取得することができます。
 * このキー情報を保持し、{@link #setKeys(String[])}で別オブジェクトに対してキーを設定することで、
 * 完了処理{@link #complete(TransferConfig, Transaction)}、エラー処理{@link #error(TransferConfig, ApplicationInfo)}
 * を呼び出しすることができます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public interface TransferRead {

	/**
	 * 読取処理を実行し、結果をテキストデータの配列として返します。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 *
	 * @return 読み取りしたデータ(配列)
	 */
	String[] read( final TransferConfig config, final Transaction tran );

	/**
	 * 更新キー(配列)を返します。
	 *
	 * @return 更新キー(配列)
	 */
	String[] getKeys();

	/**
	 * 更新キー(配列)をセットします。
	 *
	 * @param keys 更新キー配列(可変長引数)
	 */
	void setKeys( final String... keys );

	/**
	 * 完了処理を実行します。
	 * この処理は実行方法で定義された実行処理が正常終了した場合に呼び出しされます。
	 * 更新対象のキーについては、{@link #setKeys(String[])}で外部からセットすることもできます。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 */
	void complete( final TransferConfig config, final Transaction tran );

	/**
	 * エラー処理を実行します。
	 * この処理は実行方法で定義された実行処理が以上終了した場合に呼び出しされます。
	 * 更新対象のキーについては、{@link #setKeys(String[])}で外部からセットすることもできます。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param appInfo DB接続情報
	 */
	void error( final TransferConfig config, final ApplicationInfo appInfo );
}
