/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.system.Closer;

import java.sql.CallableStatement;
import java.sql.SQLException;
import java.sql.Types;

/**
 * バッチ系標準のPL/SQL をコールする Query クラスです。
 *
 * java.sql.CallableStatement を用いて、データベース検索処理を行います。
 * 引数は、従来のPL/SQLの実行が可能なように、第一引数はエラーコード、第二引数は、
 * エラーメッセージを返してきます。第三引数以降は、自由に指定できます。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 *
 * @og.formSample
 * 例：
 *     第一引数、第二引数は、通常のPL/SQLと同じ、結果(STATUS)と
 *     内容(ERR_CODE)を返します。
 *     それ以降の引数については、入力(IN)のみですが、自由に設定できます。
 *     引数に変数を使用する場合は、? 記号を当てはめます。
 *     第一引数、第二引数は、予約済みですが、それ以降は、好きな位置に割り当てられます。
 *     names 属性の順番に、値だけがセットされていきます。
 *     下記の例は、変数の引数は、使用していません。
 *
 * &lt;og:query
 *     command=&quot;NEW&quot;
 *     queryType=&quot;JDBCCallable&quot;
 *     displayMsg=&quot;&quot; &gt;
 *         { call GEP00002.GEP00002( ?,?,&#39;{&#064;GUI.KEY}&#39;,&#39;{&#064;USER.ID}&#39; ) }
 * &lt;/og:query&gt;
 *
 *    CREATE OR REPLACE PACKAGE GEP00002 AS
 *        PROCEDURE GEP00002(
 *            P_STATUS    OUT    NUMBER,
 *            P_ERR_CODE  OUT    VARCHAR2,
 *            P_MIDDB     IN     VARCHAR2,
 *            P_USRUPD    IN     VARCHAR2  );
 *    END;
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCCallable extends AbstractQuery {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Query_JDBCCallable() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.3.3.1 (2003/07/18) ＤＢ登録時の後ろスペースを削除する。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 6.3.6.1 (2015/08/28) close(),realClose() 廃止。Queryはキャッシュしません。
	 *
	 * @param   args オブジェクトの引数配列(可変長引数)
	 */
	@Override
	public void execute( final String... args ) {			// 6.1.1.0 (2015/01/17) refactoring
		CallableStatement callStmt = null ;
		try {
			callStmt  = getConnection().prepareCall( getStatement() );
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );

			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, Types.VARCHAR);
			// 6.1.1.0 (2015/01/17) refactoring. 可変引数にしたため、null は来ない。
				for( int i=0; i<args.length; i++ ) {
					callStmt.setObject( i+3,StringUtil.rTrim( args[i] ) );
				}
			callStmt.execute();

			final int rtnCode = callStmt.getInt(1);
			setErrorCode( rtnCode );

			if( rtnCode > ErrorMessage.OK ) {		// 正常以外の場合
				final String ermsg = callStmt.getString(2);
				final ErrorMessage errMessage = new ErrorMessage( "Query_JDBCCallable Error!!" );
				errMessage.addMessage( ermsg );
				setErrorMessage( errMessage );
			}
//			commit();										// 6.3.6.1 (2015/08/28)
		}
		catch( final SQLException ex ) {
			setErrorCode( ErrorMessage.EXCEPTION );
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ getStatement() + CR;
//			rollback();
//			realClose();									// 6.3.6.1 (2015/08/28) close(),realClose() 廃止。
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.stmtClose( callStmt );
		}
	}
}
