/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;				// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.OgCharacterException ;			// 6.5.0.1 (2016/10/21)
import org.opengion.fukurou.system.Closer ;
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.FileUtil;
// import org.opengion.fukurou.util.FileString;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.CommentLineParser;					// 6.3.1.1 (2015/07/10)
import org.opengion.fukurou.util.FileInfo;							// 6.4.0.2 (2015/12/11)

import java.util.Arrays;
import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import java.io.File;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.CharacterCodingException;					// 6.3.1.0 (2015/06/28)

/**
 * Process_Grep は、上流から受け取った FileLineModelから、文字列を見つけ出す
 * ChainProcess インターフェースの実装クラスです。
 *
 * 正規表現の keyword を上流から受け取った FileLineModel から検索します。
 * 見つかった対象ファイルから、指定の文字列を置換する場合は、-change か
 * -changeFile で、keyword を置換する文字列を指定して下さい。
 * 置換する文字列には、\t と \n の特殊文字が使用できます。
 *
 * 処理対象は、通常は、１行づつ読み取りながら処理を行います。存在チェックの場合は、
 * 見つかった時点で処理を中止します。これは、該当箇所をピックアップするのではなく、
 * 存在しているかどうかを判断して、あれば、下流に流すというのが目的だからです。
 * keyword を、改行を含む正規表現で、検索・置換する場合は、-useBulkRead 属性を
 * true に設定してください。これは、入力ファイルを一括して読み込みます。
 * -ignoreCase は、正規表現の検索時にキーの大文字小文字を無視するように指定します。
 * -notEquals は、結果(見つかればtrue)を反転(見つからなければtrue)します。
 * これは、行単位ではなく、ファイル単位に判定しますので、change 指定した場合
 * でも、対象行は、見つかった行です。ただし、下流に対して、見つからない
 * 場合だけ処理を継続させます。
 * -inEncode は、入力ファイルのエンコード指定になります。
 * -outEncode は、出力ファイルのエンコードや、changeFileで指定の置換文字列ファイルの
 * エンコード指定になります。(changeFile は、必ず 出力ファイルと同じエンコードです。)
 * これらのエンコードが無指定の場合は、System.getProperty("file.encoding") で
 * 求まる値を使用します。
 * -changeFile を使用することで、複数行の文字列に置換することが可能です。
 * -outfile では、処理を行ったファイル名一覧をセーブします。
 *
 * 上流(プロセスチェインのデータは上流から渡されます。)からのLineModel の
 * ファイルオブジェクトより、指定の文字列が含まれているか検索します。
 * 上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト
 * である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを
 * 使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し
 * できれば、使用可能です。
 *
 * ※ 6.3.1.1 (2015/07/10) useOmitCmnt、useAllFind 機能追加
 *
 * 引数文字列中に空白を含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_Grep -keyword=検索文字列 -ignoreCase=true -outfile=OUTFILE -encode=UTF-8
 *
 *    -keyword=キーワード        ：検索する語句
 *   [-ignoreCase=大文字小文字 ] ：検索時に大文字小文字を区別しない(true)かどうか(初期値:区別する[false])
 *   [-notEquals=判定結果の反転] ：判定結果を反転させる(true)かどうか(初期値:反転させない[false])
 *   [-inEncode=入力エンコード ] ：入力ファイルのエンコードタイプ
 *   [-outEncode=出力エンコード] ：出力ファイルや置換ファイルのエンコードタイプ
 *   [-change=置換文字列       ] ：-change="ABCD" \t や \n などの特殊文字が使用できます。
 *   [-changeFile=置換ファイル ] ：-changeFile=change.txt このファイルの記述すべてと置換します。
 *                                     -change と、-changeFile は、同時に指定できません。
 *                                     置換機能使用時は、必ず、_backup というファイルが作成されます。
 *   [-insert=[HEAD/CHANGE/BEFORE/AFTER/TAIL]   ]
 *                               : 置換でなく挿入する場合の位置を指定します(初期値:CHANGE)
 *                                 スペースで区切って数字を記述すると、挿入位置にオフセットできます。
 *   [-delete=[false/true]     ] : 置換でなく削除します(初期値:false)
 *   [-skipRowCount=ｽｷｯﾌﾟ行数  ] : 先頭行から、スキップする行数を指定します(useBulkRead時には使用されません)
 *   [-useBackup=[false/true]  ] ：trueは、backupファイルを作成します(初期値:false)
 *   [-useBulkRead=[false/true]] ：trueは、入力ファイルを一括読込します(初期値:false)
 *   [-useAllFind=[false/true] ] ：置換ではなく検索だけ最後まで行う場合、trueを指定します(初期値:false)
 *   [-useOmitCmnt=[false/true]] ：コメント部分を削除したファイルでgrep処理を行うかどうかを指定(初期値:false)
 *   [-errAbend=[true/false]   ] ：異常発生時に、処理を中断(true)するか、継続(false)するかを指定する(初期値:true[中断する])
 *   [-display=[false/true]    ] ：trueは、検索状況を表示します(初期値:false)
 *   [-debug=[false/true]      ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_Grep extends AbstractProcess implements ChainProcess {
	private static final String[] INSERT_LIST = { "HEAD","CHANGE","BEFORE","AFTER","TAIL" };	// 6.2.4.0 (2015/05/15)

	private Pattern pattern		;
	private String	keyword		;
	private boolean	ignoreCase	;
	private boolean	notEquals	;
	private String	inEncode	;
	private String	outEncode	;
	private String	change		;
	private String	insert		= "CHANGE";		// "HEAD","CHANGE","BEFORE","AFTER","TAIL" のどれか
	private int		insOffset	;				// "BEFORE","AFTER" 時のオフセット
	private boolean	useBackup	;
	private boolean	useBulkRead	;				// 4.0.1.0 (2007/12/14) 一括読込
	private boolean	delete		;
	private boolean	useAllFind	;				// 6.3.1.1 (2015/07/10) 最後まで検索
	private boolean	useOmitCmnt	;				// 6.3.1.1 (2015/07/10) コメント除外
	private boolean	errAbend	= true;			// 6.3.1.0 (2015/06/28) 中断する
	private boolean	display		;
	private boolean	debug		;				// 5.1.2.0 (2010/01/01)

	private int		inCount		;
	private int		findCount	;
	private int		cngCount	;
	private int		skipRowCount ;				// 6.2.4.0 (2015/05/15) 行スキップ

	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	/** staticイニシャライザ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "keyword",	"検索する語句(必須)" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "ignoreCase",	"検索時に大文字小文字を区別しない(true)かどうか。" +
										CR + "(初期値:区別する[false])" );
		USABLE_PROPARTY.put( "notEquals",	"検索時に判定結果を反転させる(true)かどうか。" +
										CR + "(初期値:反転させない[false])" );
		USABLE_PROPARTY.put( "inEncode",		"入力ファイルのエンコードタイプ" );
		USABLE_PROPARTY.put( "outEncode",	"出力ファイルや置換ファイルのエンコードタイプ" );
		USABLE_PROPARTY.put( "change",		"置換文字列 例: -change=\"ABCD\" \\t や \\n などの特殊文字が使用できます。" );
		USABLE_PROPARTY.put( "changeFile",	"置換文字列ファイル 例: -changeFile=change.txt" +
										CR + "-change と、-changeFile は、同時に指定できません。" +
										CR + "置換機能使用時は、必ず、_backup というファイルが作成されます。" );
		USABLE_PROPARTY.put( "insert",		"[HEAD/CHANGE/BEFORE/AFTER/TAIL]:置換でなく挿入する場合の位置を指定します(初期値:CHANGE)"  +
										CR + "スペースで区切って数字を記述すると、挿入位置にオフセットできます。" );
		USABLE_PROPARTY.put( "delete",		"[false/true]:trueは、置換でなく削除します(初期値:false)" );
		USABLE_PROPARTY.put( "skipRowCount",	"先頭行から、スキップする行数を指定します。" );		// 6.2.4.0 (2015/05/15)
		USABLE_PROPARTY.put( "useBackup",	"[false/true]:trueは、backupファイルを作成します(初期値:false)" );
		USABLE_PROPARTY.put( "useBulkRead",	"[false/true]:trueは、入力ファイルを一括読込します(初期値:false)" );
		USABLE_PROPARTY.put( "useAllFind",	"置換ではなく検索だけ最後まで行う場合、trueを指定します(初期値:false)" );		// 6.3.1.1 (2015/07/10)
		USABLE_PROPARTY.put( "useOmitCmnt",	"コメント部分を削除したファイルでgrep処理を行うかどうかを指定(初期値:false)" );	// 6.3.1.1 (2015/07/10)
		USABLE_PROPARTY.put( "errAbend",		"異常発生時に、処理を中断(true)するか、継続(false)するか" +
										CR + "(初期値:true:中断する)" );			// 6.3.1.0 (2015/06/28)
		USABLE_PROPARTY.put( "display",		"[false/true]:trueは、検索状況を表示します(初期値:false)" );
		USABLE_PROPARTY.put( "debug",		"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
											CR + "(初期値:false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_Grep() {
		super( "org.opengion.fukurou.process.Process_Grep",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 6.3.1.0 (2015/06/28) errAbend属性追加。
	 * @og.rev 6.3.1.1 (2015/07/10) useOmitCmnt、useAllFind 機能追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		keyword			= arg.getProparty( "keyword");
		ignoreCase		= arg.getProparty( "ignoreCase"	,ignoreCase	);
		notEquals		= arg.getProparty( "notEquals"	,notEquals	);
		inEncode		= arg.getProparty( "inEncode"	,System.getProperty("file.encoding"));
		outEncode		= arg.getProparty( "outEncode"	,System.getProperty("file.encoding"));
		useBackup		= arg.getProparty( "useBackup"	,useBackup	);
		useBulkRead		= arg.getProparty( "useBulkRead",useBulkRead);		// 4.0.1.0 (2007/12/14)
		delete			= arg.getProparty( "delete"		,delete		);
		insert	  		= arg.getProparty( "insert"		,insert		);
		change			= arg.getFileProparty( "change"	,"changeFile",outEncode,false );
		skipRowCount	= arg.getProparty( "skipRowCount",0			);		// 6.2.4.0 (2015/05/15)
		useAllFind		= arg.getProparty( "useAllFind"	,useAllFind);		// 6.3.1.1 (2015/07/10)
		useOmitCmnt		= arg.getProparty( "useOmitCmnt",useOmitCmnt);		// 6.3.1.1 (2015/07/10)
		errAbend		= arg.getProparty( "errAbend"	,errAbend	);		// 6.3.1.0 (2015/06/28) errAbend属性追加
		display			= arg.getProparty( "display"	,display	);
		debug			= arg.getProparty( "debug"		,debug		);		// 5.1.2.0 (2010/01/01)

		if( change != null ) {
			final int adrs = insert.indexOf( ' ' );	// オフセット数字の有無
			if( adrs > 0 ) {
				insOffset = Integer.parseInt( insert.substring( adrs+1 ) );
				insert    = insert.substring( 0,adrs );
			}

			boolean isOK = false;
			for( int i=0; i<INSERT_LIST.length; i++ ) {
				if( insert.equalsIgnoreCase( INSERT_LIST[i] ) ) {
					isOK = true; break;
				}
			}
			if( !isOK ) {
				// 実行時エラーではないので、errAbend 対象外
				final String errMsg = "insert は、" + Arrays.toString( INSERT_LIST )
									+ " から指定してください。" + CR
									+ "-insert=[" + insert + "]" ;
				throw new OgRuntimeException( errMsg );
			}

			change = StringUtil.replace( change,"\\n",CR );
			change = StringUtil.replace( change,"\\t","\t" );
		}

		if( delete ) { change = ""; }	// 削除は、"" 文字列と置換します。

		if( ignoreCase ) {
			pattern = Pattern.compile( keyword,Pattern.CASE_INSENSITIVE );
		}
		else {
			pattern = Pattern.compile( keyword );
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		// ここでは処理を行いません。
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @og.rev 4.0.1.0 (2007/12/14) ファイルの一括処理対応。
	 * @og.rev 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
	 * @og.rev 6.3.1.0 (2015/06/28) errAbend属性追加。
	 *
	 * @param	data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		inCount++ ;

		final FileLineModel fileData ;
		if( data instanceof FileLineModel ) {
			fileData = (FileLineModel)data ;
		}
		else {
			// これは、プログラマーの問題なので、errAbend 対象外
			final String errMsg = "データが FileLineModel オブジェクトではありません。" + CR ;
			throw new OgRuntimeException( errMsg );
		}

		final File file = fileData.getFile() ;
		if( ! file.isFile() ) {
			if( display ) { println( data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
			return data;
		}

//		final boolean isFind ;
		boolean isFind = false ;	// 6.3.1.0 (2015/06/28) errAbend属性追加に伴う、初期化漏れ対応
		try {
			String fileLine = null;
			int firstLineNo = -1;
			if( useBulkRead ) { fileLine    = findKeywordAsBulk( file ); }
			else			  { firstLineNo = findKeyword( file ); }

			isFind = fileLine != null || firstLineNo >= 0 ;

			// 置換処理 ただし、見つかったときのみ実行
			if( change != null && isFind ) {
				// 入力ファイルは、オリジナル_backup ファイルとする。過去のファイルを削除
				final File inFile = new File( file.getPath() + "_backup" );
				if( inFile.exists() && !inFile.delete() ) {
					final String errMsg = "過去のBKUPファイルを削除できませんでした。[" + inFile + "]" + CR
								+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					// try の中から throw するのは、行儀がよくないが、catch ブロックで errAbend処理する。
					throw new OgRuntimeException( errMsg );
				}

				// オリジナルのファイルを、_backup ファイル名に先に変換する。
				final File fromFile = new File( file.getPath() );
				if( !fromFile.renameTo( inFile ) ) {
					final String errMsg = "所定のファイルをリネームできませんでした。[" + fromFile + "]" + CR
								+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					// try の中から throw するのは、行儀がよくないが、catch ブロックで errAbend処理する。
					throw new OgRuntimeException( errMsg );
				}

				// 変換処理 本体
				if( useBulkRead ) { changeKeywordAsBulk( fileLine,file ); }
				else			  { changeKeyword( inFile,file,firstLineNo ); }

				// backup を使わない場合は、削除する。
				// 4.0.0.0 (2007/11/29) 入れ子if の統合
				if( ! useBackup && !inFile.delete() ) {
					final String errMsg = "所定のファイルを削除できませんでした。[" + inFile + "]" + CR
								+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
					// try の中から throw するのは、行儀がよくないが、catch ブロックで errAbend処理する。
					throw new OgRuntimeException( errMsg );
				}
			}
		}
		catch( final RuntimeException ex ) {
			final String errMsg = "処理中にエラーが発生しました。[" + data.getRowNo() + "]件目" + CR
						+	"data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
			// 6.3.1.0 (2015/06/28) errAbend属性追加。
			throwException( errMsg,ex,errAbend );
//			throw new OgRuntimeException( errMsg,ex );
//			if( errAbend ) { throw new OgRuntimeException( errMsg,ex ); }
//			else {
//				logging( "=================================================================" );
//				logging( errMsg );
//				logging( ex.getMessage() );
//			}
		}

		if( display && ( notEquals ^ isFind ) ) { println( data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
		return notEquals ^ isFind ? data : null ;
	}

	/**
	 * キーワードが存在しているかどうかをチェックします。
	 * ここでは、１行づつ読み取りながら、最初に見つかった時点で制御を返します。
	 * よって、複数行にまたがる keyword でのマッチングは出来ませんが、大きな
	 * ファイル等での検索には、効率的です。
	 *
	 * @og.rev 4.0.1.0 (2007/12/14) 新規追加
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 * @og.rev 6.3.1.0 (2015/06/28) errAbend属性追加。
	 * @og.rev 6.3.1.1 (2015/07/10) useOmitCmnt、useAllFind 機能追加
	 * @og.rev 6.4.0.2 (2015/12/11) CommentLineParser 改造。
	 * @og.rev 6.5.0.1 (2016/10/21) CharacterCodingException は、OgCharacterException に変換する。
	 *
	 * @param	file	検索元のファイルオブジェクト
	 *
	 * @return	最初に見つかった行番号(見つからなければ、-1 を返す)
	 */
	private int findKeyword( final File file ) {

		int firstLineNo = -1;
		final BufferedReader reader = FileUtil.getBufferedReader( file,inEncode );

		// 6.4.0.2 (2015/12/11) CommentLineParser 改造
//		final CommentLineParser clp = useOmitCmnt ? new CommentLineParser() : null;		// 6.3.1.1 (2015/07/10)
		final CommentLineParser clp = useOmitCmnt ? new CommentLineParser( FileInfo.getSUFIX( file ) ) : null;
		try {
			String line ;
			int lineNo = 0;
			while((line = reader.readLine()) != null) {
				lineNo++ ;		// 注意：ここで返す行数は、コメント行を含む行数とする。

				// 6.3.1.1 (2015/07/10) useOmitCmnt 機能。コメント行を削除する処理を入れる。
				if( useOmitCmnt ) {
					line = clp.line( line );
					if( line == null ) { continue; }	// 戻り値が null の場合は、行として不成立
				}
				final Matcher mach = pattern.matcher( line );
				if( mach.find() ) {
					if( debug ) {
						final String buf = "DEBUG:\t" + file.getPath() + "(" + lineNo + "): " + line ;
						println( buf );
					}
					// 6.3.1.1 (2015/07/10) useAllFind 機能。最後まで検索を続けます。
//					firstLineNo = lineNo;
//					break;
					if( useAllFind ) {
						final String msg = file.getAbsolutePath() + '(' + lineNo + "):" + line ;
						println( msg );
					}
					else {
						firstLineNo = lineNo;
						break;
					}
				}
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch( final CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + file.getPath() + "] , Encode=[" + inEncode + "]" ;
			// 呼出元で、errAbend処理するので、そのまま、throw しておく。
//			throw new OgRuntimeException( errMsg,ex );
			throw new OgCharacterException( errMsg,ex );	// 6.5.0.1 (2016/10/21)
		}
		catch( final IOException ex ) {
			final String errMsg = "キーワードファイル読取エラーが発生しました。" + CR
								+	" [" + file.getPath() + "] , Encode=[" + inEncode + "]" ;
			// 呼出元で、errAbend処理するので、そのまま、throw しておく。
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
		}

		return firstLineNo;
	}

	/**
	 * キーワードが存在しているかどうかをチェックします。
	 * ここでは、ファイルをすべて読み取ってから、チェックします。
	 * よって、複数行にまたがる keyword でのマッチングが可能です。
	 *
	 * @og.rev 4.0.1.0 (2007/12/14) 新規追加
	 * @og.rev 6.4.5.1 (2016/04/28) FileStringのコンストラクター変更
	 * @og.rev 6.4.5.2 (2016/05/06) fukurou.util.FileString から、fukurou.util.FileUtil に移動。
	 *
	 * @param	file	検索元のファイルオブジェクト
	 *
	 * @return	検索元のファイルの文字列化情報(ただし、見つからなければ、null)
	 */
	private String findKeywordAsBulk( final File file ) {

		boolean isFind = false;

		// 6.4.5.1 (2016/04/28) FileStringのコンストラクター変更
//		final FileString sf = new FileString();
//		sf.setFilename( file.getPath() );
//		sf.setEncode( inEncode );
//		final FileString sf = new FileString( file.getPath() , inEncode );
//		final String line = sf.getValue();
		final String line = FileUtil.getValue( file.getPath() , inEncode );		// 6.4.5.2 (2016/05/06)

		final Matcher mach = pattern.matcher( line );
		if( mach.find() ) {
			if( debug ) { println( "DEBUG:\t" + file.getPath() ); }
			isFind = true;
		}

		return isFind ? line : null;
	}

	/**
	 * キーワードを指定の文字列に置き換えます。
	 * useBackup 属性に true を指定した場合、置き換え後の、backup ファイルは、
	 * オリジナル_backup という名称に変わります。
	 * ここでは、１行づつ読み取りながら、変換処理を行います。
	 * よって、複数行にまたがる keyword でのマッチングは出来ませんが、大きな
	 * ファイル等での置換でも、メモリの使用量は抑えられます。
	 *
	 * @og.rev 4.0.1.0 (2007/12/14) 置換処理を独立させます。
	 * @og.rev 6.2.4.0 (2015/05/15) HEAD,TAIL 追加
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 * @og.rev 6.5.0.1 (2016/10/21) CharacterCodingException は、OgCharacterException に変換する。
	 *
	 * @param	inFile	検索元の入力ファイルオブジェクト
	 * @param	outFile	変換後の出力ファイルオブジェクト
	 * @param	firstLineNo	キーワードが存在した場合の最初の行番号
	 */
	private void changeKeyword( final File inFile,final File outFile,final int firstLineNo ) {

		final BufferedReader reader = FileUtil.getBufferedReader( inFile,inEncode );
		final PrintWriter    writer = FileUtil.getPrintWriter( outFile,outEncode );

		String line = null;
		try {
			// 6.2.4.0 (2015/05/15) HEAD,TAIL 追加
			if( "HEAD".equals( insert ) ) {
				writer.println( change );
			}

			int lineNo = 0;
			while((line = reader.readLine()) != null) {
				lineNo++ ;
				if( lineNo <= skipRowCount ) { continue; }		// 6.2.4.0 (2015/05/15)

				if( lineNo >= firstLineNo ) {
					final Matcher mach = pattern.matcher( line );

					String chnStr = null;
					if( "CHANGE".equals( insert ) ) {
						chnStr = strChange( mach );
					}
					else if( "BEFORE".equals( insert ) ) {
						chnStr = strBefore( line,mach );
					}
					else if( "AFTER".equals( insert ) ) {
						chnStr = strAfter( line,mach );
					}

					if( chnStr != null ) {
						line = chnStr;
						cngCount++ ;	// 変換されれば カウント
					}
				}
				writer.println( line );	// readLine() してるので、最後に改行が必要。
			}

			// 6.2.4.0 (2015/05/15) HEAD,TAIL 追加
			if( "TAIL".equals( insert ) ) {
				writer.println( change );
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch( final CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + inFile + "] , Encode=[" + inEncode + "]" ;
			// 呼出元で、errAbend処理するので、そのまま、throw しておく。
//			throw new OgRuntimeException( errMsg,ex );
			throw new OgCharacterException( errMsg,ex );	// 6.5.0.1 (2016/10/21)
		}
		catch( final IOException ex ) {
			final String errMsg = "処理中にエラーが発生しました。[" + line + "]" + CR
								+	" [" + inFile + "] , Encode=[" + inEncode + "]" ;
			// 呼出元で、errAbend処理するので、そのまま、throw しておく。
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
			Closer.ioClose( writer );
		}
	}
	/**
	 * キーワードを指定の文字列に置き換えます。
	 * useBackup 属性に true を指定した場合、置き換え後の、backup ファイルは、
	 * オリジナル_backup という名称に変わります。
	 * ここでは、ファイルをすべて読み取ってから、チェックします。
	 * よって、複数行にまたがる keyword でのマッチングが可能です。
	 *
	 * @og.rev 4.0.1.0 (2007/12/14) 置換処理を独立させます。
	 * @og.rev 6.2.4.0 (2015/05/15) HEAD,TAIL 追加
	 *
	 * @param	fileLine	検索元の行文字列
	 * @param	outFile	出力ファイルオブジェクト
	 */
	private void changeKeywordAsBulk( final String fileLine,final File outFile ) {
		final PrintWriter writer = FileUtil.getPrintWriter( outFile,outEncode );

		String line = fileLine ;
		try {
			// 6.2.4.0 (2015/05/15) HEAD,TAIL 追加
			if( "HEAD".equals( insert ) ) {
				writer.println( change );
			}

			final Matcher mach = pattern.matcher( line );

			String chnStr = null;
			if( "CHANGE".equals( insert ) ) {
				chnStr = strChange( mach );
			}
			else if( "BEFORE".equals( insert ) ) {
				chnStr = strBefore( line,mach );
			}
			else if( "AFTER".equals( insert ) ) {
				chnStr = strAfter( line,mach );
			}

			if( chnStr != null ) {
				line = chnStr;
				cngCount++ ;	// 変換されれば カウント
			}

			writer.print( line );	// 注意：改行コードは、不要

			// 6.2.4.0 (2015/05/15) HEAD,TAIL 追加
			if( "TAIL".equals( insert ) ) {
				writer.println( change );
			}
		}
		catch( final RuntimeException ex ) {
			final String errMsg = "処理中にエラーが発生しました。[" + outFile.getPath() + "]" ;
			// 呼出元で、errAbend処理するので、そのまま、throw しておく。
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( writer );
		}
	}

	/**
	 * insert が、"CHANGE" の場合の処理結果を求めます。
	 * 変換しなかった場合は、null を返します。
	 * これは、変換カウントを算出する為のフラグ代わりに使用しています。
	 *
	 * @param	mach	キーワードの正規表現
	 *
	 * @return	変換結果(対象行で無い場合は、null)
	 */
	private String strChange( final Matcher mach ) {
		String line = null;
		if( mach.find() ) {
			line = mach.replaceAll( change );
		}
		return line ;
	}

	/**
	 * insert が、"BEFORE" の場合の処理結果を求めます。
	 * 変換しなかった場合は、null を返します。
	 * これは、変換カウントを算出する為のフラグ代わりに使用しています。
	 *
	 * @param	line	検索行
	 * @param	mach	キーワードの正規表現
	 *
	 * @return	変換結果(対象行で無い場合は、null)
	 */
	private String strBefore( final String line , final Matcher mach ) {
		boolean isChng = false;
		final StringBuilder buf = new StringBuilder( line.length() );
		int indx = 0;
		while( mach.find() ) {
			isChng = true;
			final int strt = mach.start() + insOffset;
			buf.append( line.substring( indx,strt ) );
			buf.append( change );
			indx = strt;
		}

		String rtn = null;
		if( isChng ) {
			buf.append( line.substring( indx ) );
			rtn = buf.toString();
		}

		return rtn ;
	}

	/**
	 * insert が、"AFTER" の場合の処理結果を求めます。
	 * 変換しなかった場合は、null を返します。
	 * これは、変換カウントを算出する為のフラグ代わりに使用しています。
	 *
	 * @param	line	検索行
	 * @param	mach	キーワードの正規表現
	 *
	 * @return	変換結果(対象行で無い場合は、null)
	 */
	private String strAfter( final String line , final Matcher mach ) {
		boolean isChng = false;
		final StringBuilder buf = new StringBuilder( line.length() );
		int indx = 0;
		while( mach.find() ) {
			isChng = true;
			final int end = mach.end() + insOffset;
			buf.append( line.substring( indx,end ) );
			buf.append( change );
			indx = end;
		}
		String rtn = null;
		if( isChng ) {
			buf.append( line.substring( indx ) );
			rtn = buf.toString();
		}

		return rtn ;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		if( findCount < cngCount ) { findCount = cngCount; }

		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Search Keyword    : " + keyword    + CR
				+ TAB + "Search File Count : " + inCount    + CR
				+ TAB + "Key Find    Count : " + findCount  + CR
				+ TAB + "Key Change  Count : " + cngCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( 1400 )
			.append( "Process_Grep は、上流から受け取った FileLineModelから、文字列を見つけ出す"	).append( CR )
			.append( "ChainProcess インターフェースの実装クラスです。"								).append( CR )
			.append( CR )
			.append( "正規表現の keyword を上流から受け取った FileLineModel から検索します。"		).append( CR )
			.append( "見つかった対象ファイルから、指定の文字列を置換する場合は、-change か" 		).append( CR )
			.append( "-changeFile で、keyword を置換する文字列を指定して下さい。" 					).append( CR )
			.append( "置換する文字列には、\t と \n の特殊文字が使用できます。" 						).append( CR )
			.append( CR )
			.append( "処理対象は、通常は、１行づつ読み取りながら処理を行います。存在チェックの場合は、"	).append( CR )
			.append( "見つかった時点で処理を中止します。これは、該当箇所をピックアップするのではなく、"	).append( CR )
			.append( "存在しているかどうかを判断して、あれば、下流に流すというのが目的だからです。"	).append( CR )
			.append( "keyword を、改行を含む正規表現で、検索・置換する場合は、-useBulkRead 属性を"	).append( CR )
			.append( "true に設定してください。これは、入力ファイルを一括して読み込みます。"		).append( CR )
			.append( "-ignoreCase は、検索時にキーの大文字小文字を無視するように指定します。"		).append( CR )
			.append( "-notEquals は、結果(見つかればtrue)を反転(見つからなければtrue)します。"		).append( CR )
			.append( "これは、行単位ではなく、ファイル単位に判定しますので、change 指定した場合"	).append( CR )
			.append( "でも、対象行は、見つかった行です。ただし、下流に対して、見つからない"			).append( CR )
			.append( "場合だけ処理を継続させます。"													).append( CR )
			.append( "-inEncode は、入力ファイルのエンコード指定になります。" 						).append( CR )
			.append( "-outEncode は、出力ファイルのエンコードや、changeFileで指定の置換文字列" 		).append( CR )
			.append( "ファイルのエンコード指定になります。(changeFile は、必ず 出力ファイルと)" 	).append( CR )
			.append( "同じエンコードです。"															).append( CR )
			.append( "これらのエンコードが無指定の場合は、System.getProperty(\"file.encoding\") " 	).append( CR )
			.append( "で求まる値を使用します。" 													).append( CR )
			.append( "-changeFile を使用することで、複数行の文字列に置換することが可能です。" 		).append( CR )
			.append( CR )
			.append( "上流(プロセスチェインのデータは上流から渡されます。)からのLineModel の"		).append( CR )
			.append( "ファイルオブジェクトより、指定の文字列が含まれているか検索します。"			).append( CR )
			.append( "上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト"	).append( CR )
			.append( "である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを"		).append( CR )
			.append( "使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し"	).append( CR )
			.append( "できれば、使用可能です。"														).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_Grep().usage() );
	}
}
