/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.XHTMLTag;

import org.opengion.fukurou.util.StringUtil;

/**
 * 【入力】複数行の入力が可能なテキスト入力フィールドを作成するHTML拡張タグです。
 *
 * name 属性に ラベルリソース のキーを与えることで、 ロケールにあわせたリソースを使用して、
 * 画面に表示します。
 * ロケールは、ユーザー情報の lang 属性を初期値で使用し、 セットされていない場合は、
 * リクエスト情報のロケールから取得します。
 * ラベル部分と入力フィールド部分がテーブルタグの &lt;td&gt; により左右に分割されます。
 * HTML 表示時は、前後に&lt;tr&gt;タグで囲って、 整形できます。
 *
 * @og.formSample
 * ●形式：&lt;og:queryOption &gt;
 *             SELECT文 
 *         &lt;/og:queryOption &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:textarea
 *       name             ○【HTML】名前を指定します(必須)。
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       rows               【HTML】高さ(行数)を指定します
 *       cols               【HTML】横幅(カラム数)を指定します
 *       must               【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します(初期値:false)
 *       mustAny            【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します(初期値:無指定)
 *       useMustHidden      【TAG】必須の自動チェック用Hiddenを出力するかどうか[true/false]を指定します(初期値:true)
 *       aimai              【TAG】曖昧検索可能フィールドとして、曖昧検索方法を指定します(初期値:null)
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       readonly           【TAG】その部品に対して変更が出来ないように(readonly)指定します(サーバーに送信される)
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】アクセスキー(alt+キーで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       onSelect           【HTML】JavaScriptのイベント onSelect を設定します
 *       onKeydown          【HTML】JavaScriptのイベント onKeydown を設定します
 *       onKeypress         【HTML】JavaScriptのイベント onKeypress を設定します
 *       onKeyup            【HTML】JavaScriptのイベント onKeyup を設定します
 *       autofocus          【HTML5】指定した入力欄にカーソルが当たって自動的にフォーカスされます。
 *       placeholder        【HTML5】入力欄に初期表示する内容を指定します。
 *       required           【HTML5】入力必須を指定します
 *       roles              【TAG】ロールをセットします
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:textarea&gt;
 *
 * ●使用例
 *     &lt;og:textarea name="BUN" /&gt;
 *
 *     &lt;og:textarea name="BUN"&gt;            初期値に値をセットしたいとき
 *         test
 *     &lt;/og:textarea name="BUN"&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TextareaTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TextareaTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転注意
		return useTag()
				? EVAL_BODY_BUFFERED		// Body を評価する。( extends BodyTagSupport 時)
				: SKIP_BODY ;

//		if( !useTag() ) { return SKIP_BODY ; }	// 6.3.4.0 (2015/08/01)
//
//		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.0 (2003/03/28) body 属性に、 {&#064;XXXX} パース機能を付加する。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		final String str = getBodyString();

		if( str != null && str.length() > 0 ) {
			final String label = StringUtil.htmlFilter( str );
			if( label != null && label.length() > 0 ) {
				set( "body",label );
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 4.3.7.1 (2009/06/08) id=labelのclass化
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性を自動化します。
	 * @og.rev 6.2.0.0 (2015/02/27) aimai 属性を追加。
	 *
	 * @return  入力用のテキストフィールドタグ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		final String name = get( "name" );							// 6.2.0.0 (2015/02/27) 後で使うので変数化

		if( getMsglbl()  == null ) { setLbl( name ); }

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
		// 4.3.7.1 (2009/06/08) id=labelのclass化
		// rtn.append( "<td id=\"label\">" );
			.append( "<td class=\"label\">" )
			.append( getLongLabel() )		// 4.0.0 (2005/01/31)
			.append( "</td>" )
			.append( "<td>" )
			.append( XHTMLTag.textarea( getAttributes() ) )
			.append( "</td>" )
			.append( makeAimaiPicker( name ) )		// 6.2.0.0 (2015/02/27) aimai 属性を追加
			.append( makeMustHidden( name ) );		// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)

		return rtn.toString() ;
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag 名前を指定します。
	 *
	 * @param   name 名前
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】高さ(行数)を指定します。
	 *
	 * @og.tag 高さ(行数)を指定します。
	 *
	 * @param   rows 高さ(行数)
	 */
	public void setRows( final String rows ) {
		set( "rows",getRequestParameter( rows ) );
	}

	/**
	 * 【HTML】横幅(カラム数)を指定します。
	 *
	 * @og.tag 横幅を指定します。
	 *
	 * @param   cols 横幅(カラム数)
	 */
	public void setCols( final String cols ) {
		set( "cols",getRequestParameter( cols ) );
	}
}
