/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.html.ViewJsonParam;
// import org.opengion.hayabusa.html.ViewStackTableParam;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * viewタグの viewFormType が JSON の場合にパラメータを設定します。
 *
 * JOSN出力のViewForm_JSON クラスに対して、各種パラメータを設定します。
 * パラメータが設定されていない場合は、ViewJsonParam の初期値が
 * 使用されます。
 * (パラメータを使用するには、viewタグのuseParam 属性をtrueに設定する必要があります。)
 *
 * @og.formSample
 * ●形式：&lt;og:jsonParam useHead="false" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:stackParam
 *       useHead         【TAG】ヘッダ情報を出力するかどうか(初期値:true)
 *       useInfo         【TAG】INFO情報を出力するかどうか(初期値:true)
 *       useRenderer     【TAG】データ出力でレンデラを利用するかどうか(初期値:false)
 *       useNullOmit     【TAG】nullデータの出力を行わないかどうか(初期値:false)
 *       useZeroOmit     【TAG】ゼロデータの出力を行わないかどうか(初期値:false)
 *       dataName        【TAG】データ出力時の配列オブジェクトの名称指定(初期値:DATA)
 *       debug           【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * @og.rev 5.9.7.0 (2016/04/01) 新規作成
 * @og.group 画面表示
 *
 * @version  5.0
 * @author	 Takahashi Masakazu
 * @since    JDK5.0,
 */
// public class ViewJsonParamTag extends ViewParamTag {
public class ViewJsonParamTag extends ViewParamImpl {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.7.7.0 (2017/03/31)" ;
	private static final long serialVersionUID = 677020170331L ;

	/**
	 * 【TAG】JSON出力で、INFO情報を出すかどうかを指定します。
	 *
	 * @og.tag
	 * JSON中INFO情報を出力しない場合はfalseを指定します。
	 * 
	 *
	 * @param	useinfo INFO情報を出力するかどうか
	 */
	public void setUseInfo( final String useinfo ) {
		putParam( ViewJsonParam.JSON_INFO_KEY ,
				  nval( getRequestParameter( useinfo ),null ) );
	}

	/**
	 * 【TAG】JSON出力で、ヘッダ情報を出すかどうかを指定します。
	 *
	 * @og.tag
	 * JSON中にヘッダ情報を出力しない場合はfalseを指定します。
	 * 
	 *
	 * @param	usehead ヘッダ情報を出力するかどうか
	 */
	public void setUseHead( final String usehead ) {
		putParam( ViewJsonParam.JSON_HEAD_KEY ,
				  nval( getRequestParameter( usehead ),null ) );
	}

	/**
	 * 【TAG】JSON出力で、値出力にレンデラを利用するかどうかを指定します。
	 *
	 * @og.tag
	 * JSONのデータのレンデラー変換を行うかどうか。
	 * 指定しない場合は使用しない(false)です。
	 * 
	 *
	 * @param	usernd レンデラーを利用するかどうか
	 */
	public void setUseRenderer( final String usernd ) {
		putParam( ViewJsonParam.JSON_RENDERER_KEY ,
				  nval( getRequestParameter( usernd ),null ) );
	}

	/**
	 * 【TAG】JSON出力で、nullデータの出力を行わないかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * JSONのデータが、nullの場合、出力するかどうかを指定します。
	 * true を指定すると、nullデータは、出力しません。
	 * (初期値:false)
	 *
	 * @og.rev 6.7.7.0 (2017/03/31) 新規追加
	 *
	 * @param	nullOmit nullデータの出力を行わないかどうか
	 */
	public void setUseNullOmit( final String nullOmit ) {
		putParam( ViewJsonParam.JSON_NULLOMIT_KEY ,
				  nval( getRequestParameter( nullOmit ),null ) );
	}

	/**
	 * 【TAG】JSON出力で、ゼロデータの出力を行わないかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * JSONのデータが、ゼロの場合、出力するかどうかを指定します。
	 * true を指定すると、ゼロデータとnullデータを、出力しません。
	 * (初期値:false)
	 *
	 * @og.rev 6.7.7.0 (2017/03/31) 新規追加
	 *
	 * @param	zeroOmit ゼロデータの出力を行わないかどうか
	 */
	public void setUseZeroOmit( final String zeroOmit ) {
		putParam( ViewJsonParam.JSON_ZEROOMIT_KEY ,
				  nval( getRequestParameter( zeroOmit ),null ) );
	}

	/**
	 * 【TAG】JSON出力で、データ部の名前を指定します。
	 *
	 * @og.tag
	 * データ配列の名称を指定します。
	 * 指定しない場合は「DATA」です。
	 * 
	 *
	 * @param	name データ配列の名称
	 */
	public void setDataName( final String name ) {
		putParam( ViewJsonParam.JSON_DATANAME_KEY ,
				  nval( getRequestParameter( name ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "jsonParam" ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
	 *
	 * @serialData 一部のオブジェクトは、シリアライズされません
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 * @throws IOException	入出力エラーが発生した場合
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}
}
