/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.HybsDateUtil;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * TMSTMP レンデラーは、日付ネイティブのカラムのデータから、数字部分だけをピックアップし、
 * 日時(年/月/日 時:分:秒)または、日付(年/月/日)表示する場合に使用するクラスです。
 * 数字以外は、各データベースによって、まちまちなので、'/',':','-',' ' を削除します。
 * なお、並び順は、必須に、YYYYMMDDHHMISS の順番とし、January 8 などの形式や、
 * 04:05 AM 、1/8/1999 などの形式は受け付けることは出来ません。
 * あくまで、先頭から 数字を順次取得し、YYYYMMDDHHMISS 形式に当てはめていきます。
 * また、時刻部分 HHMISS が 000000 の場合は、日付のみに自動的にします。
 * これは、日付ネイティブでは、時刻情報も持っているため、使っているのかどうかの
 * 判定は、実際のデータが、000000 に設定されているかどうかでしか判断できないためです。
 * 自動判定がまずい場合は、通常のレンデラーで、表示桁数制限しても構いません。
 *
 * このクラスは、不変オブジェクトとして、共有されます。
 *
 * @og.group データ表示
 *
 * @og.rev 5.3.10.0 (2011/10/01) synchronized を、削除します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_TMSTMP extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	private static final CellRenderer DB_CELL = new Renderer_TMSTMP() ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Renderer_TMSTMP() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return DB_CELL;
	}

	/**
	 * データの表示用文字列を返します。
	 * 日時(年/月/日 時:分:秒)または、日付(年/月/日)表示する場合に使用するクラスです。
	 * 数字以外は、各データベースによって、まちまちなので、'/',':','-',' ' を削除します。
	 * なお、並び順は、必須に、YYYYMMDDHHMISS の順番とし、January 8 などの形式や、
	 * 04:05 AM 、1/8/1999 などの形式は受け付けることは出来ません。
	 * あくまで、先頭から 数字を順次取得し、YYYYMMDDHHMISS 形式に当てはめていきます。
	 * また、時刻部分 HHMISS が 000000 の場合は、日付のみに自動的にします。
	 * これは、日付ネイティブでは、時刻情報も持っているため、使っているのかどうかの
	 * 判定は、実際のデータが、000000 に設定されているかどうかでしか判断できないためです。
	 * 自動判定がまずい場合は、通常のレンデラーで、表示桁数制限しても構いません。
	 * 
	 * ここでは、規定のフォーマット(８ケタ OR １４ケタ)でない場合は、エラー表示を行います。
	 * 
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );

	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value  入力値
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
		if( value == null || value.isEmpty() ) { return ""; }

		String rtn = value;			// 初期値:isView = false の時の値

		if( rtn.length() != 8 && rtn.length() != 14 ) {
			rtn = HybsDateUtil.parseNumber( value );
		}

		char[] ch1 = new char[19];
		if( rtn.length() == 8 || rtn.length() == 14 ) {
			ch1[0] = rtn.charAt(0);
			ch1[1] = rtn.charAt(1);
			ch1[2] = rtn.charAt(2);
			ch1[3] = rtn.charAt(3);
			ch1[4] = '/' ;
			ch1[5] = rtn.charAt(4);
			ch1[6] = rtn.charAt(5);
			ch1[7] = '/' ;
			ch1[8] = rtn.charAt(6);
			ch1[9] = rtn.charAt(7);

			// 6.3.9.0 (2015/11/06) 条件は効果がない(findbugs)。rtn.length() == 14 は確定している。
			if( rtn.length() == 8 || rtn.endsWith( "000000" ) ) {
//			if( rtn.length() == 8 || ( rtn.length() == 14 && rtn.endsWith( "000000" ) ) ) {
				rtn = new String( ch1 , 0 , 10 );	// YYYY/MM/DD 形式を返す。
			}
			else {
				ch1[10] = ' ';
				ch1[11] = rtn.charAt(8);
				ch1[12] = rtn.charAt(9);
				ch1[13] = ':';
				ch1[14] = rtn.charAt(10);
				ch1[15] = rtn.charAt(11);
				ch1[16] = ':';
				ch1[17] = rtn.charAt(12);
				ch1[18] = rtn.charAt(13);
				rtn = new String( ch1 );			// YYYY/MM/DD HH:MM:SS 形式を返す。
			}
		}
		else {
			if( isView ) {
				rtn = "<span class=\"error\">" + value + "</span>";
			}
		}

		return rtn ;
	}
}
