/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval ;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.ChartCreate;
import org.opengion.hayabusa.io.ChartDataset;
import org.opengion.hayabusa.io.HybsURLTagFragmentGenerator;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Map ;
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring

import org.jfree.chart.JFreeChart;
import org.jfree.chart.ChartUtilities;

import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.entity.StandardEntityCollection;
import org.jfree.chart.imagemap.ImageMapUtilities;
import org.jfree.chart.imagemap.StandardToolTipTagFragmentGenerator;
import org.jfree.chart.imagemap.ToolTipTagFragmentGenerator;
import org.jfree.chart.imagemap.URLTagFragmentGenerator;

/**
 * BODY部に指定のSQLの検索結果をチャート(円、棒、線)で表示するタグです。
 *
 * チャート化には、JFreeChart (http://www.jfree.org/jfreechart/) を使用しています。
 * チャートの種類は、chartDataset タグで指定します。これは、複数の異なるチャートを
 * 合成表示する機能が、JFreeChart にある為です。ただし、専門的には、CategoryDataset のみ
 * いまは、合成が可能です。
 * 処理の実行にあたり、キャッシュが有効(useCache="true")であれば、キャッシュデータを
 * 使用します。
 * キャッシュは、画面ID + masterKey 属性で指定されたキーワード毎に、保管されます。
 *
 * ※ 6.0.2.0 (2014/09/19) 以降は、キャッシュは、イメージタグのみになりましたので、
 * seriesPikup 違いの場合は、再検索させる必要がありますので、masterKey を
 * 変更する必要があります。(または、seriesPikup 属性をmasterKeyに含めてください。)
 *
 * ※ 6.3.5.0 (2015/08/08) 
 *    ① chartFile属性(画像ファイル名の指定) が可能です。
 *       繰返し画像を作成する場合、無駄にファイルを生成しなくできます。
 *    ② cacheScope属性を追加しています。
 *       cacheScope="application" で、全ユーザー共有のキャッシュになります。
 *    ③ 直後の実行結果を、{&#064;CHART.TAG} 変数で取得できます。(requestスコープ)
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。
 *
 * @og.formSample
 * ●形式：&lt;og:chartCreate title="…" … /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:chartCreate
 *       title              【TAG】チャートのタイトルをセットします
 *       width              【TAG】チャートの横幅をセットします(初期値:200)
 *       height             【TAG】チャートの縦幅をセットします(初期値:200)
 *       domainLabel        【TAG】チャートのドメインラベルを指定します
 *       showLegend         【TAG】チャートの凡例の表示可否[true/false]をセットします(初期値:true[表示する])
 *       rectangleEdge      【TAG】チャートの表示箇所を、[TOP/BOTTOM/RIGHT/LEFT]で指定します(初期値:BOTTOM)
 *       plotOrientation    【TAG】チャートのプロット(Plot)表示方向を、[VERTICAL(or V),HORIZONTAL(or H)]で指定します
 *       chartBackColor     【TAG】チャートの背景色を指定します
 *       plotBackColor      【TAG】チャートの描画領域の色を指定します
 *       rotationLabel      【TAG】チャートのカテゴリーラベルの方向を指定します[1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･ ](初期値:3:60度)
 *       useVerticalLabels  【TAG】横軸ラベルの表示向きを縦にするかどうか[false/true]を指定します(初期値:false)
 *       domainMarker       【TAG】チャートの横軸の値(ドメイン)に合致する位置にマーカーラインを設定します
 *       useDomainLabel     【TAG】横軸ラベルのラベルを表示するかどうか[true/false]を指定します(初期値:true)
 *       useMarkerLabel     【TAG】マーカーラインに、その設定値を表示するかどうか[true/false]を指定します(初期値:true)
 *       useCache           【TAG】JFreeChartで作成されたイメージタグをキャッシュするかどうか[true/false]を指定します(初期値:false)
 *       cacheTime          【TAG】キャッシュ時の制限時間を、秒で指定します(初期値:CHART_CACHE_TIME[={@og.value SystemData#CHART_CACHE_TIME}])。
 *       cacheScope         【TAG】useCache="true" 時のキャッシュの保存先scopeを指定します(初期値:session)
 *       masterKey          【TAG】キャッシュ用マスタキーを指定します(初期値:imgTag)
 *       seriesPikup        【TAG】複数シリーズのピックアップを行う場合のシリーズ番号を指定します
 *       imageMapUrl        【TAG】クリッカブル・マップ用URLを指定します
 *       imageMapTarget     【TAG】クリッカブル・マップ用TARGETを指定します
 *       categoryMargin     【TAG】カテゴリマージン(0.0～1.0)を指定します
 *       lowerMargin        【TAG】下方マージン(0.0～1.0)を指定します
 *       upperMargin        【TAG】上方マージン(0.0～1.0)を指定します
 *       useDomainLine      【TAG】横軸のグリッド表示有無(垂直線)を指定します(初期値:false)
 *       domainLineColor    【TAG】横軸のグリッド線の色を指定します
 *       categorySkip       【TAG】横軸ラベルをスキップする間隔を指定します
 *       categoryCutNo      【TAG】横軸ラベルの文字位置指定のカット文字数を指定します
 *       categoryAnchor     【TAG】横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)を指定します
 *       useRangeLine       【TAG】縦軸のグリッド表示有無(水平線)を指定します(初期値:true)
 *       useToolTip         【TAG】ツールチップスの使用可否[true:利用する/false:利用しない]を指定します(初期値:false)
 *       rangeLineColor     【TAG】縦軸のグリッド線の色を指定します
 *       rangeSkip          【TAG】縦軸のグリッド線(水平線)をスキップする間隔を指定します
 *       dateAxisFormat     【TAG】Time,XYTime Polt関係の時間軸を表す場合のフォーマットを指定します(初期値:MM/dd)
 *       dateSegSize        【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのsegment size を、分単位で指定します。
 *       dateSegIncluded    【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのincluded segments(ｽﾍﾟｰｽ)を数値で指定します。
 *       dateSegExcluded    【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのexcluded segments(EE)を数値で指定します。
 *       dateStartTime      【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのStartTime(segment groupの開始位置) を、yyyyMMddHHmmss形式で指定します。
 *       chartFile          【TAG】チャートの画像ファイル名を指定します(初期値:自動)
 *       useTimeView        【TAG】処理時間を表示する TimeView を表示するかどうかを指定します
 *										(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:chartCreate&gt;
 *
 * ●使用例
 *      &lt;og:chartCreate
 *          title       = "JFreeChart Test"             チャートタイトル
 *          domainLabel = "ドメインラベル"              横軸ラベル
 *          width       = "200"                         チャート表示幅
 *          height      = "200"                         チャート表示高さ
 *          showLegend  = "[true/false]"                凡例の表示可否[true/false]
 *          rectangleEdge   = "[TOP|BOTTOM|RIGHT|LEFT]" 凡例の表示箇所
 *          plotOrientation = "[VERTICAL|HORIZONTAL]"   チャートの軸表示方向
 *          chartBackColor  = "WHITE"                   背景色
 *          plotBackColor   = "LIGHT_GRAY"              描画領域色
 *          rotationLabel   = "3"                       横軸ラベルの傾き
 *          useVerticalLabels = "false"					横軸ラベルの向き(true:縦/false:横)
 *          domainMarker    = "KING"                    横軸のマーカーライン(縦棒)
 *          useMarkerLabel  = "[true/false]"            マーカーラインの文字有無
 *          useDomainLabel  = "[true/false]"            横軸ラベルの表示有無
 *          categorySkip    = "3"                       横軸ラベルをスキップする間隔
 *          categoryCutNo   = "6"                       横軸ラベルの文字位置指定のカット文字数
 *          categoryAnchor  = "[START|MIDDLE|END]"      横軸のグリッド(垂直線)の書き出し位置
 *          useDomainLine   = "[false/true]"            横軸のグリッド表示有無(垂直線)
 *          domainLineColor = "LIGHT_GRAY"              横軸のグリッド線の色
 *          useRangeLine    = "[true/false]"            縦軸のグリッド表示有無(水平線)
 *          rangeLineColor  = "LIGHT_GRAY"              縦軸のグリッド線の色
 *          rangeSkip       = "5"                       縦軸のグリッド(水平線)をスキップする間隔
 *          useCache        = "[false|true]"            キャッシュの有無
 *          masterKey       = "{&#064;XXXX}"            キャッシュ用マスタキー(初期値:imgTag)
 *          cacheTime       = "10"                      キャッシュ時の制限時間を、秒で指定 (6.0.2.0 (2014/09/19))
 *          seriesPikup     = "2"                       ピックアップするシリーズNo
 *          imageMapUrl     = "link.jsp,BLOCK"          クリッカブル・マップ用URL
 *          imageMapTarget  = "CONTENTS"                クリッカブル・マップ用TARGET
 *          categoryMargin  = "[0.0～1.0]"              カテゴリマージン(0.0～1.0)
 *          lowerMargin     = "[0.0～1.0]"              下方マージン(0.0～1.0)
 *          upperMargin     = "[0.0～1.0]"              上方マージン(0.0～1.0)
 *       &gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType}"
 *                  valueLabel  = "{&#064;valueLabel}"
 *                  lowerBound  = "{&#064;lowerBound}"
 *                  upperBound  = "{&#064;upperBound}"
 *                  markValues  = "{&#064;markValues}"
 *                  markColors  = "{&#064;markColors}"
 *                  useGradient   = "{&#064;useGradient}"
 *                  shapesVisible = "{&#064;shapesVisible}"
 *                  useDottedLine = "{&#064;useDottedLine}"
 *                  seriesColors  = "{&#064;seriesColors}"
 *                  valueLabelsVisible = "[true|false]"
 *                  valueMarksVisible  = "[true|false]"
 *              &gt;
 *                         {&#064;SQL}
 *              &lt;/og:chartDataset&gt;
 *      &lt;/og:chartCreate&gt;
 *
 *    複数のグラフを重ね合わせる場合は、chartDataset タグを chartCreate のBODY部に
 *    複数記述します。
 *      &lt;og:chartCreate
 *          title       = "{&#064;title}"
 *          domainLabel = "{&#064;domainLabel}"
 *          width       = "{&#064;width}"
 *          height      = "{&#064;height}" &gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType1}"
 *                  valueLabel  = "{&#064;valueLabel1}"
 *                  lowerBound  = "{&#064;lowerBound1}"
 *                  upperBound  = "{&#064;upperBound1}"
 *                  markValues  = "{&#064;markValues1}"
 *                  markColors  = "{&#064;markColors1}"
 *              &gt;
 *                         {&#064;SQL1}
 *              &lt;/og:chartDataset&gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType2}"
 *                  valueLabel  = "{&#064;valueLabel2}"
 *                  lowerBound  = "{&#064;lowerBound2}"
 *                  upperBound  = "{&#064;upperBound2}"
 *                  markValues  = "{&#064;markValues2}"
 *                  markColors  = "{&#064;markColors2}"
 *              &gt;
 *                         {&#064;SQL2}
 *              &lt;/og:chartDataset&gt;
 *      &lt;/og:chartCreate&gt;
 *
 *    rectangleEdge属性 は、凡例の表示箇所を示す、RectangleEdge クラスの値を設定します。
 *    [rectangleEdge属性]
 *      TOP      上側
 *      BOTTOM   下側
 *      RIGHT    右側
 *      LEFT     左側
 *
 *    plotOrientation属性 は、チャートの軸表示方向を設定します。
 *    [renderer属性]
 *      VERTICAL    縦方向
 *      HORIZONTAL  横方向
 *
 * @og.group 画面表示
 *
 * @version  0.9.0	2007/06/19
 * @author	 Nakamura
 * @since	 JDK1.4,
 */
public class ChartCreateTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.7.3.0 (2017/01/27)" ;
	private static final long serialVersionUID = 673020170127L ;

	private static final String FILE_URL = HybsSystem.sys( "CHART_TEMP_DIR" );

	private static final String CHART_CACHE_KEY = HybsSystem.JF_CHART_CACHE_KEY;

	private	transient List<ChartDataset> chartDataset	;		// 6.3.9.0 (2015/11/06) transient 追加

	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private	transient Map<String,ChartCache> chrtMap ;	// キャッシュ用マップ			// 6.4.3.1 (2016/02/12) 変数名も変えておきます。

	private	String	title			;
	private	int		width			= 200;
	private	int		height			= 200;
	private	String	domainLabel		;
	private	boolean	showLegend		= true;
	private	String	rectangleEdge	= "BOTTOM";
	private	String	plotOrientation	= "VERTICAL";
	private	String	chartBackColor	;				// 背景色の設定
	private	String	plotBackColor	;				// 描画領域の設定
	private int		rotationLabel	= 3;			// 横軸ラベルの傾き(1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･ )
	private boolean	useVerticalLabels;				// 5.6.4.3 (2013/05/24) 横軸ラベルの表示向き(横書き:false/縦書き:true) 
	private	String	domainMarker	;				// 横軸のマーカーライン
	private	boolean	useDomainLabel	= true;			// 横軸ラベルの表示有無
	private	boolean	useMarkerLabel	= true;			// マーカーライン時の文字有無
	private	boolean	useCache		;				// JFreeChart オブジェクトをキャッシュするかどうか
	private int		seriesPikup		= -1;			// ピックアップするシリーズ番号
	private	String	masterKey		= "imgTag";		// キャッシュ用マスタキー			// 6.3.5.1 (2015/08/16) 初期値指定
	private int		cacheTime		= HybsSystem.sysInt( "CHART_CACHE_TIME" );			// 6.0.2.0 (2014/09/19) キャッシュ時の制限時間を、秒で指定
	private	String	imgTag			;				// キャッシュ用イメージタグ
	private	String	imageMapUrl		;				// クリッカブル・マップ用URL
	private	String	imageMapTarget	;				// クリッカブル・マップ用TARGET
	private long	dyStart			;				// 3.5.4.7 (2004/02/06) 実行時間測定用のDIV要素を出力します。
	private boolean	useTimeView		= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );		// 6.3.6.0 (2015/08/16)

	// 4.0.2.0 (2007/12/20)
	private	String	categoryMargin	;				// カテゴリマージン(0.0～1.0)
	private	String	lowerMargin		;				// 下方マージン(0.0～1.0)
	private	String	upperMargin		;				// 上方マージン(0.0～1.0)

	private	boolean	useDomainLine	;				// 4.0.3.0 (2008/01/07) 横軸のグリッド表示有無(垂直線)
	private	String	domainLineColor	;				// 4.0.3.0 (2008/01/07) 横軸のグリッド線の色
	private	boolean	useRangeLine	= true;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド表示有無(水平線)
	private	String	rangeLineColor	;				// 4.0.3.0 (2008/01/07) 縦軸のグリッド線の色
	private int		categorySkip	= 1;			// 4.0.3.0 (2008/01/07) 横軸ラベルをスキップする間隔
	private int		categoryCutNo	= -1;			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のカット文字数
	private String	categoryAnchor	;				// 4.1.1.0 (2008/02/14) 横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)
	private int		rangeSkip		= -1;			// 4.1.1.0 (2008/02/04) 縦軸のグリッドをスキップする間隔(初期値:-1)
	private	boolean	useToolTip		;				// 4.3.1.0 (2008/08/09) ツールチップスを利用するか

	private String	dateAxisFormat	= "MM/dd";		// 5.6.1.0 (2013/02/01) Time,XYTime Polt関係の時間軸を表す場合のフォーマット
	private int		dateSegSize		;				// 6.7.3.0 (2017/01/27) SegmentedTimelineのsegment size を、分単位で指定します。
	private int		dateSegIncluded ;				// 6.7.3.0 (2017/01/27) SegmentedTimelineのincluded segments(ｽﾍﾟｰｽ)を数値で指定します。
	private int		dateSegExcluded ;				// 6.7.3.0 (2017/01/27) SegmentedTimelineのexcluded segments(EE)を数値で指定します。
	private String	dateStartTime	;				// 6.7.3.0 (2017/01/27) SegmentedTimelineのStartTime(segment groupの開始位置) を、yyyyMMddHHmmss形式で指定します。 

	private String	chartFile		;				// 6.3.5.0 (2015/08/08) チャートの画像ファイル名

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ChartCreateTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.3.5.1 (2015/08/16) キャッシュ処理のロジックを整理します。
	 * @og.rev 6.3.6.0 (2015/08/16) キャッシュ処理のロジックを整理します。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。変数名も変えておきます。
	 *
	 * @return	後続処理の指示
	 */
	@SuppressWarnings(value={"unchecked"})
	@Override
	public int doStartTag() {
		if( !useTag() ) { return SKIP_BODY ; }	// 6.3.4.0 (2015/08/01)
		dyStart = System.currentTimeMillis();

		if( useCache ) {
			// キャッシュから、取り出す。
			chrtMap = (Map<String,ChartCache>)getObject( CHART_CACHE_KEY );						// 6.4.3.1 (2016/02/12) 変数名も変えておきます。
			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
			if( chrtMap == null ) {
				// 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
				chrtMap = new ConcurrentHashMap<>();	// キャッシュ用のMapが未作成の場合。	// 6.4.3.1 (2016/02/12) 変数名も変えておきます。
			}
			else {
				// キャッシュのタイムアウトを判定し、削除します。
				chrtMap.values().removeIf( cache -> !cache.isAvailable() );		// jdk1.8 		// 6.4.3.1 (2016/02/12) 変数名も変えておきます。

				final String cacheKey = getGUIInfoAttri( "KEY" ) + masterKey ;	// 6.4.3.1 (2016/02/12) 位置変更
				final ChartCache cache = chrtMap.get( cacheKey );				// 6.4.3.1 (2016/02/12) 変数名も変えておきます。
				if( cache != null && cache.isAvailable() ) {	// キャッシュ有効
					// キャッシュから、イメージを取り出す。存在しなければ、null
					imgTag = cache.getImageTag();				// 6.3.5.1 (2015/08/16) メソッド名を変更します。
					return SKIP_BODY ;							// キャッシュ使用
				}
			}
		}

		return EVAL_BODY_BUFFERED;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) debugPrint() メソッドの処理条件見直し
	 * @og.rev 6.0.2.0 (2014/09/19) キャッシュ時の制限時間対応
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.3.5.0 (2015/08/08) クリッカブル・マップ処理のロジックを整理します。
	 * @og.rev 6.3.5.1 (2015/08/16) キャッシュ処理のロジックを整理します。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。変数名も変えておきます。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 5.2.1.0 (2010/10/01) debugPrint() メソッド自体に、isDebug() が組み込まれている。
		if( !useTag() ) { return EVAL_PAGE ; }	// 6.3.4.0 (2015/08/01)

		final long queryEnd = System.currentTimeMillis();
		if( useTimeView ) {		// 6.3.6.0 (2015/08/16)
			jspPrint( "<div id=\"queryTime\" value=\"" + (queryEnd-dyStart) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		}
		// イメージタグのキャッシュがあれば、それを返す。
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( imgTag == null ) {
			// 6.0.2.0 (2014/09/19) キャッシュは、イメージタグのみ
			// チャートがない、または、無効になっている場合は、新規作成する。
			final ChartCreate chartCreate = create();
			final JFreeChart chart = chartCreate.getChart();

			final File file = getTempFile( FILE_URL );
			try {
				// クリッカブル・マップ
				final String fname = file.getName();
				final String filename = getContextPath() + "/" + FILE_URL + fname;

				// 6.3.5.0 (2015/08/08) クリッカブル・マップ処理のロジックを整理します。
				if( imageMapUrl != null || useToolTip ) {
					final String keyword = fname.substring( 0, fname.length() - 4 ); // ".png" 分を削除
					// 4.3.1.0 (2008/08/09) ツールチップスのマップ情報取得
					final ToolTipTagFragmentGenerator toolTipGen =
										useToolTip ? new StandardToolTipTagFragmentGenerator() : null ;
					// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
					final URLTagFragmentGenerator urlGen =
										imageMapUrl == null ? null : new HybsURLTagFragmentGenerator( imageMapTarget ) ;
					final ChartRenderingInfo objCri = new ChartRenderingInfo( new StandardEntityCollection() );

					ChartUtilities.saveChartAsPNG( file, chart, width, height, objCri );
					imgTag = ImageMapUtilities.getImageMap( keyword, objCri, toolTipGen, urlGen ) + makeTag2( filename, keyword );
				}
				else {
					ChartUtilities.saveChartAsPNG( file, chart, width, height );
					imgTag = makeTag2( filename, null );
				}

			}
			catch( final IOException ex ) {
				final String errMsg = "ファイル I/O が実行できませんでした。"
					+ CR + file + CR
						+ ex.getMessage();
				throw new HybsSystemException( errMsg,ex );
			}

			jspPrint( imgTag );

			// キャッシュ時にセーブする。
			if( useCache ) {
				final ChartCache cache = new ChartCache( imgTag,cacheTime );
				final String cacheKey = getGUIInfoAttri( "KEY" ) + masterKey ;
				chrtMap.put( cacheKey,cache );						// 6.4.3.1 (2016/02/12) 変数名も変えておきます。
				setObject( CHART_CACHE_KEY,chrtMap );				// 6.4.3.1 (2016/02/12) 変数名も変えておきます。
			}
		}
		// イメージタグのキャッシュがあれば、それを返す。
		else {
			jspPrint( imgTag );
		}

		// 6.3.5.1 (2015/08/16) 
		// ③ 直後の実行結果を、{&#064;CHART.TAG} 変数で取得できます。(requestスコープ)
		setRequestAttribute( "CHART.TAG" , imgTag );

		// 3.5.4.7 (2004/02/06)
		if( useTimeView ) {		// 6.3.6.0 (2015/08/16)
			final long dyEnd = System.currentTimeMillis();
			jspPrint( "<div id=\"viewTime\" value=\"" + (dyEnd-queryEnd) + "\"></div>" );	// 6.3.6.0 (2015/08/16)
		}
		return EVAL_PAGE;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.1.0 (2007/12/13) categoryMargin、lowerMargin、upperMargin 属性を追加
	 * @og.rev 4.1.1.0 (2008/02/04) categoryCutNo , categoryAnchor , rangeSkip 属性を追加
	 * @og.rev 5.6.1.0 (2013/02/01) dateAxisFormat 属性を追加
	 * @og.rev 5.6.4.3 (2013/05/24) useVerticalLabels 属性を追加
	 * @og.rev 6.0.2.0 (2014/09/19) rangeSkip 属性 の初期値を、-1(設定なし)に変更
	 * @og.rev 6.0.2.0 (2014/09/19) cacheTime 属性 の追加
	 * @og.rev 6.3.5.0 (2015/08/08) チャートの画像ファイル名指定に対応。
	 * @og.rev 6.3.5.1 (2015/08/16) cache は、ローカル変数とする。masterKey に、初期値指定。
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。変数名も変えておきます。
	 * @og.rev 6.7.3.0 (2017/01/27) 日付軸(dateAxis)の指定で、dateSegSize,dateSegIncluded,dateSegExcluded,dateStartTime を追加します。
	 */
	@Override
	protected void release2() {
		super.release2();
		chartDataset	= null;
		title			= null;
		width			= 200;
		height			= 200;
		domainLabel		= null;
		showLegend		= true;
		rectangleEdge	= "BOTTOM";
		plotOrientation	= "VERTICAL";
		chartBackColor	= null;			// 背景色の設定
		plotBackColor	= null;			// 描画領域の設定
		rotationLabel	= 3;			// 横軸ラベルの傾き(1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･ )
		useVerticalLabels = false;		// 5.6.4.3 (2013/05/24) 横軸ラベルの表示向き(横書き:false/縦書き:true) 
		domainMarker	= null;			// 横軸のマーカーライン
		useDomainLabel	= true;			// 横軸ラベルの表示有無
		useMarkerLabel	= true;			// マーカーライン時の文字有無
		useCache		= false;		// キャッシュするかどうか
		seriesPikup		= -1;			// ピックアップするシリーズ番号
		masterKey		= "imgTag";		// キャッシュ用マスタキー			// 6.3.5.1 (2015/08/16) 初期値指定
		cacheTime		= HybsSystem.sysInt( "CHART_CACHE_TIME" );			// 6.0.2.0 (2014/09/19) キャッシュ時の制限時間を、秒で指定
		chrtMap			= null;			// キャッシュ用マップ				// 6.4.3.1 (2016/02/12) 変数名も変えておきます。
		imgTag			= null;			// キャッシュ用イメージタグ
		imageMapUrl		= null;			// クリッカブル・マップ
		imageMapTarget	= null;			// クリッカブル・マップ用TARGET
		categoryMargin	= null;			// 4.0.2.0 (2007/12/20) カテゴリマージン(0.0～1.0)
		lowerMargin		= null;			// 4.0.2.0 (2007/12/20) 下方マージン(0.0～1.0)
		upperMargin		= null;			// 4.0.2.0 (2007/12/20) 上方マージン(0.0～1.0)
		useDomainLine	= false;		// 4.0.3.0 (2008/01/07) 横軸のグリッド表示有無(垂直線)
		domainLineColor	= null;			// 4.0.3.0 (2008/01/07) 横軸のグリッド線の色
		useRangeLine	= true;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド表示有無(水平線)
		rangeLineColor	= null;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド線の色
		categorySkip	= 1;			// 4.0.3.0 (2008/01/07) 横軸ラベルをスキップする間隔
		categoryCutNo	= -1;			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のカット文字数
		categoryAnchor	= null;			// 4.1.1.0 (2008/02/14) 横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)
		rangeSkip		= -1;			// 4.1.1.0 (2008/02/04) 縦軸のグリッドをスキップする間隔(初期値:-1)
		dateAxisFormat	= "MM/dd";		// 5.6.1.0 (2013/02/01) Time,XYTime Polt関係の時間軸を表す場合のフォーマット
		dateSegSize		= 0;			// 6.7.3.0 (2017/01/27) SegmentedTimelineのsegment size を、分単位で指定します。
		dateSegIncluded = 0;			// 6.7.3.0 (2017/01/27) SegmentedTimelineのincluded segments(ｽﾍﾟｰｽ)を数値で指定します。
		dateSegExcluded = 0;			// 6.7.3.0 (2017/01/27) SegmentedTimelineのexcluded segments(EE)を数値で指定します。
		dateStartTime	= null;			// 6.7.3.0 (2017/01/27) SegmentedTimelineのStartTime(segment groupの開始位置) を、yyyyMMddHHmmss形式で指定します。 
		chartFile		= null;			// 6.3.5.0 (2015/08/08) チャートの画像ファイル名
		useTimeView		= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );	// 6.3.6.0 (2015/08/16)
	}

	/**
	 * チャートを表示するためのタグを作成します。
	 *
	 * @param	filename	画像ファイル
	 * @param	keyword 	クリッカブル・マップの対応づける為のキーワード
	 *
	 * @return	タグ文字列
	 * @og.rtnNotNull
	 */
	private String makeTag2( final String filename,final String keyword ) {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( "<img" );

		if( keyword != null ) {
			rtn.append( " usemap=\"#").append( keyword ).append( '"');		// 6.0.2.5 (2014/10/31) char を append する。
		}
		rtn.append( " width=\""  ).append( width  )
			.append( "px\" height=\"" ).append( height  )
			.append( "px\" src=\""    ).append( filename )
			.append( "\" /> ");

		return rtn.toString();
	}

	/**
	 * ChartCreate のオブジェクトを生成します。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) categoryCutNo , categoryAnchor 属性を追加
	 * @og.rev 5.6.1.0 (2013/02/01) dateAxisFormat 属性を追加
	 * @og.rev 5.6.1.0 (2013/02/01) useVerticalLabels 属性を追加
	 * @og.rev 6.7.3.0 (2017/01/27) 日付軸(dateAxis)の指定で、dateSegSize,dateSegIncluded,dateSegExcluded,dateStartTime を追加します。
	 *
	 * @return  ChartCreateオブジェクト
	 * @og.rtnNotNull
	 */
	private ChartCreate create() {
		final ChartCreate tempCreate = new ChartCreate();

		tempCreate.setTitle( title );
		tempCreate.setDomainLabel( domainLabel );
		tempCreate.setShowLegend( showLegend );
		tempCreate.setRectangleEdge( rectangleEdge );
		tempCreate.setPlotOrientation( plotOrientation );
		tempCreate.setChartBackColor( chartBackColor );			// 背景色の設定
		tempCreate.setPlotBackColor( plotBackColor );			// 描画領域の設定
		tempCreate.setRotationLabel( rotationLabel );			// 横軸ラベルの傾き
		tempCreate.setUseVerticalLabels( useVerticalLabels );	// 5.6.4.3 (2013/05/24)) ラベルの表示向き
		tempCreate.setDomainMarker( domainMarker );				// ドメインマーカー
		tempCreate.setUseDomainLabel( useDomainLabel );			// 横軸ラベルの表示有無
		tempCreate.setUseMarkerLabel( useMarkerLabel );			// マーカーの設定値表示
		tempCreate.setSeriesPikup( seriesPikup );				// ピックアップするシリーズ番号
		tempCreate.setImageMapUrl( imageMapUrl );				// クリッカブル・マップ用URL
		tempCreate.setCategoryMargin( categoryMargin );			// 4.0.2.0 (2007/12/20) カテゴリマージン(0.0～1.0)
		tempCreate.setLowerMargin( lowerMargin );				// 4.0.2.0 (2007/12/20) 下方マージン(0.0～1.0)
		tempCreate.setUpperMargin( upperMargin );				// 4.0.2.0 (2007/12/20) 上方マージン(0.0～1.0)
		tempCreate.setUseDomainLine( useDomainLine );			// 4.0.3.0 (2008/01/07) 横軸のグリッド表示有無(垂直線)
		tempCreate.setDomainLineColor( domainLineColor );		// 4.0.3.0 (2008/01/07) 横軸のグリッド線の色
		tempCreate.setUseRangeLine( useRangeLine );				// 4.0.3.0 (2008/01/07) 縦軸のグリッド表示有無(水平線)
		tempCreate.setRangeLineColor( rangeLineColor );			// 4.0.3.0 (2008/01/07) 縦軸のグリッド線の色
		tempCreate.setCategorySkip( categorySkip );				// 4.0.3.0 (2008/01/07) 横軸ラベルをスキップする間隔
		tempCreate.setCategoryCutNo( categoryCutNo );			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のカット文字数
		tempCreate.setCategoryAnchor( categoryAnchor );			// 4.1.1.0 (2008/02/04) 横軸のグリッド(垂直線)の書き出し位置
		tempCreate.setRangeSkip( rangeSkip );					// 4.1.1.0 (2008/02/04) 縦軸のグリッド線(水平線)をスキップする間隔
		tempCreate.setUseToolTip( useToolTip );					// 4.3.1.0 (2008/08/09) ツールチップスの利用
		tempCreate.setDateAxisFormat( dateAxisFormat );			// 5.6.1.0 (2013/02/01) Time,XYTime Polt関係の時間軸を表す場合のフォーマット指定
		tempCreate.setSegmentedTimelineInfo( dateSegSize,dateSegIncluded,dateSegExcluded,dateStartTime );	// 6.7.3.0 (2017/01/27) SegmentedTimeline 関係

		if( isDebug() ) { tempCreate.setDebug( true );	}		// 4.0.2.0 (2007/12/20)

		tempCreate.setDatasetList( chartDataset );

		return tempCreate ;
	}

	/**
	 * テンポラリFile を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した テンポラリFile を作成します。
	 *
	 * ※ 6.3.5.0 (2015/08/08) で、chartFile属性を指定する事で、任意のファイルで
	 * 画像を作成できるようになりました。ただし、出力先フォルダは、CHART_TEMP_DIR
	 * です。
	 *	(CHART_TEMP_DIR[={@og.value SystemData#CHART_TEMP_DIR}])。
	 *
	 * @og.rev 6.3.5.0 (2015/08/08) チャートの画像ファイル名指定に対応。
	 *
	 * @param	fileURL	ファイルを作成するディレクトリ
	 *
	 * @return	テンポラリFile
	 * @og.rtnNotNull
	 */
	private File getTempFile( final String fileURL ) {

		final String directory = HybsSystem.url2dir( fileURL );
		final File dir = new File( directory );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
			throw new HybsSystemException( errMsg );
		}

		final File file ;
		try {
			// 6.3.5.0 (2015/08/08) チャートの画像ファイル名指定に対応。
			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
			if( chartFile == null ) {
				file = File.createTempFile( "JFree",".png",dir );
				file.deleteOnExit();
			}
			else {
				file = new File( dir , chartFile );
			}

		}
		catch( final IOException ex ) {
			final String errMsg = "ファイル名がオープン出来ませんでした。"
				+ CR
					+ "Url:" + fileURL ;
			throw new HybsSystemException( errMsg,ex );
		}

		return file ;
	}

	/**
	 * 【TAG】チャートのタイトルをセットします。
	 *
	 * @og.tag チャートのタイトルをセットします。
	 *
	 * @param   ttl タイトル
	 */
	public void setTitle( final String ttl ) {
		title = nval( getRequestParameter( ttl ),title );
	}

	/**
	 * 【TAG】チャートの横幅をセットします(初期値:200)。
	 *
	 * @og.tag
	 * タイトルや凡例も含んだ大きさです。データ領域は自動計算されます。
	 *
	 * @og.rev 4.0.1.0 (2007/12/13) 引数に "px" を使用できるようにする。
	 *
	 * @param   wd 横幅
	 */
	public void setWidth( final String wd ) {
		// 引数に、"px" が使用されていた場合は、それを取り除く。
		if( wd != null && wd.length() >= 3 && wd.endsWith( "px" ) ) {
			width = Integer.parseInt( wd.substring( 0,wd.length()-2 ) );		// 6.0.2.4 (2014/10/17) メソッド間違い
		}
		else {
			width = nval( getRequestParameter( wd ),width );
		}
	}

	/**
	 * 【TAG】チャートの縦幅をセットします(初期値:200)。
	 *
	 * @og.tag
	 * タイトルや凡例も含んだ大きさです。データ領域は自動計算されます。
	 *
	 * @og.rev 4.0.1.0 (2007/12/13) 引数に "px" を使用できるようにする。
	 *
	 * @param   ht 縦幅
	 */
	public void setHeight( final String ht ) {
		// 引数に、"px" が使用されていた場合は、それを取り除く。
		if( ht != null && ht.length() >= 3 && ht.endsWith( "px" ) ) {
			height = Integer.parseInt( ht.substring( 0,ht.length()-2 ) );		// 6.0.2.4 (2014/10/17) メソッド間違い
		}
		else {
			height = nval( getRequestParameter( ht ),height );
		}
	}

	/**
	 * 【TAG】チャートのドメインラベルを指定します。
	 *
	 * @og.tag
	 * チャートのドメインラベルを指定します。
	 *
	 * @param   dmLbl ドメインラベル
	 */
	public void setDomainLabel( final String dmLbl ) {
		domainLabel = nval( getRequestParameter( dmLbl ),domainLabel );
	}

	/**
	 * 【TAG】チャートの凡例の表示可否[true/false]をセットします(初期値:true[表示する])。
	 *
	 * @og.tag
	 * 初期値は、表示する(true) です。
	 *
	 * @param   swLegend 凡例の表示可否 [true:表示する/それ以外:しない]
	 */
	public void setShowLegend( final String swLegend ) {
		showLegend = nval( getRequestParameter( swLegend ),showLegend );
	}

	/**
	 * 【TAG】チャートの表示箇所を、[TOP/BOTTOM/RIGHT/LEFT]で指定します(初期値:BOTTOM)。
	 *
	 * @og.tag
	 * 表示箇所は、org.jfree.ui.RectangleEdge クラスの設定値を使用します。
	 * 指定できるのは、TOP、BOTTOM、RIGHT、LEFT で、各文字の頭一文字で
	 * 判定してます。つまり、T,B,R,L で、判定を行い、それ以外はエラーになります。
	 * 初期値は、BOTTOM です。
	 *
	 * @param   rectEdge 凡例表示方向 [TOP/BOTTOM/RIGHT/LEFT]
	 */
	public void setRectangleEdge( final String rectEdge ) {
		rectangleEdge = nval( getRequestParameter( rectEdge ),rectangleEdge );
	}

	/**
	 * 【TAG】チャートのプロット(Plot)表示方向を、[VERTICAL(or V)/HORIZONTAL(or H)]で指定します(初期値:VERTICAL)。
	 *
	 * @og.tag
	 * 軸表示方向は、org.jfree.chart.plot.PlotOrientation クラスの設定値を使用します。
	 * 指定できるのは、VERTICAL、HORIZONTALで、各文字の頭一文字で
	 * 判定してます。つまり、V,H で、判定を行い、それ以外はエラーになります。
	 * 初期値は、VERTICAL です。
	 *
	 * @param   orientation 表示方向 [VERTICAL(or V)/HORIZONTAL(or H)]
	 */
	public void setPlotOrientation( final String orientation ) {
		plotOrientation = nval( getRequestParameter( orientation ),plotOrientation );
	}

	/**
	 * 【TAG】チャートの背景色を指定します。
	 *
	 * @og.tag
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @param   chBackClr 背景色
	 * @see		java.awt.Color#BLACK
	 */
	public void setChartBackColor( final String chBackClr ) {
		chartBackColor = nval( getRequestParameter( chBackClr ),chartBackColor );
	}

	/**
	 * 【TAG】チャートの描画領域の色を指定します。
	 *
	 * @og.tag
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @param   plBackClr 描画領域色
	 * @see		java.awt.Color#BLACK
	 */
	public void setPlotBackColor( final String plBackClr ) {
		plotBackColor = nval( getRequestParameter( plBackClr ),plotBackColor );
	}

	/**
	 * 【TAG】チャートのカテゴリーラベルの方向を指定します(初期値は、3:60度)。
	 *
	 * @og.tag
	 * 方向は、上方向に対して、(PI / 指示数) で求まる値に設定します。
	 * この指示数に相当する値を設定します。
	 * 1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･
	 * マイナスは、下方向に回転させます。
	 * 0 を指定した場合は、何も設定しません。
	 * 初期値は、3:60度です。
	 *
	 * @param	rttLabel	ラベルの方向 (例:1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･)
	 */
	public void setRotationLabel( final String rttLabel ) {
		rotationLabel = nval( getRequestParameter( rttLabel ),rotationLabel );
	}

	/**
	 * 【TAG】ラベルの表示向きを縦にするかどうか[false/true]を指定します(初期値:false[横書き])。
	 *
	 * @og.tag
	 * ChartCreate の rotationLabel は、角度を指定できましたが、NumberAxis,DateAxis では、
	 * 縦にするかどうかの指定しかできません。
	 * ここでは、true を指定するとラベルは、縦書きになります。
	 * 初期値は、false(横書き)です。
	 *
	 * @og.rev 5.6.4.3 (2013/05/24)) 新規追加
	 *
	 * @param	useVLavels	ラベルの表示向き [false:横書き/true:縦書き]
	 */
	public void setUseVerticalLabels( final String useVLavels ) {
		useVerticalLabels = nval( getRequestParameter( useVLavels ),useVerticalLabels );
	}

	/**
	 * 【TAG】チャートの横軸の値(ドメイン)に合致する位置にマーカーラインを設定します。
	 *
	 * @og.tag
	 * この属性には、マーカーラインを設定する値を記述します。
	 *
	 * @param   marker 横軸マーカーライン
	 */
	public void setDomainMarker( final String marker ) {
		domainMarker = nval( getRequestParameter( marker ),domainMarker );
	}

	/**
	 * 【TAG】横軸ラベルのラベルを表示するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * ドメイン(横軸)が、多数存在する場合、ドメインラベルが見えにくくなります。
	 * そのようなケースで、横軸のラベルそのものを表示しない場合に、false を
	 * 設定します。
	 * 初期値は、表示する(true)です。
	 *
	 * @param   flag 横軸ラベルの表示有無 [true:表示する/false:表示しない]
	 */
	public void setUseDomainLabel( final String flag ) {
		useDomainLabel = nval( getRequestParameter( flag ),useDomainLabel );
	}

	/**
	 * 【TAG】マーカーラインに、その設定値を表示するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * ドメイン(横軸)が、多数存在する場合、ドメインラベルが見えない場合があります。
	 * そのようなケースで、見たい値にマーカーラインを設定し、その横に、ドメインラベル
	 * を表示する事で、ピックアップしている軸の値を容易に知ることが可能です。
	 * 初期値は、表示する(true)です。
	 *
	 * @param   flag 設定値表示 [true:する/false:しない]
	 */
	public void setUseMarkerLabel( final String flag ) {
		useMarkerLabel = nval( getRequestParameter( flag ),useMarkerLabel );
	}

	/**
	 * 【TAG】JFreeChartで作成されたイメージタグをキャッシュするかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * useCache="true" を設定すると、session に、HybsSystem.JF_CHART_CACHE_KEY
	 * キーで、Mapが、キャッシュされます。そのMap に、画面ID + masterKey 単位にキャッシュされます。
	 * 6.0.2.0 (2014/09/19) 以降は、イメージタグのみキャッシュしますので、
	 * domainMarker 属性と、seriesPikup 属性を再設定する場合は、画像を再作成する
	 * 必要がありますので、masterKey を変更する必要があります。
	 * (または、seriesPikup 属性をmasterKeyに含めてください。)
	 *
	 * 初期値は、キャッシュしない(false)です。
	 *
	 * @param   flag キャッシュ可否 [true:する/false:しない]
	 * @see     #setDomainMarker( String )
	 * @see     #setSeriesPikup( String )
	 * @see		#setCacheTime( String )
	 */
	public void setUseCache( final String flag ) {
		useCache = nval( getRequestParameter( flag ),useCache );
	}

	/**
	 * 【TAG】キャッシュ時の制限時間を、秒で指定します
	 *		(初期値:CHART_CACHE_TIME[={@og.value SystemData#CHART_CACHE_TIME}])。
	 *
	 * @og.tag
	 * useCache="true" に設定した場合、キャッシュを使用します。
	 * この場合、cacheTime で指定した時間は、キャッシュが使用されます。
	 * 制限時間を超えたキャッシュは破棄され、新しいキャッシュにセットされます。
	 * 
	 * 初期値:システム定数のCHART_CACHE_TIME[={@og.value SystemData#CHART_CACHE_TIME}])。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 *
	 * @param	ccTime	キャッシュ時の制限時間(秒)
	 * @see		#setMasterKey( String )
	 * @see		#setUseCache( String )
	 */
	public void setCacheTime( final String ccTime ) {
		cacheTime = nval( getRequestParameter( ccTime ),cacheTime );
	}

	/**
	 * 【TAG】useCache="true" 時のキャッシュの保存先scopeを指定します(初期値:session)。
	 *
	 * @og.tag
	 * 通常のグラフのキャッシュは、ユーザー単位に行う(session)スコープに保管されます。
	 * 現状の JFreeChart でのグラフは、作成毎に、実イメージファイルを作成し、それを
	 * 返すため、ほとんど書き換えが無く、多くにユーザーがアクセスする様なケース・・例えば、
	 * トップメニューに月次損益のグラフを表示するなどの場合、無駄にファイルが作成されます。
	 * そこで、ユーザー間も共有する様なグラフには、cacheScope="application" とすることで、
	 * 全ユーザー共通のキャッシュにすることができます。
	 * 一応、スコープとしては、"request","page","session","application" が指定できます。
	 *
	 * @og.rev 6.3.5.1 (2015/08/16) キャッシュ処理のロジックを整理します。新規追加。
     *
	 * @param  scp スコープ [request/page/session/application]
	 * @see		#getScope()
	 */
	public void setCacheScope( final String scp ) {
		setScope( scp );
	}

	/**
	 * 【TAG】キャッシュ用マスタキーを指定します(初期値:imgTag)。
	 *
	 * @og.tag
	 * useCache="true" に設定した場合、このキーを用いて、キャッシュします。
	 * 実際のキャッシュは、画面ID ＋ masterKey になります。
	 *
	 * ※ 6.0.2.0 (2014/09/19) 以降は、イメージタグのみキャッシュしますので、
	 * 従来の様に、seriesPikup 違いに応じた処理は行いません。
	 * よって、useCache="true"  で、seriesPikup 違いの場合は、再検索させる
	 * 必要がありますので、masterKey を変更する必要があります。
	 * (または、seriesPikup 属性をmasterKeyに含めてください。)
	 * なお、キャッシュ保持時間は、#setCacheTime( String ) で指定できます。
	 *
	 * @og.rev 6.3.5.1 (2015/08/16) masterKey に、初期値指定。
	 *
	 * @param	mkey	キャッシュ用マスタキー
	 * @see		#setUseCache( String )
	 * @see		#setCacheTime( String )
	 */
	public void setMasterKey( final String mkey ) {
		masterKey = nval( getRequestParameter( mkey ),masterKey );
	}

	/**
	 * 【TAG】複数シリーズのピックアップを行う場合のシリーズ番号を指定します。
	 *
	 * @og.tag
	 * 複数シリーズ(検索時に複数項目を同時に検索する場合)では、チャート上に
	 * 複数のグラフが表示されますが、その中の一つをピックアップする場合に、
	 * シリーズ番号を指定します。
	 * シリーズ番号は、0 から始まる数字です。
	 * ここでは、ピックアップされたシリーズは、赤色で表示されます。
	 * それ以外は、グレー色での表示になります。
	 * seriesPikup を使用すると、chartDataset タグの useValueVisible 属性が
	 * 影響を受けます。この属性は、データの値(itemText)を表示しますが、
	 * seriesPikup が指定された場合は、そのシリーズのみにラベル表示します。
	 *
	 * @param	pikup	シリーズ番号
	 */
	public void setSeriesPikup( final String pikup ) {
		seriesPikup = nval( getRequestParameter( pikup ),seriesPikup );
	}

	/**
	 * 【TAG】クリッカブル・マップ用URLを指定します。
	 *
	 * @og.tag
	 * 画像に、クリッカブル・マップを作成する場合の、URL を指定します。
	 * これは、画像上にエリア指定でリンク引数を作成することが可能です。
	 * URL 自身は、? 付きで固定値の引数を連結することが可能です。
	 * クリックしたエリアのカテゴリやインデックスの値(引数)は、自動的に
	 * 設定されます。(指定しない場合はチャートによって異なります)
	 * <pre>
	 * ・Pie      ：category、pieIndex
	 * ・XY       ：series、item
	 * ・Category ：series、category
	 * </pre>
	 * この引数の URL の名称を変更したい場合は、URL に続けて、カンマ(,) で、
	 * 名称を記述してください。
	 * 例：link.jsp,BLOCK
	 *
	 * @param	imap	クリッカブル・マップ用URL
	 * @see #setImageMapTarget( String )
	 */
	public void setImageMapUrl( final String imap ) {
		imageMapUrl = nval( getRequestParameter( imap ),imageMapUrl );
	}

	/**
	 * 【TAG】クリッカブル・マップ用TARGETを指定します。
	 *
	 * @og.tag
	 * 画像に、クリッカブル・マップを作成する場合の、TARGET を指定します。
	 * これは、画像上にエリア指定でリンクを作成する場合のフレーム指定です。
	 *
	 * @param	target	クリッカブル・マップ用TARGET
	 * @see #setImageMapUrl( String )
	 */
	public void setImageMapTarget( final String target ) {
		imageMapTarget = nval( getRequestParameter( target ),imageMapTarget );
	}

	/**
	 * 【TAG】カテゴリマージン(0.0～1.0)を指定します(初期値:null 自動調整)。
	 *
	 * @og.tag
	 * カテゴリ(グラフの横軸に相当)の表示間隔(マージン)の比率を指定します。
	 * この比率は、％ ではなく、数字(double)での設定になります。
	 * 何も指定しない場合は、デフォルトで自動調整されます。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	margin	カテゴリマージン(0.0～1.0)
	 */
	public void setCategoryMargin( final String margin ) {
		categoryMargin = nval( getRequestParameter( margin ),categoryMargin );
	}

	/**
	 * 【TAG】下方マージン(0.0～1.0)を指定します(初期値:null 自動調整)。
	 *
	 * @og.tag
	 * カテゴリ(グラフの横軸に相当)の下方側(左側)のマージンの比率を指定します。
	 * この比率は、％ ではなく、数字(double)での設定になります。
	 * 何も指定しない場合は、デフォルトで自動調整されます。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	margin	下方マージン(0.0～1.0)
	 */
	public void setLowerMargin( final String margin ) {
		lowerMargin = nval( getRequestParameter( margin ),lowerMargin );
	}

	/**
	 * 【TAG】上方マージン(0.0～1.0)を指定します(初期値:null 自動調整)。
	 *
	 * @og.tag
	 * カテゴリ(グラフの横軸に相当)の上方側(右側)のマージンの比率を指定します。
	 * この比率は、％ ではなく、数字(double)での設定になります。
	 * 何も指定しない場合は、デフォルトで自動調整されます。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	margin	上方マージン(0.0～1.0)
	 */
	public void setUpperMargin( final String margin ) {
		upperMargin = nval( getRequestParameter( margin ),upperMargin );
	}

	/**
	 * 【TAG】横軸のグリッド表示有無(垂直線)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ドメイン(横軸)に対する、グリッドライン(垂直線)を表示するかどうか指定します。
	 *
	 * 何も指定しない場合は、表示しません。(false)
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	useLine	横軸のグリッド表示有無 [true:する/false:しない]
	 */
	public void setUseDomainLine( final String useLine ) {
		useDomainLine = nval( getRequestParameter( useLine ),useDomainLine );
	}

	/**
	 * 【TAG】横軸のグリッド線の色を指定します。
	 *
	 * @og.tag
	 * ドメイン(横軸)に対する、グリッドラインの表示色を指定します。
	 * 何も指定しない場合は、デフォルトで自動設定されます。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	color	横軸のグリッド線の色
	 */
	public void setDomainLineColor( final String color ) {
		domainLineColor = nval( getRequestParameter( color ),domainLineColor );
	}

	/**
	 * 【TAG】横軸ラベルをスキップする間隔を指定します(初期値:1 すべて表示)。
	 *
	 * @og.tag
	 * 横軸ラベル(カテゴリラベル表示)する際に、スキップする間隔を指定します。
	 * "1" (初期値)では、１つづつ表示(つまり、すべて表示する)します。
	 * "2" とすると、１つおきに、"3" とすると、２つおきに表示します。
	 * 初期値は、"1" (すべて表示)です。
	 * なお、先頭から表示を開始します。
	 *
	 * 注意：これとは別に、ラベル先頭に "_" を付けた場合は、ラベルを表示しません。
	 * また、categoryCutNo が指定された場合は、categorySkip は使用されません。
	 *
	 * @param	step	スキップする間隔
	 * @see		#setCategoryCutNo( String )
	 */
	public void setCategorySkip( final String step ) {
		categorySkip = nval( getRequestParameter( step ),categorySkip );
	}

	/**
	 * 【TAG】横軸ラベルの文字位置指定のカット文字数を指定します(初期値:-1 すべて表示)。
	 *
	 * @og.tag
	 * 横軸ラベル(カテゴリラベル表示)する際に、ラベルの先頭から、この指定文字数だけ
	 * カットして、表示します。
	 * その際、前回作成したカットラベルと、同一ラベルの場合は、表示しません。
	 * 例えば、データは、年月日で、年と月のみ(先頭６文字)を指定すると、
	 * 日のデータは、ラベルが表示されません。
	 * 指定される数字は、１以上の整数としてください。
	 * 初期値は、すべて表示です。
	 *
	 * 注意：これとは別に、ラベル先頭に "_" を付けた場合は、ラベルを表示しません。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param	cutNo	カット文字数位置
	 * @see		#setCategorySkip( String )
	 */
	public void setCategoryCutNo( final String cutNo ) {
		categoryCutNo = nval( getRequestParameter( cutNo ),categoryCutNo );
	}

	/**
	 * 【TAG】横軸のグリッド(垂直線)の書き出し位置[START/MIDDLE/END]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 横軸のグリッド(垂直線)を、グラフのどの位置に記述するかを指定します。
	 * 具体的な値は、CategoryAnchor オブジェクトの値になります。
	 * ここでは、文字列で(START,MIDDLE,END)を指定します。(先頭一文字で判定)
	 * 何も指定しない場合は、デフォルト(MIDDLE)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/14) 新規追加
	 *
	 * @param	anchor	書き出し位置 [START/MIDDLE/END]
	 */
	public void setCategoryAnchor( final String anchor ) {
		categoryAnchor = nval( getRequestParameter( anchor ),categoryAnchor );

		if( categoryAnchor != null && categoryAnchor.length() > 0 ) {
			if( ! "START".equals( categoryAnchor )  &&
				! "MIDDLE".equals( categoryAnchor ) &&
				! "END".equals( categoryAnchor ) ) {
					final String errMsg = "指定のAnchorは適用できません。[" + anchor + "]"
							+ CR
							+ "START,MIDDLE,END の中から、指定してください。" ;
					throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】縦軸のグリッド表示有無(水平線)を指定します(初期値:true)。
	 *
	 * @og.tag
	 * レンジ(縦軸)に対する、グリッドライン(垂直線)を表示するかどうか指定します。
	 *
	 * 何も指定しない場合は、表示しません。(false)
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	useLine	グリッド表示有無 [true:する/false:しない]
	 */
	public void setUseRangeLine( final String useLine ) {
		useRangeLine = nval( getRequestParameter( useLine ),useRangeLine );
	}

	/**
	 * 【TAG】縦軸のグリッド線(水平線)の色を指定します(初期値:null 自動設定)。
	 *
	 * @og.tag
	 * レンジ(縦軸)に対する、グリッドラインの表示色を指定します。
	 * 何も指定しない場合は、デフォルトで自動設定されます。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	color	縦軸のグリッド線(水平線)の色
	 */
	public void setRangeLineColor( final String color ) {
		rangeLineColor = nval( getRequestParameter( color ),rangeLineColor );
	}

	/**
	 * 【TAG】縦軸のグリッド線(水平線)をスキップする間隔を指定します(初期値:-1)。
	 *
	 * @og.tag
	 * 縦軸のグリッド線(水平線)を表示する際に、スキップする間隔を指定します。
	 * 通常は、ラベルと同じだけのグリッド線が掛かれますが、ラベルよりも
	 * 少ない数のグリッド線(例えば、２つおき)を出す場合に、値を設定します。
	 * "1" (初期値)では、１つづつ表示(つまり、すべて表示する)します。
	 * "2" とすると、１つおきに、"3" とすると、２つおきに表示します。
	 * なお、先頭から表示を開始します。
	 *
	 * 6.0.2.0 (2014/09/19) 前回の JFreeChart のVerUpで、グリッド線の表示が
	 * ５個おきに変わったようです。設定する値を マイナスにすると、初期設定の値を
	 * 使用するように変更します。微調整は、個々にお願いします。
	 *
	 * 初期値は、"-1" (設定しない)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 * @og.rev 6.0.2.0 (2014/09/19) rangeSkip 属性 の初期値を、-1(設定なし)に変更
	 *
	 * @param	rngSkip	縦軸のグリッド線(水平線)をスキップする間隔
	 */
	public void setRangeSkip( final String rngSkip ) {
		rangeSkip = nval( getRequestParameter( rngSkip ),rangeSkip );
	}

	/**
	 * 【TAG】ツールチップスの使用可否[true:利用する/false:利用しない]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ラベルを利用する際に、ラベルと図面、隣のラベル同士が重なることがあります。
	 * この場合、ツールチップスの利用をお勧めします。
	 * 初期値は'false'です。
	 *
	 * @og.rev 4.3.1.0 (2008/08/09) 新規追加
	 *
	 * @param   toolTip ツールチップス [true:利用/false:利用しない]
	 */
	public void setUseToolTip( final String toolTip ) {
		useToolTip = nval( getRequestParameter( toolTip ), useToolTip );
	}

	/**
	 * 【TAG】Time,XYTime Polt関係の時間軸を表す場合のフォーマットを指定します(初期値:MM/dd)。
	 *
	 * @og.tag
	 * TimeSeries 関連のデータを扱う場合の横軸の日付フォーマットを指定します。
	 * 日付フォーマットは、java.text.SimpleDateFormat で使われる yyyy/MM/dd HH:mm:ss 表記です。
	 * 
	 * 初期値は、"MM/dd" です。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) 新規追加
	 *
	 * @param	dtFormat	時間軸を表すフォーマット
	 */
	public void setDateAxisFormat( final String dtFormat ) {
		dateAxisFormat = nval( getRequestParameter( dtFormat ), dateAxisFormat );
	}

	/**
	 * 【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのsegment size を、分単位で指定します。
	 *
	 * @og.tag
	 * 日単位を指定する場合は、24*60 = 1440 を指定します。
	 * 0 または、何も指定しない場合は、SegmentedTimeline を使用しません。
	 * <pre>
	 *   start time
	 *     |
	 *     v
	 *     0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 ...
	 *   +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+...
	 *   |  |  |  |  |  |EE|EE|  |  |  |  |  |EE|EE|  |  |  |  |  |EE|EE|
	 *   +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+...
	 *   [_____________] [___]            [_]
	 *          |          |               |
	 *       included   excluded        segment
	 *       segments   segments         size
	 *   [___________________]
	 *              |
	 *         segment group
	 * </pre>
	 *
	 * @og.rev 6.7.3.0 (2017/01/27) 新規追加
	 *
	 * @param	segSize	SegmentedTimelineのsegment size を、分単位で指定
	 */
	public void setDateSegSize( final String segSize ) {
		dateSegSize = nval( getRequestParameter( segSize ), dateSegSize );
	}

	/**
	 * 【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのincluded segments(ｽﾍﾟｰｽ)を数値で指定します。
	 *
	 * @og.tag
	 * segment group内で、スペース(空ける) になる segment数 を指定します。
	 * この値は、dateSegSize を指定しない限り、使用されません。
	 *
	 * @og.rev 6.7.3.0 (2017/01/27) 新規追加
	 *
	 * @param	segIncluded	SegmentedTimelineのincluded segments(ｽﾍﾟｰｽ)を数値で指定
	 * @see		#setDateSegSize( String )
	 */
	public void setDateSegIncluded( final String segIncluded ) {
		dateSegIncluded = nval( getRequestParameter( segIncluded ), dateSegIncluded );
	}

	/**
	 * 【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのexcluded segments(EE)を数値で指定します。
	 *
	 * @og.tag
	 * segment group内で、excluded する segment数 を指定します。
	 * この値は、dateSegSize を指定しない限り、使用されません。
	 *
	 * @og.rev 6.7.3.0 (2017/01/27) 新規追加
	 *
	 * @param	segExcluded	SegmentedTimelineのexcluded segments(EE)を数値で指定
	 * @see		#setDateSegSize( String )
	 */
	public void setDateSegExcluded( final String segExcluded ) {
		dateSegExcluded = nval( getRequestParameter( segExcluded ), dateSegExcluded );
	}

	/**
	 * 【TAG】Time,XYTime Polt関係の時間軸で、SegmentedTimelineのStartTime(segment groupの開始位置) を、yyyyMMddHHmmss形式で指定します。
	 *
	 * @og.tag
	 * TimeSeries 関連のデータを扱う場合の横軸に、SegmentedTimelineを指定します。
	 * 日付フォーマットは、yyyyMMddHHmmss形式で指定します。
	 * 
	 * この値は、dateSegSize を指定しない限り、使用されません。
	 * 初期値は、null(自動設定) です。
	 *
	 * @og.rev 6.7.3.0 (2017/01/27) 新規追加
	 *
	 * @param	startTime	SegmentedTimelineのStartTime(segment groupの開始位置)
	 * @see		#setDateSegSize( String )
	 */
	public void setDateStartTime( final String startTime ) {
		dateStartTime = nval( getRequestParameter( startTime ), dateStartTime );
	}

	/**
	 * 【TAG】チャートの画像ファイル名を指定します(初期値:自動)。
	 *
	 * @og.tag
	 * チャートは、png 形式のファイルとして作成されます。そのファイル名を指定します。
	 * ファイルは、CHART_TEMP_DIR 以下に、作成されます。
	 * 原仕様では、作成するフォルダを指定できません。
	 * 何も指定しない場合は、JFreeXXXXX.png 形式のテンポラリファイル名で作成されます。
	 * この場合は、File#deleteOnExit() を付与するので、Tomcat停止時に自動的に削除されます。
	 * chartFile 属性で指定した場合は、自動削除されませんので、ご注意ください。
	 * 
	 * 初期値は、自動( File.createTempFile( "JFree",".png" , CHART_TEMP_DIR ) ) です。
	 *	(CHART_TEMP_DIR[={@og.value SystemData#CHART_TEMP_DIR}])。
	 *
	 * @og.rev 6.3.5.0 (2015/08/08) チャートの画像ファイル名指定に対応。
	 *
	 * @param	file	時間軸を表すフォーマット
	 */
	public void setChartFile( final String file ) {
		chartFile = nval( getRequestParameter( file ), chartFile );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します
	 *		(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するバーイメージが表示されます。
	 * これは、DB検索、APサーバー処理、画面表示の各処理時間をバーイメージで
	 * 表示させる機能です。処理時間の目安になります。
	 * (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * BODY要素に記述した、chartDataset タグの属性設定クラスをListに追加します。
	 *
	 * BODY部に記述された、chartDataset タグの属性は、チャートのデータ取得Queryや、
	 * チャートの種類などの情報を管理しています。
	 * チャートの種類(Category系)によっては、チャートを合成して表示させることが可能です。
	 * それらの複数のチャートの指定が、chartDataset タグ で行われます。
	 *
	 * @param	chDataset	chartDatasetタグの属性設定クラス
	 */
	protected void addChartDataset( final ChartDataset chDataset ) {
		if( chartDataset == null ) { chartDataset = new ArrayList<>(); }
		chartDataset.add( chDataset );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {

		// 6.0.2.5 (2014/10/31) char を append する。
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( '[' ).append( this.getClass().getName() ).append( ']' ).append( CR )
			.append( "title           [" ).append( title           ).append( ']' ).append( CR )
			.append( "width           [" ).append( width           ).append( ']' ).append( CR )
			.append( "height          [" ).append( height          ).append( ']' ).append( CR )
			.append( "domainLabel     [" ).append( domainLabel     ).append( ']' ).append( CR )
			.append( "showLegend      [" ).append( showLegend      ).append( ']' ).append( CR )
			.append( "rectangleEdge   [" ).append( rectangleEdge   ).append( ']' ).append( CR )
			.append( "plotOrientation [" ).append( plotOrientation ).append( ']' ).append( CR )
			.append( "chartBackColor  [" ).append( chartBackColor  ).append( ']' ).append( CR )
			.append( "plotBackColor   [" ).append( plotBackColor   ).append( ']' ).append( CR );

		return rtn.toString();
	}

	/**
	 * 6.0.2.0 (2014/09/19) キャッシュ時間の外部設定、キャッシュ対象の変更
	 * 
	 * イメージタグをキャッシュするクラスです。
	 * キャッシュの保持時間は、コンストラクタで指定します。
	 * 従来は、JFreeChart と、画像ファイルのキャッシュを行っていましたが
	 * JFreeChart オブジェクトをキャッシュするメリットがあまりない為、
	 * イメージタグのキャッシュのみに変更しました。
	 * キャッシュの有効期間の初期値は、システムリソースの CHART_CACHE_TIME で定義します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) キャッシュは、イメージタグのみ
	 * @og.rev 6.3.5.1 (2015/08/16) キャッシュ処理のロジックを整理します。
	 */
	private static final class ChartCache {

		// 6.0.2.0 (2014/09/19) イメージタグのみのキャッシュなので、seriesPikup は関係なくなった。
		private final	String		imgTag	;		// キャッシュのキーをタグそのものに変更
		private final	long		createLimitTime	;

		/**
		 * コンストラクター
		 * JFreeChartで作成sれたイメージタグをキャッシュするキーを指定します。
		 * 内部的には、このキー(mkey)で、比較や hashCode が計算されています。
		 * キャッシュの有効期間は、ccTime (秒) で定義します。
		 * イメージタグの作成時間が、それを超える場合は、キャッシュを初期化します。
		 *
		 * @og.rev 6.0.2.0 (2014/09/19) キャッシュ時の制限時間対応
		 * @og.rev 6.3.5.1 (2015/08/16) コンストラクターで、イメージタグの設定と有効時間の算出を行います。
		 *
		 * @param	imgTag	イメージタグ
		 * @param	ccTime	キャッシュの制限時間(秒)
		 */
		public ChartCache( final String imgTag,final int ccTime ) {
			this.imgTag	= imgTag;
			createLimitTime = System.currentTimeMillis() + ccTime * 1000L;
		}

		/**
		 * イメージタグを返します。
		 * キャッシュされていれば、そのキャッシュを、そうでなければ、null を
		 * 返します。
		 *
		 * @og.rev 6.3.5.1 (2015/08/16) メソッド名を変更します。
		 *
		 * @return   イメージタグ
		 */
		public String getImageTag() {
			return imgTag ;
		}

		/**
		 * キャッシュが、有効かどうかを返します。(有効:true)
		 * キャッシュは、コンストラクタ引数で指定された時間が経過すると、
		 * 無効と判断されます。
		 * 有効であれば、true を返します。
		 *
		 * @return	有効かどうか[有効:true/無効:false]
		 */
		public boolean isAvailable() {
			return System.currentTimeMillis() < createLimitTime ;
		}

	}
}
