/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.develop;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;

import org.opengion.hayabusa.develop.AbstractJspCreate;
import org.opengion.hayabusa.develop.JspConvertEntity;
import org.opengion.fukurou.xml.OGElement;
import org.opengion.fukurou.xml.OGNode;
import org.opengion.fukurou.xml.OGNodeType;

/**
 * query.jspの&lt;og:column&gt;タグを作成します。
 * column タグは、部分置換ではなく、table 部分からの全面置換です。（部分置換は難しかったので）
 * table には、tr を配置し、column タグを、TD_COUNT（初期値=3）の繰り返し数だけ配置します。
 * それを超えると、新しい tr に書き出していきます。
 * TR_COUNT（初期値=2）を超える検索条件は、og:hideMenu として、作成します。
 *
 * ●使用例
 *      &lt;table summary = "layout" &gt;
 *          &lt;tr&gt;
 *              &lt;og:column
 *                  name       = column.getColumnName() 
 *                  defaultVal = column.getDefaultValue()
 *                  must       = "true"         ("1".equals( column.getMust() ))
 *                  clazz      = "aimai"        (ope.startsWith( "lk" ))
 *              /&gt;
 *              &lt;og:column
 *                  ・・・・
 *              /&gt;
 *          &lt;/tr&gt;
 *          &lt;tr&gt;
 *                  ・・・・
 *          &lt;/tr&gt;
 *      &lt;/table&gt;
 *
 * @og.rev 5.6.1.2 (2013/02/22) 文字列連結から、XML処理するように変更します。
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public class JspCreate_COLUMN extends AbstractJspCreate {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.9.1 (2015/11/27)" ;

	public static final int TD_COUNT = 3 ;		// tdタグを繰り返す数
	public static final int TR_COUNT = 2 ;		// trタグを繰り返す数

	// 6.3.9.1 (2015/11/27) Variables should start with a lowercase character(PMD)
	private List<JspConvertEntity> queryROWS ;
	private boolean isNULL ;

	/**
	 * コンストラクター
	 *
	 * インスタンス構築時に、タグ名(key)とファイル名(names)を指定します。
	 *
	 * @og.rev 6.3.9.1 (2015/11/27) コンストラクタを用意して、KEY,NAME をセットするように変更します。
	 */
	public JspCreate_COLUMN() {
		super( "table" , "query" );
	}

	/**
	 * タグ名とファイル名を指定するコンストラクター
	 *
	 * インスタンス構築時に、タグ名(key)とファイル名(names)を指定します。
	 * これは、JspCreate_HIDEMENU が、このクラスを継承しているため、用意しました。
	 *
	 * @og.rev 6.3.9.1 (2015/11/27) コンストラクタを用意して、KEY,NAME をセットするように変更します。
	 *
	 * @param	key		タグ名の指定(og:XXXX , :XXX , XXX いづれの形式でも可)
	 * @param	names	処理対象のファイル名(複数のファイル名をCSV形式で指定可能)
	 */
	protected JspCreate_COLUMN( final String key , final String names ) {
		super( key , names );
	}

	/**
	 * 初期化メソッド
	 *
	 * 内部で使用する JspConvertEntity の リスト のマップを受け取り、初期化を行います。
	 *
	 * @og.rev 5.6.4.4 (2013/05/31) 検索カラムは、名称で重複している場合は、片方だけでよい。
	 *
	 * @param	master	JspConvertEntityのリストのマップ
	 */
	@Override
	protected void init( final Map<String,List<JspConvertEntity>> master ) {
		queryROWS = master.get("QUERY");						// 6.3.9.1 (2015/11/27)
		isNULL = !isNotEmpty( queryROWS );						// 6.3.9.1 (2015/11/27)

		// 検索カラムは、名称で重複している場合は、片方だけでよい。
		if( !isNULL ) {											// 6.3.9.1 (2015/11/27)
			final Set<String> keys = new HashSet<>();
			int size = queryROWS.size();
			int idx  = 0;
			while( idx < size ) {
				final String key = queryROWS.get(idx).getColumnName();		// 6.3.9.1 (2015/11/27)
				if( keys.add( key ) ) {				// 正常にセットできれば、true
					idx++ ;
				}
				else {								// すでに同じキーが存在する場合
					queryROWS.remove(idx);						// 6.3.9.1 (2015/11/27)
					size-- ;
				}
			}
		}
	}

	/**
	 * JSPに出力するタグの内容を作成します。
	 * 引数より作成前のタグの属性内容を確認するする事が出来ます。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) メソッドの引数を、OGAttributes から OGElement に変更します。
	 * @og.rev 5.2.1.0 (2010/10/01) 名前空間を、og 決め打ちから、引数を使用するように変更します。
	 * @og.rev 5.6.1.2 (2013/02/22) XML処理するように変更します。
	 * @og.rev 5.6.4.4 (2013/05/31) hideMenu の対応
	 *
	 * @param ele OGElementエレメントオブジェクト
	 * @param	nameSpace	このドキュメントのnameSpace( og とか mis とか )
	 *
	 * @return	変換された文字列
	 * @og.rtnNotNull
	 * @throws Throwable 変換時のエラー
	 */
	@Override
	protected String execute( final OGElement ele , final String nameSpace )  throws Throwable {
		if( isNULL ) { return ""; }

		// table タグの親タグが、hideMenu の場合は、処理しません。
		final OGNode paraNode = ele.getParentNode() ;
		// 6.0.2.5 (2014/10/31) refactoring: getNodeType でチェックしているので間違いはないが、findBugs対応
		if( paraNode != null && paraNode.getNodeType() == OGNodeType.Element &&
				paraNode instanceof OGElement &&
				"og:hideMenu".equals( ((OGElement)paraNode).getTagName() ) ) {
			return "";
		}

		// 既存の設定値をすべて削除します。ホントは自動登録した分だけを削除すべき。
		final OGElement tblEle  = new OGElement( "table" );
		tblEle.addAttr( "summary","layout" );

		OGElement tr = null;
		for( int i=0; i<queryROWS.size() && i<(TD_COUNT*TR_COUNT); i++ ) {
			final JspConvertEntity column = queryROWS.get(i);
			if( i%TD_COUNT == 0 ) {
				tr = new OGElement( "tr" );
				tblEle.addNode( tr );
			}

			tr = trElement( tr,column );
		}

		return tblEle.getText( 0 );
	}

	/**
	 * TRエレメントにカラムタグを追加していきます。
	 * 行列の判定処理を上位で行い、ここでは、カラムタグを追加する処理に専念します。
	 *
	 * @og.rev 5.6.4.4 (2013/05/31) hideMenu の対応
	 *
	 * @param	tr		OGElementエレメントオブジェクト
	 * @param	column	カラムタグの元情報
	 *
	 * @return	カラムタグが追加されたTRエレメント
	 * @og.rtnNotNull
	 */
	protected OGElement trElement( final OGElement tr , final JspConvertEntity column ) {
		final OGElement clm  = new OGElement( "og:column" );
		clm.addAttr( "name",column.getColumnName() );

		if( column.getDefaultValue() != null && column.getDefaultValue().trim().length() > 0) {
			clm.addAttr( "defaultVal",column.getDefaultValue() );
		}
		if( "1".equals( column.getMust() )){
			clm.addAttr( "must","true" );
		}
		final String ope = column.getRemarks();
		if( ope != null && ope.startsWith( "lk" ) ) {
			clm.addAttr( "clazz","aimai" );
		}

		tr.addNode( clm );

		return tr ;
	}
}
