package org.opengion.fukurou.model;

import java.io.File;

import org.opengion.fukurou.util.StringUtil;


/**
 * ファイル操作インスタンスのファクトリークラス
 * 
 * システムリソースの「CLOUD_TARGET」を参照して、
 * 処理対象のファイル操作クラスを生成します。
 * 
 * デフォルトはローカルサーバのファイル操作を行う、
 * DefaultFileOperationクラスを生成します。
 * 
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @sinse JDK7.0
 */
public class FileOperationFactory {
	private static final int BUFFER_MIDDLE = 200;
	
	/**
	 * 引数を元に、ファイル操作クラスを生成します。
	 * 
	 * @param path ファイルパス
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(String path) {
		return newStorageOperation(null, null, path.toString());
	}
	
	/**
	 * 引数を元に、ファイル操作クラスを生成します。
	 * 
	 * @param plugin 利用プラグイン
	 * @param buket バケット名
	 * @param dir ディレクトリ
	 * @param fileName ファイル名
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(String plugin, String buket, String dir, String fileName) {
		StringBuilder path = new StringBuilder(BUFFER_MIDDLE);
		path.append(dir).append(File.separator).append(fileName);

		return newStorageOperation(path.toString());
	}

	/**
	 * 引数を元に、ファイル操作クラスを生成します。
	 * 
	 * @param plugin 利用プラグイン
	 * @param buket バケット名
	 * @param file ファイル情報
	 * @param fileName ファイル名
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(String plugin, String buket, FileOperation file, String fileName) {
		StringBuilder path = new StringBuilder(BUFFER_MIDDLE);
		path.append(file.getPath()).append(File.separator).append(fileName);

		return newStorageOperation(path.toString());
	}

	/**
	 * 引数を元に、ファイル操作クラスを生成します。
	 * CLOUD_TARGETがNULLの場合はローカルファイル用のクラスを利用します。
	 * 
	 * @param plugin 利用プラグイン
	 * @param buket バケット名
	 * @param path ファイルパス
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(String plugin, String buket, String path) {
		FileOperation rtn;
		String cloudTarget = null;

		Object[] args = new Object[] { buket, path };

		// 対象のクラウドサービスを取得(大文字化)。
		// 未指定の場合は、ローカルディレクトリを利用。
		if ( plugin != null && plugin.length() > 0 ) {
			cloudTarget = plugin.toUpperCase();
		}

		try {
			StringBuilder sb = new StringBuilder(BUFFER_MIDDLE);

			if (StringUtil.isNull(cloudTarget)) {
				sb.append("org.opengion.fukurou.model.DefaultFileOperation");
			} else {
				sb.append("org.opengion.plugin.cloud.");
				sb.append("FileOperation_");
				sb.append(cloudTarget);
			}

			rtn = (FileOperation) Class.forName(sb.toString())
					.getConstructor(String.class, String.class)
					.newInstance(args);
		} catch (Exception e) {
			StringBuilder errMsg = new StringBuilder(BUFFER_MIDDLE);
			errMsg.append("ファイルストレージの操作クラス生成に失敗しました。target:").append(cloudTarget);
			errMsg.append(" システムエラー情報：").append(e.getMessage());
			throw new RuntimeException(errMsg.toString());
		}

		return rtn;
	}
}
