/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.Set;
import java.util.HashSet;
import java.util.Arrays;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.CodeData;
import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_LARGE;		// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのチェックボックスを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 *
 * ※ このクラスは、CHBOX 用ではなく、CHBOX2 用です。
 *
 * @og.group 選択データ制御
 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
 *
 * @version  6.4
 * @author   Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class Selection_CHBOX extends Selection_NULL {
	private final CodeData codeData ;

	/**
	 * コンストラクター
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 *
	 * @param	cdData	コードデータオブジェクト
	 *
	 */
	public Selection_CHBOX( final CodeData cdData ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		if( cdData == null ) {
			final String errMsg = "コードリソースが定義されていません。" + CR ;
			throw new HybsSystemException( errMsg );
		}

		codeData = cdData ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) CHBOX2は、コードリソースも使用できるように変更。
	 *
	 * @param	selectValue	選択されている値
	 * @param	seqFlag	シーケンスアクセス機能の指定
	 * @param	useShortLabel	短ラベルの指定
	 *
	 * @return  オプションタグ
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		final String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * 選択されている値は、複数指定が可能です。カンマ区切りデータとして渡された場合は、
	 * 個別に分解して、ラベル化します。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) CHBOX2は、コードリソースも使用できるように変更。
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String name,final String selectValue,final boolean useLabel ) {
		// 選択値を、カンマで分解(split)して、Listを作成し、HashSetに渡している。存在チェック用
		final Set<String> valSet = selectValue == null
										? new HashSet<>()
										: new HashSet<>( Arrays.asList( selectValue.split( "," ) ) );

		final String inputTag = "<input type=\"checkbox\" name=\"" + name + "\" value=\"" ;
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
		final int size = codeData.getSize();
		for( int i=0; i<size; i++ ) {
			final String value = codeData.getCodeKey(i);
			if( useLabel ) { buf.append( "<label>" ); }
			buf.append( inputTag ).append( value ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			if( valSet.contains( value ) ) {
				buf.append( " checked=\"checked\"" );
			}
//			buf.append( "/>" );
			buf.append( '>' );				// 7.0.1.0 (2018/10/15)
			if( useLabel ) { buf.append( codeData.getShortLabel(i) ).append( '　' ).append( "</label>" ); }
		}

		return buf.toString();

	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) CHBOX2は、コードリソースも使用できるように変更。
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを使用する [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		// 選択値を、カンマで分解(split)して、Listを作成し、HashSetに渡している。存在チェック用
		final Set<String> valSet = selectValue == null
										? new HashSet<>()
										: new HashSet<>( Arrays.asList( selectValue.split( "," ) ) );

		final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
		final int size = codeData.getSize();
		for( int i=0; i<size; i++ ) {
			final String value = codeData.getCodeKey(i);
		if( valSet.contains( value ) ) {
				buf.append( '■' );
			}
			else {
				buf.append( '□' );
			}

			if( isSLbl ) { buf.append( codeData.getShortLabel(i) ).append( '　' ); }
		}
		return buf.toString();

	}
}
