/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.util.StringUtil ;						// 6.2.2.0 (2015/03/27)

import static org.opengion.fukurou.system.HybsConst.CR ;			// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_LARGE;	// 6.1.0.0 (2014/12/26) refactoring

import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 * QUERYの第１カラムは、選択キーになります。第２カラムはラベルです。ここまでは必須です。
 * 第３カラムが存在する場合は、短縮カラムとして認識されます。存在しない場合は、
 * 短縮ラベルは使用しません。
 *
 * メニュー作成用に、SELECT文を与えます。
 * SELECT 値,ラベル[,Sラベル][,グループ][,クラス] FROM XXXX で指定され、値、ラベルまでは必須、
 * グループは、optgroup に対して指定するラベルです。クラスは、そのオプションに
 * 色づけなどを行う為の指定です。
 * なお、グループ、クラス は、NULL(または、ゼロ文字列)の場合は、適用されません。)
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_DB extends Selection_NULL {
	// 3.5.4.8 (2004/02/23) USE_MULTI_KEY_SELECT を定義しておきます。
	// 6.1.0.0 (2014/12/26) Column 側に移動。の処理忘れ
	private final long	  DB_CACHE_TIME			= (long)HybsSystem.sysInt( "DB_CACHE_TIME" ) ;

	private final boolean	isShortLavel ;		// 短縮ラベルを使用できるかどうか
	private final long		createTime ;		// キャッシュの破棄タイミングを計るための作成時間

	private final int[]		ADRS  ;
	private final String	CACHE ;
	private final int		LEN	 ;
	private final int[]		LADRS  ;	// 5.1.3.0 (2010/02/01)
	private final String	LCACHE ;	// 5.1.3.0 (2010/02/01)
	private final int		LLEN   ;	// 5.1.3.0 (2010/02/01)
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private final ConcurrentMap<String,Integer>	adrsMap  ;			// 6.4.3.1 (2016/02/12) ついでに変数名も変えておきます。

	private final String[]	value ;		// 値
	private final String[]	label ;		// ラベル
	private final String[]	slabel ;	// 短縮ラベル
	private final String[]	desc ;		// 6.2.0.0 (2015/02/27) 概要説明 追加

	private final String addKeyLabel ;	// 6.2.0.0 (2015/02/27) キー:ラベル形式

	private static final int VAL  = 0;
	private static final int LBL  = 1;
	private static final int SLBL = 2;
	private static final int GRP  = 3;
	private static final int CLS  = 4;
	private static final int DESC = 5;	// 6.2.0.0 (2015/02/27) 概要説明 追加

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private static final ApplicationInfo APP_INFO;										// 6.4.1.1 (2016/01/16) appInfo → APP_INFO refactoring
	static {
		/** コネクションにアプリケーション情報を追記するかどうか指定 */
		final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			final String SYSTEM_ID = HybsSystem.sys( "SYSTEM_ID" );
			APP_INFO = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			APP_INFO.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			APP_INFO.setModuleInfo( "Selection_DB",null,null );
		}
		else {
			APP_INFO = null;
		}
	}

	/**
	 * コンストラクター
	 *
	 * ＤＢ検索用のＳＱＬ文を与えて、初期化します。
	 * SQL文は、KEY,LNAME [,SNAME] で、第３項がなければ、LNAME を使用します。
	 * LNAME は、通常の値を返す場合に、SNAME は、一覧表示の値を返す場合に使用します。
	 * 特別に、KEY のみの場合は、lang に基づく ResourceManager からﾗﾍﾞﾙを取得します。
	 * ただし、その場合は、オーナー(SYSTEM_ID)は選べません。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) コンストラクター 新規追加
	 * @og.rev 3.6.0.9 (2004/12/03) isMultiSelect の判定をラベル部のユニーク度で判定します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 3.8.9.2 (2007/07/28) グループと、クラスを追加。Select文の第３、第４引数として指定。
	 * @og.rev 4.0.0.0 (2006/11/15) lang 属性を追加します。
	 * @og.rev 4.3.8.0 (2009/08/01) ツールチップ表示機能追加
	 * @og.rev 5.1.3.0 (2010/02/01) ラベル(短)がnullの場合でも、ラベル(短)で表示されてしまうバグを修正
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示以外は、ツールチップ表示しない
	 * @og.rev 6.1.0.0 (2014/12/26) Column 側に移動。の処理忘れ
	 * @og.rev 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param newQuery	DB検索(SQL)文字列
	 * @param dbid		データベース接続先ID
	 * @param lang		リソースを使用する場合の言語
	 * @param addKeyLabel キー:ラベル形式で表示するかどうか[true/false/null]
	 */
	public Selection_DB( final String newQuery,final String dbid,final String lang,final String addKeyLabel ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		this.addKeyLabel = addKeyLabel;			// 6.2.0.0 (2015/02/27) キー:ラベル形式

		final String[][] cols = DBUtil.dbExecute( newQuery,null,APP_INFO,dbid );	// 3.8.7.0 (2006/12/15)
		final int count = cols.length;

		value	= new String[count];
		label	= new String[count];
		slabel	= new String[count];
		desc	= new String[count];				// 6.2.0.0 (2015/02/27) 概要説明 追加
		ADRS	= new int[count];
		adrsMap	= new ConcurrentHashMap<>(count);	// 6.4.3.1 (2016/02/12) ついでに変数名も変えておきます。

		final int len = count > 0 ? cols[0].length : 0 ;
		isShortLavel        = len > SLBL ;		// >= 3 と同意
		final boolean isGrp = len > GRP ;		// >= 4 と同意
		final boolean isCls = len > CLS ;		// >= 5 と同意
		final boolean isDesc= len > DESC ;		// >= 6 と同意 6.2.0.0 (2015/02/27)

		boolean useLabelData = false ;
		ResourceManager resource = null;
		if( len == 1 ) {						// キーしか存在しない場合は、ラベルをキーから求める。
			useLabelData = true;
			resource = ResourceFactory.newInstance( lang );
		}

		// 3.6.0.9 (2004/12/03)
		// 6.1.0.0 (2014/12/26) Column 側に移動。の処理忘れ

		final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
		String bkGroupKey = "";
		for( int i=0; i<count; i++ ) {
			value[i] = cols[i][VAL];
			if( useLabelData ) {
				label[i] = resource.getLabel( value[i] );
			}
			else {
				label[i] = cols[i][LBL];
				if( isShortLavel ) { slabel[i] = cols[i][SLBL]; }
			}
			adrsMap.put( value[i], Integer.valueOf( i ) );

			// 3.8.9.2 (2007/07/28)
			if( isGrp ) {
				final String groupKey = cols[i][GRP];
				if( !groupKey.equals( bkGroupKey ) ) {	// キーブレイク
					if( ! "".equals( bkGroupKey ) ) {
						buf.append( "</optgroup>" );
					}
					if( ! "".equals( groupKey ) ) {
						buf.append( "<optgroup label=\"" + groupKey + "\">" );
					}
					bkGroupKey = groupKey;
				}
			}

			// 6.0.2.5 (2014/10/31) char を append する。
			buf.append( "<option value=\"" ).append( value[i] ).append( '"' );
			ADRS[i] = buf.length() ;
			if( isCls ) {
				// 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
				setCodeParam( buf,cols[i][CLS] );
			}

			// 6.2.0.0 (2015/02/27) Description があれば、優先して title 属性に設定します。
			boolean useTitle = false;
			if( isDesc ){
				desc[i] = cols[i][DESC];
				if( desc[i] != null && desc[i].length() > 0 ) {
					// 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
					buf.append( " title=\"" ).append( StringUtil.htmlFilter( desc[i],true ) ).append( '"' );
					useTitle = true;
				}
			}

			// 6.2.0.0 (2015/02/27) キー:ラベル形式
			final String kv = "true".equalsIgnoreCase( addKeyLabel ) ? value[i] + ':' : "" ;

			// 4.3.8.0 (2009/08/01) slabel利用の場合はlabelをtitle属性にセット
			//buf.append( ">" ).append( label[i] ).append( "</option>" );
			// 6.0.2.5 (2014/10/31) char を append する。
			if( isShortLavel && slabel[i] != null && slabel[i].length() > 0 ){ // 5.1.3.0 (2010/02/01)
				if( !useTitle && !label[i].equals( slabel[i] ) ){ // slabelとlabelが異なる場合のみ
					buf.append( " title=\"" ).append( StringUtil.htmlFilter( label[i],true ) ).append( '"' );
				}
				buf.append( '>' ).append( kv ).append( slabel[i] );	// 6.2.0.0 (2015/02/27) キー:ラベル形式
			}
			else{
				buf.append( '>' ).append( kv ).append( label[i] );	// 6.2.0.0 (2015/02/27) キー:ラベル形式
			}
			buf.append( "</option>" );

			// 3.6.0.9 (2004/12/03)
			// 6.1.0.0 (2014/12/26) Column 側に移動。の処理忘れ
		}
		if( isGrp && ! "".equals( bkGroupKey ) ) {
			buf.append( "</optgroup>" );
		}

		CACHE = buf.toString();
		LEN   = CACHE.length() + 30;

		// 5.1.3.0 (2010/02/01) ツールチップ表示が適用されている場合のみ、ツールチップなしの状態のoptionをキャッシュする。
		if( CACHE.indexOf( "title=\"" ) < 0 ) {
			LADRS  = null;
			LCACHE = null;
			LLEN   = 0;
		}
		else {
			LADRS  = new int[count];
			final StringBuilder lbuf = new StringBuilder( BUFFER_LARGE );

			bkGroupKey = "";
			// 6.0.2.5 (2014/10/31) char を append する。
			for( int i=0; i<count; i++ ) {
				if( isGrp ) {
					final String groupKey = cols[i][GRP];
					if( !groupKey.equals( bkGroupKey ) ) {
						if( ! "".equals( bkGroupKey ) )	{ lbuf.append( "</optgroup>" ); }
						if( ! "".equals( groupKey ) )	{ lbuf.append( "<optgroup label=\"" + groupKey + "\">" ); }
						bkGroupKey = groupKey;
					}
				}
				lbuf.append( "<option value=\"" ).append( value[i] ).append( '"' );
				LADRS[i] = lbuf.length() ;
				if( isCls && ! "".equals( cols[i][CLS] ) ) {
					// 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
					setCodeParam( lbuf,cols[i][CLS] );
				}
				// 6.2.0.0 (2015/02/27) キー:ラベル形式
				final String kv = "true".equalsIgnoreCase( addKeyLabel ) ? value[i] + ':' : "" ;
				lbuf.append( '>' ).append( kv ).append( label[i] ).append( "</option>" );
			}
			if( isGrp && ! "".equals( bkGroupKey ) ) {
				lbuf.append( "</optgroup>" );
			}
			LCACHE = lbuf.toString();
			LLEN   = LCACHE.length() + 30;
		}

		// 6.1.0.0 (2014/12/26) Column 側に移動。の処理忘れ
		createTime = System.currentTimeMillis() ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 追加
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 6.4.3.2 (2016/02/19) ConcurrentHashMap は、key,val ともに、NOT NULL制限あり。
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		final int[] adrs   ;
		final String cache ;
		final int len      ;
		if( !useShortLabel && LCACHE != null && LCACHE.length() > 0 ) {
			adrs  = LADRS;
			cache = LCACHE;
			len   = LLEN;
		}
		else {
			adrs  = ADRS;
			cache = CACHE;
			len   = LEN;
		}

		// 6.4.3.2 (2016/02/19) ConcurrentHashMap は、key,val ともに、NOT NULL制限あり。
		if( selectValue == null ) {
			final String errMsg = "選択されている値に、null は指定できません。" + CR ;
			LogWriter.log( errMsg );
			return cache;
		}

		// マッチするアドレスを探す。
		final Integer sel = adrsMap.get( selectValue );			// 6.4.3.1 (2016/02/12) ついでに変数名も変えておきます。

		if( sel == null ) {
			// 4.0.0 (2005/01/31)
			// 6.9.8.0 (2018/05/28) FindBugs:null でないことがわかっている値の冗長な null チェック。
//			if( selectValue != null && selectValue.length() > 0 ) {
				final String errMsg = "DBコードに存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ CR ;
				LogWriter.log( errMsg );
//			}
			return cache;
		}
		else {
			final int selected = sel.intValue();
			final StringBuilder buf = new StringBuilder( len + 100 );						// 6.1.0.0 (2014/12/26) refactoring
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( value[selected] ).append( '"' );	// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( cache.substring( 0,adrs[selected] ) );
			}
			buf.append( " selected=\"selected\"" )
				.append( cache.substring( adrs[selected] ) );
			return buf.toString() ;
		}
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 * @og.rev 5.3.5.0 (2011/05/01) 名称(短)表示時に名称(長)をツールチップで表示する。
	 * @og.rev 6.2.0.0 (2015/02/27) Description があれば、優先して title 属性に設定します。
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のチェック追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを使用する [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		// 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限
		// 元々の仕様どおりになりますので、エラーにはしません。
		if( selectValue == null ) { return selectValue; }

		// マッチするアドレスを探す。
		final Integer sel = adrsMap.get( selectValue );			// 6.4.3.1 (2016/02/12) ついでに変数名も変えておきます。

		if( sel == null ) {
			return selectValue;
		}
		else {
			// 6.2.0.0 (2015/02/27) Description があれば、優先して title 属性に設定します。
			final int adrs = sel.intValue();		// 6.2.0.0 (2015/02/27) 変数使用
			String title = desc[adrs];
			if( isShortLavel && isSLbl && !label[adrs].equals( slabel[adrs] )
				 && title != null && !title.isEmpty() ) {
				title = label[adrs];
			}

			// 6.2.0.0 (2015/02/27) キー:ラベル形式
			final String kv = "true".equalsIgnoreCase( addKeyLabel ) ? selectValue + ':' : "" ;

			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
			return title == null || title.isEmpty()
						? kv + label[adrs]
						: "<span title=\"" + StringUtil.htmlFilter( title,true ) + "\">" + kv + slabel[adrs] + "</span>";
		}
	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	@Override
	public boolean isTimeOver() {
		return ( System.currentTimeMillis() - createTime ) > DB_CACHE_TIME ;
	}
}
