/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.File;
import java.io.IOException;
import java.util.Locale;

/**
 * FileInfo.java は、共通的に使用される File関連の情報管理を行うクラスです。
 *
 * ファイルオブジェクトより、ファイル名、ファイル名(拡張子無し)、拡張子(ピリオド無)、 親パス
 * の情報を、分割します。
 * また、一部のFileUtilのファイル属性を扱うメソッド類を移植してきました。
 *
 * 絶対パス は、File#getCanonicalPath() の値を設定します。
 * ../AAA/BBB/CCC.XLS というファイルオブジェクトに対して、
 *
 *   FILE    : CCC.xls            ファイル名
 *   NAME    : CCC                拡張子なしのファイル名
 *   SUFIX   : xls                ピリオド無しの拡張子(小文字に統一)
 *   DIR     : C:\tmp\BBB         親パス
 *
 * ファイル名と、拡張子は、File#getName() の結果から、一番最後に見つかった
 * ピリオドで、前後を分割します。拡張子にはピリオドを含みません。
 * ピリオドがなければ、File#getName() の結果をファイル名に設定し、
 * 拡張子は、空文字列を設定します。
 *
 * @og.rev 5.9.0.0 (2015/09/04) Ver.6からの逆移植
 * @og.group ユーティリティ
 *
 * @version  6.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public final class FileInfo {
	/** ファイル情報のキー */
	public static final String[] KEYS = { "FILE","NAME","SUFIX","DIR" };

	/** CCC.xls            ファイル名 */
	public final String FILE   ;
	/** CCC                拡張子なしのファイル名 */
	public final String NAME   ;
	/** xls                ピリオド無しの拡張子(小文字に統一) */
	public final String SUFIX  ;
	/** C:\tmp\BBB         親パス */
	public final String DIR    ;

	/**
	 * ファイルオブジェクトを引数に取るコンストラクター
	 *
	 * 絶対パス は、File#getCanonicalPath() の値を設定します。
	 * ../AAA/BBB/CCC.XLS というファイルオブジェクトに対して、
	 *   FILE   : Fileオブジェクト   File#getCanonicalFile() の値
	 *
	 *   FILE    : CCC.xls            ファイル名
	 *   NAME    : CCC                拡張子なしのファイル名
	 *   SUFIX   : xls                ピリオド無しの拡張子(小文字に統一)
	 *   DIR     : C:\tmp\BBB         親パス
	 *
	 * ファイルオブジェクトは、File#getCanonicalFile(File) で、絶対パスに変換後、内部処理されます。
	 * 必要な情報は、final String に設定されるため、そこから情報を取得してください。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) 新規作成
	 *
	 * @param	file	ファイルオブジェクト
	 */
	public FileInfo( final File file ) {
		final File ccFile ;
		try {
			ccFile = file.getCanonicalFile();
		}
		catch( IOException ex ) {
			final String errMsg = "ファイルの正式パス名が取得できません。[" + file + "]";
			throw new RuntimeException( errMsg,ex );
		}

		FILE = ccFile.getName();
		final int extPos = FILE.lastIndexOf( '.' );
		if( extPos >= 0 ) {
			NAME  = FILE.substring( 0,extPos );
			SUFIX = FILE.substring( extPos + 1 ).toLowerCase( Locale.JAPAN );		// ピリオドを含まない
		}
		else {
			NAME  = FILE;
			SUFIX = "";									// 拡張子が無ければ、空文字列
		}
		DIR = ccFile.getParent();
	}

	/**
	 * キーに対応した値を返します。
	 *
	 * キーは、値が文字列の分だけです。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) 新規作成
	 *
	 * @param	key	ファイル情報のキー
	 * @return	ファイル情報の値(キーが未設定の場合はnull)
	 * @throws NullPointerException	key に null がセットされた場合
	 */
	public String getValue( final String key ) {
		final String rtn ;
//		switch( key ) {
//			case "FILE"		: rtn = FILE;		break;
//			case "NAME"		: rtn = NAME;		break;
//			case "SUFIX"	: rtn = SUFIX;		break;
//			case "DIR"		: rtn = DIR;		break;
//			default			: rtn = null;		break;
//		}
		if( "FILE".equals( key )){ rtn = FILE; }
		else if( "NAME".equals( key ) ){ rtn = NAME; } 
		else if( "SUFIX".equals( key ) ){ rtn = SUFIX; } 
		else if( "DIR".equals( key ) ){ rtn = DIR; } 
		else { rtn = null; }
			

		return rtn ;
	}

	/**
	 * ファイルオブジェクトから、名前部分のみを取得します。
	 *
	 * 名前だけ取得したい場合の、簡易スタティックメソッドです。
	 * ピリオドがない場合は、ファイル名を、先頭がピリオドの場合は、
	 * ゼロ文字列です。
	 * オブジェクトを作成しません。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) 新規作成
	 *
	 * @param	file	ファイルオブジェクト
	 * @return	ファイルの名称(引数がnullの場合はnull)
	 */
	public static String getNAME( final File file ) {
		return file == null ? null : getNAME( file.getName() ) ;
	}

	/**
	 * ファイル文字列から、名前部分のみを取得します。
	 *
	 * 名前だけ取得したい場合の、簡易スタティックメソッドです。
	 * ピリオドがない場合は、ファイル名を、先頭がピリオドの場合は、
	 * ゼロ文字列です。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) 新規作成
	 *
	 * @param	filename	ファイル文字列
	 * @return	ファイルの名称(引数がnullの場合はnull)
	 */
	public static String getNAME( final String filename ) {
		String name = null;
		if( filename != null ) {
			final int extPos = filename.lastIndexOf( '.' );
			if( extPos >= 0 ) {
				name = filename.substring( 0,extPos );
			}
		}
		return name ;
	}

	/**
	 * ファイルオブジェクトから、拡張子(小文字に統一)のみを取得します。
	 *
	 * 拡張子だけ取得したい場合の、簡易スタティックメソッドです。
	 * ピリオドがない場合は、ゼロ文字列を返します。
	 * オブジェクトを作成しません。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) 新規作成
	 *
	 * @param	file	ファイルオブジェクト
	 * @return	ファイルの拡張子(小文字に統一,引数がnullの場合はnull)
	 */
	public static String getSUFIX( final File file ) {
		return file == null ? null : getSUFIX( file.getName() ) ;
	}

	/**
	 * ファイル文字列から、拡張子(小文字に統一)のみを取得します。
	 *
	 * 拡張子だけ取得したい場合の、簡易スタティックメソッドです。
	 * ピリオドがない場合は、ゼロ文字列を返します。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) 新規作成
	 *
	 * @param	filename	ファイル文字列
	 * @return	ファイルの拡張子(小文字に統一,引数がnullの場合はnull)
	 */
	public static String getSUFIX( final String filename ) {
		String sufix = null;
		if( filename != null ) {
			final int extPos = filename.lastIndexOf( '.' );
			if( extPos >= 0 ) {
				sufix = filename.substring( extPos + 1 ).toLowerCase( Locale.JAPAN );	// ピリオドを含まない
			}
			else {
				sufix = "";									// 拡張子が無ければ、空文字列
			}
		}
		return sufix ;
	}

	/**
	 * ファイルオブジェクトから、指定の拡張子かどうかを判定します。
	 *
	 * 拡張子判定に特化した、簡易スタティックメソッドです。
	 * 引数の拡張子には、ピリオドは含めません。
	 * 比較する拡張子の大文字小文字は区別しません。
	 *
	 * @og.rev 6.2.4.0 (2015/05/15) 新規作成
	 *
	 * @param	file	ファイルオブジェクト
	 * @param	sufix	拡張子群
	 * @return	true:拡張子が含まれる/false:含まれない
	 */
	public static boolean isSUFIX( final File file , final String... sufix ) {
		final String fsfx = getSUFIX( file );
		if( fsfx != null && !fsfx.isEmpty() ) {
			for( int i=0; i<sufix.length; i++ ) {
				if( fsfx.equalsIgnoreCase( sufix[i] ) ) { return true; }
			}
		}
		return false;
	}

	/**
	 * ファイル文字列から、指定の拡張子かどうかを判定します。
	 *
	 * 拡張子判定に特化した、簡易スタティックメソッドです。
	 * 引数の拡張子には、ピリオドは含めません。
	 * 比較する拡張子の大文字小文字は区別しません。
	 *
	 * @og.rev 6.2.4.0 (2015/05/15) 新規作成
	 *
	 * @param	filename	ファイル文字列
	 * @param	sufix		拡張子群
	 * @return	true:拡張子が含まれる/false:含まれない
	 */
	public static boolean isSUFIX( final String filename , final String... sufix ) {
		final String fsfx = getSUFIX( filename );
		if( fsfx != null && !fsfx.isEmpty() ) {
			for( int i=0; i<sufix.length; i++ ) {
				if( fsfx.equalsIgnoreCase( sufix[i] ) ) { return true; }
			}
		}
		return false;
	}
}
