/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Random ;
import java.util.Set;
// import java.util.HashMap;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;	// 5.5.2.6 (2012/05/25) findbugs対応

import javax.servlet.http.HttpServletRequest;

import org.opengion.fukurou.model.FileOperation;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.io.HybsFileOperationFactory;
import org.opengion.hayabusa.servlet.multipart.FilePart;
import org.opengion.hayabusa.servlet.multipart.MultipartParser;
import org.opengion.hayabusa.servlet.multipart.ParamPart;
// import org.opengion.fukurou.util.ZipArchive;				// 5.7.1.2 (2013/12/20) zip 対応
import org.opengion.hayabusa.servlet.multipart.Part;

/**
 * ファイルをサーバーにアップロードする場合に使用されるマルチパート処理サーブレットです。
 *
 * 通常のファイルアップロード時の、form で使用する、enctype="multipart/form-data"
 * を指定した場合の、他のリクエスト情報も、取り出すことが可能です。
 *
 * ファイルをアップロード後に、指定のファイル名に変更する機能があります。
 * file 登録ダイアログで指定した name に、"_NEW" という名称を付けたリクエスト値を
 * ファイルのアップロードと同時に送信することで、この名前にファイルを付け替えます。
 * また、アップロード後のファイル名は、name 指定の名称で、取り出せます。
 * クライアントから登録したオリジナルのファイル名は、name に、"_ORG" という名称
 * で取り出すことが可能です。
 *
 * maxPostSize : 最大転送サイズ(Byte)を指定します。 0,またはマイナスで無制限です。
 * useBackup   : ファイルアップロード時に、すでに同名のファイルが存在した場合に、
 *               バックアップ処理(renameTo)するかどうか[true/false]を指定します(初期値:false)
 *
 * ファイルアップロード時に、アップロード先に、同名のファイルが存在した場合は、既存機能は、そのまま
 * 置き換えていましたが、簡易バージョンアップ機能として、useBackup="true" を指定すると、既存のファイルを
 * リネームして、バックアップファイルを作成します。
 * バックアップファイルは、アップロードフォルダを基準として、_backup/ファイル名.拡張子_処理時刻のlong値.拡張子 になります。
 * オリジナルのファイル名（拡張子付）を残したまま、"_処理時刻のlong値" を追加し、さらに、オリジナルの拡張子を追加します。
 * バックアップファイルの形式は指定できません。
 *
 * @og.rev 5.10.9.0 (2019/03/01) oota クラウドストレージ対応を追加。(Fileクラスを拡張)
 * 
 * @og.group その他機能
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class MultipartRequest {
//	private static volatile int dumyNewFileCnt = 1 ;	// 3.8.1.4 (2006/03/17)
	private static AtomicInteger dumyNewFileCnt = new AtomicInteger(1);		// 5.5.2.6 (2012/05/25) findbugs対応

	private static String RANDOM_KEY = new Random().nextInt( Integer.MAX_VALUE ) + "_" ;		// 5.6.5.3 (2013/06/28) アップロード時のダミーファイル名をもう少しだけランダムにする。

//	private final Map<String,List<String>> parameters	= new HashMap<String,List<String>>();
//	private final Map<String,UploadedFile> files		= new HashMap<String,UploadedFile>();
	private final Map<String,List<String>> parameters	= new TreeMap<String,List<String>>();		// 5.6.5.2 (2013/06/21) ソートします。

	// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
//	private final Map<String,UploadedFile> files		= new TreeMap<String,UploadedFile>();		// 5.6.5.2 (2013/06/21) ソートします。
	private final List<UploadedFile> files				= new ArrayList<UploadedFile>();			// 5.7.1.1 (2013/12/13) HTML5対応

	/**
	 * MultipartRequest オブジェクトを構築します。
	 *
	 * 引数として、ファイルアップロード時の保存フォルダ、最大サイズ、エンコード、
	 * 新しいファイル名などを指定できます。新しいファイル名は、アップロードされる
	 * ファイルが一つだけの場合に使用できます。複数のファイルを同時に変更したい
	 * 場合は、アップロードルールにのっとり、リクエストパラメータで指定してください。
	 *
	 * HTML5 では、ファイルアップロード時に、multiple 属性（inputタグのtype="file"）を
	 * 付ける事で、ファイルを複数選択できます。
	 * その場合は、inputのname属性は、一つなので、_NEW による名前の書き換えはできません。
	 *
	 * @og.rev 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。staticフィールドへの書き込みに、AtomicInteger を利用します。
	 * @og.rev 5.6.5.3 (2013/06/28) useBackup引数追加
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 * @og.rev 5.7.4.3 (2014/03/28) inputFilename のリクエスト変数処理追加
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用処理追加
	 * @og.rev 5.10.9.0 (2019/03/01) クラウドストレージ対応を追加。
	 *
	 * @param	request	HttpServletRequestオブジェクト
	 * @param	saveDirectory	ファイルアップロードがあった場合の保存フォルダ名
	 * @param	maxPostSize	ファイルアップロード時の最大ファイルサイズ(Byte)0,またはマイナスで無制限
	 * @param	encoding	ファイルのエンコード
	 * @param	inputFilename	アップロードされたファイルの新しい名前
	 * @param	useBackup		ファイルアップロード時に、バックアップ処理するかどうか[true/false]を指定
	 * @param  fileURL   クラウドストレージ用のURL
	 * @param  storageType クラウドストレージのタイプ(plugin)
	 * @param  bucketName クラウドストレージのバケット名
	 * @throws IOException 入出力エラーが発生したとき
	 */
	public MultipartRequest(final HttpServletRequest request,
							final String saveDirectory,
							final int maxPostSize,
							final String  encoding,
//							final String  inputFilename ) throws IOException {
							final String  inputFilename,
							final boolean useBackup,							// 5.6.5.3 (2013/06/28) 追加
//							final String fileURL) throws IOException {			// (2017/10/06) 追加
							final String fileURL,
							final String storageType,
							final String bucketName) throws IOException {		// 5.10.9.0 (2019/03/01) ADD

		if(request == null) {
			throw new IllegalArgumentException("request cannot be null");
		}

		if(saveDirectory == null) {
			throw new IllegalArgumentException("saveDirectory cannot be null");
		}
		// 5.5.2.6 (2012/05/25) 0,またはマイナスで無制限
//		if(maxPostSize <= 0) {
//			throw new IllegalArgumentException("maxPostSize must be positive");
//		}

		// Save the dir
		// 5.10.9.0 (2019/03/01) クラウドストレージ対応 oota tmp 
		// File dir = new File(saveDirectory);
		FileOperation dir = HybsFileOperationFactory.create(storageType, bucketName, saveDirectory);

		
		// 5.10.9.0 (2019/03/01) if条件を追加。チェックはローカルストレージの場合のみ行います。 oota tmp
		if(dir.isLocal()) {
			// Check saveDirectory is truly a directory
			if(!dir.isDirectory()) {
				throw new IllegalArgumentException("Not a directory: " + saveDirectory);
			}
	
			// Check saveDirectory is writable
			if(!dir.canWrite()) {
				throw new IllegalArgumentException("Not writable: " + saveDirectory);
			}
		}

		// Parse the incoming multipart, storing files in the dir provided,
		// and populate the meta objects which describe what we found
		MultipartParser parser = new MultipartParser(request, maxPostSize);
		if(encoding != null) {
			parser.setEncoding(encoding);
		}

		// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
//		List<String> list = new ArrayList<String>();

		Part part;
		while ((part = parser.readNextPart()) != null) {
			String name = part.getName();
			if( part.isParam() && part instanceof ParamPart ) {
				ParamPart paramPart = (ParamPart)part;
				String value = paramPart.getStringValue();
				List<String> existingValues = parameters.get(name);
				if(existingValues == null) {
					existingValues = new ArrayList<String>();
					parameters.put(name, existingValues);
				}
				existingValues.add(value);
			}
			else if( part.isFile() && part instanceof FilePart ) {
				FilePart filePart = (FilePart)part;
//				String fileName = filePart.getFilename();
				String orgName = filePart.getFilename();		// 5.7.1.1 (2013/12/13) 判りやすいように変数名変更
//				if(fileName != null) {
				if(orgName != null) {
					// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
					// 同一 name で、複数ファイルを扱う必要があります。
//					list.add( name );		// 3.5.6.5 (2004/08/09) 指定の name 属性
					// 3.8.1.2 (2005/12/19) 仮ファイルでセーブする。
//					String newName = String.valueOf( dumyNewFileCnt++ ) ;	// 3.8.1.4 (2006/03/17)
//					String newName = String.valueOf( dumyNewFileCnt.getAndIncrement() ) ;	// 5.5.2.6 (2012/05/25) findbugs対応
					String uniqKey = RANDOM_KEY + dumyNewFileCnt.getAndIncrement() ;		// 5.6.5.3 (2013/06/28) アップロード時のダミーファイル名をもう少しだけランダムにする。
//					filePart.setFilename( newName );			// 5.6.5.3 (2013/06/28) newName はややこしいので、変更
					filePart.setFilename( uniqKey );

					// 標準のﾌｧｲﾙ書き込み 2017/10/06 DELETE ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用判定を追加
					// filePart.writeTo(dir);

					// 2017/10/06 ADD ｼｽﾃﾑﾘｿｰｽにｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用が登録されている場合は、ｸﾗｳﾄﾞｽﾄﾚｰｼﾞを利用する
					
					// ﾌｧｲﾙ書き込み
					// 5.10.9.0 (2019/03/01) クラウドストレージ対応。oota tmp
//					filePart.writeTo(dir);
					filePart.writeTo(dir, storageType, bucketName);
					
					// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
					files.add( new UploadedFile(
											uniqKey,		// 5.7.1.1 (2013/12/13) 順番変更
											dir.toString(),
											name,			// 5.7.1.1 (2013/12/13) 項目追加
//											fileName,
											orgName,
											filePart.getContentType()));

//					files.put(name,
//							  new UploadedFile( dir.toString(),
////												newName,	// 3.8.1.2 (2005/12/19)
//												tempName,		// 3.8.1.2 (2005/12/19)
//												fileName,
//												filePart.getContentType()));
				}
//				else {
//					files.put(name, new UploadedFile(null, null, null, null));
//				}
			}
			else {
				String errMsg = "Partオブジェクトが、ParamPartでもFilePartでもありません。"
							+ " class=[" + part.getClass() + "]";
				throw new RuntimeException( errMsg );
			}
		}

		// 5.7.4.3 (2014/03/28) inputFilename は、リクエスト変数が使えるようにします。
		String filename = getReqParamFileName( inputFilename ) ;

		// 3.5.6.5 (2004/08/09) 登録後にファイルをリネームします。
		// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
//		int size = list.size();
		int size = files.size();

		// 5.7.1.2 (2013/12/20) zip 対応
		// 5.9.25.0 (2017/10/06) FileをString型に変更
		//File[] tgtFiles = new File[size];
		String[] tgtFiles = new String[size];

//		boolean isZip = ( inputFilename != null && inputFilename.endsWith( ".zip" ) );
		boolean isZip = ( filename != null && filename.endsWith( ".zip" ) );

		for( int i=0; i<size; i++ ) {
//			String name = list.get(i);
//			File file = getFile( name );
			UploadedFile upFile = files.get(i);
			String name = upFile.getName();		// 5.7.1.1 (2013/12/13)

//			String newName = (isZip) ? null : inputFilename ;
			String newName = (isZip) ? null : filename ;
			if( newName == null && name != null ) {
				int adrs = name.lastIndexOf( HybsSystem.JOINT_STRING );	// カラム__行番号 の __ の位置
				if( adrs < 0 ) {
					newName = getParameter( name + "_NEW" );
				}
				else {
					newName = getParameter( name.substring( 0,adrs ) + "_NEW" + name.substring( adrs ) );
				}
			}

			// 5.7.1.1 (2013/12/13) UploadedFile 内で処理するように変更
			// 5.9.25.0 (2017/10/06) MODIFY fileURLとsessionを追加
//			tgtFiles[i] = upFile.renameTo( newName,useBackup);
			// 5.10.9.0 (2019/03/01) クラウドストレージ対応。sessionは不要になったため除去。 ootat tmp
//			tgtFiles[i] = upFile.renameTo( newName,useBackup,fileURL,request.getSession(true));
			tgtFiles[i] = upFile.renameTo( newName, useBackup, fileURL, storageType, bucketName);

//			// 3.8.1.3 (2006/02/06) 新ファイル名に拡張子がないとき
//			// 旧ファイル名から拡張子取得し新ファイル名に文字列連結
//			if( newName != null && newName.length() > 0 ) {
//				// 新ファイル名から拡張子取得
//				String newExt = getExtension( newName );
//				if( newExt == null || newExt.length() == 0 ) {
////					String oldExt = getExtension( getOriginalFileName( name ) );		// 5.7.1.1 (2013/12/13)
//					String oldExt = getExtension( upFile.getOriginalFileName() );
////					newName = new StringBuilder().append( newName ).append( "." ).append( oldExt ).toString();
//					newName = newName + "." + oldExt ;
//				}
//			}
//			else {
////				newName = getOriginalFileName( name );		// 5.7.1.1 (2013/12/13)
//				newName = upFile.getOriginalFileName();
//			}
//
//			// 3.8.1.2 (2005/12/19) 基本的にはすべてのケースでファイル名変更が発生する。
//			File file = upFile.getFile();		// 5.7.1.1 (2013/12/13)
//			if( file != null && newName != null && newName.length() > 0 ) {
//				File newFile = new File( dir,newName );
//
//				// 5.6.5.3 (2013/06/28) useBackup ファイルアップロード時に、バックアップ処理するかどうか[true/false]を指定
////				if( newFile.exists() && !newFile.delete() ) {
////					String errMsg = "既存のファイル[" + newName + "]が削除できませんでした。";
////					throw new RuntimeException( errMsg );
////				}
//				if( newFile.exists() ) {
//					if( useBackup ) {
//						// newName にフォルダ階層を含む場合に、そなえて。
//						File parent = newFile.getParentFile();			// バックアップすべきファイルのフォルダ
//						File backup = new File( parent , "_backup" );	// その直下に、"_backup" フォルダを作成
//						if( backup != null && !backup.exists() && !backup.mkdirs() ) {
//							String errMsg = "バックアップ処理でbackupフォルダの作成に失敗しました。[" + backup + "]";
//							throw new RuntimeException( errMsg );
//						}
//						// バックアップファイル名は、元のファイル名(拡張子含む) ＋ "_" + 現在時刻のlong値 + "." + 元のファイルの拡張子
//						String bkupName = newFile.getName() + "_" + System.currentTimeMillis() + "."  + getExtension( newName ) ;
//						File fromFile = new File( dir,newName );		// オリジナルの newFile をrename するとまずいので、同名のFileオブジェクトを作成
//						File bkupFile = new File( backup,bkupName );
//
//						if( !fromFile.renameTo( bkupFile ) ) {
//							String errMsg = "バックアップ処理でバックアップファイルをリネームできませんでした。[" + bkupFile + "]" ;
//							throw new RuntimeException( errMsg );
//						}
//					}
//					else if( !newFile.delete() ) {
//						String errMsg = "既存のファイル[" + newName + "]が削除できませんでした。";
//						throw new RuntimeException( errMsg );
//					}
//				}
//
////				file.renameTo( newFile );
//				if( !file.renameTo( newFile ) ) {
//					String errMsg = "所定のファイルをリネームできませんでした。[" + file + "]" ;
//					throw new RuntimeException( errMsg );
//				}
////				UploadedFile fup = files.get( name );
////				fup.setFilesystemName( newName );
//				upFile.setFilesystemName( newName );
//			}
		}
		// 5.7.1.2 (2013/12/20) zip 対応
		// 5.7.4.3 (2014/03/28) 一旦保留にしていましたが、復活します。
	//	if( isZip ) {
//	//		File zipFile = new File( saveDirectory,inputFilename );
	//		File zipFile = new File( saveDirectory,filename );
	//		ZipArchive.compress( tgtFiles,zipFile );
	//	}
	}

	/**
	 * リクエストパラメータの名前配列を取得します。
	 *
	 * @return	リクエストパラメータの名前配列
	 */
	public String[] getParameterNames() {
		Set<String> keyset = parameters.keySet();
		return keyset.toArray( new String[keyset.size()] );
	}

	/**
	 * ファイルアップロードされたファイル群のファイル名配列を取得します。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応の為、廃止
	 *
	 * @return	アップロードされたファイル名配列
	 */
//	public String[] getFilenames() {
//		Set<String> keyset = files.keySet();
//		return keyset.toArray( new String[keyset.size()] );
//	}

	/**
	 * ファイルアップロードされたファイル群のファイル配列を取得します。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 *
	 * @return	アップロードされたファイル群
	 */
	public UploadedFile[] getUploadedFile() {
		return files.toArray( new UploadedFile[files.size()] );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を取得します。
	 *
	 * 複数存在する場合は、一番最後の値を返します。
	 *
	 * @param	name	リクエストパラメータ名
	 *
	 * @return	パラメータの値
	 */
	public String getParameter( final String name ) {
		List<String> values = parameters.get(name);
		if( values == null || values.isEmpty() ) {
			return null;
		}
		return values.get(values.size() - 1);
	}

	/**
	 * 指定の名前のリクエストパラメータの値を配列型式で取得します。
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) 新規作成
	 *
	 * @param	name	リクエストパラメータ名
	 *
	 * @return	パラメータの値配列
	 */
	public String[] getParameters( final String name ) {
		List<String> values = parameters.get(name);
		if( values == null || values.isEmpty() ) {
			return null;
		}
//		return values.toArray( new String[0] );
		return values.toArray( new String[values.size()] );
	}

	/**
	 * 指定の名前のリクエストパラメータの値を配列(int)型式で取得します。
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) 新規作成
	 * @og.rev 5.3.6.0 (2011/06/01) 配列値が""の場合にNumberFormatExceptionが発生するバグを修正
	 *
	 * @param	name	リクエストパラメータ名
	 *
	 * @return	パラメータの値配列
	 */
	public int[] getIntParameters( final String name ) {
		List<String> values = parameters.get(name);
		if( values == null || values.isEmpty() ) {
			return null;
		}

//		int[] rtn = new int[values.size()];
//		for( int i=0; i<values.size(); i++ ) {
//			rtn[i] = Integer.valueOf( values.get(i) );
//		}

		// 5.3.6.0 (2011/06/01) ゼロストリング("")はint変換対象から予め除外する
		List<Integer> intVals = new ArrayList<Integer>();
		for( int i=0; i<values.size(); i++ ) {
			String str = values.get(i);
			if( str != null && str.length() > 0 ) {
				intVals.add( Integer.parseInt( str ) );
			}
		}
		if( intVals.isEmpty() ) {
			return null;
		}

		int[] rtn = new int[intVals.size()];
		for( int i=0; i<intVals.size(); i++ ) {
			rtn[i] = intVals.get(i).intValue();
		}

		return rtn;
	}

	/**
	 * 指定の名前の ファイル名のリクエスト変数処理を行います。
	 *
	 * filename 属性のみ、{&#064;XXXX} のリクエスト変数が使えるようにします。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) 新規追加
	 *
	 * @param	fname	ファイル名
	 * @return	リクエスト変数を処理したファイル名
	 */
	private String getReqParamFileName( final String fname ) {

		String rtn = fname ;
		if( fname != null ) {
			StringBuilder filename = new StringBuilder( fname ) ;
			int st = filename.indexOf( "{@" );
			while( st >= 0 ) {
				int ed = filename.indexOf( "}",st );
				if( ed < 0 ) {
					String errMsg = "{@XXXX} の対応関係が取れていません。"
								+ " filename=[" + fname + "]";
					throw new RuntimeException( errMsg );
				}
				String key = filename.substring( st+2,ed );		// "}" は切り出し対象外にする。
				String val = getParameter( key );
				filename.replace( st,ed+1,val );				// "}" を含めて置換したいので、ed+1
				// 次の "{@" を探す。開始は置換文字数が不明なので、st から始める。
				st = filename.indexOf( "{@",st );
			}
			rtn = filename.toString();
		}
		return rtn ;
	}

	/**
	 * 指定の名前の UploadedFile オブジェクトから 登録されるファイル名を取得します。
	 *
	 * 登録されるファイル名とは、新たに書き換えられたファイル名のことです。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) null 対策
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応のため廃止
	 *
	 * @param	name	キー情報
	 *
	 * @return	新たに書き換えられたファイル名
	 */
//	public String getFilesystemName( final String name ) {
//		UploadedFile file = files.get(name);
////		return file.getFilesystemName();  // may be null
//		return (file == null) ? null : file.getFilesystemName();  // may be null
//	}

	/**
	 * 指定の名前の UploadedFile オブジェクトから アップロードされたファイル名を取得します。
	 *
	 * アップロードされたファイル名とは、オリジナルのファイル名のことです。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) null 対策
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応のため廃止
	 *
	 * @param	name	キー情報
	 *
	 * @return	オリジナルのファイル名
	 */
//	public String getOriginalFileName( final String name ) {
//		UploadedFile file = files.get(name);
////		return file.getOriginalFileName();  // may be null
//		return (file == null) ? null : file.getOriginalFileName();  // may be null
//	}

	/**
	 * 指定の名前の UploadedFile オブジェクトから File オブジェクトを取得します。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) null 対策
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 *
	 * @param	name	キー情報
	 *
	 * @return	Fileオブジェクト
	 */
//	public File getFile( final String name ) {
//		UploadedFile file = files.get(name);
////		return file.getFile();  // may be null
//		return (file == null) ? null : file.getFile();  // may be null
//	}

	/**
	 * ファイル名から 拡張子を取得します。
	 *
	 * @og.rev 5.7.1.1 (2013/12/13) UploadedFileクラスに移動
	 *
	 * @param	fileName	ファイル名
	 * @return	拡張子
	 */
//	private String getExtension( final String fileName ) {
//		int index = fileName.lastIndexOf('.');
//		if(index!=-1) {
//			return fileName.substring(index + 1, fileName.length());
//		}
//		return "";
//	}
}

/**
 * ファイルをサーバーにアップロードする場合に使用されるファイル管理内部クラスです。
 *
 * @og.group その他機能
 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応のため、public化
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
//final class UploadedFile {
//
//	private String filename;
//	private final String name;
//	private final String dir;
//	private final String original;
//	private final String type;
//
//	/**
//	 * アップロードファイルの管理オブジェクトを作成します。
//	 *
//	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
//	 *
//	 * @param	dir	ファイルを保管するフォルダ
//	 * @param	name		ファイルアップロードされた時のname属性
//	 * @param	filename	ファイル名(置き換え後)
//	 * @param	original	ファイル名(オリジナル)
//	 * @param	type	コンテントタイプ
//	 */
//	UploadedFile( final String dir, final String name, final String filename, final String original, final String type) {
//		this.dir		= dir;
//		this.name		= name;
//		this.filename	= filename;
//		this.original	= original;
//		this.type		= type;
//	}
//
//	/**
//	 * ファイルアップロードされた時のname属性を取得します。
//	 *
//	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
//	 *
//	 * @return	ファイルアップロードされた時のname属性
//	 */
//	public String getName() {
//		return name;
//	}
//
//	/**
//	 * コンテントタイプを取得します。
//	 *
//	 * @return	コンテントタイプ
//	 */
//	public String getContentType() {
//		return type;
//	}
//
//	/**
//	 * ファイル名(置き換え後)を取得します。
//	 *
//	 * @return	ファイル名(置き換え後)
//	 */
//	public String getFilesystemName() {
//		return filename;
//	}
//
//	/**
//	 * ファイル名(置き換え後)をセットします。
//	 *
//	 * @param	name	ファイル名(置き換え後)
//	 */
//	public void setFilesystemName( final String name ) {
//		filename = name;
//	}
//
//	/**
//	 * ファイル名(オリジナル)を取得します。
//	 *
//	 * @return	ファイル名(オリジナル)
//	 */
//	public String getOriginalFileName() {
//		return original;
//	}
//
//	/**
//	 * ファイル名(置き換え後)の File オブジェクトを取得します。
//	 *
//	 * @return File	File オブジェクト
//	 */
//	public File getFile() {
//		if(dir == null || filename == null) {
//			return null;
//		}
//		else {
//			return new File(dir + File.separator + filename);
//		}
//	}
//}
