/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;

import org.opengion.fukurou.model.CloudFileOperation;
import org.opengion.fukurou.model.FileOperation;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.io.HybsFileOperationFactory;
import org.opengion.hayabusa.report2.DocConverter_OOO;

/**
 * ドキュメントの変換・マージを行うタグです。
 *
 * 変換を行うことのできる入出力のフォーマット以下の通りです。
 *
 * [対応フォーマット]
 *  入力[Calc(ods)   ,Excel(xls)     ] ⇒ 出力[Calc(ods)   ,Excel(xls)     ,PDF]
 *  入力[Writer(odt) ,Word(doc)      ] ⇒ 出力[Writer(odt) ,Word(doc)      ,PDF]
 *  入力[Impress(odp),PowerPoint(ppt)] ⇒ 出力[Impress(odp),PowerPoint(ppt),PDF]
 *
 * 入出力のフォーマットは、入出力ファイルの拡張子により自動判別されます。
 * また、入出力ファイル名が同じ場合は何も処理されません。
 *
 * 入力ファイルを、カンマ区切りで複数指定することで、複数の入力ファイルをマージして
 * 出力することもできます。
 *
 * ※2 現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
 * ※1 この機能を利用するには、OpenOfficeが正しく設定されている必要があります。
 *
 * @og.formSample
 * ●形式：&lt;og:docConvert fileURL="…" inputFile="…" outputFile="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:docConvert
 *       fileURL            【TAG】操作するファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       inputFile        ○【TAG】入力ファイル名を指定します(必須)。
 *       inputStorageType   【TAG】入力ファイルのストレージタイプを指定します。(初期値:CLOUD_TARGET)
 *       inputBucketName     【TAG】入力ファイルのバケット名を指定します。(初期値:CLOUD_BUCKET)
 *       outputFile       ○【TAG】出力ファイル名を指定します(必須)。
 *       outputStorageType  【TAG】出力ファイルのストレージタイプを指定します。(初期値:CLOUD_TARGET)
 *       outputBucketName    【TAG】出力ファイルのバケット名を指定します。(初期値:CLOUD_BUCKET)
 *       delInput           【TAG】ドキュメント変換後、元のファイルを削除するかどうかを指定します(初期値:false[=削除しない])
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ・Calc(ods)ファイルをPDFに変換
 *        &lt;og:docConvert inputFile="temp.ods" outputFile="out.pdf" /&gt;
 *
 *    ・Excel(xls)ファイルを連結
 *        &lt;og:docConvert inputFile="temp1.xls,temp2.xls" outputFile="out.xls" /&gt;
 *
 * @og.rev 5.10.9.0 (2019/03/01) oota クラウドストレージ対応を追加。(Fileクラスを拡張)
 * 
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class DocConvertTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.6.0 (2010/05/06)" ;

	private static final long serialVersionUID = 516020100506L ;

	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String[]inputFile	= null;
	private String	outputFile	= null;
	private boolean	delInput	= false;
	private String	inputStorageType	= null;		// 5.10.9.0 (2019/03/01) ADD
	private String	inputBucketName		= null;		// 5.10.9.0 (2019/03/01) ADD
	private String	outputStorageType	= null;		// 5.10.9.0 (2019/03/01) ADD
	private String	outputBucketName		= null;		// 5.10.9.0 (2019/03/01) ADD

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		return ( SKIP_BODY );
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		// 出力ファイルで拡張子だけが指定された場合は、入力ファイル名を付加する
		if( outputFile.indexOf( '.' ) < 0 ) {
			int inSufIdx = inputFile[0].lastIndexOf( '.' );
			if( inSufIdx >= 0 ) {
				outputFile = inputFile[0].substring( 0, inSufIdx ) + "." + outputFile;
			}

		}

		String directory = HybsSystem.url2dir( fileURL );
		String[] input = new String[inputFile.length];
		for( int i=0; i<input.length; i++ ) {
			input[i] = StringUtil.urlAppend( directory,inputFile[i] );
		}
		String output = StringUtil.urlAppend( directory,outputFile );

		FileOperation inFile = HybsFileOperationFactory.create(null, null, input[0]);
		
		// 5.10.9.0 (2019/03/01) ADD クラウドストレージ指定の場合、ファイルをローカルにダウンロードします。
		if(!inFile.isLocal()) {
			for(String in: input) {
				File fromFile = HybsFileOperationFactory.create(inputStorageType, inputBucketName, in);
				File toFile = new File(in);
				// toFileのディレクトリを作成する必要あり？
				FileUtil.copy(fromFile, toFile);
			}
		}
		
		// ドキュメント変換
		DocConverter_OOO.convert( input, output );

		// 5.10.9.0 (2019/03/01) ADD クラウドストレージ指定の場合、ファイルをクラウドストレージにアップロードします。
		FileOperation outFile = HybsFileOperationFactory.create(outputStorageType, outputBucketName, output);
		if(!outFile.isLocal()) {
			File localFile = new File(output); 
			FileUtil.copy(localFile, outFile);
			localFile.delete();
		}
		
		if( delInput ) {
			for( int i=0; i<input.length; i++ ) {
				if( !( new File( input[i] ) ).delete() ) {
					System.err.println( "入力ファイルの削除に失敗しました。file=[" + input[i] + "]" );
				}
			}
		}

		return( EVAL_PAGE );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 	= HybsSystem.sys( "FILE_URL" );
		inputFile	= null;
		outputFile	= null;
		delInput	= false;
		inputStorageType	= null;		// 5.10.9.0 (2019/03/01) ADD
		inputBucketName		= null;		// 5.10.9.0 (2019/03/01) ADD
		outputStorageType	= null;		// 5.10.9.0 (2019/03/01) ADD
		outputBucketName		= null;		// 5.10.9.0 (2019/03/01) ADD
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * (初期値:システム定数のFILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】入力ファイル名を指定します。
	 *
	 * @og.tag
	 * 入力ファイル名を指定します。
	 *
	 * 入力ファイルは、カンマ区切りで複数指定することができます。
	 * この場合、複数の入力ファイルをマージして出力を行います。
	 *
	 * ※現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
	 *   また、マージを行う場合、印刷範囲など、ドキュメントに関連する情報は、1つ目のファイルの
	 *   情報が使用されます。
	 *
	 * @param	fname 入力ファイル名
	 */
	public void setInputFile( final String fname ) {
		inputFile = getCSVParameter( fname );
	}

	/**
	 * 【TAG】出力ファイル名を指定します。
	 *
	 * @og.tag
	 * 出力ファイル名を指定します。
	 * 出力ファイルには、拡張子のみ(xls,ods等)を指定することもできます。
	 * この場合、出力ファイル名は、入力ファイル名と同じになります。(拡張子のみが変換される)
	 *
	 * @param	fname 出力ファイル名
	 */
	public void setOutputFile( final String fname ) {
		outputFile = nval( getRequestParameter( fname ),outputFile );
	}

	/**
	 * 【TAG】ドキュメント変換後、元のファイルを削除するかどうかを指定します(初期値:false[=削除しない])。
	 *
	 * @og.tag
	 * ドキュメント変換後、元のファイルを削除するかどうかを指定します。
	 * (初期値:false(削除しない))
	 *
	 * @param	flg 出力元のファイルを削除するかどうか名
	 */
	public void setDelInput( final String flg ) {
		delInput = nval( getRequestParameter( flg ),delInput );
	}

	/**
	 * 【TAG】読み取り元ストレージタイプを設定します。
	 *  
	 * @og.tag
	 * ファイルを読み取り元の、ストレージタイプを設定します。
	 * 未設定の場合は、システムリソースの「CLOUD_TARGET」が参照されます。
	 * 自身のサーバを指定する場合は、「default」を設定してください。
	 * 
	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
	 * 
	 * @param inStorage ストレージタイプ
	 */
	public void setInputStorageType( final String inStorage ) {
		inputStorageType = nval( getRequestParameter( inStorage ), inputStorageType );
	}
	
	/**
	 * 【TAG】読み取り元バケット名を設定します。
	 * 
	 * @og.tag
	 * ファイルを読み取り元の、バケット名を指定します。
	 * クラウドストレージ利用時のみ有効です。
	 * 未設定の場合は、システムリソースの「CLOUD_BUKET」が参照されます。
	 * 
	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
	 * 
	 * @param inBucket バケット名
	 */
	public void setInputBucketName( final String inBucket ) {
		inputBucketName = nval( getRequestParameter( inBucket ), inputBucketName );
	}
	
	/**
	 * 【TAG】保存先ストレージタイプを設定します。
	 *  
	 * @og.tag
	 * ファイルを保存するストレージタイプを設定します。
	 * 未設定の場合は、システムリソースの「CLOUD_TARGET」が参照されます。
	 * 自身のサーバを指定する場合は、「default」を設定してください。
	 * 
	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
	 * 
	 * @param outStorage ストレージタイプ
	 */
	public void setOutputStorageType( final String outStorage ) {
		outputStorageType = nval( getRequestParameter( outStorage ), outputStorageType );
	}
	
	/**
	 * 【TAG】保存先バケット名を設定します。
	 * 
	 * @og.tag
	 * ファイルを保存するバケット名を指定します。
	 * クラウドストレージ利用時のみ有効です。
	 * 未設定の場合は、システムリソースの「CLOUD_BUKET」が参照されます。
	 * 
	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
	 * 
	 * @param outBucket バケット名
	 */
	public void setOutputBucketName(final String outBucket ) {
		this.outputBucketName = nval( getRequestParameter( outBucket ), outputBucketName );;
	}
	
	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @og.rev 5.10.9.0 (2019/03/01) inputStorageType,inputBucketName, outputStorageType,outputBucketNameを属性に追加。
	 * 
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"		,fileURL	)
				.println( "inputFile"	,inputFile		)
				.println( "outputFile"	,outputFile		)
				.println( "inputStorageType",	inputStorageType	)
				.println( "inputBucketName",		inputBucketName		)
				.println( "outputStorageType",	outputStorageType	)
				.println( "outputBucketName",	outputBucketName		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
