/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
import java.util.Locale;
import java.util.Map;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;			// 5.5.2.6 (2012/05/25)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_TABLE は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブル一覧の検索結果より、GF05 のテーブルカラム定義テーブルから
 * 必要な情報を取得し、テーブル作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"S.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,NEXT_EXTENT,COMMENTS)
 * の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/install/01_TABLE ;    出力ファイルの基準フォルダ(必須)
 *       XML : false ;                                    XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,NEXT_EXTENT,COMMENTS from GF02
 * 
 *      ① &lt;og:tableFilter classId="TABLE" keys="DIR,XML" vals='"{&#064;BASE_DIR}/sql/install/01_TABLE,"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="TABLE" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/install/01_TABLE ;
 *                   XML : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_TABLE extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.2 (2013/07/19)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

	private static final String[] DBKEY = {"SYSTEM_ID","TBLSYU","TABLE_NAME","NAME_JA",
							"TABLESPACE_NAME","INITIAL_EXTENT","NEXT_EXTENT","COMMENTS" };

	// 5.1.1.0 (2009/12/01) データのアクセス用の配列番号のIDを private ⇒ protected にします。
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SYSTEM_ID		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TBLSYU			= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_NAME		= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int NAME_JA			= 3;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLESPACE_NAME	= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INITIAL_EXTENT	= 5;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int NEXT_EXTENT		= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int COMMENTS			= 7;

	private static final String GF05_SEL = "SELECT CLM,SEQNO,NAME_JA,CLS_NAME,USE_LENGTH,DATA_DEFAULT,NOT_NULL,'' AS OPTS"
											+ " FROM GF05"
											+ " WHERE SYSTEM_ID=? AND TBLSYU=? AND TABLE_NAME=?"
											+ " AND   FGJ='1'"
											+ " ORDER BY SEQNO" ;

	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_CLM			= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_SEQNO		= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_NAME_JA		= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_CLS_NAME	= 3;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_USE_LENGTH	= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_DATA_DEFAULT= 5;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_NOT_NULL	= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int GF05_OPTIONS		= 7;

 //	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ; // 4.3.6.6 (2009/05/15)

	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int S = FixLengthData.S ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数
	private static final int T = FixLengthData.T ;		// addLen 定数
	private static final int T2= FixLengthData.T2 ;		// addLen 定数

	/** 各種定数  */
	protected static final String XML_START_TAG	= "<?xml version='1.0' encoding='UTF-8'?>" + CR + "<ROWSET tableName='xxx'>";
	protected static final String XML_END_TAG	= "</ROWSET>";
	protected static final String EXEC_START_TAG= "<EXEC_SQL>";
	protected static final String EXEC_END_TAG	= "</EXEC_SQL>";

	/** XML形式かどうか */
	protected boolean		isXml				= false; // 4.3.7.0 (2009/06/01)

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.3.7.0 (2009/06/01) トリガー、SEQUENCE作成機能、XML出力機能追加
	 * @og.rev 5.1.1.0 (2009/12/01) XML_START_TAG に、tableName をセットします。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		isXml = StringUtil.nval( getValue( "XML" ), false );

		int[] clmNo = getTableColumnNo( DBKEY );
		int rowCnt = table.getRowCount();

		File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		// カンマ,カラム,クラス,(,桁数,),初期値,NOT_NULL,拡張機能,コメント開始,行番号,名称,コメント終了
		int[] addLen = new int[] { 0,T,0,0,0,T2,T,T,T2,0,1,T,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,X,X,X,S,X, X,X,X, X,S,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[] data  = null;
		Transaction tran = getTransaction();	// 5.5.2.6 (2012/05/25)
		for( int row=0; row<rowCnt; row++ ) {
			String tableName = null;
			try {
				data  = table.getValues( row );
				String systemId	= data[clmNo[SYSTEM_ID]];
				String tblsyu	= data[clmNo[TBLSYU]];
				tableName		= data[clmNo[TABLE_NAME]];

				PrintWriter writer = FileUtil.getPrintWriter( new File( dir,tableName + ( isXml ? "S.xml" : "S.sql" ) ),ENCODE );

				if( isXml ) { writer.println( XML_START_TAG.replace( "xxx",tableName ) ); }		// 5.1.1.0 (2009/12/01) tableName をセット
				writer.print( makeHeadLine( clmNo,data ) );

				String[] vals = new String[] { systemId,tblsyu,tableName };
				String[][] gf05 = DBUtil.dbExecute( GF05_SEL,vals,tran );	// 5.1.9.0 (2010/08/01) Transaction 対応

				String uniqName = null;
				fixData.clear();
				// 値セット：まずは、最大長を求める必要がある。
				for( int i=0; i<gf05.length; i++ ) {
					String[] outData = makeLineList( gf05[i],i==0 );
					fixData.addListData( outData );

					// 4.3.7.0 (2009/06/01)
					if( "UNIQ".equalsIgnoreCase( gf05[i][GF05_CLM] ) || "UNIQSEQ".equalsIgnoreCase( gf05[i][GF05_CLM] ) ) {
						uniqName = gf05[i][GF05_CLM].toUpperCase( Locale.JAPAN );
					}
				}
				// 固定長化：最大長であわせた文字列を出力します。
				for( int i=0; i<gf05.length; i++ ) {
					writer.println( fixData.getFixData( i ) );
				}
				writer.println( makeEndLine( clmNo,data ) );

				// 4.3.7.0 (2009/06/01) UNIQ項目のSEQとトリガーを作成
				if( uniqName != null ) {
					writer.println( makeUniqSeq( clmNo,data ) );
					writer.println( makeUniqTrig( clmNo,data, uniqName ) );
				}

				if( isXml ) { writer.println( XML_END_TAG ); }
				writer.close();
			}
			catch( RuntimeException ex ) {
				ErrorMessage errMessage = makeErrorMessage( "TableFilter_TABLE Error",ErrorMessage.NG );
				errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE",ex.toString() );
				errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE",StringUtil.array2csv( data ) );
				errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE","TABLE=[" + tableName + "]" );
				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
				System.out.println( errMessage );
			}
		}

		return table;
	}

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 * @og.rev 5.6.6.2 (2013/07/19) EXEC_START_TAG の付け忘れ
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		final String TBL_NAME = data[clmNo[TABLE_NAME]];

		String LINE1 = TBL_NAME + " ( " + data[clmNo[NAME_JA]] + " )" ;
		String LINE2 = data[clmNo[COMMENTS]] ;
		String LINE3 = "Created : " + HybsSystem.getDate() ;

		// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
		int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/* ",	LINE3,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		fixData.addAllListData( outData );

		StringBuilder buf = new StringBuilder();
		fixData.getAllFixData( buf );

		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }			// 5.6.6.2 (2013/07/19) 出力忘れ

		buf.append( "CREATE TABLE " ).append( TBL_NAME ).append( " (" ).append( CR );

		return buf.toString();
	}

	/**
	 * 各行部分(カラム定義)の処理を実行します。
	 * カンマ,カラム,クラス,(,桁数,),初期値,NOT_NULL,拡張機能,コメント開始,行番号,名称,コメント終了
	 * の順に配列にセットします。
	 *
	 * @og.rev 5.5.1.9 (2012/04/18) useLen.length=0対応
	 * @og.rev 5.9.13.2 (2016/10/28) シングルクォート対応
	 * @param	data	１行分のデータ配列
	 * @param	first	最初の行かどうか[true:最初/false:それ以降]
	 *
	 * @return	各行部分(カラム定義)配列
	 */
	protected String[] makeLineList( final String[] data,final boolean first ) {
		// カンマ,カラム,クラス(桁数),初期値,NOT_NULL,独自拡張,行番号,名称,終了
		String[] outData = new String[13];
		String clsName = data[GF05_CLS_NAME];

		outData[0] = first ? "   " : " , " ;				// 0:カンマ
		outData[1] = data[GF05_CLM] ;							// 1:カラム

		if( clsName.startsWith( "CLOB" ) || clsName.startsWith( "DATE" ) ) {
			data[GF05_USE_LENGTH] = null;
		}
		String useLen = data[GF05_USE_LENGTH];
		if( useLen != null && ! useLen.equals( "0" ) && useLen.length() > 0 ) { // 5.5.1.9 (2012/04/18)
			outData[2] = clsName ;								// 2:クラス
			outData[3] = " ( " ;								// 3:(
			outData[4] = useLen ;								// 4:桁数
			outData[5] = " )" ;									// 5:)
		}
		else {
			outData[2] = clsName ;								// NUMBER型の桁数指定なしのケース
		}

		String def = data[GF05_DATA_DEFAULT];
		if( def != null && def.length() > 0 ) {
//			String comma = ( clsName.indexOf( "CHAR" ) >= 0 ) ? "'" : "" ;
			String comma = ( clsName.indexOf( "CHAR" ) >= 0 && def.indexOf( "'" ) < 0  ) ? "'" : "" ; // 5.9.13.2 (2016/10/28)
			outData[6] = "DEFAULT " + comma + def + comma ;		// 6:初期値
		}

		String notNull = data[GF05_NOT_NULL];
		if( notNull != null && notNull.equals( "1" ) ) {
			outData[7] = "NOT NULL"	;							// 7:NOT_NULL
		}

		String options = data[GF05_OPTIONS];
		if( options != null ) {
			outData[8] = options	;							// 8:拡張機能
		}

		String nameJA = data[GF05_NAME_JA] ;					// 名称
		if( nameJA != null ) {
			outData[9]  = "/* " ;								// 9:コメント開始
			outData[10] = data[GF05_SEQNO] ;					// 10:行番号
			outData[11] = nameJA ;								// 11:名称
			outData[12] = "*/" ;								// 12:コメント終了
		}

		return outData ;
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 * 5.8.3.2 (2015/01/23)より
	 * １．TABLESPACE_NAME を指定しない場合は、TABLESPACE 句を出力しません。
	 * ２．INITIAL_EXTENT,NEXT_EXTENT を 0 で指定した場合は該当箇所を出力しません。
	 * ３．PCTINCREASE は、出力しません。
	 * 
	 * @og.rev 5.8.3.1 (2015/01/23) 不要項目を出さないようにする(V6の6.1.0.0と類似の対応)
	 * 						V6ではNEXT_EXTENT完全削除ですが、V5では互換性のため一旦残しておきます。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 */
	protected String makeEndLine( final int[] clmNo,final String[] data ) {
		StringBuilder buf = new StringBuilder();
		
		final String tblSpcse = data[clmNo[TABLESPACE_NAME]] ;
		final String initExt  = data[clmNo[INITIAL_EXTENT]] ;
		final String nextExt  = data[clmNo[NEXT_EXTENT]] ;

		buf.append( ")" ).append( CR );
		if( !StringUtil.isNull( tblSpcse ) ) {
			buf.append( "TABLESPACE " ).append( tblSpcse ).append( CR );
		}
		if( !StringUtil.isNull( initExt ) && initExt.charAt(0) != '0' ) {
			buf.append( "STORAGE( INITIAL " ).append(  initExt ).append("K");
			if( !StringUtil.isNull( nextExt ) && nextExt.charAt(0) != '0' ) {
				buf.append( " NEXT " ).append(  nextExt ).append( "K" );
			}
//			buf.append( "K PCTINCREASE 0 )" );
			buf.append( " )" );
		}

		if( isXml )	{ buf.append( CR ).append( EXEC_END_TAG ); }
		else		{ buf.append( ";" ); }

		return buf.toString();
	}

	/**
	 * ユニークシーケンスの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ユニークシーケンス
	 */
	protected String makeUniqSeq( final int[] clmNo,final String[] data ) {
		StringBuilder buf = new StringBuilder();

		buf.append( CR );
		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		// 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
		buf.append( "CREATE SEQUENCE " ).append( data[clmNo[TABLE_NAME]] ).append( "S00 " ).append( CR );
		buf.append( "  INCREMENT BY 1 START WITH 1 MAXVALUE 999999999 CYCLE NOCACHE" );

		if( isXml )	{ buf.append( CR ).append( EXEC_END_TAG ); }
		else		{ buf.append( ";" ); }

		return buf.toString();
	}

	/**
	 * ユニークシーケンスと関連付けるトリガの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 * @param   uniqName	ユニークトリガ名
	 *
	 * @return	ユニークシーケンスと関連付けるトリガ
	 */
	protected String makeUniqTrig( final int[] clmNo,final String[] data, final String uniqName ) {
		final String TBL_NAME = data[clmNo[TABLE_NAME]] ;
		StringBuilder buf = new StringBuilder();

		buf.append( CR );
		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		// 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
		buf.append( "CREATE OR REPLACE TRIGGER " ).append( TBL_NAME ).append( "T00 " ).append( CR );
		buf.append( "  BEFORE INSERT ON ").append( TBL_NAME ).append( CR );
		buf.append( "  FOR EACH ROW " ).append( CR );
		buf.append( "  BEGIN " ).append( CR );
		// 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
		buf.append( "    SELECT " ).append( TBL_NAME ).append( "S00.NEXTVAL INTO :NEW." )
			.append( uniqName ).append( " FROM DUAL; " ).append( CR );
		buf.append( "  END; " ).append( CR );

		if( isXml )	{ buf.append( EXEC_END_TAG ); }
		else		{ buf.append( "/" ); }

		return buf.toString();
	}
}
