/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.io.File;
import java.util.Locale;
import org.opengion.fukurou.util.StringUtil;

import static org.opengion.fukurou.system.HybsConst.CR;

/**
 * ファイル操作のファクトリークラス
 *
 * デフォルトはローカルのファイル操作を行うFileOperationクラスを生成します。
 * 利用プラグイン、バケット、パス等を指定する事でクラウドのオブジェクトストレージに対応した
 * クラスを生成します。
 *
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @og.rev 5.10.9.0 (2019/03/01) 変更対応
 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
 * @author oota
 * @since JDK7.0
 */
// public class FileOperationFactory {
public final class FileOperationFactory {	// 7.2.9.4 (2020/11/20) PMD:A class which only has private constructors should be final
	private static final int BUFFER_MIDDLE = 200;

	private static final String CLOUD_PLUGIN = "org.opengion.plugin.cloud.FileOperation_" ;

	/**
	 * オブジェクトを作らせない為の、private コンストラクタ。
	 *
	 * @og.rev 7.2.9.4 (2020/11/20) オブジェクトを作らせない為の、private コンストラクタ
	 */
	private FileOperationFactory() {}

	/**
	 * インスタンス生成(ﾛｰｶﾙFile)。
	 *
	 * 引数を元に、ファイル操作インスタンスを生成します。
	 * ローカルのファイル操作を行うFileOperationクラスを返します。
	 *
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
	 *
	 * @param path ファイルパス
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(final String path) {
//		return newStorageOperation( (String)null, null, path.toString());
		return new FileOperation( path );
	}

	/**
	 * インスタンス生成(ｸﾗｳﾄﾞFile)。
	 *
	 * 引数を元に、ファイル操作クラスを生成します。
	 * new File( dir,fileName ).getPath() で求めたパスで、生成します。
	 * プラグインとバケットを指定する事で、plugin.cloud内のクラウド用のクラスを返します。
	 *
	 * ディレクトリとファイル名からパスを生成します。
	 *
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
	 *
	 * @param plugin 利用プラグイン
	 * @param buket バケット名
	 * @param dir ディレクトリ
	 * @param fileName ファイル名
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(final String plugin, final String buket, final String dir, final String fileName) {
		return newStorageOperation(plugin, buket, new File(dir,fileName).getPath());

//		final StringBuilder path = new StringBuilder(BUFFER_MIDDLE);
//		path.append( dir );
//
//		if(fileName != null) {
//			path.append(File.separator).append(fileName);
//		}
//
//		return newStorageOperation(plugin, buket, path.toString());
	}

	/**
	 * インスタンス生成(ｸﾗｳﾄﾞFile/ﾛｰｶﾙFile)。
	 *
	 * 引数を元に、ファイル操作クラスを生成します。
	 * プラグインとバケットを指定する事で、plugin.cloud内のクラウド用のクラスを返します。
	 * プラグインがnull、バケットがnull、もしくはDEFAULTの場合は標準のFileOperation(ﾛｰｶﾙﾌｧｲﾙ用)を返します。
	 *
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
	 *
	 * @param plugin 利用プラグイン
	 * @param buket バケット名
	 * @param path ファイルパス
	 * @return ファイル操作インスタンス
	 */
	public static FileOperation newStorageOperation(final String plugin, final String buket, final String path) {
		// buket の null 判定も条件に加えます。
		if( StringUtil.isNull(plugin) || "DEFAULT".equalsIgnoreCase(plugin) || StringUtil.isNull(buket) ) {
			return new FileOperation( path );		// ﾛｰｶﾙFile
		}

		final String cloudTarget = plugin.toUpperCase( Locale.JAPAN );		// 先に null 判定済み

		try {
			final Object[] args = new Object[] { buket, path };

			return (FileOperation)Class.forName( CLOUD_PLUGIN + cloudTarget )
						.getConstructor(String.class, String.class)
						.newInstance( args );
		}
		catch ( final Throwable th ) {
			final String errMsg = "FileOperation 生成で、失敗しました。" + CR
								+ " plugin=" + plugin + " , buket=" + buket + CR
								+ " path=" + path + CR ;

			throw new RuntimeException( errMsg,th );
		}

//		FileOperation rtn;
//		String cloudTarget = null;
//
//		final Object[] args = new Object[] { buket, path };
//
//		// 対象のクラウドサービスを取得(大文字化)。
//		// 未指定の場合は、ローカルディレクトリを利用。
//		if ( plugin != null && plugin.length() > 0 ) {
//			cloudTarget = plugin.toUpperCase( Locale.JAPAN );
//		}
//
//		try {
//			final StringBuilder sb = new StringBuilder(BUFFER_MIDDLE);
//
//			if (StringUtil.isNull(cloudTarget) || "DEFAULT".equals(cloudTarget)) {
//				sb.append("org.opengion.fukurou.model.FileOperation");
//			} else {
//				sb.append("org.opengion.plugin.cloud.FileOperation_")
//					.append(cloudTarget);
//			}
//			rtn = (FileOperation)Class.forName(sb.toString())
//					.getConstructor(String.class, String.class)
//					.newInstance(args);
////		} catch (Exception e) {
//		} catch (final Throwable th) {
//			// キャッチしたエラー情報をスロー
//			throw new RuntimeException(th);
//		}
//
//		return rtn;
	}

	/**
	 * 引数を元に、ファイル操作クラスをコピー生成します。
	 *
	 * 与えたfileオブジェクトがFileOperationだった場合は、プラグインとバケットを取得して
	 * それに基づいたFileOperationを返します。
	 * 標準のFileの場合は、defaultのFileOperationを返します。
	 * 元がnullの場合はnullを返します。
	 * new File( dir,fileName ).getPath() で求めたパスで、生成します。
	 *
	 * @og.rev 7.2.9.4 (2020/11/20) PMD:Avoid declaring a variable if it is unreferenced before a possible exit point.
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
	 *
	 * @param file コピー元
	 * @param dir 親パス(ディレクトリ)
	 * @param fileName 子パス
	 * @return 設定をコピーしたのFileOperation
	 */
//	public static FileOperation newStorageOperation(final File file, final String dir, final String fileName) {
	public static FileOperation copyFile(final File file, final String dir, final String fileName) {
		return copyFile( file, new File(dir,fileName).getPath() );

//		if( file == null) { return null; }
//
//		String plugin = null;
//		String buket  = null;
//
////		if( file == null) { return null; }		// PMD:Avoid declaring a variable if it is unreferenced before a possible exit point.
//
//		// FileOperation型の場合にプラグインを判定する
//		if( file instanceof FileOperation ) {
//			plugin = ((FileOperation)file).getPlugin();
//			buket  = ((FileOperation)file).getBucket();
//		}
//
//		return newStorageOperation( plugin, buket, dir, fileName);
	}

	/**
	 * 引数を元に、ファイル操作クラスをコピー生成します。
	 *
	 * 与えたfileオブジェクトがFileOperationだった場合は、プラグインとバケットを取得して
	 * それに基づいたFileOperationを返します。
	 * 標準のFileの場合は、defaultのFileOperationを返します。
	 * 元がnullの場合はnullを返します。
	 *
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
	 *
	 * @param file コピー元
	 * @param path パス
	 * @return 設定をコピーしたのFileOperation
	 */
//	public static FileOperation newStorageOperation(final File file, final String path) {
	public static FileOperation copyFile(final File file, final String path) {
//		return newStorageOperation( file, path, null);
		if( file == null) { return null; }		// 元がnullの場合はnullを返します。

//		String plugin = null;
//		String buket  = null;

//		if( file == null) { return null; }		// PMD:Avoid declaring a variable if it is unreferenced before a possible exit point.

		// FileOperation型の場合にプラグインを判定する
		if( file instanceof FileOperation ) {
			final String plugin = ((FileOperation)file).getPlugin();
			final String buket  = ((FileOperation)file).getBucket();

			return newStorageOperation( plugin, buket, path );		// ｸﾗｳﾄﾞFile/(ﾛｰｶﾙFile もありうる)
		}
		else {
			return newStorageOperation( path );						// ﾛｰｶﾙFile
		}
	}
}
