/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
// package org.opengion.fukurou.model;
package org.opengion.cloud;

import org.opengion.fukurou.model.FileOperation;
// import org.opengion.fukurou.model.CloudFileOperation;		// 8.0.0.2 (2021/10/15) fukurou.model → plugin.cloud にパッケージ移動
import org.opengion.fukurou.model.FileOperationFactory;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

/**
 * CloudFileOperation用のファイル情報の格納クラス
 *
 * listFilesで取得した、ディレクトリとファイル一覧情報を格納します。
 *
 * パフォーマンスや分かりやすさを考慮してCloudFileOperationからは分離しています
 *
 * @og.group ファイル操作
 *
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @og.rev 5.10.9.0 (2019/03/01) 変更対応
 * @author oota
 * @since JDK7.0
 */
public class FileOperationInfo extends CloudFileOperation {
	//* このプログラムのVERSION文字列を設定します。{@VALUE} */
	private static final String VERSION = "8.0.0.1 (2021/10/08)" ;
	private static final long serialVersionUID = 800120211008L ;

	/** クラス変数 */
	private final String plugin;

	private long size;
	// 8.0.0.0 (2021/07/31) Field ** has the same name as a method
//	private long lastModified;
//	private boolean isFile;
//	private boolean isDirectory;
	private long lastTime;				// 最終時刻
	private boolean isFil;				// ファイルか？
	private boolean isDir;				// フォルダか？
	private FileOperation file;			// ファイルオペレータ

	/**
	 * コンストラクタ
	 *
	 * 生成時の初期処理。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param path ファイルパス
	 */
	public FileOperationInfo(final String plugin, final String bucket, final String path) {
		super(bucket, path);
		this.plugin	= plugin;
		size		= 0;
//		lastModified = 0;
//		isFile = false;
//		isDirectory = false;
		lastTime	= 0L;
		isFil		= false;
		isDir		= false;
		file		= null;
	}

	/**
	 * FileOperationクラスの遅延生成
	 *
	 * 呼び出し時に、FileOperationインスタンスが未生成の場合は、
	 * 生成を行います。
	 */
	private void setFileOperation() {
		if(file == null) {
			file = FileOperationFactory.newStorageOperation( plugin, conBucket, conPath );
		}
	}

	/** Method */

	/**
	 * 書き込み処理(評価用)
	 *
	 * Fileを書き込みます。
	 *
	 * @og.rev 8.0.0.1 (2021/10/08) 新規追加
	 *
	 * @param inFile 書き込みFile
	 * @throws IOException ファイル関連エラー情報
	 */
	@Override
	public void write(final File inFile) throws IOException {
		setFileOperation();
		file.write(inFile);
	}

	/**
	 * 書き込み処理
	 *
	 * InputStreamのデータを書き込みます。
	 *
	 * @param is 書き込みデータのInputStream
	 * @throws IOException ファイル関連エラー情報
	 */
	@Override
	public void write(final InputStream is) throws IOException {
		setFileOperation();
		file.write(is);
	}

	/**
	 * 読み込み処理
	 *
	 * データを読み込み、InputStreamとして、返します。
	 *
	 * @return 読み込みデータのInputStream
	 * @throws FileNotFoundException ファイル非存在エラー情報
	 */
	@Override
	public InputStream read() throws FileNotFoundException {
		setFileOperation();
		return file.read();
	}

	/**
	 * 削除処理
	 *
	 * ファイルを削除します。
	 *
	 * @return 成否フラグ
	 */
	@Override
	public boolean delete() {
		setFileOperation();
		return file.delete();
	}

	/**
	 * コピー処理
	 *
	 * ファイルを指定先に、コピーします。
	 *
	 * @param afPath コピー先
	 * @return 成否フラグ
	 */
	@Override
	public boolean copy(final String afPath) {
		setFileOperation();
		return file.copy(afPath);
	}

	/**
	 * 一覧取得(Fileｸﾗｽ)
	 *
	 * この抽象パス名が示すディレクトリ内のファイルを示す抽象パス名の配列を返します。
	 * １つ下の、ディレクトリ・ファイル一覧を取得します。
	 *
	 * @param filter フィルタ情報
	 * @return ファイル一覧
	 * @see java.io.File#listFiles(FileFilter)
	 */
	@Override
	public File[] listFiles(final FileFilter filter) {
		setFileOperation();
		return file.listFiles(filter);
	}

	/**
	 * ファイルサイズ取得(Fileｸﾗｽ)
	 *
	 * ファイルサイズを取得します。
	 *
	 * @return ファイルサイズ
	 * @see java.io.File#length()
	 */
	@Override
	public long length() {
		return size;
	}

	/**
	 * ファイルサイズ設定
	 *
	 * ファイルサイズを設定します。
	 *
	 * @param size ファイルサイズ
	 */
	protected void setSize(final long size) {
		this.size = size;
	}

	/**
	 * 最終更新時刻の取得(Fileｸﾗｽ)
	 *
	 * 最終更新時刻を取得します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @return 最終更新時刻
	 * @see java.io.File#lastModified()
	 */
	@Override
	public long lastModified() {
//		return lastModified;
		return lastTime;
	}

	/**
	 * 最終更新時刻の設定
	 *
	 * 最終更新時刻を設定します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param lastModified 最終更新時刻
	 */
	protected void setLastModifiedValue(final long lastModified) {
//		this.lastModified = lastModified;
		lastTime = lastModified;
	}

	/**
	 * ファイル判定取得(Fileｸﾗｽ)
	 *
	 * ファイルであるかの判定を返します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @return ファイル判定
	 * @see java.io.File#isFile()
	 */
	@Override
	public boolean isFile() {
//		return isFile;
		return isFil;
	}

	/**
	 * ファイル判定設定
	 *
	 * ファイルであるかの判定を設定します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param isFile ファイル判定
	 */
	protected void setFile(final boolean isFile) {
//		this.isFile = isFile;
		isFil = isFile;
	}

	/**
	 * ディレクトリ判定取得(Fileｸﾗｽ)
	 *
	 * ディレクトリであるかの判定を返します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @return ディレクトリ判定
	 * @see java.io.File#isDirectory()
	 */
	@Override
	public boolean isDirectory() {
//		return isDirectory;
		return isDir;
	}

	/**
	 * ディレクトリ判定設定
	 *
	 * ディレクトリであるかの判定を設定します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param isDirectory ディレクトリ判定
	 */
	protected void setDirectory(final boolean isDirectory) {
//		this.isDirectory = isDirectory;
		isDir = isDirectory;
	}

	/**
	 * 親情報の取得(Fileｸﾗｽ)
	 *
	 * 親情報を返します。
	 *
	 * @return 親情報
	 * @see java.io.File#getParentFile()
	 */
	@Override
	public File getParentFile() {
//		return  FileOperationFactory.newStorageOperation( file , this.getParent() );
		return  FileOperationFactory.resolveFile( file , this.getParent() );
	}
}
