/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

/**
 * TableFilter_TABLE_DERBY は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。とくに、JavaDB(Derby)用のDB定義スクリプトを作成します。
 *
 * ここでは、テーブル一覧の検索結果より、GF05 のテーブルカラム定義テーブルから
 * 必要な情報を取得し、テーブル作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"S.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,COMMENTS)
 * の項目を取得する必要があります。
 *
 * @og.rev 6.4.5.0 (2016/04/08) 新規作成
 * @og.rev 6.8.1.0 (2017/07/14) JAVADB → DERBY に変更
 *
 * @version  6.4.5.0 (2016/04/08)
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class TableFilter_TABLE_DERBY extends TableFilter_TABLE {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.1.0.3 (2022/01/21)" ;

	/** 32,672Byte以上のカラムは、CLOBで定義する	{@value} */
	/* default */ static final int CLM_MAX_SIZE = 32672;

	private String tableName ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.5.0 (2016/04/08) 新規作成
	 * @og.rev 6.8.1.0 (2017/07/14) JAVADB → DERBY に変更
	 */
	public TableFilter_TABLE_DERBY() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * 単に、プライマリキー名を取得するために、テーブル名を設定します。
	 *
	 * @og.rev 6.4.5.0 (2016/04/08) 新規作成
	 * @og.rev 8.1.0.3 (2022/01/21) EXEC_SQLに、exists属性追加。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		tableName = data[clmNo[TABLE_NAME]];

		// 8.1.0.3 (2022/01/21) EXEC_SQLに、exists属性追加。
		execExistsSQL = "SELECT COUNT(*) FROM SYS.SYSTABLES WHERE TABLENAME=UPPER('" ;

		return super.makeHeadLine( clmNo,data );
	}

	/**
	 * 各行部分(カラム定義)の処理を実行します。
	 *
	 * @og.rev 6.4.5.0 (2016/04/08) 新規作成
	 *
	 * @param	data	１行分のデータ配列
	 * @param	first	最初の行かどうか[true:最初/false:それ以降]
	 *
	 * @return	各行部分(カラム定義)配列
	 */
	@Override
	protected String[] makeLineList( final String[] data,final boolean first ) {

		final String clsName = data[GF05_CLS_NAME] ;

		// 5.1.1.0 (2009/12/01) 各カラムの説明文を 入れたくない場合は、null を設定する。
		if( isXml ) { data[GF05_NAME_JA] = null; }

		// 5.6.9.2 (2013/10/18) useLen が、nullやゼロ文字列の場合の考慮
		String useLen = data[GF05_USE_LENGTH] ;
		if( useLen != null && useLen.contains( "." ) ) { useLen = useLen.replace( '.' , ',' ); }	// 6.4.9.3 (2016/08/26)

		if( "UNIQ".equalsIgnoreCase( data[GF05_CLM] ) || "UNIQSEQ".equalsIgnoreCase( data[GF05_CLM] ) ) {
			data[GF05_CLS_NAME]     = "BIGINT";
			data[GF05_DATA_DEFAULT] = null;
			data[GF05_NOT_NULL]     = "1";
			data[GF05_USE_LENGTH]   = null;
			data[GF05_OPTIONS]      = " generated always as identity (start with 1, increment by 1),"
										+ "constraint " + tableName + " primary key(" + data[GF05_CLM] + ") " ;
		}
		else if( clsName == null || clsName.isEmpty() || clsName.startsWith( "VARCHAR" ) ) {				// 6.1.0.0 (2014/12/26) refactoring
			// 5.6.9.2 (2013/10/18) 上でuseLen変数にセットしているので、その値を使います。
			if( useLen != null && useLen.length() > 0 && Integer.parseInt( useLen ) > CLM_MAX_SIZE ) {		// 6.0.2.4 (2014/10/17) メソッド間違い
				data[GF05_CLS_NAME]     = "CLOB";
				data[GF05_DATA_DEFAULT] = null;
				data[GF05_NOT_NULL]     = null;
				data[GF05_USE_LENGTH]   = null;
			}
			else {
				data[GF05_CLS_NAME] = "VARCHAR" ;
			}
		}
		else if( clsName.startsWith( "NUMBER" ) ) {
			// 5.6.9.2 (2013/10/18) 初期値をあらかじめ設定しておきます。
			data[GF05_CLS_NAME] = "INTEGER";
			data[GF05_USE_LENGTH] = null;

			// 5.6.9.2 (2013/10/18) 上でuseLen変数にセットしているので、その値を使います。
			if( useLen != null && useLen.length() > 0 ) {
				if( useLen.indexOf( ',' ) >= 0 ) {		// カンマがある
					data[GF05_CLS_NAME] = "DOUBLE";
				}
				else if( Integer.parseInt( useLen ) >= 10 ) {	// １０桁以上
					data[GF05_CLS_NAME] = "BIGINT";
				}
			}
		}

		return super.makeLineList( data,first );
	}

//	/**
//	 * 定義の最後の部分の処理を実行します。
//	 *
//	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
//	 *
//	 * @param	clmNo	カラム番号配列
//	 * @param	data	１行分のデータ配列
//	 *
//	 * @return	定義の最後の部分
//	 * @og.rtnNotNull
//	 */
//	@Override
//	protected String makeEndLine( final int[] clmNo,final String[] data ) {
//		return ")" + execEndTag ;						// 6.0.2.3 (2014/10/10) 改行を含める。
//	}

//	/**
//	 * ユニークシーケンスの作成処理を実行します。
//	 *
//	 * @og.rev 8.1.0.3 (2022/01/21) 引数にtableNameを渡すように変更。
//	 *
////	 * @param	clmNo	カラム番号配列
////	 * @param	data	１行分のデータ配列
//	 * @param	tableName	ﾃｰﾌﾞﾙ名
//	 *
//	 * @return	ユニークシーケンス
//	 * @og.rtnNotNull
//	 */
//	@Override
////	protected String makeUniqSeq( final int[] clmNo,final String[] data ) {
//	protected String makeUniqSeq( final String tableName ) {
//		return "";
//	}

//	/**
//	 * ユニークシーケンスと関連付けるトリガの作成処理を実行します。
//	 *
////	 * @param	clmNo		カラム番号配列
////	 * @param	data		１行分のデータ配列
//	 * @param	tableName	ﾃｰﾌﾞﾙ名
//	 * @param   uniqName	ユニークトリガ名
//	 *
//	 * @og.rev 8.1.0.3 (2022/01/21) 引数にtableNameを渡すように変更。
//	 *
//	 * @return	ユニークシーケンスと関連付けるトリガ
//	 * @og.rtnNotNull
//	 */
//	@Override
////	protected String makeUniqTrig( final int[] clmNo,final String[] data, final String uniqName ) {
//	protected String makeUniqTrig( final String tableName, final String uniqName ) {
//		return "";
//	}
}
