/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.ToString;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.StringUtil;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.Locale;

/**
 * jQueryのdraggableを使用したｵﾌﾞｼﾞｪｸﾄの移動を行う個々の div 要素を作成します。
 *
 * viewFormType="CustomData" の BODY 部に記述された、class="dragDiv"（固定） を持った、
 * DIV要素を出力します。
 * BODY部に、出力する際に、GE18 データベースとのインターフェースになります。
 * 実際には、このクラスは文字列の出力のみで、DBアクセスや、TableModelアクセスは発生しません。
 *
 * @og.formSample
 * ●形式：&lt;og:dragDiv &gt;[ICON]&lt;/og:dragDiv&gt;
 *     → &lt;div class="dragDiv" id="[UNIQ]" style="left:[LOC_COL]px; top:[LOC_ROW]px;" &gt;[ICON]&lt;/div&gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:dragDiv
 *       id                 【TAG】id属性をセットします(初期値:[UNIQ])
 *       style              【TAG】style属性をセットします(初期値:left:[LOC_COL]px; top:[LOC_ROW]px;)
 *       names              【TAG】divタグに追加する標準属性を指定します。
 *                                   LOC_GRP,LOC_KEY,NAME_JA,LABEL_NAME,BIKO,ICON,R_KEY1,R_KEY2,R_KEY3 など
 *       optionAttributes   【TAG】JavaScript などの HTML基本タグ以外の属性を、そのままタグとして使用します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ※ tbody の中を、og:dragDiv にした場合。
 *    &lt;og:dragView&gt;
 *      &lt;og:view
 *          viewFormType	= "CustomData"
 *          ･･･
 *      &gt;
 *          &lt;og:tbody rowspan="1" &gt;
 *              &lt;og:dragDiv &gt;[ICON]&lt;/og:dragDiv&gt;
 *          &lt;/og:tbody&gt;
 *      &lt;/og:view&gt;
 *    &lt;/og:dragView&gt;
 *
 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
 * @og.group 画面部品
 *
 * @version  7.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class DragDiv extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.1.0 (2018/10/15)" ;
	private static final long serialVersionUID = 701020181015L ;

	private static final String	DEF_ID		= "[UNIQ]"	;								// id属性
	private static final String	DEF_STYLE	= "left:[LOC_COL]px; top:[LOC_ROW]px;"	;	// style属性
	private String[]	names	;										// divタグに追加する標準属性を指定します。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
	 */
	public DragDiv() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		if( useTag() ) {
			jspPrint( makeDiv() );			// divタグを出力します。
			return EVAL_BODY_INCLUDE ;		// Body インクルード( extends TagSupport 時)
		}
		return SKIP_BODY ;					// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( useTag() ) {
			jspPrint( "</div>" );	// 最後の divタグを出力します。
		}

		return EVAL_PAGE ;			// ページの残りを評価する。( extends TagSupport 時)
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
	 */
	@Override
	protected void release2() {
		super.release2();
		names		= null;										// divタグに追加する標準属性を指定します。
	}

	/**
	 * dragDivタグを生成します。
	 *
	 * ﾄﾞﾗｯｸﾞ処理に必要な、DIVタグを出力します。
	 *
	 * @return  divタグ
	 */
	private String makeDiv() {
		final TagBuffer tagBuf = new TagBuffer()
				.startTag( "div" )
				.add( "class"	, "dragDiv"							)
				.add( "id"		, nval( get("id"    ) , DEF_ID )	)
				.add( "style"	, nval( get("style" ) , DEF_STYLE )	)
				.addOptions( get( "optionAttributes" )				);

		if( names != null ) {
			for( final String name : names ) {
				tagBuf.add( name.toLowerCase(Locale.JAPAN) , '[' + name + "]" );	// キーと値を追加
			}
		}

		return tagBuf.toBefore();			// タグの前半部分のみ生成します。
	}

	/**
	 * 【HTML】要素に対して固有の名前(id)をつける場合に設定します。
	 *
	 * @og.tag
	 * 特別な使用方法として、id="FOCUS" とすることで、フィールド系要素に
	 * フォーカスを移動させます。これは、そのページ内で唯一の id 属性として使用ください。
	 *
	 * ※ HTML5 より、autofocus属性の使用が可能になりました。
	 *
	 * @param   id 固有の名前
	 */
	@Override
	public void setId( final String id ) {
		set( "id",getRequestParameter( id ) );
	}

	/**
	 * 【HTML】この要素に対して適用させるスタイルシート(style)を設定します。
	 *
	 * @og.tag
	 * タグにstyle属性を設定します。これは、キー:値; のセットを複数記述できます。
	 * 通常は、class属性や、id属性で登録しておき、&lt;style&gt; で
	 * 外部から指定する方がソースは読みやすくなります。
	 *
	 * @param   style スタイルシート (例 style="color:red; font-size:24pt;" など)
	 */
	public void setStyle( final String style ) {
		set( "style",getRequestParameter( style ) );
	}

	/**
	 * 【TAG】JavaScript などの HTML基本タグ以外の属性を、そのままタグとして使用します。
	 *
	 * @og.tag
	 * JavaScript などの HTML基本タグ以外の属性を、そのままタグとして使用します。
	 *
	 * @param   optionAttributes オプション属性
	 */
	public void setOptionAttributes( final String optionAttributes ) {
		set( "optionAttributes",getRequestParameter( optionAttributes ) );
	}

	/**
	 * 【TAG】処理対象のカラム名をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * divタグに属性として追記します。
	 * 標準のカラム(LOC_GRP,LOC_KEY,NAME_JA,LABEL_NAME,BIKO,ICON,R_KEY1,R_KEY2,R_KEY3) の
	 * 場合は、標準の JavaScript common/jquery/dragView.js と、common/dragUpdate.jsp で
	 * 処理可能です。
	 * それ以外に、JOIN したカラムを使用する場合は、両方とも書き換える必要があります。
	 * 通常は、ここで、カラムを指定するより、dragUpdate.jsp をカスタマイズして、
	 * ｷｰﾜｰﾄﾞから、JOIN して値を取得するほうが、わかりやすいと思います。
	 * ちなみに、単に、値を使用するだけなら、div属性に追記する必要はありません。
	 *
	 * div要素への指定方法は、カラム="[カラム]" ですが、キーとなるカラムを小文字にしています。
	 * 例えば、カラムに、NAME_JA,LABEL_NAME を指定する場合、
	 * name_ja="[NAME_JA]" label_name="[LABEL_NAME]" という属性が追加されます。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
	 *
	 * @param	nms カラム名 (CSV形式)
	 */
	public void setNames( final String nms ) {
		names = StringUtil.csv2Array( getRequestParameter( nms ) );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "id"			,get("id" 	 )	)
				.println( "style"		,get("style" )	)
				.println( "names"		,String.join( ", " , names ) )
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
