/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;		// 6.1.0.0 (2014/12/26)

/**
 * option/ogTbodyMovable.js JavaScript を利用したマウスによる上下移動にシンクロ等させます。
 *
 * headタグで、useTbodyMovable="true" をセットすると、マウスまたはカーソルで、
 * tbody 部を上下に移動させることが可能です。
 * このタグは、その動作に関連付けて、同期して移動させたり、内容を入れ替えるカラムや
 * 内容に、＋１、－１するカラム、移動の制限（入替対象にしない、ドラッグできないなど）
 * を指定します。
 * 記述は、view の後で構いません。
 *
 * このタグを有効にするには、headタグで、useTbodyMovable="true" をセットする必要があります。
 * なお、この JavaScript は、１行単位に tbody を記述している view であれば、有効です。
 * 内部で、tr ではなく、tbody を処理しているためです。
 * （つまり、ガント専用ではありませんが、HTMLTableには使えません。）
 *
 * このタグは、span タグを出力し、ogTbodyMovable.js で処理しています。その際のキーとして、
 * span タグ には、class="tbodyMovable" を付けています。
 *
 * @og.formSample
 * ●形式：&lt;og:tbodyMovable  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、useTbodyMovable="true" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:tbodyMovable
 *       seqclm             【TAG】移動時に値を入れ替えするカラム名(CSV形式で複数指定可能,先頭名称で判断)(初期値:SEQ)
 *       noMove             【TAG】入替制限する tbodyのクラス名(CSV形式で複数指定可能)
 *       noDrag             【TAG】マウスでドラッグできない tbodyのクラス名(CSV形式で複数指定可能)
 *       adjustSeq          【TAG】上移動(-1)、下移動(+1)で、値をセットするカラム名(１つのみ、seqclmで指定されている必要あり)
 *       sync               【TAG】他のTABLEを同期して移動させる場合に、クラス/IDを指定
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *   &lt;og:tbodyMovable
 *      seqclm      = "DYORDER,SEQ"
 *      noMove      = "disMovable"
 *      noDrag      = "disDragable"
 *      adjustSeq   = "DYORDER"
 *      sync        = "fixBodyDiv"
 *   /&gt;
 *
 * @og.rev 5.6.3.3 (2013/04/19) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class TbodyMovableTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	private transient TagBuffer tag = new TagBuffer( "span" ) ;		// 6.0.2.4 (2014/10/17) transient化

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TbodyMovableTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		tag.add( "class","tbodyMovable" );
		jspPrint( tag.makeTag() );

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "span" );
	}

	/**
	 * 【TAG】移動時に値を入れ替えするカラム名を指定します(CSV形式,先頭名称で判断)(初期値:SEQ)。
	 *
	 * @og.tag
	 * 該当タグのseqclm属性値が存在する場合はシーケンスカラム名とします。
	 * 上下移動で、同期して動きますが、このカラムの値も入れ替わります。判りにくいですが、
	 * つまり位置は入れ替わらないという事です。
	 * 例えば、レコードに連番を振っておき、このカラムだけ移動しないので、当初の連番のままとなり、
	 * その値を元に、計算しなおすことで、納期調整を行ったりします。
	 * シーケンスは指定カラムの先頭名称で判断するので注意して下さい。
	 * 　例：SEQの指定でSEQUENCEカラムが判定されてしまう。 
	 * seqclm属性はCSV形式で複数指定可能です。 
	 * 初期値 は、"SEQ" です。
	 *
	 * @param   seqclm SEQカラム名 移動時に値を入れ替えたいカラム名(CSV形式)
	 */
	public void setSeqclm( final String seqclm ) {
		tag.add( "seqclm",nval( getRequestParameter( seqclm ),null ) );
	}

	/**
	 * 【TAG】入替制限(移動できない)する tbodyのクラス名を指定します(CSV形式)。
	 *
	 * @og.tag
	 * tbodyが 入替制限する場合に、tbodyにここで指定した class 属性を付与しておくことで、
	 * 入れ替え対象から除外されます。（簡単にいうと、これを飛び越えて移動できません）
	 * 例えば、グループ化しているレコードの範囲でしか入れ替えたくない・・・などの時に使用します。
	 * 
	 * ここでは、CSV形式で複数のクラス名を指定可能です。
	 *
	 * @param   noMove 入替制限クラス (CSV形式)
	 */
	public void setNoMove( final String noMove ) {
		tag.add( "noMove",nval( getRequestParameter( noMove ),null ) );
	}

	/**
	 * 【TAG】マウスでドラッグできない tbodyのクラス名を指定します(CSV形式)。
	 *
	 * @og.tag
	 * tbodyを ドラッグしたくない場合に、tbodyにここで指定した class 属性を付与しておくことで、
	 * ドラッグ対象から除外されます。（簡単にいうと、掴めません）
	 * これは、対象の tbody がつかめないだけで、その前後の tbody と入れ替えはされます。
	 * 位置はどこに行くか判りませんが、ここで指定した tbody の順番は保証されます。
	 *
	 * 例えば、納期確定しているレコードは入れ替えたくない・・・などの時に使用します。
	 * 
	 * ここでは、CSV形式で複数のクラス名を指定可能です。
	 *
	 * @param   noDrag ドラッグ制限クラス (CSV形式)
	 */
	public void setNoDrag( final String noDrag ) {
		tag.add( "noDrag",nval( getRequestParameter( noDrag ),null ) );
	}

	/**
	 * 【TAG】上移動(-1)、下移動(+1)で、値をセットするカラム名(１つのみ、seqclmで指定されている必要あり)。
	 *
	 * @og.tag
	 * adjustSeq属性で指定したカラム（seqclmで指定されている必要あり）は、
	 * 上に動かした時は上の値から-1された値を、下に動かした時は下の値に+1した値を
	 * セットするような動作になります。これは１つのみ指定可能です。
	 * 
	 * 移動時に値を入れ替えするカラム名の特殊版といえます。
	 * ここでは、カラム名は１つのみ指定可能で、かつ、seqclmで指定されている必要があります。
	 *
	 * @param   adjustSeq 値設定カラム (上移動(-1)、下移動(+1)で、値をセットするカラム名)
	 */
	public void setAdjustSeq( final String adjustSeq ) {
		tag.add( "adjustSeq",nval( getRequestParameter( adjustSeq ),null ) );
	}

	/**
	 * 【TAG】他のTABLEを同期して移動させる場合に、クラス/IDを指定します。
	 *
	 * @og.tag
	 * 左右分割やガント系のテーブル部とガント部に分かれているようなテーブルで、上下移動を
	 * 行う場合に、ドラッグしているレコードに同期させて移動させることが可能です。
	 * ここで指定するクラスは、tableか、table直上の要素のクラスのみ指定可能です。
	 * また、tbodyMovableが複数ある場合は先頭の属性値がセットされます。
	 *
	 * @param   sync テーブル同期カラム (他のTABLEを同期して移動させる場合に指定するクラス/ID)
	 */
	public void setSync( final String sync ) {
		tag.add( "sync",nval( getRequestParameter( sync ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "tbodyMovable" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
