/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.HybsConst ;						// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.db.Transaction;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.html.ViewMarker;						// 8.0.0.0 (2021/09/30)

import java.util.Locale;
import java.util.Map;
import java.util.LinkedHashMap ;
import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring
import java.util.Collections;										// 6.4.3.1 (2016/02/12) refactoring
import java.util.List;												// 8.2.1.0 (2022/07/15)
import java.util.ArrayList;											// 8.2.1.0 (2022/07/15)

/**
 * AbstractTableFilter は､TableUpda ｲﾝﾀｰﾌｪｰｽを継承した､DBTableModel 処理用の
 * Abstract実装ｸﾗｽです｡
 *
 * @og.rev 5.5.2.6 (2012/05/25) protected変数をprivateに変更｡ｲﾝﾀｰﾌｪｰｽにﾒｿｯﾄﾞ追加
 * @og.rev 6.4.1.1 (2016/01/16) keysMap を､ｻﾌﾞｸﾗｽから設定させるように変更｡
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public abstract class AbstractTableFilter implements TableFilter {
	/** ｼｽﾃﾑの改行ｺｰﾄﾞを設定します｡*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します｡	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

	// 5.5.2.6 (2012/05/25) protected変数をprivateに変更｡ｲﾝﾀｰﾌｪｰｽにﾒｿｯﾄﾞ追加
	private DBTableModel	table		;
	private String			modifyType	;
	private int[]			rowNo		;
	private boolean			useDebug	;		// 6.0.2.5 (2014/10/31) refactoring ﾒｿｯﾄﾞと同じなので名称変更
	private Transaction		tran		;		// 5.1.9.0 (2010/08/01) 追加
	private String			sql			;		// 4.2.4.0 (2008/06/23)
	private String			dbid		;		// 4.2.4.0 (2008/06/23)
	private ResourceManager resource	;		// 4.3.7.4 (2009/07/01)
	private ViewMarker		viewMarker	;		// 8.0.0.0 (2021/09/30)

	private int 			errCode		= ErrorMessage.OK;
	private ErrorMessage	errMessage	;

	// 8.2.1.0 (2022/07/15) TableFilter_MAPCLM 専用のMapを管理するList
	private List<Map<String,String>> mapList ;

	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え｡  */
	private final ConcurrentMap<String,String> keyValMap = new ConcurrentHashMap<>();

	/** 7.4.0.1 (2021/04/16) 値を返すための変数  */
	private final ConcurrentMap<String,String> rtnValMap = new ConcurrentHashMap<>();

	// 5.6.6.0 (2013/07/05) keys の整合性ﾁｪｯｸを行います｡
	// 6.4.1.1 (2016/01/16) keysMap を､ｻﾌﾞｸﾗｽから設定させるように変更
	/** 6.4.3.1 (2016/02/12) Collections.synchronizedMap で同期処理を行います｡  */
	private final Map<String,String> keysMap = Collections.synchronizedMap( new LinkedHashMap<>() ) ;

	// 6.0.2.3 (2014/10/10) plugin.table.TableFilter_XXXX から移動
	/** XML宣言定数 */
	protected static final String XML_START_TAG	= "<?xml version='1.0' encoding='UTF-8'?>" + CR + "<ROWSET tableName='xxx'>";
	/** ROWSET 終了TAG {@value} */
	protected static final String XML_END_TAG	= "</ROWSET>";
	/** EXEC_SQL 開始TAG {@value} */
	protected static final String EXEC_START_TAG= "<EXEC_SQL>";
	/** EXEC_SQL exists="0" 開始TAG {@value} */
	protected static final String EXEC_EXISTS_0_TAG= "<EXEC_SQL exists=\"0\">";		// 8.1.0.3 (2022/01/21)
	/** EXEC_SQL 終了TAG {@value} */
	protected static final String EXEC_END_TAG	= "</EXEC_SQL>";

	// 6.0.2.3 (2014/10/10) isXml で､CR + EXEC_END_TAG のｷｬｯｼｭを作成します｡
	/** XML形式かどうか  */
	protected boolean	isXml		;		// 6.0.2.3 (2014/10/10)
	/** exists判定に使用するSQL文 */
	protected String 	execExistsSQL;		// 8.1.0.3 (2022/01/21)
	/** 終了ﾀｸﾞ */
	protected String 	execEndTag	;		// 6.0.2.3 (2014/10/10)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	protected AbstractTableFilter() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * keys の整合性ﾁｪｯｸを行うための初期設定を行います｡
	 * ｻﾌﾞｸﾗｽのｺﾝｽﾄﾗｸﾀ内で､設定するようにしてください｡
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keys の整合性ﾁｪｯｸ対応
	 * @og.rev 6.4.3.1 (2016/02/12) ConcurrentMap 系は､key,val ともに not null 制限です｡
	 *
	 * @param	key  整合性ﾁｪｯｸを行うための keysMap に設定するｷｰ
	 * @param	cmnt 整合性ﾁｪｯｸを行うための ｷｰ の説明
	 */
	protected void initSet( final String key , final String cmnt ) {
		if( key != null && cmnt != null ) {
			keysMap.put( key , cmnt );
		}
	}

	/**
	 * DBTableModel をｾｯﾄします｡
	 *
	 * @param	table DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public void setDBTableModel( final DBTableModel table ) {
		this.table = table;
	}

	/**
	 * DBTableModel を取得します｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return	内部のDBTableModel
	 */
	@Override	// TableFilter
	public DBTableModel getDBTableModel() {
		return table;
	}

	/**
	 * ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)を指定します｡
	 *
	 * 通常は､DBTableModel に自動設定されている modifyType を元に､ﾃﾞｰﾀ処理方法を
	 * 選別します｡(A:追加 C:更新 D:削除)
	 * この場合､行単位で modifyType の値を取得して判別する必要がありますが､一般には
	 * 処理対象は､全件おなじ modifyType である可能性が高いです｡
	 * また､selectedAll などで強制的に全件処理対象とする場合は､modifyType に値が
	 * 設定さていません｡その様な場合に外部より modifyType を指定します｡
	 * 初期値は､自動判定 です｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) 廃止
	 *
	 * @param  type ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)
	 */
	@Override	// TableFilter
	public void setModifyType( final String type ) {
		modifyType = type;
	}

	/**
	 * ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)を取得します｡
	 *
	 * 初期値は､自動判定 です｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return  ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)
	 */
	@Override	// TableFilter
	public String getModifyType() {
		return modifyType ;
	}

	/**
	 * ｷｰと値のﾍﾟｱの変数配列を受け取ります｡
	 *
	 * ここでは､この方式以外に､ﾊﾟﾗﾒｰﾀｰMapを受け取る方法もあります｡
	 * この受け取る時に､ｷｰを大文字化します｡TableFilter の keys は､
	 * 大文字のみで定義しておくことで､HTMLやWindows世代の曖昧な表記方法に
	 * 対応しています｡(unixやxmlのような厳格な方が好きですけど)
	 *
	 * keys,vals とﾊﾟﾗﾒｰﾀｰMapを同時に指定した場合は､両方とも有効です｡
	 * ただし､ｷｰが重複した場合は､不定と考えてください｡
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性ﾁｪｯｸを行います｡
	 *
	 * @param   keys ｷｰ配列
	 * @param   vals 値配列
	 * @see		#setParamMap( ConcurrentMap )
	 */
	@Override	// TableFilter
	public void setKeysVals( final String[] keys,final String[] vals ) {
		if( keys != null && vals != null ) {
			for( int i=0; i<keys.length; i++ ) {
				// 5.6.6.0 (2013/07/05) 共通のｾｯﾀｰﾒｿｯﾄﾞ経由で登録します｡
				setKeyVal( keys[i],vals[i] );
			}
		}
	}

	/**
	 * ｷｰと値のﾍﾟｱを受け取り､内部の keyValMap ﾏｯﾌﾟに追加します｡
	 *
	 * ｷｰか値のどちらかが null の場合は､何もしません｡つまり､val に
	 * null をｾｯﾄすることはできません｡
	 *
	 * このﾒｿｯﾄﾞは､setKeysVals( String[] ,String[] ) ﾒｿｯﾄﾞと､
	 * setParamMap( Map<String,String> ) ﾒｿｯﾄﾞの両方から､使用します｡
	 * 処理を行うに当たり､下記の処理を行います｡
	 * １．ｷｰを大文字化します｡
	 * ２．各ｸﾗｽの keys と整合性ﾁｪｯｸを行います｡
	 *
	 * ただし､setKeysVals と setParamMap の登録順は､不定と考えてください｡
	 * 両方に同じｷｰを指定すると､どちらの値がｾｯﾄされたかは､不定です｡
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性ﾁｪｯｸを行います｡
	 * @og.rev 6.4.3.4 (2016/03/12) Map#forEach で対応する｡
	 * @og.rev 6.7.9.1 (2017/05/19) keysMap が､空の場合も､keyValMap に登録する｡(initSet 未登録時)
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の､"／＞" 止めを､"＞" に変更します)｡
	 * @og.rev 8.0.0.0 (2021/09/30) brﾀｸﾞを止めてを､preﾀｸﾞに変更します｡
	 *
	 * @param   key ｷｰ文字列(null の場合は､処理しない)
	 * @param   val 値文字列(null の場合は､処理しない)
	 * @see		#setKeysVals( String[] ,String[] )
	 * @see		#setParamMap( ConcurrentMap )
	 */
	private void setKeyVal( final String key,final String val ) {
		// key か val かどちらかが null の場合は､処理を行わない｡
		if( key == null || val == null ) { return; }

		final String upKey = key.toUpperCase(Locale.JAPAN);

		// 6.7.9.1 (2017/05/19) keysMap が､空の場合も､keyValMap に登録する｡(initSet 未登録時)
		if(  keysMap.isEmpty() || keysMap.containsKey( upKey ) ) {		// keysMap は､各ｻﾌﾞｸﾗｽで定義
			keyValMap.put( upKey,val );
		}
		else {
//			final String BR = "<br />" + CR ;
//			final String BR = "<br>" + CR ;				// 7.0.1.0 (2018/10/15)
			final String BR = CR ;						// 8.0.0.0 (2021/09/30)
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE );
			// 6.0.2.5 (2014/10/31) char を append する｡
//			errMsg.append( BR )
			errMsg.append( "<pre>" ).append( BR )
				  .append( "指定のｷｰは､この tableFilter では､使用できません｡" ).append( BR )
				  .append( "  class=[" ).append( getClass().getName() ).append( ']' ).append( BR )
				  .append( "  key  =[" ).append( key				  ).append( ']' ).append( BR )
				  .append( "  ======== usage keys ======== " ).append( BR ) ;
			// 6.4.3.4 (2016/03/12) Map#forEach で対応する｡
			keysMap.forEach( (k,v) -> { errMsg.append( ' ' ).append( k ).append( ':' ).append( v ).append( BR ); } );
			errMsg.append( "  ============================ " ).append( BR ).append( "</pre>" );

			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 選択された行番号の配列をｾｯﾄします｡
	 *
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SELECTED_KEY を元に､選ばれた 行を
	 * 処理の対象とします｡
	 *
	 * @param   rowNoTmp 行番号配列(可変長引数)
	 */
	@Override	// TableFilter
	public void setParameterRows( final int... rowNoTmp ) {
		if( rowNoTmp != null && rowNoTmp.length > 0 ) {		// 6.1.1.0 (2015/01/17) 可変長引数でもnullは来る｡
			final int size = rowNoTmp.length ;
			rowNo = new int[size];
			System.arraycopy( rowNoTmp,0,rowNo,0,size );
		}
	}

	/**
	 * 選択された行番号の配列を取得します｡
	 *
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY を元に､選ばれた 行を
	 * 処理の対象とします｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 * @og.rev 6.0.2.5 (2014/10/31) null ではなく､ｻｲｽﾞ０の配列を返すように変更｡
	 *
	 * @return   行番号の配列(選ばれていない場合は､ｻｲｽﾞ０の配列を返す)
	 * @og.rtnNotNull
	 */
	@Override	// TableFilter
	public int[] getParameterRows() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return rowNo == null ? new int[0] : rowNo.clone() ;
	}

	/**
	 * ｱｸｾｽﾛｸﾞ取得の為,Transactionｵﾌﾞｼﾞｪｸﾄを設定します｡
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 *
	 * @param   tran Transactionｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public void setTransaction( final Transaction tran ) {
		this.tran = tran;
	}

	/**
	 * ｱｸｾｽﾛｸﾞ取得の為,Transactionｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return   Transactionｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public Transaction getTransaction() {
		return tran;
	}

	/**
	 * DBIDを指定します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param dbid 接続先ID
	 */
	@Override	// TableFilter
	public void setDbid( final String dbid ) {
		this.dbid = dbid;
	}

	/**
	 * DBIDを取得します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return DBID(接続先情報)
	 */
	@Override	// TableFilter
	public String getDbid() {
		return dbid;
	}

	/**
	 * ﾎﾞﾃﾞｨｰ部分のSQLを指定します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param sql ﾎﾞﾃﾞｨｰ部分のSQL
	 */
	@Override	// TableFilter
	public void setSql( final String sql ) {
		this.sql = sql;
	}

	/**
	 * ﾎﾞﾃﾞｨｰ部分のSQLを取得します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return ﾎﾞﾃﾞｨｰ部分のSQL
	 */
	@Override	// TableFilter
	public String getSql() {
		return sql;
	}

	/**
	 * ﾊﾟﾗﾒｰﾀｰMapを指定します｡
	 *
	 * keys,vals と ﾊﾟﾗﾒｰﾀｰMapを同時に指定した場合は､両方とも有効です｡
	 * ただし､ｷｰが重複した場合は､不定と考えてください｡
	 *
	 * この受け取る時に､ｷｰを大文字化します｡TableFilter の keys は､
	 * 大文字のみで定義しておくことで､HTMLやWindows世代の曖昧な表記方法に
	 * 対応しています｡(unixやxmlのような厳格な方が好きですけど)
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規追加
	 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性ﾁｪｯｸを行います｡
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap を受け取ることを明確にするため､I/FをConcurrentMapに変更します｡
	 *
	 * @param paramMap ﾊﾟﾗﾒｰﾀｰMap
	 * @see		#setKeysVals( String[] ,String[] )
	 */
	@Override	// TableFilter
	public void setParamMap( final ConcurrentMap<String,String> paramMap ) {
		// 6.4.3.3 (2016/03/04) Map#forEach に変更
		if( paramMap != null ) {
			paramMap.forEach( (k,v) -> setKeyVal( k,v ) );
		}
	}

	/**
	 * ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄを指定します｡
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 *
	 * @param resource ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public void setResource( final ResourceManager resource ) {
		this.resource = resource;
	}

	/**
	 * ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public ResourceManager getResource() {
		return resource;
	}

	/**
	 * viewMarkerｵﾌﾞｼﾞｪｸﾄを指定します｡
	 *
	 * @og.rev 8.0.0.0 (2021/09/30) viewMarker追加
	 *
	 * @param marker viewMarkerｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public void setViewMarker( final ViewMarker marker ) {
		viewMarker = marker;
	}

	/**
	 * viewMarkerｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 8.0.0.0 (2021/09/30) viewMarker追加
	 *
	 * @return viewMarkerｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public ViewMarker getViewMarker() {
		return viewMarker;
	}

	/**
	 * 値を返すためのMapを返します｡
	 *
	 * Mapそのものを返しますので､中身の書き換えは行わないでください｡
	 *
	 * @og.rev 7.4.0.1 (2021/04/16) 値を返すための変数
	 *
	 * @return Mapｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public Map<String,String> getReturnMap() {
		return rtnValMap;
	}

	/**
	 * TableFilter_MAPCLM 専用のMapを管理するListを設定します｡
	 *
	 * @og.rev 8.2.1.0 (2022/07/15) TableFilter_MAPCLM 専用のMapを管理するList
	 *
	 * @param	valList Mapｵﾌﾞｼﾞｪｸﾄのﾘｽﾄ
	 */
	@Override	// TableFilter
	public void setMapList( final List<Map<String,String>> valList ) {
		mapList = valList ;
	}

	/**
	 * TableFilter_MAPCLM 専用のMapを管理するListを返します｡
	 *
	 * ﾘｽﾄの登録順は､vals で指定されたMapｵﾌﾞｼﾞｪｸﾄの登録順です｡
	 * 指定がない場合は､null が返ります｡
	 *
	 * @og.rev 8.2.1.0 (2022/07/15) TableFilter_MAPCLM 専用のMapを管理するList
	 *
	 * @return  Mapｵﾌﾞｼﾞｪｸﾄ
	 */
	protected List<Map<String,String>> getMapList() {
		return mapList;
	}

	/**
	 * ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true:する/false:しない]を指定します｡
	 * true でﾃﾞﾊﾞｯｸﾞ情報を表示します｡
	 *
	 * @param   flag  ﾃﾞﾊﾞｯｸﾞ出力するか [true:する/false:しない]
	 */
	@Override	// TableFilter
	public void setDebug( final boolean flag ) {
		useDebug = flag;	// 6.0.2.5 (2014/10/31) refactoring ﾒｿｯﾄﾞと同じなので名称変更
	}

	/**
	 * ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true:する/false:しない]を取得します｡
	 * true でﾃﾞﾊﾞｯｸﾞ情報を表示します｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return  ﾃﾞﾊﾞｯｸﾞ出力 [true:する/false:しない]
	 */
	@Override	// TableFilter
	public boolean isDebug() {
		return useDebug ;	// 6.0.2.5 (2014/10/31) refactoring ﾒｿｯﾄﾞと同じなので名称変更
	}

	/**
	 * ｴﾗｰｺｰﾄﾞ を取得します｡
	 * ｴﾗｰｺｰﾄﾞ は､ErrorMessage ｸﾗｽで規定されているｺｰﾄﾞです｡
	 *
	 * @return   ｴﾗｰｺｰﾄﾞ
	 */
	@Override	// TableFilter
	public int getErrorCode() {
		return errCode;
	}

	/**
	 * ｴﾗｰﾒｯｾｰｼﾞｵﾌﾞｼﾞｪｸﾄ を取得します｡
	 *
	 * @return   ｴﾗｰﾒｯｾｰｼﾞｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override	// TableFilter
	public ErrorMessage getErrorMessage() {
		return errMessage;
	}

	/**
	 * ﾀｲﾄﾙとｴﾗｰｺｰﾄﾞを指定して､ｴﾗｰﾒｯｾｰｼﾞｵﾌﾞｼﾞｪｸﾄ を作成します｡
	 * すでに､作成済みの場合は､作成済みのｵﾌﾞｼﾞｪｸﾄを､まだ､未作成の場合は､
	 * 新規に作成します｡
	 *
	 * @param	title	ﾀｲﾄﾙ
	 * @param	code	ｴﾗｰｺｰﾄﾞ
	 *
	 * @return	ｴﾗｰﾒｯｾｰｼﾞｵﾌﾞｼﾞｪｸﾄ
	 */
	protected ErrorMessage makeErrorMessage( final String title,final int code ) {
		if( errMessage == null ) {
			errMessage = new ErrorMessage( title );
		}
		if( errCode < code ) { errCode = code; }
		return errMessage;
	}

	/**
	 *  ｶﾗﾑ名配列(String[])より､対応するｶﾗﾑNo配列(int[])を作成します｡
	 *
	 * @param	nameArray ｶﾗﾑ名配列
	 *
	 * @return	ｶﾗﾑNo配列(可変長引数)
	 */
	protected int[] getTableColumnNo( final String... nameArray ) {
		int[] clmNo = new int[nameArray.length];
		for( int i=0; i<clmNo.length; i++ ) {
			clmNo[i] = table.getColumnNo( nameArray[i] );
		}
		return clmNo;
	}

	/**
	 * 設定されたﾊﾟﾗﾒｰﾀｷｰに対する値を取得します｡
	 * 引数､および､ﾊﾟﾗﾒｰﾀが null の場合は､ null を返します｡
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のﾁｪｯｸ追加
	 *
	 * @param	key	ﾊﾟﾗﾒｰﾀｷｰ
	 *
	 * @return	ﾊﾟﾗﾒｰﾀ値
	 */
	protected String getValue( final String key ) {
		return key == null ? null : keyValMap.get( key );
	}

	/**
	 * ﾌｨﾙﾀｰからtaglibのﾘｸｴｽﾄ変数に値を書き戻したい場合に､key と val をｾｯﾄします｡
	 * 引数､および､ﾊﾟﾗﾒｰﾀが null の場合は､何もしません｡
	 *
	 * @og.rev 7.4.0.1 (2021/04/16) 値を返すための変数ｾｯﾄ
	 *
	 * @param	key	戻しｷｰ
	 * @param	val	戻し値
	 */
	protected void setValue( final String key ,final String val ) {
		if( key != null && val != null ) {
			rtnValMap.put( key,val );
		}
	}

	/**
	 * keyValMapに持っているｷｰの配列を取得します｡
	 * これは､ｻﾌﾞｸﾗｽで､initSet(String,String) を行わない場合､keys には
	 * 値を自由に設定できます｡
	 * その値を取り出すためです｡
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) 新規追加
	 *
	 * @return	ｷｰ値の配列(keyValMapに持っているｷｰ)
	 */
	protected String[] getKeys() {
		return keyValMap.keySet().toArray( new String[keyValMap.size()] );
	}
}
