/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.html.ViewMarker;
import org.opengion.hayabusa.html.ViewMarker_MARKER;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Set;											// 6.4.3.4 (2016/03/11)
import java.util.Locale ;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * 複数のcolumnMarker を統合して、検索結果に対して様々な属性を付加するタグです。(参照:columnMarker)
 *
 * DBTableModelオブジェクトの表示にHTMLタグをマークします。
 * 子タグとして、ColumnMarkerTag のBODY要素をパースして、タグを作成します。
 *
 * @og.formSample
 * ●形式：&lt;og:viewMarker &gt; ･･･ &lt;/og:viewMarker &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:viewMarker
 *       command          ○【TAG】コマンド (NEW,RENEW,RESET,REVIEW)をセットします(必須)。
 *       viewMarkerId       【TAG】(通常は使いません)requestから取得する ViewMarker に対応する Attributes オブジェクトの ID
 *       isRenderer         【TAG】マーカーのタイプが renderer かどうか[true/false]を指定します(初期値:true)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:viewMarker&gt;
 *
 * ●使用例
 *     商品CD２(CDSYHN02)は赤字で表示する。
 *     商品CD３(CDSYHN03)は-----と表示する。
 *         &lt;og:viewMarker command="{&#064;command}"&gt;
 *             &lt;og:columnMarker column="CDSYHN02" onMark="true" &gt;
 *                 &lt;font color='red'&gt;[VCDSYHN02]&lt;/font&gt;
 *             &lt;/og:columnMarker&gt;
 *             &lt;og:columnMarker column="CDSYHN03" onMark="true" &gt;
 *                 &lt;CENTER&gt;-----&lt;/CENTER&gt;
 *             &lt;/og:columnMarker&gt;
 *         &lt;/og:viewMarker&gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewMarkerTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.4.3 (2019/06/24)" ;
	private static final long serialVersionUID = 704320190624L ;

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	= "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW	= "RENEW" ;
	/** command 引数に渡す事の出来る コマンド  リセット {@value}  */
	public static final String CMD_RESET  = "RESET" ;			// 3.5.4.0 (2003/11/25)
	/** command 引数に渡す事の出来る コマンド  再表示 {@value}  */
	public static final String CMD_REVIEW  = "REVIEW" ;			// 3.5.4.0 (2003/11/25)

	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> COMMAND_SET = new ArraySet<>( CMD_NEW , CMD_RENEW ,CMD_RESET , CMD_REVIEW );

	private transient ViewMarker	viewMarker		= new ViewMarker_MARKER();
	private String	viewMarkerId	;
	private String	command			;
	private boolean	isRenderer		= true;						// 3.8.6.1 (2006/10/20)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ViewMarkerTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 戻り値を、EVAL_BODY_INCLUDE → EVAL_BODY_BUFFERED に変更
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return useTag() && check( command, COMMAND_SET )
					? EVAL_BODY_BUFFERED
					: SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.6.1 (2006/10/20) isRenderer 属性でマーカーの種類を指定
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 * @og.rev 6.7.2.0 (2017/01/16) ColumnMarkerTagなどが、caseKey,caseVal等で未使用のときの対応。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( useTag() && check( command, COMMAND_SET ) && viewMarker.isUsable() ) {		// 6.7.2.0 (2017/01/16)
			final String mid ;
			if( isRenderer ) {
				mid = nval( viewMarkerId,HybsSystem.VIEW_MARK_KEY );
			}
			else {
				mid = nval( viewMarkerId,HybsSystem.EDIT_MARK_KEY );
			}
			setRequestAttribute( mid,viewMarker );
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.6.1 (2006/10/20) isRenderer属性追加、viewMarkerId属性初期値変更
	 */
	@Override
	protected void release2() {
		super.release2();
		viewMarker		= new ViewMarker_MARKER();
		viewMarkerId	= null;
		command			= null;
		isRenderer		= true;						// 3.8.6.1 (2006/10/20)
	}

	/**
	 * 内部タグの ColumnMarkerTag より、個々のカラムの値を書き換える 為の
	 * マーカー文字列を受け取る。
	 *
	 * 複数の値を受け取って、後ほど、すべてのカラムに対して処理を行います。
	 *
	 * @og.rev 3.1.2.0 (2003/04/07) taglib パッケージ内部で使用している箇所を protected 化する。
	 * @og.rev 4.0.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
	 *
	 * @param	attri	マーク処理属性
	 */
	protected void addAttribute( final Attributes attri ) {
		viewMarker.addAttribute( attri );
	}

	/**
	 * 【TAG】コマンド (NEW,RENEW,RESET,REVIEW)をセットします。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ViewMarkerTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】(通常は使いません)requestから取得する ViewMarker に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewForm オブジェクトに、ViewMarker オブジェクトをこのキーより取得することにより、
	 * カラムに対して、マーカー情報を付加して表示させる。
	 * 初期値は、HybsSystem.VIEW_MARK_KEY です。
	 *
	 * @og.rev 3.1.4.0 (2003/04/18) 新規追加
	 * @og.rev 3.5.6.3 (2004/07/12) {&#064;XXXX} 変数を使用できるように変更。
	 * @og.rev 3.5.6.4 (2004/07/16) MARK_ID を付加して、他のid と混同しないようにします。
	 * @og.rev 3.8.6.1 (2006/10/20) 初期値:null (editMarker を考慮)
	 * @og.rev 7.0.4.3 (2019/06/24) MARK_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id マーカーID
	 */
	public void setViewMarkerId( final String id ) {
//		viewMarkerId = nval( getRequestParameter( id ),null ) ;

		final String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewMarkerId = temp + TaglibUtil.MARK_ID;
		}
	}

	/**
	 * 【TAG】マーカーのタイプが renderer かどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * このMarker オブジェクトが、ViewMarker か、EditMarker かを指定します。
	 * 内部的には、ViewMarker オブジェクトを構築しており、viewForm で、どちらにセット
	 * するかを決めているだけです。
	 * true にセットすると、従来からある、viewMarker(renderer) を指定します。
	 * false は、editMarker として作用し、オブジェクトが書き込み可能な場合に表示されます。
	 * 初期値は、true:renderer です。
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param	flag	マーカータイプ [true:renderer/false:editor]
	 */
	public void setIsRenderer( final String flag ) {
		isRenderer = nval( getRequestParameter( flag ),isRenderer ) ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		viewMarker = new ViewMarker_MARKER();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "viewMarkerId"	,viewMarkerId	)
				.println( "command"			,command		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
