/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;
import java.util.List;
import java.util.Map ;
import java.util.LinkedHashMap ;

import org.opengion.fukurou.util.HybsEntry;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableWriter をXML形式で出力する為の実装ｸﾗｽです。
 * DefaultTableWriter を継承していますので,ﾗﾍﾞﾙ,名前,ﾃﾞｰﾀの出力部のみ
 * ｵｰﾊﾞｰﾗｲﾄﾞして,XML形式ﾌｧｲﾙの出力機能を実現しています。
 *
 * 出力のXML形式は、拡張ｵﾗｸﾙ XDK形式のXMLﾌｧｲﾙです。
 * ｵﾗｸﾙXDK形式のXMLとは、下記のような ROWSET をﾄｯﾌﾟとする ROW の
 * 集まりで1ﾚｺｰﾄﾞを表し、各ROWには、ｶﾗﾑ名をｷｰとするXMLになっています。
 *
 *   &lt;ROWSET&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;ｶﾗﾑ1&gt;値1&lt;/ｶﾗﾑ1&gt;
 *             ･･･
 *           &lt;ｶﾗﾑn&gt;値n&lt;/ｶﾗﾑn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *   &lt;ROWSET&gt;
 *
 * この形式であれば、XDK(Oracle XML Developer's Kit)を利用すれば、非常に簡単に
 * ﾃﾞｰﾀﾍﾞｰｽとXMLﾌｧｲﾙとの交換が可能です。
 * <a href="https://docs.oracle.com/cd/F19136_01/adxdk/introduction-to-XDK.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * 拡張XDK形式とは、ROW 以外に、SQL処理用ﾀｸﾞ(EXEC_SQL)を持つ XML ﾌｧｲﾙです。
 * これは、ｵﾗｸﾙXDKで処理する場合、無視されますので、同様に扱うことが出来ます。
 * この、EXEC_SQL は、それそれの XMLﾃﾞｰﾀをﾃﾞｰﾀﾍﾞｰｽに登録する際に、
 * SQL処理を自動的に流す為の、SQL文を記載します。
 * この処理は、ｲﾍﾞﾝﾄ毎に実行される為、その配置順は重要です。
 * このﾀｸﾞは、複数記述することも出来ますが、BODY部には、1つのSQL文のみ記述します。
 *
 *   &lt;ROWSET tablename="GEXX" &gt;
 *       &lt;EXEC_SQL&gt;                    最初に記載して、初期処理(ﾃﾞｰﾀｸﾘｱ等)を実行させる。
 *           delete from GEXX where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *       &lt;MERGE_SQL&gt;                   このSQL文で UPDATEして、結果が0件ならINSERTを行います。
 *           update GEXX set AA=[AA] , BB=[BB] where CC=[CC]
 *       &lt;/MERGE_SQL&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;ｶﾗﾑ1&gt;値1&lt;/ｶﾗﾑ1&gt;
 *             ･･･
 *           &lt;ｶﾗﾑn&gt;値n&lt;/ｶﾗﾑn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *       &lt;EXEC_SQL&gt;                    最後に記載して、項目の設定(整合性登録)を行う。
 *           update GEXX set AA='XX' , BB='XX' where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *   &lt;ROWSET&gt;
 *
 * @og.group ﾌｧｲﾙ出力
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class TableWriter_XML extends TableWriter_Default {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	// 5.6.6.1 (2013/07/12) keys の整合性ﾁｪｯｸを行います。
	/** staticｲﾆｼｬﾗｲｻﾞ後、読み取り専用にするので、ConcurrentHashMap を使用しません。 */
	private static final Map<String,String> KEYS_MAP = new LinkedHashMap<>();
	static {
		KEYS_MAP.put( "TABLENAME"	, "処理を実施するテーブル名を指定"											);
		KEYS_MAP.put( "FIRST"		, "最初に記載して、初期処理(データクリア等)を実行させる、EXEC_SQLを指定"	);
		KEYS_MAP.put( "LAST"		, "最後に記載して、項目の設定(整合性登録)を行う、EXEC_SQLを指定"			);
		KEYS_MAP.put( "MERGESQL"	, "このSQL文で UPDATEして、結果が０件ならINSERTを行う、MERGE_SQLを指定"		);
	}

	private String	tableName		= "" ;					// 4.0.0 (2005/01/31)
	private String	firstExecSql	= "" ;					// 4.0.0 (2005/01/31)
	private String	lastExecSql		= "" ;					// 4.0.0 (2005/01/31)
	private String	mergeSql		= "" ;					// 5.6.6.1 (2013/07/12) MERGE_SQL 対応

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_XML() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBTableModel から ﾃﾞｰﾀを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期ﾒｿｯﾄﾞ(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 *
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	public void writeDBTable( final PrintWriter writer )  {
		super.setHeaderSequence( "D" );
		super.writeDBTable( writer );
	}

	/**
	 * PrintWriter に DBTableModelのﾍｯﾀﾞｰ情報を書き込みます。
	 * XML のﾍｯﾀﾞｰは、&lt;?xml version='1.0' encoding='encode値'?&gt; になります。
	 * encoding属性には、encode値をｾｯﾄします。
	 * encoding属性に設定できる値は、"UTF-8","UTF-16","Shift_JIS" です。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期ﾒｿｯﾄﾞ(synchronized付き)を非同期に変更する。
	 * @og.rev 4.0.0.0 (2005/01/31) XML宣言部のencoding属性を、encode値をｾｯﾄする。
	 *
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeHeader( final PrintWriter writer ) {
		final String encoding = getEncode();
		writer.println( "<?xml version='1.0' encoding='" + encoding + "'?>" ) ;
	}

	/**
	 * PrintWriter に DBTableModelのﾃｰﾌﾞﾙ情報を書き込みます。
	 *
	 * 出力のXML形式は、ORACLE XDK での出力ﾌｧｲﾙと同じ形式ですので、直接ﾃﾞｰﾀﾍﾞｰｽに
	 * 登録することができます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期ﾒｿｯﾄﾞ(synchronized付き)を非同期に変更する。
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 4.0.0.0 (2005/01/31) EXEC_SQL ﾀｸﾞ機能の追加
	 * @og.rev 5.1.6.0 (2010/05/01) DbType の初期値(dbType)を利用する。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 5.6.6.1 (2013/07/12) MERGE_SQL 対応
	 * @og.rev 5.6.7.0 (2013/07/27) 改行処理の見直し
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) ﾃﾞｰﾀ出力用のﾚﾝﾃﾞﾗｰ
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows =  table.getRowCount();
		final boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		writer.print( "<ROWSET" );
		writer.print( tableName );
		writer.println( ">" );

		// 4.0.0 (2005/01/31)
		writer.print( firstExecSql );						// 5.6.7.0 (2013/07/27) 改行処理の見直し

		// 5.6.6.1 (2013/07/12) MERGE_SQL 対応
		writer.print( mergeSql );							// 5.6.7.0 (2013/07/27) 改行処理の見直し

		for( int row=0; row<numberOfRows; row++ ) {
			writer.print( "<ROW num=\"" );
			writer.print( row+1 );
			writer.println( "\">" );
			for( int i=0; i<numberOfColumns; i++ ) {
				final int clm = clmNo[i];
				if( clm < 0 ) { continue; }					// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし

				String val = table.getValue(row,clm);
				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) ﾃﾞｰﾀ出力用のﾚﾝﾃﾞﾗｰ
					val = dbColumn[clm].getWriteValue( val );
				}

				writer.print( "<" );
				writer.print( table.getColumnName(clm) );
				writer.print( ">" );
				writer.print( StringUtil.htmlFilter( val ) );
				writer.print( "</" );
				writer.print( table.getColumnName(clm) );
				writer.println( ">" );
			}
			writer.println( "</ROW>" );
		}

		// 4.0.0 (2005/01/31)
		writer.print( lastExecSql );											// 5.6.7.0 (2013/07/27) 改行処理の見直し
		writer.println( "</ROWSET>" );
	}

	/**
	 * ﾊﾟﾗﾒｰﾀｰﾘｽﾄをｾｯﾄします。
	 * 内部は、HybsEntry ｸﾗｽを持っています。
	 * 引数が、null の場合は、何もしません。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 5.6.6.1 (2013/07/12) MERGE_SQL 対応
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性ﾁｪｯｸを行います。
	 *
	 * @param	listParam	HybsEntryﾘｽﾄ
	 */
	@Override
	public void setParam( final List<HybsEntry> listParam ) {
		if( listParam != null && ! listParam.isEmpty() ) {
			final StringBuilder first = new StringBuilder( BUFFER_MIDDLE ) ;
			final StringBuilder last  = new StringBuilder( BUFFER_MIDDLE ) ;
			final StringBuilder merge = new StringBuilder( BUFFER_MIDDLE ) ;	// 5.6.6.1 (2013/07/12) MERGE_SQL 対応

			for( final HybsEntry entry : listParam ) {
				final String key = entry.getKey() ;								// 5.6.6.1 (2013/07/12) keys の整合性ﾁｪｯｸ
				checkParam( key , KEYS_MAP );

				final String val = entry.getValue() ;							// 5.6.6.1 (2013/07/12) val を先に取得
				if( val != null && val.length() > 0 ) {
					// 処理を実施するﾃｰﾌﾞﾙ名を指定
					// 複数の key="TableName" があれば、最後のみが有効。
					if( "TableName".equalsIgnoreCase( key ) ) {
						tableName = " tableName=\"" + val + "\"" ;
					}
					// 最初に記載して、初期処理(ﾃﾞｰﾀｸﾘｱ等)を実行させる、EXEC_SQLを指定
					else if( "First".equalsIgnoreCase( key ) ) {
						first.append( "<EXEC_SQL>" )
							.append( StringUtil.htmlFilter( val ) )
							.append( "</EXEC_SQL>" )
							.append( CR );
					}
					// 最後に記載して、項目の設定(整合性登録)を行う、EXEC_SQLを指定
					else if( "Last".equalsIgnoreCase( key ) ) {
						last.append( "<EXEC_SQL>" )
							.append( StringUtil.htmlFilter( val ) )
							.append( "</EXEC_SQL>" )
							.append( CR );
					}
					// このSQL文で UPDATEして、結果が0件ならINSERTを行う、MERGE_SQLを指定
					else if( "MergeSql".equalsIgnoreCase( key ) ) {
						merge.append( "<MERGE_SQL>" )
							.append( StringUtil.htmlFilter( val ) )
							.append( "</MERGE_SQL>" )
							.append( CR );
					}
				}
			}
			firstExecSql = first.toString();
			lastExecSql  = last.toString();
			mergeSql     = merge.toString();									// 5.6.6.1 (2013/07/12) MERGE_SQL 対応
		}
	}
}
