/*
 * Copyright (c) 2017 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.fileexec;

import java.nio.file.Path;
import java.util.Set;									// 7.2.1.0 (2020/03/13) ｽｷｬﾝ対象外判定
// import java.util.HashSet;							// 7.2.2.0 (2020/03/27) javadoc用のjdk8に対応します。

/**
 * システムのベースフォルダ基準で、各種パスを管理するクラスです。
 *
 *<pre>
 * 本システムでは、ベースフォルダを基準として、各種管理フォルダを規定しています。
 * それらのパスオブジェクトを管理します。
 *
 *</pre>
 *
 * <table class="plain" style="font-family: monospace;">
 *  <caption>各種管理フォルダの規定</caption>
 *  <tr><th>フォルダ	</th><th>説明</th></tr>
 *  <tr><td>BASE_PATH	</td><td>基本となるフォルダパス</td></tr>
 *  <tr><td>SUB_PATH	</td><td>ベースパスの下位のサブパス。これがスキャン対象になる。</td></tr>
 *  <tr><td>WORK_PATH	</td><td>中間パス。サブパスのファイルを一旦このワークに移動してから、取込処理を行います</td></tr>
 *  <tr><td>OK_PATH		</td><td>処理が成功した場合のファイルの移動先</td></tr>
 *  <tr><td>NG_PATH		</td><td>処理が失敗した場合のファイルの移</td></tr>
 *</table>
 *
 * @og.rev 7.0.0.0 (2017/07/07) 新規作成
 *
 * @version  7.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public final class BasePath {
	private static final XLogger LOGGER= XLogger.getLogger( BasePath.class.getSimpleName() );		// ログ出力

	/** システム依存の改行記号(String)。	*/
	public static final String CR = System.getProperty("line.separator");

	/** BASEフォルダの相対パスの処理ﾌｫﾙﾀﾞ(WORK)の初期値 {@value} */
	public static final String DEF_WORK_DIR	= "work" ;
	/** BASEフォルダの相対パスの処理済ﾌｫﾙﾀﾞ(正常)の初期値 {@value} */
	public static final String DEF_OK_DIR	= "bkup_ok" ;
	/** BASEフォルダの相対パスの処理済ﾌｫﾙﾀﾞ(異常)の初期値 {@value} */
	public static final String DEF_NG_DIR	= "bkup_ng" ;

	/** ベースパス */
	public final Path BASE_PATH	;

	/** サブパス(実際にｽｷｬﾝを行うﾄｯﾌﾟﾊﾟｽ) */
	public final Path SUB_PATH	;

	/** 処理ﾌｫﾙﾀﾞ(WORK)のパス */
	public final Path WORK_PATH	;
	/** 処理済ﾌｫﾙﾀﾞ(正常)のパス */
	public final Path OK_PATH ;
	/** 処理済ﾌｫﾙﾀﾞ(異常)のパス */
	public final Path NG_PATH	;

//	private final Set<String> omitScanDir ;
	private final Set<Path> omitScanDir ;				// 7.2.9.4 (2020/11/20) Path を直接管理します。

	/**
	 * フォルダを指定して、オブジェクトを作成します。
	 *
	 * DIR_BASE,DIR_SUB,DIR_WORK,DIR_BKUP_OK,DIR_BKUP_NG の順にセットされます。
	 * 指定されていない(null)の場合は、初期設定を使用します。
	 *
	 * @og.rev 7.2.1.0 (2020/03/13) FileUtil#isAbsolute(String)を利用します。
	 * @og.rev 7.2.2.0 (2020/03/27) javadoc用のjdk8に対応します。
	 * @og.rev 7.2.9.4 (2020/11/20) スキャン除外判定は、Path を直接管理します。
	 *
	 * @param	dirs	フォルダ配列
	 */
	public BasePath( final String... dirs ) {
		if( dirs == null || dirs.length != 5 ) {
			// MSG0005 = ﾌｫﾙﾀﾞのﾌｧｲﾙ読み込み時にｴﾗｰが発生しました。file=[{0}]
			throw MsgUtil.throwException( "MSG0005" , "DIR_BASE,DIR_SUB,DIR_WORK,DIR_BKUP_OK,DIR_BKUP_NG" );
		}

		final String baseDir = dirs[0];

		BASE_PATH	= FileUtil.writePath( baseDir );						// ベースフォルダのチェック

		SUB_PATH	= dirs[1] == null || dirs[1].isEmpty()
							? BASE_PATH
							: FileUtil.writePath( baseDir , dirs[1] );

		WORK_PATH	= makePath( baseDir,dirs[2],DEF_WORK_DIR );
		OK_PATH		= makePath( baseDir,dirs[3],DEF_OK_DIR );
		NG_PATH		= makePath( baseDir,dirs[4],DEF_NG_DIR );

//		omitScanDir = new HashSet<>();
//		omitScanDir.add( WORK_PATH.getFileName().toString()  );				// ミス？ OK_PATH → WORK_PATH に変更
//		omitScanDir.add( OK_PATH.getFileName().toString()  );
//		omitScanDir.add( NG_PATH.getFileName().toString()  );

		omitScanDir = Set.of( WORK_PATH , OK_PATH , NG_PATH );				// 7.2.9.4 (2020/11/20)

		LOGGER.debug( () ->	"③ [BasePath]"						+ CR
						+	"\t  BASE_PATH="	+ BASE_PATH		+ CR
						+	"\t  SUB_PATH ="	+ SUB_PATH		+ CR
						+	"\t  WORK_PATH="	+ WORK_PATH		+ CR
						+	"\t  OK_PATH  ="	+ OK_PATH		+ CR
						+	"\t  NG_PATH  ="	+ NG_PATH
		);
	}

	/**
	 * WORK,OK,NG の各Pathパスを生成するロジックのまとめメソッド。
	 *
	 * パスの元となるフォルダ(dir)が、nullかゼロ文字列の場合は、ベース＋初期DIRを返し
	 * dirが、絶対パスの場合は、そのパスを、そうでない場合は、baseDir + dir でパスを生成します。
	 * ここで生成するパスは、FileUtil#writePath で指定し、フォルダ作成も同時に行います。
	 *
	 * @og.rev 7.2.1.0 (2020/03/13) FileUtil#isAbsolute(String)を利用します。
	 *
	 * @param	baseDir	基準フォルダ
	 * @param	dir		パスの元となるフォルダ
	 * @param	def		パスが未設定時のフォルダ
	 * @return	path	生成したPathオブジェクト
	 */
	private Path makePath( final String baseDir , final String dir , final String def ) {
		return dir == null || dir.isEmpty()
					? FileUtil.writePath( baseDir , def )			// ベース＋初期DIR
					: FileUtil.isAbsolute( dir )
						? FileUtil.writePath( dir )					// 絶対パス
						: FileUtil.writePath( baseDir , dir );		// ベース＋相対パス
	}

	/**
	 * WORK,OK,NG,LOG をスキャン対象外にするための簡易判定メソッド。
	 *
	 * 指定のパスが、WORK_PATH,OK_PATH,NG_PATH,LOG_PATH の場合は、false を返します。
	 *
	 * @og.rev 7.2.9.4 (2020/11/20) スキャン除外判定は、Path を直接管理します。
	 *
	 * @param	path	判定するパス
	 * @return	スキャン対象なら、true
	 */
	public boolean isScanPath( final Path path ) {
		return !omitScanDir.contains( path );

//		final String fname = path.getFileName().toString();
//		return !omitScanDir.contains( fname );
	}
}
