/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.io.InputStream;
import java.io.FileInputStream;
import java.io.BufferedReader;													// 6.2.2.0 (2015/03/27)
import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;														// 6.2.2.0 (2015/03/27)
import java.nio.charset.Charset;												// 6.2.2.0 (2015/03/27)

import java.util.zip.ZipException;												// 8.5.0.0 (2023/04/21)
import java.util.Set;															// 6.0.2.3 (2014/10/10)
import java.util.TreeSet;														// 6.0.2.3 (2014/10/10)
import java.util.List;															// 6.4.6.0 (2016/05/27) poi-3.15
// import java.util.ArrayList;													// 8.0.1.0 (2021/10/29)

// import org.apache.xmlbeans.XmlException;										// 8.0.0.0 (2021/07/31) Delete
// import org.apache.poi.POITextExtractor;
import org.apache.poi.extractor.POITextExtractor;								// 7.0.0.0 (2018/10/01) poi-3.17.jar → poi-4.0.0.jar
// import org.apache.poi.extractor.ExtractorFactory;
// import org.apache.poi.ooxml.extractor.ExtractorFactory;						// 7.0.0.0 (2018/10/01) poi-ooxml-3.17.jar → poi-ooxml-4.0.0.jar
import org.apache.poi.ooxml.extractor.POIXMLExtractorFactory;					// 8.0.0.0 (2021/07/31) poi-ooxml-4.0.0.jar → poi-ooxml-5.0.0.jar
import org.apache.poi.hwpf.HWPFDocument;
import org.apache.poi.hwpf.usermodel.Range;
import org.apache.poi.hwpf.usermodel.Paragraph;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hslf.usermodel.HSLFTextParagraph;							// 6.4.6.0 (2016/05/27) poi-3.15
import org.apache.poi.hslf.usermodel.HSLFSlide;									// 6.4.6.0 (2016/05/27) poi-3.15
import org.apache.poi.hslf.usermodel.HSLFSlideShow;								// 6.4.6.0 (2016/05/27) poi-3.15

import org.apache.poi.xwpf.usermodel.XWPFDocument;								// 6.2.0.0 (2015/02/27)
import org.apache.poi.xwpf.usermodel.XWPFParagraph;								// 6.2.0.0 (2015/02/27)
import org.apache.poi.xwpf.model.XWPFCommentsDecorator;							// 8.5.0.0 (2023/04/21) Wordのﾃｷｽﾄ抜出を改造
import org.apache.poi.xwpf.usermodel.IBodyElement;								// 8.5.0.0 (2023/04/21)
import org.apache.poi.xwpf.usermodel.XWPFTable;									// 8.5.0.0 (2023/04/21)
import org.apache.poi.xwpf.usermodel.XWPFTableCell;								// 8.5.0.0 (2023/04/21)
import org.apache.poi.xwpf.usermodel.XWPFTableRow;								// 8.5.0.0 (2023/04/21)
import org.apache.poi.xwpf.usermodel.XWPFSDT;									// 8.5.0.0 (2023/04/21)

import org.openxmlformats.schemas.drawingml.x2006.main.CTHyperlink;				// 8.1.0.1 (2022/01/07)
import org.apache.poi.openxml4j.opc.PackageRelationshipTypes;					// 8.1.0.1 (2022/01/07)

import org.apache.poi.xslf.usermodel.XMLSlideShow;								// 6.2.0.0 (2015/02/27)
// import org.apache.poi.xslf.extractor.XSLFPowerPointExtractor;				// 7.0.0.0 (2018/10/01) POI4.0.0 deprecation
// import org.apache.poi.sl.extractor.SlideShowExtractor;						// 7.0.0.0 (2018/10/01) POI4.0.0 XSLFPowerPointExtractor → SlideShowExtractor 8.5.0.0 (2023/04/21) Delete
import org.apache.poi.xslf.usermodel.XSLFSlide;									// 8.5.0.0 (2023/04/21)
import org.apache.poi.xslf.usermodel.XSLFShape;									// 7.0.0.0 (2018/10/01) POI4.0.0 XSLFPowerPointExtractor → SlideShowExtractor
import org.apache.poi.xslf.usermodel.XSLFTextParagraph;							// 7.0.0.0 (2018/10/01) POI4.0.0 XSLFPowerPointExtractor → SlideShowExtractor
import org.apache.poi.sl.usermodel.TableCell;									// 8.5.0.0 (2023/04/21)
import org.apache.poi.sl.usermodel.ShapeContainer;								// 8.5.0.0 (2023/04/21)
import org.apache.poi.sl.usermodel.TableShape;									// 8.5.0.0 (2023/04/21)
import org.apache.poi.sl.usermodel.Shape;										// 8.5.0.0 (2023/04/21)
import org.apache.poi.sl.usermodel.TextShape;									// 8.5.0.0 (2023/04/21)
import org.apache.poi.sl.usermodel.TextRun;										// 8.5.0.0 (2023/04/21)

import org.apache.pdfbox.pdmodel.PDDocument;									// 8.5.0.0 (2023/05/12) PDFﾌｧｲﾙ処理
// import org.apache.pdfbox.Loader;												// 8.5.0.0 (2023/05/12) 3.0.0
import org.apache.pdfbox.text.PDFTextStripper;									// 8.5.0.0 (2023/05/12)

// 8.5.0.0 (2023/05/12) pdfboxの警告抑止 … ただし、Log4J に切り替えると、制御できない。
import java.util.logging.Logger;												// 8.5.0.0 (2023/05/12) pdfboxの警告抑止
import java.util.logging.Level;													// 8.5.0.0 (2023/05/12)

import org.apache.poi.xssf.usermodel.XSSFSimpleShape;							// 8.1.0.1 (2022/01/07) ﾃｷｽﾄ変換処理

// import org.apache.poi.openxml4j.exceptions.InvalidFormatException;			// 8.0.0.0 (2021/07/31) Delete
// import org.apache.poi.openxml4j.exceptions.OpenXML4JException ;				// 6.1.0.0 (2014/12/26) findBugs 8.0.0.0 (2021/07/31) Delete
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CreationHelper;								// 8.1.2.3 (2022/05/20) 復活
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.FormulaEvaluator;							// 8.1.2.3 (2022/05/20) 復活
import org.apache.poi.ss.usermodel.CellValue;									// 8.1.2.3 (2022/05/20)
import org.apache.poi.ss.usermodel.Name;										// 6.0.2.3 (2014/10/10)
import org.apache.poi.ss.usermodel.CellType;									// 6.5.0.0 (2016/09/30) poi-3.15
import org.apache.poi.ss.util.SheetUtil;

import org.opengion.fukurou.system.OgRuntimeException ;							// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.FileInfo;										// 6.2.3.0 (2015/05/01)
// import org.opengion.fukurou.system.ThrowUtil;								// 6.4.2.0 (2016/01/29) 8.5.0.0 (2023/04/21) Delete
import org.opengion.fukurou.system.Closer;										// 6.2.0.0 (2015/02/27)
import static org.opengion.fukurou.system.HybsConst.CR;							// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;				// 6.4.2.1 (2016/02/05) refactoring
// import org.apache.poi.sl.usermodel.Slide;									// 8.5.0.0 (2023/04/21) Delete

/**
 * POI による、Excel/Word/PoworPoint等に対する、ﾕｰﾃｨﾘﾃｨｸﾗｽです。
 *
 * 基本的には、ﾈｲﾃｨﾌﾞﾌｧｲﾙを読み取り、ﾃｷｽﾄを取得する機能が主です。
 * Excel、Word、PowerPoint、Visio、Publisher からのﾃｷｽﾄ取得が可能です。
 *
 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
 * @og.rev 6.2.0.0 (2015/02/27) ﾊﾟｯｹｰｼﾞ変更(util → model)
 * @og.group その他
 *
 * @version	6.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK7.0,
 */
public final class POIUtil {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.5.0.0 (2023/05/12)" ;

	// 6.2.3.0 (2015/05/01)
	/** POI対象ｻﾌｨｯｸｽ {@value} */
	public static final String POI_SUFIX = "ppt,pptx,doc,docx,xls,xlsx,xlsm" ;

	// 8.5.0.0 (2023/04/21)
	/** ﾃｷｽﾄ対象ｻﾌｨｯｸｽ {@value} */
	public static final String TXT_SUFIX = "txt,csv,jsp,java,html,xml,css,js,json,py" ;

	/**
	 * すべてが staticﾒｿｯﾄﾞなので、ｺﾝｽﾄﾗｸﾀを呼び出さなくしておきます。
	 *
	 */
	private POIUtil() {}

	/**
	 * 引数ﾌｧｲﾙが、POI関連の拡張子ﾌｧｲﾙかどうかを判定します。
	 *
	 * Excel、Word、PowerPoint、Visio、Publisher からのﾃｷｽﾄ取得が可能です。
	 * ﾌｧｲﾙの拡張子が、{@value #POI_SUFIX} の場合、true を返します。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) POI関連の拡張子ﾌｧｲﾙかどうかを判定
	 *
	 * @param	file	判定するﾌｧｲﾙ
	 * @return	POI関連の拡張子の場合、true
	 */
	public static boolean isPOI( final File file ) {
		return POI_SUFIX.contains( FileInfo.getSUFIX( file ) );
	}

	/**
	 * 引数ﾌｧｲﾙが、処理対象となるﾃｷｽﾄ関連の拡張子ﾌｧｲﾙかどうかを判定します。
	 *
	 * ﾌｧｲﾙの拡張子が、{@value #TXT_SUFIX} の場合、true を返します。
	 *
	 * @og.rev 8.5.0.0 (2023/04/21) 新規追加
	 *
	 * @param	file	判定するﾌｧｲﾙ
	 * @return	ﾃｷｽﾄ関連の拡張子の場合、true
	 */
	public static boolean isText( final File file ) {
		return TXT_SUFIX.contains( FileInfo.getSUFIX( file ) );
	}

	/**
	 * 引数ﾌｧｲﾙが、textReader の読み取り対象となる拡張子ﾌｧｲﾙかどうかを判定します。
	 *
	 * ﾌｧｲﾙの拡張子が {@value #POI_SUFIX}、{@value #TXT_SUFIX}、pdf の場合、true を返します。
	 *
	 * @og.rev 8.5.0.0 (2023/05/12) 新規追加
	 *
	 * @param	file	判定するﾌｧｲﾙ
	 * @return	ﾃｷｽﾄ関連の拡張子の場合、true
	 */
	public static boolean isReadText( final File file ) {
		final String sufix = FileInfo.getSUFIX( file );		// sufix は小文字変換されてくる。

		return POI_SUFIX.contains( sufix )
			|| TXT_SUFIX.contains( sufix )
			|| "pdf".equals( sufix ) ;
	}

	/**
	 * 引数ﾌｧｲﾙを、POITextExtractor を使用してﾃｷｽﾄ化します。
	 *
	 * Excel、Word、PowerPoint、Visio、Publisher からのﾃｷｽﾄ取得が可能です。
	 * 拡張子から、ﾌｧｲﾙの種類を自動判別します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) getText → extractor に変更
	 * @og.rev 8.0.0.0 (2021/07/31) ExtractorFactory → POIXMLExtractorFactory に変更
	 *
	 * @param	file	入力ﾌｧｲﾙ名
	 * @return	変換後のﾃｷｽﾄ
	 * @og.rtnNotNull
	 */
	public static String extractor( final File file ) {
	//	InputStream fis = null;
		POITextExtractor extractor = null;
		try {
	//		fis = new BufferedInputStream( new FileInputStream( file ) );
	//		extractor = ExtractorFactory.createExtractor( fis );
	//		extractor = ExtractorFactory.createExtractor( file );
			extractor = new POIXMLExtractorFactory().create( file , null );		// 8.0.0.0 (2021/07/31) poi-ooxml-4.0.0.jar → poi-ooxml-5.0.0.jar
			return extractor.getText();
		}
		catch( final FileNotFoundException ex ) {
			final String errMsg = "ﾌｧｲﾙが存在しません[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ処理ｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		// 8.0.0.0 (2021/07/31) poi-ooxml-4.0.0.jar → poi-ooxml-5.0.0.jar
//		catch( final InvalidFormatException ex ) {
//			final String errMsg = "ﾌｧｲﾙﾌｫｰﾏｯﾄｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
//			throw new OgRuntimeException( errMsg,ex );
//		}
//		catch( final OpenXML4JException ex ) {
//			final String errMsg = "ODF-XML処理ｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
//			throw new OgRuntimeException( errMsg,ex );
//		}
//		catch( final XmlException ex ) {
//			final String errMsg = "XML処理ｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
//			throw new OgRuntimeException( errMsg,ex );
//		}
		finally {
			Closer.ioClose( extractor );
	//		Closer.ioClose( fis );
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(Text)を、ﾃｷｽﾄ化します。
	 *
	 * ここでは、ﾌｧｲﾙとｴﾝｺｰﾄﾞを指定して、ﾌｧｲﾙのﾃｷｽﾄ全てを読み取ります。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) 引数ﾌｧｲﾙ(Text)を、ﾃｷｽﾄ化。
	 * @og.rev 6.2.3.0 (2015/05/01) textReader → extractor に変更
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	encode	ｴﾝｺｰﾄﾞ名
	 * @return	ﾌｧｲﾙのﾃｷｽﾄ
	 */
	public static String extractor( final File file , final String encode ) {
		try {
			// 指定のﾌｧｲﾙをﾊﾞｲﾄ列として読み込む
			final byte[] bytes = Files.readAllBytes( file.toPath() );
			// 読み込んだﾊﾞｲﾄ列を ｴﾝｺｰﾄﾞして文字列にする
			return new String( bytes, encode );
		}
	//	catch( final UnsupportedEncodingException ex ) {
	//		final String errMsg = "ｴﾝｺｰﾄﾞが不正です[" + file + "] , ENCODE=[" + encode + "]"  ;
	//		throw new OgRuntimeException( errMsg,ex );
	//	}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "] , ENCODE=[" + encode + "]"  ;
			throw new OgRuntimeException( errMsg,ex );
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(Text)を、ﾃｷｽﾄ化します。
	 *
	 * ここでは、ﾌｧｲﾙとｴﾝｺｰﾄﾞを指定して、ﾌｧｲﾙのﾃｷｽﾄ全てを読み取ります。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) 引数ﾌｧｲﾙ(Text)を、ﾃｷｽﾄ化。
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 * @param	encode	ｴﾝｺｰﾄﾞ名
	 */
	public static void textReader( final File file , final TextConverter<String,String> conv , final String encode ) {
		BufferedReader reader = null ;

		int rowNo = 0;		// 6.2.0.0 (2015/02/27) 行番号
		try {
			reader = Files.newBufferedReader( file.toPath() , Charset.forName( encode ) );

			String line ;
			while((line = reader.readLine()) != null) {
				conv.change( line,String.valueOf( rowNo++ ) );
			}
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "] , ENCODE=[" + encode + "] , ROW=[" + rowNo + "]"  ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(Word,PoworPoint,Excel)を、TableModelHelper を使用してﾃｷｽﾄ化します。
	 *
	 * ここでは、ﾌｧｲﾙ名の拡張子で、処理するﾒｿｯﾄﾞを選別します。
	 * 拡張子が、対象かどうかは、#isPOI( File ) ﾒｿｯﾄﾞで判定できます。
	 *
	 * TableModelHelper によるｲﾍﾞﾝﾄ処理できますが、TEXTというｶﾗﾑ名を持つ
	 * 表形式ｵﾌﾞｼﾞｪｸﾄの形で処理されます。
	 * また、内部的に、先頭に、# がある場合や、行ﾃﾞｰﾀが存在しない場合は、
	 * ｽｷｯﾌﾟされます。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) 新規作成
	 * @og.rev 6.2.5.0 (2015/06/05) xls,xlsxは、それぞれ excelReader1,excelReader2 で処理します。
	 * @og.rev 8.5.0.0 (2023/04/21) txt,csv,jsp,java,xml,css,js は、UTF-8 固定で、textReader を呼び出す。
	 * @og.rev 8.5.0.0 (2023/05/12) pdfReader1 追加
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 */
	public static void textReader( final File file , final TextConverter<String,String> conv ) {
		final String sufix = FileInfo.getSUFIX( file );				// sufix は小文字変換されてくる。

		if( "doc".equalsIgnoreCase( sufix ) ) {
			wordReader1( file,conv );
		}
		else if( "docx".equalsIgnoreCase( sufix ) ) {
			wordReader2( file,conv );
		}
		else if( "ppt".equalsIgnoreCase( sufix ) ) {
			pptReader1( file,conv );
		}
		else if( "pptx".equalsIgnoreCase( sufix ) ) {
			pptReader2( file,conv );
		}
		else if( "xls".equalsIgnoreCase( sufix ) ) {
			excelReader1( file,conv );								// 6.2.5.0 (2015/06/05)
		}
		else if( "xlsx".equalsIgnoreCase( sufix ) || "xlsm".equalsIgnoreCase( sufix ) ) {
			excelReader2( file,conv );								// 6.2.5.0 (2015/06/05)
		}
		else if( "pdf".equalsIgnoreCase( sufix ) ) {
			pdfReader1( file,conv );								// 8.5.0.0 (2023/05/12)
		}
		// 8.5.0.0 (2023/04/21) txt,csv,jsp,java,xml,css,js は、UTF-8 固定で、textReader を呼び出す。
		else if( TXT_SUFIX.contains( sufix ) ) {
			try {
				textReader( file,conv,"UTF-8" );
			}
			catch( final OgRuntimeException ex ) {			// ほとんどのｹｰｽでencode違い
				conv.change( ex.getMessage() , file.getAbsolutePath() );

				textReader( file,conv,"Windows-31J" );		// Windows-31J で読み直し。
			}
		}
		else {
//			final String errMsg = "拡張子は、" +  POI_SUFIX + " にしてください。[" + file + "]" ;
//			throw new OgRuntimeException( errMsg );
			try {
				final String filename = file.getCanonicalPath();
				conv.change( "ﾃｷｽﾄ化対象外です" , filename );		// text,cmnt の順
			}
			catch( final IOException ex ) {
				final String errMsg = "正規のパス名取得ｴﾗｰ[" + file + "]" ;
				throw new OgRuntimeException( errMsg,ex );
			}
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(Word)を、HWPFDocument を使用してﾃｷｽﾄ化します。
	 *
	 * 拡張子(.doc)のﾌｧｲﾙを処理します。
	 * TableModelHelper によるｲﾍﾞﾝﾄ処理できますが、TEXTというｶﾗﾑ名を持つ
	 * 表形式ｵﾌﾞｼﾞｪｸﾄの形で処理されます。
	 * また、内部的に、先頭に、# がある場合や、行ﾃﾞｰﾀが存在しない場合は、
	 * ｽｷｯﾌﾟされます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) POIEvent → TableModelHelper に変更
	 * @og.rev 6.2.4.2 (2015/05/29) 改行以外に、｢。｣で分割します。
	 *
	 * @param	file	入力ﾌｧｲﾙ名
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 */
	private static void wordReader1( final File file , final TextConverter<String,String> conv ) {
		InputStream fis  = null;

		try {
			// 6.2.0.0 (2015/02/27) TableModelHelper 変更に伴う修正

			fis = new BufferedInputStream( new FileInputStream( file ) );		// 6.2.0.0 (2015/02/27)
			final HWPFDocument doc = new HWPFDocument( fis );

	//		int rowNo = 0;		// 6.2.0.0 (2015/02/27) 行番号

	//		// WordExtractor を使ったｻﾝﾌﾟﾙ
	//		WordExtractor we = new WordExtractor( doc );
	//		for( String txt : we.getParagraphText() ) {
	//			String text = WordExtractor.stripFields( txt )
	//							.replaceAll( "\\x13[^\\x01]+\\x01\\x14" , "" )
	//							.replaceAll( "\\x0b" , "\n" ).trim();
	//			helper.value( text.trim(),rowNo++,0 );				// 6.2.0.0 (2015/02/27) ｲﾍﾞﾝﾄ変更
	//		}

			// Range,Paragraph を使ったｻﾝﾌﾟﾙ
			final Range rng = doc.getRange();
			for( int pno=0; pno<rng.numParagraphs(); pno++ ) {
				final Paragraph para = rng.getParagraph(pno);
				final String text = Range.stripFields( para.text() )
								.replaceAll( "\\x13[^\\x01]+\\x01\\x14" , "" )
								.replaceAll( "\\x0b" , "\n" ).trim();
	//			conv.change( text, String.valueOf( rowNo++ ) );
				if( text.length() > 0 ) {								// 8.5.0.0 (2023/04/21) 存在する場合のみ抜き出す
					conv.change( text, String.valueOf( pno ) );			// 行番号代わりの数値は飛び番となる
				}
			}

			// Range,Paragraph,CharacterRun を使ったｻﾝﾌﾟﾙ(変な個所で文字が分断される)
	//		final Range rng = doc.getRange();
	//		for( int pno = 0; pno < rng.numParagraphs(); pno++ ) {
	//			final Paragraph para = rng.getParagraph(pno);
	//			for( int cno = 0; cno < para.numCharacterRuns(); cno++ ) {
	//				final CharacterRun crun = para.getCharacterRun(cno);
	//				String text = Range.stripFields( crun.text() )
	//								.replaceAll( "\\x13[^\\x01]+\\x01\\x14" , "" )
	//								.replaceAll( "\\x0b" , "\n" ).trim();
	//				helper.value( text,rowNo++,0 );				// 6.2.0.0 (2015/02/27) ｲﾍﾞﾝﾄ変更
	//			}
	//		}
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(Word)を、XWPFDocument を使用してﾃｷｽﾄ化します。
	 *
	 * 拡張子(.docx)のﾌｧｲﾙを処理します。
	 * TableModelHelper によるｲﾍﾞﾝﾄ処理できますが、TEXTというｶﾗﾑ名を持つ
	 * 表形式ｵﾌﾞｼﾞｪｸﾄの形で処理されます。
	 * また、内部的に、先頭に、# がある場合や、行ﾃﾞｰﾀが存在しない場合は、
	 * ｽｷｯﾌﾟされます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) POIEvent → TableModelHelper に変更
	 * @og.rev 6.2.4.2 (2015/05/29) 改行以外に、｢。｣で分割します。
	 * @og.rev 8.5.0.0 (2023/04/21) XWPFWordExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 */
	private static void wordReader2( final File file , final TextConverter<String,String> conv ) {
		InputStream fis  = null;

		try {
			// 6.2.0.0 (2015/02/27) TableModelHelper 変更に伴う修正

			fis = new BufferedInputStream( new FileInputStream( file ) );		// 6.2.0.0 (2015/02/27)
			final XWPFDocument doc = new XWPFDocument( fis );

			int rowNo = 0;		// 6.2.0.0 (2015/02/27) 行番号

//			for( final XWPFParagraph para : doc.getParagraphs() ) {				// 8.5.0.0 (2023/04/21) 削除
//				final String text = para.getParagraphText().trim();
//				conv.change( text, String.valueOf( rowNo++ ) );
//			}

			// 8.5.0.0 (2023/04/21) XWPFWordExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
			// \poi-ooxml\src\main\java\org\apache\poi\xwpf\extractor\XWPFWordExtractor.java
			for( final IBodyElement ibody : doc.getBodyElements() ) {
				if (ibody instanceof XWPFParagraph) {
					appendParagraphText(conv,rowNo, (XWPFParagraph) ibody);
				} else if (ibody instanceof XWPFTable) {
					appendTableText(conv,rowNo, (XWPFTable) ibody);
				} else if (ibody instanceof XWPFSDT) {
					final String text = ((XWPFSDT) ibody).getContent().getText().trim();
					if( text.length() > 0 ) {
						conv.change( text, "XWPFSDT " + rowNo );
					}
				}
				rowNo++ ;
			}
		}
		catch( final ZipException ex ) {
			final String errMsg = "ﾌｧｲﾙがすでにｵｰﾌﾟﾝされています[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * wordReader2 から派生した、部分処理
	 *
	 * 引数ﾌｧｲﾙ(Word)を、XWPFDocument を使用してﾃｷｽﾄ化するにあたり
	 * XWPFParagraph のﾃｷｽﾄ化を行います。
	 *
	 * @og.rev 8.5.0.0 (2023/04/21) XWPFWordExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	conv		ｲﾍﾞﾝﾄ処理させるI/F
	 * @param	rowNo		検索における連番
	 * @param	paragraph	XWPFParagraphｵﾌﾞｼﾞｪｸﾄ
	 */
	private static void appendParagraphText(final TextConverter<String,String> conv, final int rowNo, final XWPFParagraph paragraph) {
//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

//		for (final IRunElement run : paragraph.getIRuns()) {
//			if (run instanceof XWPFSDT) {
//				buf.append( ((XWPFSDT)run).getContent().getText() );
//			}
//			else if (run instanceof XWPFRun) {
//				buf.append( ((XWPFRun)run).text() );
//			}
//			else {
//				buf.append( String.valueOf( run ) );
//			}
//		}
//		final String text = buf.toString().trim();		// Paragraph は、1行にまとめる

		final String text = paragraph.getText().trim();
		if( text.length() > 0 ) {
			conv.change( text, "Paragraph " + rowNo );
		}

		// FootnoteText
		final String note = paragraph.getFootnoteText().trim();
		if( note.length() > 0 ) {
			conv.change( note, "Footnote " + rowNo );
		}

		// Add comments
		final XWPFCommentsDecorator decorator = new XWPFCommentsDecorator(paragraph, null);
		final String cmnt = decorator.getCommentText().trim();
		if( cmnt.length() > 0 ) {
			conv.change( cmnt, "Comment " + rowNo );
		}
	}

	/**
	 * wordReader2 から派生した、部分処理
	 *
	 * 引数ﾌｧｲﾙ(Word)を、XWPFDocument を使用してﾃｷｽﾄ化するにあたり
	 * XWPFTable のﾃｷｽﾄ化を行います。
	 *
	 * @og.rev 8.5.0.0 (2023/04/21) XWPFWordExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 * @param	rowNo	検索における連番
	 * @param	table	XWPFTableｵﾌﾞｼﾞｪｸﾄ
	 */
	private static void appendTableText(final TextConverter<String,String> conv, final int rowNo, final XWPFTable table) {
		int subNo = 0;
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		//this works recursively to pull embedded tables from tables
		for (final XWPFTableRow row : table.getRows()) {
			// ﾃｰﾌﾞﾙのｾﾙは、ﾀﾌﾞで結合する。
			for( final XWPFTableCell cell : row.getTableCells() ) {
				buf.append( cell.getText().trim() ).append( '\t' );
			}

//			final List<ICell> cells = row.getTableICells();
//			// ﾃｰﾌﾞﾙのｾﾙは、ﾀﾌﾞで結合する。
//			for (int i = 0; i < cells.size(); i++) {
//				final ICell cell = cells.get(i);
//				if (cell instanceof XWPFTableCell) {
//					final String text = ((XWPFTableCell) cell).getTextRecursively();
//					buf.append( text.trim() ).append( '\t' );
//				} else if (cell instanceof XWPFSDTCell) {
//					final String text = ((XWPFSDTCell) cell).getContent().getText();
//					buf.append( text.trim() ).append( '\t' );
//				}
//			}

			final String text = buf.toString().trim();	// 先に trim することで、空行を除く
			if( text.length() > 0 ) {
				conv.change( text , "TableRow " + rowNo + ":" + subNo );
			}
			buf.setLength(0);		// Clearの事
			subNo++ ;
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(PoworPoint)を、HSLFSlideShow を使用してﾃｷｽﾄ化します。
	 *
	 * 拡張子(.ppt)のﾌｧｲﾙを処理します。
	 * TableModelHelper によるｲﾍﾞﾝﾄ処理できますが、TEXTというｶﾗﾑ名を持つ
	 * 表形式ｵﾌﾞｼﾞｪｸﾄの形で処理されます。
	 * また、内部的に、先頭に、# がある場合や、行ﾃﾞｰﾀが存在しない場合は、
	 * ｽｷｯﾌﾟされます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) POIEvent → TableModelHelper に変更
	 * @og.rev 6.4.6.0 (2016/05/27) poi-3.15 準備
	 * @og.rev 8.5.0.0 (2023/04/21) conv.change の cmnt 修正
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 */
	private static void pptReader1( final File file , final TextConverter<String,String> conv ) {
		InputStream fis  = null;

		try {
			// 6.2.0.0 (2015/02/27) TableModelHelper 変更に伴う修正

			fis = new BufferedInputStream( new FileInputStream( file ) );		// 6.2.0.0 (2015/02/27)

	//		6.4.6.0 (2016/05/27) poi-3.15
			final HSLFSlideShow ss = new HSLFSlideShow( fis );
			final List<HSLFSlide> slides = ss.getSlides();						// 6.4.6.0 (2016/05/27) poi-3.15
			int rowNo = 0;		// 6.2.0.0 (2015/02/27) 行番号
			for( final HSLFSlide  slide : slides ) {							// 6.4.6.0 (2016/05/27) poi-3.15
				int subNo = 0;	// 8.5.0.0 (2023/04/21)
				for( final List<HSLFTextParagraph> txtList : slide.getTextParagraphs() ) {	// 6.4.6.0 (2016/05/27) poi-3.15
					final String text = HSLFTextParagraph.getText( txtList ).trim();
					if( text.length() > 0 ) {
//						conv.change( text, String.valueOf( rowNo++ ) );
						conv.change( text, "Slide " + rowNo + ":" + subNo );	// 8.5.0.0 (2023/04/21) cmnt 修正
					}
					subNo++;	// 8.5.0.0 (2023/04/21)
				}
				rowNo++ ;		// 8.5.0.0 (2023/04/21)
			}

	//		6.4.6.0 (2016/05/27) poi-3.12
	//		final SlideShow ss = new SlideShow( new HSLFSlideShow( fis ) );
	//		final Slide[] slides = ss.getSlides();
	//		int rowNo = 0;		// 6.2.0.0 (2015/02/27) 行番号
	//		for( int sno=0; sno<slides.length; sno++ ) {
	//			final TextRun[] textRun = slides[sno].getTextRuns();
	//			for( int tno=0; tno<textRun.length; tno++ ) {
	//				final String text = textRun[tno].getText();
	//				// ﾃﾞｰﾀとして設定されているﾚｺｰﾄﾞのみｲﾍﾞﾝﾄを発生させる。
	//				if( text.length() > 0 ) {
	//					conv.change( text, String.valueOf( rowNo++ ) );
	//				}
	//			}
	//		}
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(PoworPoint)を、XMLSlideShow を使用してﾃｷｽﾄ化します。
	 *
	 * 拡張子(.pptx)のﾌｧｲﾙを処理します。
	 * TableModelHelper によるｲﾍﾞﾝﾄ処理できますが、TEXTというｶﾗﾑ名を持つ
	 * 表形式ｵﾌﾞｼﾞｪｸﾄの形で処理されます。
	 * また、内部的に、先頭に、# がある場合や、行ﾃﾞｰﾀが存在しない場合は、
	 * ｽｷｯﾌﾟされます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) POIEvent → TableModelHelper に変更
	 * @og.rev 6.2.4.2 (2015/05/29) 行単位に、取り込むようにします。
	 * @og.rev 7.0.0.0 (2018/10/01) 警告:[deprecation] org.apache.poi.xslf.extractorのXSLFPowerPointExtractorは推奨されません (POI4.0.0)
	 * @og.rev 8.5.0.0 (2023/04/21) SlideShowExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 */
	private static void pptReader2( final File file , final TextConverter<String,String> conv ) {
		InputStream fis  = null;

		try {
			// 6.2.0.0 (2015/02/27) TableModelEvent 変更に伴う修正

			fis = new BufferedInputStream( new FileInputStream( file ) );		// 6.2.0.0 (2015/02/27)
			final XMLSlideShow ss = new XMLSlideShow( fis );
	//		final SlideShowExtractor<XSLFShape,XSLFTextParagraph> ext = new SlideShowExtractor<>( ss );		// 7.0.0.0 (2018/10/01) poi-4.0.0 Deprecated.
	//		final String[] vals = ext.getText().split( "\\n" );		// 6.2.4.2 (2015/05/29) 行単位に、取り込むようにします。
	//		for( int row=0; row<vals.length; row++ ) {
	//			conv.change( vals[row], String.valueOf( row ) );
	//		}
			int rowNo = 0;
			for (final XSLFSlide slide : ss.getSlides()) {
				conv.change( slide.getSlideName(), "Slide " + rowNo );
				printShapeText( slide,conv,rowNo );
				rowNo ++ ;
			}
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * pptReader2 から派生した、部分処理
	 *
	 * 引数ﾌｧｲﾙ(PowerPoint)を、XSLFSlide を使用してﾃｷｽﾄ化します。
	 *
	 * @og.rev 8.5.0.0 (2023/04/21) SlideShowExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	container	XSLFShapeを含むShapeContainerｵﾌﾞｼﾞｪｸﾄ
	 * @param	conv		ｲﾍﾞﾝﾄ処理させるI/F
	 * @param	rowNo		検索における連番
	 */
	@SuppressWarnings("unchecked")
	private static void printShapeText(final ShapeContainer<XSLFShape,XSLFTextParagraph> container, final TextConverter<String,String> conv,final int rowNo ) {
		int subNo = 0;
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		for (final Shape<XSLFShape,XSLFTextParagraph> shape : container) {
			if (shape instanceof TextShape) {
				buf.setLength(0);
				printTextParagraphs(((TextShape<XSLFShape,XSLFTextParagraph>)shape).getTextParagraphs(),buf);
				final String text = buf.toString().trim();
				if( text.length() > 0 ) {
					conv.change( text, "Shape " + rowNo + ":" + subNo );
				}
				subNo++ ;
			} else if (shape instanceof TableShape) {
				printTableShape((TableShape<XSLFShape,XSLFTextParagraph>)shape,conv,rowNo);
			} else if (shape instanceof ShapeContainer) {
				printShapeText((ShapeContainer<XSLFShape,XSLFTextParagraph>)shape,conv,rowNo);
			}
		}
	}

	/**
	 * pptReader2 から派生した、部分処理
	 *
	 * 引数ﾌｧｲﾙ(PowerPoint)を、XSLFSlide を使用してﾃｷｽﾄ化します。
	 *
	 * @og.rev 8.5.0.0 (2023/04/21) SlideShowExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	shape	XSLFShapeを含むTableShapeｵﾌﾞｼﾞｪｸﾄ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 * @param	rowNo	検索における連番
	 */
	private static void printTableShape(final TableShape<XSLFShape,XSLFTextParagraph> shape, final TextConverter<String,String> conv,final int rowNo) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		final int nrows = shape.getNumberOfRows();
		final int ncols = shape.getNumberOfColumns();
		for (int row = 0; row < nrows; row++) {
			for (int col = 0; col < ncols; col++){
				final TableCell<XSLFShape,XSLFTextParagraph> cell = shape.getCell(row, col);
				//defensive null checks; don't know if they're necessary
				if (cell != null) {
					printTextParagraphs( cell.getTextParagraphs(),buf );
					buf.append('\t');
				}
			}
			final String text = buf.toString().trim();
			if( text.length() > 0 ) {
				conv.change( text, "TableRow " + rowNo + ":" + row );
			}
			buf.setLength(0);		// Clearの事
		}
	}

	/**
	 * pptReader2 から派生した、部分処理
	 *
	 * 引数ﾌｧｲﾙ(PowerPoint)を、XSLFSlide を使用してﾃｷｽﾄ化します。
	 *
	 * @og.rev 8.5.0.0 (2023/04/21) SlideShowExtractor を参考に、ﾃｷｽﾄ化を詳細化した。
	 *
	 * @param	paras	XSLFTextParagraphｵﾌﾞｼﾞｪｸﾄ のﾘｽﾄ
	 * @param	buf		文字列ﾊﾞｯﾌｧ
	 */
	private static void printTextParagraphs( final List<XSLFTextParagraph> paras ,final StringBuilder buf ) {
		for (final XSLFTextParagraph para : paras) {
			for (final TextRun run : para) {
				buf.append( run.getRawText().trim() );
			}
		}
	}

	/**
	 * 引数ﾌｧｲﾙ(Excel)を、ﾃｷｽﾄ化します。
	 *
	 * TableModelHelper を与えることで、EXCELﾃﾞｰﾀをﾃｷｽﾄ化できます。
	 * ここでは、HSSF(.xls)形式を処理します。
	 * ｼｰﾄ名、ｾﾙ、ﾃｷｽﾄｵﾌﾞｼﾞｪｸﾄをﾃｷｽﾄ化します。
	 *
	 * @og.rev 6.2.5.0 (2015/06/05) 新規作成
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 * @see		org.opengion.fukurou.model.ExcelModel
	 */
	public static void excelReader1( final File file , final TextConverter<String,String> conv ) {
		excelReader2( file , conv );
	}

	/**
	 * 引数ﾌｧｲﾙ(Excel)を、ﾃｷｽﾄ化します。
	 *
	 * TableModelHelper を与えることで、EXCELﾃﾞｰﾀをﾃｷｽﾄ化できます。
	 * ここでは、ExcelModelを使用して、(.xlsx , .xlsm)形式を処理します。
	 * ｼｰﾄ名、ｾﾙ、ﾃｷｽﾄｵﾌﾞｼﾞｪｸﾄをﾃｷｽﾄ化します。
	 *
	 * @og.rev 6.2.5.0 (2015/06/05) 新規作成
	 * @og.rev 6.3.1.0 (2015/06/28) TextConverterに、引数(cmnt)を追加
	 * @og.rev 6.3.9.0 (2015/11/06) Java 8 ﾗﾑﾀﾞ式に変更
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 * @see		org.opengion.fukurou.model.ExcelModel
	 */
	public static void excelReader2( final File file , final TextConverter<String,String> conv ) {
		final ExcelModel excel = new ExcelModel( file, true );

		// 6.3.9.0 (2015/11/06) Java 8 ﾗﾑﾀﾞ式に変更
		// textConverter を使いますが、ﾃｷｽﾄを読み込むだけで、変換しません。
		excel.textConverter(
			( val,cmnt ) -> {
				conv.change( val,cmnt );	// 変換したくないので、引数の TextConverter を直接渡せない。
				return null;				// nullを返せば、変換しません。
			}
		);
	}

	/**
	 * 引数ﾌｧｲﾙ(PDF)を、ﾃｷｽﾄ化します。
	 *
	 * 引数ﾌｧｲﾙ(PDF)を、pdfbox を使用してﾃｷｽﾄ化します。
	 *
	 * @og.rev 8.5.0.0 (2023/05/12) pdfReader1 追加
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @param	conv	ｲﾍﾞﾝﾄ処理させるI/F
	 */
	public static void pdfReader1( final File file , final TextConverter<String,String> conv ) {
		Logger.getLogger("org.apache.fontbox").setLevel(Level.OFF);		// 警告抑止

//		// pdfbox-3.0.0-alpha3.jar
//		try( PDDocument document = Loader.loadPDF( file )) {
//			final PDFTextStripper stripper = new PDFTextStripper();
//			for( final PDPage page : document.getPages() ) {
//				stripper.processPage( page );
//				final String text = stripper.getText(document);
//				final int no = stripper.getStartPage();
//				conv.change( text,String.valueOf( no ) );
//			}
//		}
//		catch( final IOException ex ) {
//			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
//			throw new OgRuntimeException( errMsg,ex );
//		}

		// pdfbox-2.0.28.jar
		try( final PDDocument document = PDDocument.load(file) ) {
			final int psize  = document.getNumberOfPages();			// 総ページ数
			final PDFTextStripper stripper = new PDFTextStripper();
			final String sep = stripper.getLineSeparator();
			for( int pno=1; pno<=psize; pno++ ) {
				stripper.setStartPage(pno);
				stripper.setEndPage(pno);
				int   lno = 0;
				final String text = stripper.getText(document);
				for( final String line : text.split( sep ) ) {		// PDFTextStripper を使うと、ﾃｷｽﾄ行のみ抜き出すので
		//			if( line.length() > 0 ) {						// 行番号は取れない…感じ。
						final String cmnt = "Page" + pno + " : " + lno ;
						conv.change( line,cmnt );
		//			}
					lno++ ;
				}
			}
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
	}

	/**
	 * Excelの行列記号を、行番号と列番号に分解します。
	 *
	 * Excelの行列記号とは、A1 , B5 , AA23 などの形式を指します。
	 * これを、行番号と列番号に分解します。例えば、A1→0行0列、B5→4行1列、AA23→22行26列 となります。
	 * 分解した結果は、内部変数の、rowNo と colNo にｾｯﾄされます。
	 * これらは、0 から始まる int型の数字で表します。
	 *
	 *   ①行-列形式
	 *     行列は、EXCELｵﾌﾞｼﾞｪｸﾄに準拠するため、０から始まる整数です。
	 *     0-0 ⇒ A1 , 1-0 ⇒ A2 , 0-1 ⇒ B1 になります。
	 *   ②EXCEL表記
	 *     EXCEL表記に準拠した、A1,A2,B1 の記述も処理できるように対応します。
	 *     なお、A1,A2,B1 の記述は、必ず、英字1文字＋数字 にしてください。(A～Zまで)
	 *   ③EXCELｼｰﾄ名をｷｰに割り当てるために、"SHEET" という記号に対応します。
	 *     rowNo = -1 をｾｯﾄします。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 * @og.rev 6.2.6.0 (2015/06/19) 行-列形式と、SHEET文字列判定を採用。
	 *
	 * @param	kigo	Excelの行列記号( A1 , B5 , AA23 など )
	 * @return	行と列の番号を持った配列([0]=行=ROW , [1]=列=COL)
	 * @og.rtnNotNull
	 */
	public static int[] kigo2rowCol( final String kigo ) {
		int rowNo = 0;
		int colNo = -1;										// +1 して、26 かける処理をしているので、辻褄合わせ

		// 6.2.6.0 (2015/06/19) 行-列形式と、SHEET文字列判定を採用。
		if( "SHEET".equalsIgnoreCase( kigo ) ) {
			rowNo = -1;
		}
		else {
			final int adrs = kigo.indexOf( '-' );
			if( adrs > 0 ) {
				rowNo = Integer.parseInt( kigo.substring( 0,adrs ) );
				colNo = Integer.parseInt( kigo.substring( adrs+1 ) );
			}
			else {
				for( int i=0; i<kigo.length(); i++ ) {
					final char ch = kigo.charAt(i);
					if( 'A' <= ch && ch <= 'Z' ) { colNo = (colNo+1)*26 + ch-'A'; }
					else {
						// ｱﾙﾌｧﾍﾞｯﾄでなくなったら、残りは 行番号(ただし、-1する)
						rowNo = Integer.parseInt( kigo.substring( i ) ) -1;
						break;
					}
				}
			}
		}
		return new int[] { rowNo,colNo };
	}

	/**
	 * ｾﾙｵﾌﾞｼﾞｪｸﾄ(Cell)から値を取り出します。
	 *
	 * ｾﾙｵﾌﾞｼﾞｪｸﾄが存在しない場合は、null を返します。
	 * それ以外で、うまく値を取得できなかった場合は、ｾﾞﾛ文字列を返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 取り出し方法を少し修正
	 * @og.rev 5.5.1.2 (2012/04/06) ﾌｫｰﾏｯﾄｾﾙを実行して、その結果を再帰的に処理する。
	 * @og.rev 6.0.3.0 (2014/11/13) ｾﾙﾌｫｰﾏｯﾄｴﾗｰ時に、RuntimeException を throw しない。
	 * @og.rev 6.4.2.0 (2016/01/29) StringUtil#ogStackTrace(Throwable) を、ThrowUtil#ogStackTrace(String,Throwable) に置き換え。
	 * @og.rev 6.5.0.0 (2016/09/30) poi-3.15 対応(Cell.CELL_TYPE_XXXX → CellType.XXXX)
	 * @og.rev 7.0.0.0 (2018/10/01) 警告:[deprecation] CellのgetCellTypeEnum()は推奨されません (POI4.0.0)
	 * @og.rev 7.3.0.0 (2021/01/06) ﾌｫｰﾏｯﾄｴﾗｰ時に、ｴﾗｰﾒｯｾｰｼﾞ取得でもｴﾗｰになる。
	 * @og.rev 8.0.0.0 (2021/07/31) FORMULA処理のｴﾗｰ対応(出来るだけ…)
	 * @og.rev 8.1.2.3 (2022/05/20) 計算式の計算を行う。
	 * @og.rev 8.5.0.0 (2023/04/21) ｾﾙﾌｫｰﾏｯﾄ処理ｴﾗｰ時には、ｽﾀｯｸﾄﾚｰｽは出さずに計算式を返す。
	 *
	 * @param	oCell	EXCELのｾﾙｵﾌﾞｼﾞｪｸﾄ
	 * @return	ｾﾙの値
	 */
	public static String getValue( final Cell oCell ) {
		if( oCell == null ) { return null; }
		String strText = "";
	//	final int nCellType = oCell.getCellType();									// 6.5.0.0 (2016/09/30) poi-3.12
	//	switch(nCellType) {															// 6.5.0.0 (2016/09/30) poi-3.12
//		switch( oCell.getCellTypeEnum() ) {											// 6.5.0.0 (2016/09/30) poi-3.15
		switch( oCell.getCellType() ) {												// 7.0.0.0 (2018/10/01) poi-4.0.0 Deprecated.
	//		case Cell.CELL_TYPE_NUMERIC:											// 6.5.0.0 (2016/09/30) poi-3.12
			case NUMERIC:															// 6.5.0.0 (2016/09/30) poi-3.15
					strText = getNumericTypeString( oCell );
					break;
	//		case Cell.CELL_TYPE_STRING:												// 6.5.0.0 (2016/09/30) poi-3.12
			case STRING:															// 6.5.0.0 (2016/09/30) poi-3.15
	// POI3.0		strText = oCell.getStringCellValue();
					final RichTextString richText = oCell.getRichStringCellValue();
					if( richText != null ) {
						strText = richText.getString();
					}
					break;
	//		case Cell.CELL_TYPE_FORMULA:											// 6.5.0.0 (2016/09/30) poi-3.12
			case FORMULA:															// 6.5.0.0 (2016/09/30) poi-3.15
	// POI3.0		strText = oCell.getStringCellValue();
					// 5.5.1.2 (2012/04/06) ﾌｫｰﾏｯﾄｾﾙを実行して、その結果を再帰的に処理する。
				//	final Workbook wb = oCell.getSheet().getWorkbook();
				//	final CreationHelper crateHelper = wb.getCreationHelper();
				//	final FormulaEvaluator evaluator = crateHelper.createFormulaEvaluator();

					try {
						// 8.1.2.3 (2022/05/20) 計算式の計算を行う。
						final Workbook wb = oCell.getSheet().getWorkbook();
						final CreationHelper crateHelper = wb.getCreationHelper();
						final FormulaEvaluator evaluator = crateHelper.createFormulaEvaluator();
						final CellValue value = evaluator.evaluate(oCell);

						switch (value.getCellType()) {
							case STRING:
								strText = value.getStringValue();
								break;
							case NUMERIC:
								strText = Double.toString(value.getNumberValue());
								break;
							case BOOLEAN:
								strText = Boolean.toString(value.getBooleanValue());
								break;
							default:
								strText = oCell.getCellFormula();		// 計算式のまま
								break;
						}

		//				strText = oCell.getCellFormula();				// 8.1.2.3 (2022/05/20) 計算式は返さない
						// 8.0.0.0 (2021/07/31) FORMULA処理のｴﾗｰ対応(出来るだけ…)
				//		final Cell fCell = evaluator.evaluateInCell(oCell);
				//		strText = getValue( fCell );
					}
					catch( final Throwable th ) {
						// 7.3.0.0 (2021/01/06) ﾌｫｰﾏｯﾄｴﾗｰ時に、ｴﾗｰﾒｯｾｰｼﾞ取得でもｴﾗｰになる。
				//		final String errMsg = "ｾﾙﾌｫｰﾏｯﾄが解析できません。";
						// 8.5.0.0 (2023/04/21) ｾﾙﾌｫｰﾏｯﾄ処理ｴﾗｰ時には、ｽﾀｯｸﾄﾚｰｽは出さずに計算式を返す。
						final String errMsg = "ｾﾙﾌｫｰﾏｯﾄが解析できません。"
									+ CR + "  Formula=[" + oCell.getCellFormula() + "]"
									+ CR + getCellMsg( oCell );
	//					throw new OgRuntimeException( errMsg,th );
				//		System.err.println( ThrowUtil.ogStackTrace( errMsg,th ) );	// 6.4.2.0 (2016/01/29) , 8.5.0.0 (2023/04/21) Delete
						System.err.println( errMsg );		// 8.5.0.0 (2023/04/21) Add
						strText = oCell.toString();			// 8.5.0.0 (2023/04/21) Add
					}
					break;
	//		case Cell.CELL_TYPE_BOOLEAN:											// 6.5.0.0 (2016/09/30) poi-3.12
			case BOOLEAN:															// 6.5.0.0 (2016/09/30) poi-3.15
					strText = String.valueOf(oCell.getBooleanCellValue());
					break;
	//		case Cell.CELL_TYPE_BLANK :												// 6.5.0.0 (2016/09/30) poi-3.12
			case BLANK :															// 6.5.0.0 (2016/09/30) poi-3.15
					break;
	//		case Cell.CELL_TYPE_ERROR:												// 6.5.0.0 (2016/09/30) poi-3.12
			case ERROR:																// 6.5.0.0 (2016/09/30) poi-3.15
					break;
			default :
				final String errMsg = "ｾﾙﾀｲﾌﾟが不明です。CellType=" + oCell.getCellType()
										+ CR + getCellMsg( oCell );		// 8.5.0.0 (2023/04/21) Add
				System.err.println( errMsg );							// 8.5.0.0 (2023/04/21) Add
				strText = oCell.toString();								// 8.5.0.0 (2023/04/21) Add
				break;
		}
		return strText ;
	}

	/**
	 * ｾﾙｵﾌﾞｼﾞｪｸﾄ(Cell)に、値をｾｯﾄします。
	 *
	 * ｾﾙｵﾌﾞｼﾞｪｸﾄが存在しない場合は、何もしません。
	 * 引数は、文字列で渡しますが、ｾﾙの形式に合わせて、変換します。
	 * 変換がうまくいかなかった場合は、ｴﾗｰになりますので、ご注意ください。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) 新規追加
	 * @og.rev 6.5.0.0 (2016/09/30) poi-3.15 対応(Cell.CELL_TYPE_XXXX → CellType.XXXX)
	 * @og.rev 7.0.0.0 (2018/10/01) 警告:[deprecation] CellのgetCellTypeEnum()は推奨されません (POI4.0.0)
	 * @og.rev 7.3.0.0 (2021/01/06) setCellType( CellType.BLANK )(Deprecated) → setBlank() (poi-4.1.2)
	 *
	 * @param	oCell	EXCELのｾﾙｵﾌﾞｼﾞｪｸﾄ
	 * @param	val		ｾｯﾄする値
	 */
	public static void setValue( final Cell oCell , final String val ) {
		if( oCell == null ) { return ; }
	//	if( val == null || val.isEmpty() ) { oCell.setCellType( Cell.CELL_TYPE_BLANK ); }		// 6.5.0.0 (2016/09/30) poi-3.12
	//	if( val == null || val.isEmpty() ) { oCell.setCellType( CellType.BLANK ); }				// 6.5.0.0 (2016/09/30) poi-3.15
		if( val == null || val.isEmpty() ) { oCell.setBlank(); }								// 7.3.0.0 (2021/01/06) poi-4.1.2

	//	switch( oCell.getCellType() ) {										// 6.5.0.0 (2016/09/30) poi-3.12
//		switch( oCell.getCellTypeEnum() ) {									// 6.5.0.0 (2016/09/30) poi-3.15
		switch( oCell.getCellType() ) {										// 7.0.0.0 (2018/10/01) poi-4.0.0 Deprecated.
	//		case Cell.CELL_TYPE_NUMERIC:									// 6.5.0.0 (2016/09/30) poi-3.12
			case NUMERIC:													// 6.5.0.0 (2016/09/30) poi-3.15
//					oCell.setCellValue( Double.valueOf( val ) );
					oCell.setCellValue( Double.parseDouble( val ) );		// 7.3.0.0 (2021/01/06) SpotBugs 疑わしいﾌﾟﾘﾐﾃｨﾌﾞ値のﾎﾞｸｼﾝｸﾞ
					break;
	//		case Cell.CELL_TYPE_BOOLEAN:									// 6.5.0.0 (2016/09/30) poi-3.12
			case BOOLEAN:													// 6.5.0.0 (2016/09/30) poi-3.15
					oCell.setCellValue( "true".equalsIgnoreCase( val ) );
					break;
			default :
					oCell.setCellValue( val );
					break;
		}
	}

	/**
	 * ｾﾙ値が数字の場合に、数字か日付かを判断して、対応する文字列を返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 6.3.1.0 (2015/06/28) ExcelStyleFormat を使用します。
	 *
	 * @param	oCell	EXCELのｾﾙｵﾌﾞｼﾞｪｸﾄ
	 * @return	数字の場合は、文字列に変換した結果を、日付の場合は、"yyyyMMddHHmmss" 形式で返します。
	 */
	public static String getNumericTypeString( final Cell oCell ) {
		final String strText ;

		final double dd = oCell.getNumericCellValue() ;
		if( DateUtil.isCellDateFormatted( oCell ) ) {
	//		strText = DateSet.getDate( DateUtil.getJavaDate( dd ).getTime() , "yyyyMMddHHmmss" );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
			strText = ExcelStyleFormat.dateFormat( dd );
		}
		else {
	//		final NumberFormat numFormat = NumberFormat.getInstance();
	//		if( numFormat instanceof DecimalFormat ) {
	//			((DecimalFormat)numFormat).applyPattern( "#.####" );
	//		}
	//		strText = numFormat.format( dd );
			final String fmrs = oCell.getCellStyle().getDataFormatString();
			strText = ExcelStyleFormat.getNumberValue( fmrs,dd );
		}
		return strText ;
	}

	/**
	 * 全てのSheetに対して、autoSizeColumn設定を行います。
	 *
	 * 重たい処理なので、ﾌｧｲﾙの書き出し直前に一度だけ実行するのがよいでしょう。
	 * autoSize設定で、ｶﾗﾑ幅が大きすぎる場合、現状では、
	 * 初期ｶﾗﾑ幅のmaxColCount倍を限度に設定します。
	 * ただし、maxColCount がﾏｲﾅｽの場合は、無制限になります。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.8.2.4 (2017/11/20) rowObj のnull対策(poi-3.17)
	 *
	 * @param	wkbook		処理対象のWorkbook
	 * @param	maxColCount	最大幅を標準ｾﾙ幅の何倍にするかを指定。ﾏｲﾅｽの場合は、無制限
	 * @param	dataStRow	ﾃﾞｰﾀ行の開始位置。未設定時は、-1
	 */
	public static void autoCellSize( final Workbook wkbook , final int maxColCount , final int dataStRow ) {
		final int shCnt = wkbook.getNumberOfSheets();

		for( int shNo=0; shNo<shCnt; shNo++ ) {
			final Sheet sht = wkbook.getSheetAt( shNo );
			final int defW = sht.getDefaultColumnWidth();		// 標準ｶﾗﾑの文字数
			final int maxWidth = defW*256*maxColCount ;			// Widthは、文字数(文字幅)*256*最大ｾﾙ数

			int stR = sht.getFirstRowNum();
			final int edR = sht.getLastRowNum();

			// 6.8.2.4 (2017/11/20) rowObj のnull対策(poi-3.17)
			// poi-3.15 でも同じ現象が出ていますが、sht.getRow( stR ) で、Rowｵﾌﾞｼﾞｪｸﾄにnullが返ってきます。
			// なんとなく、最後の行だけ、返ってきている感じです。
			// 頻繁には使わないと思いますので、最大ｶﾗﾑ数=256 として処理します。

			final Row rowObj = sht.getRow( stR );
	//		Row rowObj = sht.getRow( stR );
	//		if( rowObj == null ) {
	//			for( int i=stR+1; i<edR; i++ ) {
	//				rowObj = sht.getRow( i );
	//				if( rowObj != null ) { break; }
	//			}
	//		}

			final int stC = rowObj == null ? 0   : rowObj.getFirstCellNum();	// 6.8.2.4 (2017/11/20) rowObj のnull対策
			final int edC = rowObj == null ? 256 : rowObj.getLastCellNum();		// 含まない (xlsxでは、最大 16,384 列

			// SheetUtil を使用して、計算範囲を指定します。
			if( stR < dataStRow ) { stR = dataStRow; }		// 計算範囲
			for( int colNo=stC; colNo<edC; colNo++ ) {
				final double wpx = SheetUtil.getColumnWidth( sht,colNo,true,stR,edR );
				if( wpx >= 0.0d ) {							// Cellがないと、ﾏｲﾅｽ値が戻る。
					int wd = (int)Math.ceil(wpx * 256) ;
					if( maxWidth >= 0 && wd > maxWidth ) { wd = maxWidth; }	// 最大値が有効な場合は、置き換える
					sht.setColumnWidth( colNo,wd );
				}
			}

			// Sheet#autoSizeColumn(int) を使用して、自動計算させる場合。
	//		for( int colNo=stC; colNo<edC; colNo++ ) {
	//			sht.autoSizeColumn( colNo );
	//			if( maxWidth >= 0 ) {					// 最大値が有効な場合は、置き換える
	//				int wd = sht.getColumnWidth( colNo );
	//				if( wd > maxWidth ) { sht.setColumnWidth( colNo,maxWidth ); }
	//			}
	//		}
		}
	}

//	/**
//	 * 指定の Workbook の全Sheetを対象に、実際の有効行と有効ｶﾗﾑを取得します。
//	 *
//	 * ※ 現在、唯一LibreOfficeでのみ、xslx 変換できますが、有効行とｶﾗﾑが
//	 *    ｼｭﾘﾝｸされず、無駄な行とｶﾗﾑが存在します。
//	 *    これは、xsl で出力されたﾌｧｲﾙから有効な値を取得して、xslxに適用させるための
//	 *    機能で、本来きちんとした有効範囲の xslx が生成されれば、不要な処理です。
//	 *
//	 * 配列は、[0]=行の最大値(Sheet#getLastRowNum())と、[1]は有効行の中の列の
//	 * 最大値(Row#getLastCellNum())を、ｼｰﾄごとにListに追加していきます。
//	 *
//	 * @og.rev 8.0.1.0 (2021/10/29) 全Sheetを対象に、実際の有効行と有効ｶﾗﾑを取得
//	 * @og.rev 8.0.3.0 (2021/12/17) 処理が中途半端だったので、廃止します。
//	 *
//	 * @param	wkbook		処理対象のWorkbook
//	 * @return	ｼｰﾄごとの有効行の配列ﾘｽﾄ
//	 * @see		#activeWorkbook( Workbook,List )
//	 */
//	public static List<int[]> getLastRowCellNum( final Workbook wkbook ) {
//		final List<int[]> rcList = new ArrayList<>();					// ｼｰﾄごとの有効行の配列ﾘｽﾄ
//
//		final int shCnt = wkbook.getNumberOfSheets();
//		for( int shNo=0; shNo<shCnt; shNo++ ) {
//			final Sheet sht = wkbook.getSheetAt( shNo );
//			final int stR = sht.getFirstRowNum();
//			final int edR = sht.getLastRowNum();
//			int lastNo = 0;												// 行の有効最大値
//			for( int rowNo=edR; rowNo>=stR && rowNo>=0; rowNo-- ) {		// 逆順に処理します。
//				final Row rowObj = sht.getRow( rowNo );
//				if( rowObj != null ) {
//					final int edC = rowObj.getLastCellNum();			// 列の有効最大値
//					if( lastNo < edC ) { lastNo = edC; }				// ｼｰﾄ内での列の最大有効値
//				}
//			}
//			rcList.add( new int[] {edR,lastNo} );						// 有効行の配列
//		}
//		return rcList;
//	}

	/**
	 * 指定の Workbook の全Sheetを対象に、空行を取り除き、全体をｼｭﾘﾝｸします。
	 *
	 * この処理は、#saveFile( String ) の直前に行うのがよいでしょう。
	 *
	 * ここでは、Row を逆順にｽｷｬﾝし、Cellが 存在しない間は、行を削除します。
	 * 途中の空行の削除ではなく、最終行からの連続した空行の削除です。
	 *
	 * isCellDel=true を指定すると、Cellの末尾削除を行います。
	 * 有効行の最後のCellから空ｾﾙを削除していきます。
	 * 表形式などの場合は、Cellのあるなしで、ﾚｲｱｳﾄが崩れる場合がありますので
	 * 処理が不要な場合は、isCellDel=false を指定してください。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.0.2.3 (2014/10/10) CellStyle の有無も判定基準に含めます。
	 * @og.rev 6.0.2.5 (2014/10/31) Cellの開始、終了番号が、ﾏｲﾅｽのｹｰｽの対応
	 * @og.rev 6.5.0.0 (2016/09/30) poi-3.15 対応(Cell.CELL_TYPE_XXXX → CellType.XXXX)
	 * @og.rev 7.0.0.0 (2018/10/01) 警告:[deprecation] CellのgetCellTypeEnum()は推奨されません (POI4.0.0)
	 * @og.rev 8.0.1.0 (2021/10/29) CellStyle は not null になったための修正
	 *
	 * @param	wkbook		処理対象のWorkbook
	 * @param	isCellDel	Cellの末尾削除を行うかどうか(true:行う/false:行わない)
	 * @see		#activeWorkbook( Workbook,List )
	 */
	public static void activeWorkbook( final Workbook wkbook , final boolean isCellDel ) {
		final int shCnt = wkbook.getNumberOfSheets();
		for( int shNo=0; shNo<shCnt; shNo++ ) {
			final Sheet sht = wkbook.getSheetAt( shNo );
			final int stR = sht.getFirstRowNum();
			final int edR = sht.getLastRowNum();

			boolean isRowDel = true;											// 行の削除は、Cellが見つかるまで。
			for( int rowNo=edR; rowNo>=stR && rowNo>=0; rowNo-- ) {				// 逆順に処理します。
				final Row rowObj = sht.getRow( rowNo );
				if( rowObj != null ) {
					final int stC = rowObj.getFirstCellNum();
					final int edC = rowObj.getLastCellNum();
					// 8.0.1.0 (2021/10/29) 各行の最初のｾﾙｽﾀｲﾙをﾍﾞｰｽとして比較する。

					if( stC >= 0 && edC >= 0 ) {								// 8.0.3.0 (2021/12/17) 存在しない場合もある。
						final CellStyle endCellStyle = rowObj.getCell( stC ).getCellStyle();	// nullﾁｪｯｸ入れてない…
						for( int colNo=edC; colNo>=stC && colNo>=0; colNo-- ) {		// 6.0.2.5 (2014/10/31) stC,edC が、ﾏｲﾅｽのｹｰｽがある。
							final Cell colObj = rowObj.getCell( colNo );
							if( colObj != null ) {
								final String val = getValue( colObj );
								if( colObj.getCellType() != CellType.BLANK && val != null && val.length() > 0 ) { 			// 7.0.0.0 (2018/10/01) poi-4.0.0 Deprecated.
									isRowDel = false;					// 一つでも現れれば、行の削除は中止
									break;
								}
								// 6.0.2.3 (2014/10/10) CellStyle の有無も判定基準に含めます。
								// 8.0.1.0 (2021/10/29) 各行の最初のｾﾙｽﾀｲﾙをﾍﾞｰｽとして比較する。
		//						else if( colObj.getCellStyle() != null ) {
								else if( ! endCellStyle.equals(colObj.getCellStyle()) ) {
									isRowDel = false;					// 一つでも現れれば、行の削除は中止
									break;
								}
								else if( isCellDel ) {
									rowObj.removeCell( colObj );		// CELL_TYPE_BLANK の場合は、削除
								}
							}
						}
					}
					if( isRowDel ) { sht.removeRow( rowObj );	}
					else if( !isCellDel ) { break; }				// Cell の末尾削除を行わない場合は、break すればよい。
				}
			}
		}
	}

	/**
	 * 指定の Workbook の全Sheetを対象に、実際の有効行と有効ｶﾗﾑを元に全体をｼｭﾘﾝｸします。
	 *
	 * ※ 現在、唯一LibreOfficeでのみ、xslx 変換できますが、有効行とｶﾗﾑが
	 *    ｼｭﾘﾝｸされず、無駄な行とｶﾗﾑが存在します。
	 *    これは、xsl で出力されたﾌｧｲﾙから有効な値を取得して、xslxに適用させるための
	 *    機能で、本来きちんとした有効範囲の xslx が生成されれば、不要な処理です。
	 *
	 * 引数のListｵﾌﾞｼﾞｪｸﾄに従って、無条件に処理を行います。
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) 全Sheetを対象に、実際の有効行と有効ｶﾗﾑを取得
	 * @og.rev 8.0.3.0 (2021/12/17) ｼｰﾄ毎の行数Listに変更。
	 *
	 * @param	wkbook		処理対象のWorkbook
//	 * @param	rcList		ｼｰﾄごとの有効行の配列ﾘｽﾄ
	 * @param	rowCntList	ｼｰﾄごとの有効行の配列ﾘｽﾄ
//	 * @see		#getLastRowCellNum( Workbook )
	 * @see		#activeWorkbook( Workbook,boolean )
	 */
//	 public static void activeWorkbook( final Workbook wkbook , final List<int[]> rcList ) {
	 public static void activeWorkbook( final Workbook wkbook , final List<Integer> rowCntList ) {
		final int shCnt = wkbook.getNumberOfSheets();
		for( int shNo=0; shNo<shCnt; shNo++ ) {
			final Sheet sht = wkbook.getSheetAt( shNo );
//			final int[] rowcol = rcList.get(shNo);						// ｼｰﾄ内の有効行と列
			final int stR = rowCntList.get(shNo);						// ｼｰﾄ内の有効行と列

//			final int stR = rowcol[0];
			final int edR = sht.getLastRowNum();						// 参考程度
			// edR～stRまでの行は、無条件に削除します。
			for( int rowNo=edR; rowNo>=stR && rowNo>=0; rowNo-- ) {		// 逆順に処理します。
				final Row rowObj = sht.getRow( rowNo );
				if( rowObj != null ) {
					sht.removeRow( rowObj );
				}
			}

		//	カラム列の削除は保留
		//	// stR～0までの行は、有効行なので、ｶﾗﾑの処理を考えます。
//		//	final int stC = rowcol[1];									// ｼｰﾄの中での有効ｶﾗﾑの最大値
		//	final int stC = 0;
		//	for( int rowNo=stR; rowNo>=0; rowNo-- ) {					// 逆順に処理します。
		//		final Row rowObj = sht.getRow( rowNo );
		//		if( rowObj != null ) {
		//			final int edC = rowObj.getLastCellNum();			// 参考程度
		//			// edC～stCまでのｶﾗﾑは、無条件に削除します。
		//			for( int colNo=edC; colNo>=stC && colNo>=0; colNo-- ) {	// 6.0.2.5 (2014/10/31) stC,edC が、ﾏｲﾅｽのｹｰｽがある。
		//				final Cell colObj = rowObj.getCell( colNo );
		//				if( colObj != null ) {
		//					if( colObj.getCellType() == CellType.BLANK ) {
		//						rowObj.removeCell( colObj );
		//					}
		//					else {
		//						break;
		//					}
		//				}
		//			}
		//		}
		//	}
		}
	}

	/**
	 * ﾌｧｲﾙから、Workbookｵﾌﾞｼﾞｪｸﾄを新規に作成します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) ﾌｧｲﾙ引数を、String → File に変更
	 * @og.rev 7.0.0.0 (2018/10/01) poi-4.0.0 例外InvalidFormatExceptionは対応するtry文の本体ではｽﾛｰされません
	 *
	 * @param	file	入力ﾌｧｲﾙ
	 * @return	Workbookｵﾌﾞｼﾞｪｸﾄ
	 * @og.rtnNotNull
	 */
	public static Workbook createWorkbook( final File file ) {
		InputStream fis = null;
		try {
			// File ｵﾌﾞｼﾞｪｸﾄでcreate すると、ﾌｧｲﾙがｵｰﾌﾟﾝされたままになってしまう。
			fis = new BufferedInputStream( new FileInputStream( file ) );
			return WorkbookFactory.create( fis );
		}
		catch( final IOException ex ) {
			final String errMsg = "ﾌｧｲﾙ読込みｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
		// 7.0.0.0 (2018/10/01) poi-4.0.0 対応するtry文の本体ではｽﾛｰされません
//		catch( final InvalidFormatException ex ) {
//			final String errMsg = "ﾌｧｲﾙ形式ｴﾗｰ[" + file + "]" + CR + ex.getMessage() ;
//			throw new OgRuntimeException( errMsg,ex );
//		}
		finally {
			Closer.ioClose( fis );
		}
	}

	/**
	 * ｼｰﾄ一覧を、Workbook から取得します。
	 *
	 * 取得元が、Workbook なので、xls , xlsx どちらの形式でも取り出せます。
	 *
	 * EXCEL上のｼｰﾄ名を、配列で返します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param	wkbook	Workbookｵﾌﾞｼﾞｪｸﾄ
	 * @return	ｼｰﾄ名の配列
	 */
	public static String[] getSheetNames( final Workbook wkbook ) {
		final int shCnt = wkbook.getNumberOfSheets();

		String[] shtNms = new String[shCnt];

		for( int i=0; i<shCnt; i++ ) {
			final Sheet sht = wkbook.getSheetAt( i );
			shtNms[i] = sht.getSheetName();
		}

		return shtNms;
	}

	/**
	 * 名前定義一覧を取得します。
	 *
	 * EXCEL上に定義された名前を、配列で返します。
	 * ここでは、名前とFormulaをﾀﾌﾞで連結した文字列を配列で返します。
	 * Name ｵﾌﾞｼﾞｪｸﾄを削除すると、EXCELが開かなくなったりするので、
	 * 取りあえず一覧を作成して、手動で削除してください。
	 * なお、名前定義には、非表示というのがありますので、ご注意ください。
	 *
	 * ◆ 非表示になっている名前の定義を表示にする EXCEL VBA ﾏｸﾛ
	 * http://dev.classmethod.jp/tool/excel-delete-name/
	 *    Sub VisibleNames()
	 *        Dim name
	 *        For Each name In ActiveWorkbook.Names
	 *            If name.Visible = False Then
	 *                name.Visible = True
	 *            End If
	 *        Next
	 *        MsgBox "すべての名前の定義を表示しました。", vbOKOnly
	 *    End Sub
	 *
	 * ※ EXCEL2010 数式ﾀﾌﾞ→名前の管理 で、複数選択で、削除できます。
	 *    ただし、非表示の名前定義は、先に表示しないと、削除できません。
	 * ◆ 名前の一括削除 EXCEL VBA ﾏｸﾛ
	 * http://komitsudo.blog70.fc2.com/blog-entry-104.html
	 *    Sub DeleteNames()
	 *        Dim name
	 *        On Error Resume Next
	 *        For Each name In ActiveWorkbook.Names
	 *            If Not name.BuiltIn Then
	 *                name.Delete
	 *            End If
	 *        Next
	 *        MsgBox "すべての名前の定義を削除しました。", vbOKOnly
	 *    End Sub
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 * @og.rev 7.0.0.0 (2018/10/01) 警告:[deprecation] WorkbookのgetNameAt(int)は推奨されません (POI4.0.0)
	 *
	 * @param	wkbook	Workbookｵﾌﾞｼﾞｪｸﾄ
	 * @return	名前定義(名前+TAB+Formula)の配列
	 * @og.rtnNotNull
	 */
	public static String[] getNames( final Workbook wkbook ) {
//		final int cnt = wkbook.getNumberOfNames();

		final Set<String> nmSet = new TreeSet<>();

		// 7.0.0.0 (2018/10/01) 警告:[deprecation] WorkbookのgetNameAt(int)は推奨されません (POI4.0.0)
//		for( int i=0; i<cnt; i++ ) {
		for( final Name nm : wkbook.getAllNames() ) {
			String name	= null;
			String ref	= null;

//			final Name nm = wkbook.getNameAt(i);
			try {
				name = nm.getNameName();
				ref  = nm.getRefersToFormula();
			}
	//		catch( final Exception ex ) {					// 6.1.0.0 (2014/12/26) refactoring
			catch( final RuntimeException ex ) {
				final String errMsg = "POIUtil:RefersToFormula Error! name=[" + name + "]" + ex.getMessage() ;
				System.out.println( errMsg );
				// Excel97形式の場合、getRefersToFormula() でｴﾗｰが発生することがある。
			}

			nmSet.add( name + "\t" + ref );

			// 削除するとEXCELが壊れる？ なお、削除時には逆順で廻さないとｱﾄﾞﾚｽがずれます。
			// if( nm.isDeleted() ) { wkbook.removeName(i); }
		}

		return nmSet.toArray( new String[nmSet.size()] );
	}

	/**
	 * 書式のｽﾀｲﾙ一覧を取得します。
	 *
	 * EXCEL上に定義された書式のｽﾀｲﾙを、配列で返します。
	 * 書式のｽﾀｲﾙの名称は、CellStyle にﾒｿｯﾄﾞが定義されていません。
	 * 実ｸﾗｽである HSSFCellStyle にｷｬｽﾄして使用する
	 * 必要があります。(XSSFCellStyle にも名称を取得するﾒｿｯﾄﾞがありません。)
	 *
	 * ※ EXCEL2010 ﾎｰﾑﾀﾌﾞ→ｾﾙのｽﾀｲﾙ は、一つづつしか削除できません。
	 *    ﾏｸﾛは、開発ﾀﾌﾞ→Visual Basic で、挿入→標準ﾓｼﾞｭｰﾙ を開き
	 *    ﾃｷｽﾄを張り付けてください。
	 *    実行は、開発ﾀﾌﾞ→ﾏｸﾛ で、ﾏｸﾛ名を選択して、実行します。
	 *    最後は、削除してください。
	 *
	 * ◆ ｽﾀｲﾙの一括削除 EXCEL VBA ﾏｸﾛ
	 * http://komitsudo.blog70.fc2.com/blog-entry-104.html
	 *    Sub DeleteStyle()
	 *        Dim styl
	 *        On Error Resume Next
	 *        For Each styl In ActiveWorkbook.Styles
	 *            If Not styl.BuiltIn Then
	 *                styl.Delete
	 *            End If
	 *        Next
	 *        MsgBox "すべての追加ｽﾀｲﾙを削除しました。", vbOKOnly
	 *    End Sub
	 *
	 * ◆ 名前の表示、削除、ｽﾀｲﾙの削除の一括実行 EXCEL VBA ﾏｸﾛ
	 *    Sub AllDelete()
	 *        Call VisibleNames
	 *        Call DeleteNames
	 *        Call DeleteStyle
	 *        MsgBox "すべての処理を完了しました。", vbOKOnly
	 *    End Sub
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param	wkbook	Workbookｵﾌﾞｼﾞｪｸﾄ
	 * @return	書式のｽﾀｲﾙ一覧
	 * @og.rtnNotNull
	 */
	public static String[] getStyleNames( final Workbook wkbook ) {
		final int cnt = wkbook.getNumCellStyles();		// return 値は、short

		final Set<String> nmSet = new TreeSet<>();

		for( int s=0; s<cnt; s++ ) {
			final CellStyle cs = wkbook.getCellStyleAt( (short)s );
			if( cs instanceof HSSFCellStyle ) {
				final HSSFCellStyle hcs = (HSSFCellStyle)cs;
				final String name = hcs.getUserStyleName();
				// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
				if( name == null ) {							// この処理は不要かも。
					final HSSFCellStyle pst = hcs.getParentStyle();
					if( pst != null ) {
						final String pname = pst.getUserStyleName();
						if( pname != null ) { nmSet.add( pname ); }
					}
				}
				else {
					nmSet.add( name );
				}
			}
		}

		return nmSet.toArray( new String[nmSet.size()] );
	}

	/**
	 * ｾﾙ情報を返します。
	 *
	 * ｴﾗｰ発生時に、どのｾﾙでｴﾗｰが発生したかの情報を取得できるようにします。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.0.3.0 (2014/11/13) ｾﾙ情報を作成する時に、値もｾｯﾄします。
	 * @og.rev 6.2.2.0 (2015/03/27) celKigo を求めるﾛｼﾞｯｸ変更
	 * @og.rev 6.3.1.0 (2015/06/28) rowNo(行番号)も引数に取るようにします。
	 *
	 * @param	oCell	EXCELのｾﾙｵﾌﾞｼﾞｪｸﾄ
	 * @return	ｾﾙ情報の文字列
	 */
	public static String getCellMsg( final Cell oCell ) {
		String lastMsg = null;

		if( oCell != null ) {
			final String shtNm = oCell.getSheet().getSheetName();
			final int  rowNo = oCell.getRowIndex();
			final int  celNo = oCell.getColumnIndex();

			// 6.0.3.0 (2014/11/13) ｾﾙ情報を作成する時に、値もｾｯﾄします。
			lastMsg = "  Sheet=" + shtNm + ", Row=" + rowNo + ", Cel=" + celNo
						 + "(" + getCelKigo(rowNo,celNo) + "), Val=" + oCell.toString() ;
		}

		return lastMsg;
	}

	/**
	 * Excelの行番号,列番号より、ｾﾙ記号を求めます。
	 *
	 * 行番号は、0から始まる数字ですが、記号化する場合は、１から始まります。
	 * Excelの列記号とは、A,B,C,…,Z,AA,AB,…,ZZ,AAA,AAB,… と続きます。
	 * つまり、ｱﾙﾌｧﾍﾞｯﾄだけの、２６進数になります。(ｾﾞﾛの扱いが少し特殊です)
	 * 列番号は、0から始まる数字で、0=A,1=B,2=C,…,25=Z,26=AA,27=AB,…,701=ZZ,702=AAA,703=AAB,…
	 * EXCELの行列記号にする場合は、この列記号に、行番号を、＋１して付ければよいだけです。
	 * (※ 列番号に＋１するのは、内部では０から始まる列番号ですが、表示上は１から始まります)
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) celKigo を求めるﾛｼﾞｯｸ変更
	 * @og.rev 6.3.1.0 (2015/06/28) rowNo(行番号)も引数に取るようにします。
	 *
	 * @param	rowNo	行番号(0,1,2,…)
	 * @param	colNo	列番号(0,1,2,…)
	 * @return	Excelの列記号(A1,B2,C3,…)
	 */
	public static String getCelKigo( final int rowNo,final int colNo ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		int cnt = colNo;
		while( cnt >= 26 ) {
			buf.append( (char)('A'+cnt%26) );
			cnt = cnt/26-1;
		}
		buf.append( (char)('A'+cnt%26) )
			.reverse()								// append で逆順に付けているので、反転して返します。
			.append( rowNo+1 );

		return buf.toString();
	}

	/**
	 * XSSFSimpleShapeｵﾌﾞｼﾞｪｸﾄにﾘﾝｸを設定します。
	 *
	 * 処理の簡素化のために、url引数が null の場合は、何もせず処理を終了します。
	 *
	 * @og.rev 8.1.0.1 (2022/01/07) ﾃｷｽﾄﾍﾞｰｽのﾘﾝｸ作成
	 *
	 * @param	shape	XSSFSimpleShapeｵﾌﾞｼﾞｪｸﾄ
	 * @param	url		ﾘﾝｸ文字列
	 */
	public static void makeShapeLink( final XSSFSimpleShape shape , final String url ) {
		if( url == null ) { return; }

		final String rid = shape.getDrawing()			// XSSFDrawing	XSSFShape#getDrawing()
							.getPackagePart()			// PackagePart	POIXMLDocumentPart#getPackagePart()
							.addExternalRelationship(	// PackageRelationship	PackagePart#addExternalRelationship(String,String)
									url, PackageRelationshipTypes.HYPERLINK_PART)
							.getId();					// String		PackageRelationship#getId()

		final CTHyperlink hyperlink = CTHyperlink.Factory.newInstance();
		hyperlink.setId(rid);

		shape.getCTShape()								// CTShape					XSSFSimpleShape#getCTShape()
			.getNvSpPr()								// CTShapeNonVisual			CTShape#getNvSpPr()
			.getCNvPr()									// CTNonVisualDrawingProps	CTShapeNonVisual#getCNvPr()
			.setHlinkClick( hyperlink );				// void						CTNonVisualDrawingProps#setHlinkClick(CTHyperlink)
	}

	/**
	 * XSSFSimpleShapeｵﾌﾞｼﾞｪｸﾄにｶﾗｰを設定します。
	 *
	 * 処理の簡素化のために、col引数が null の場合は、何もせず処理を終了します。
	 * col配列は、[0]:red [1]:blue [2] green です。
	 *
	 * ※ ｶﾗｰの設定を、XSSFSimpleShape#setFillColor(int,int,int) で行うと、XSLXﾌｧｲﾙが
	 * 壊れるようです。POIが対応できていないのか、ｶﾗｰ化設定方法を間違っているのか…
	 *
	 * @og.rev 8.1.0.1 (2022/01/07) ﾃｷｽﾄﾍﾞｰｽのｶﾗｰ作成
	 *
	 * @param	shape	XSSFSimpleShapeｵﾌﾞｼﾞｪｸﾄ
	 * @param	col		色配列
	 */
	public static void makeShapeColor( final XSSFSimpleShape shape , final int[] col ) {
		if( col != null ) {
			shape.setFillColor(col[0],col[1],col[2]);
		}
	}

	/**
	 * ｱﾌﾟﾘｹｰｼｮﾝのｻﾝﾌﾟﾙです。
	 *
	 * 入力ﾌｧｲﾙ名 は必須で、第一引数固定です。
	 * 第二引数は、処理方法で、-ALL か、-LINE を指定します。何も指定しなければ、-ALL です。
	 * 第三引数を指定した場合は、Encode を指定します。
	 *
	 * Usage: java org.opengion.fukurou.model.POIUtil 入力ﾌｧｲﾙ名 [処理方式] [ｴﾝｺｰﾄﾞ]
	 *   -A(LL)        ･･･ ALL 一括処理(初期値)
	 *   -L(INE)       ･･･ LINE 行単位処理
	 *   -S(heet)      ･･･ Sheet名一覧
	 *   -N(AME)       ･･･ NAME:名前定義
	 *   -C(ellStyle)  ･･･ CellStyle:書式のｽﾀｲﾙ
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.3.0 (2015/05/01) ﾊﾟﾗﾒｰﾀ変更、textReader → extractor に変更
	 * @og.rev 6.2.4.2 (2015/05/29) 引数判定の true/false の処理が逆でした。
	 * @og.rev 6.3.9.0 (2015/11/06) Java 8 ﾗﾑﾀﾞ式に変更
	 *
	 * @param	args	ｺﾏﾝﾄﾞ引数配列
	 */
	public static void main( final String[] args ) {
		final String usageMsg = "Usage: java org.opengion.fukurou.model.POIUtil 入力ﾌｧｲﾙ名 [処理方式] [ｴﾝｺｰﾄﾞ]" + "\n" +
								"\t -A(LL)        ･･･ ALL 一括処理(初期値)      \n" +
								"\t -L(INE)       ･･･ LINE 行単位処理           \n" +
								"\t -S(heet)      ･･･ Sheet名一覧               \n" +
								"\t -N(AME)       ･･･ NAME:名前定義             \n" +
								"\t -C(ellStyle)  ･･･ CellStyle:書式のｽﾀｲﾙ  \n" ;
		if( args.length == 0 ) {
			System.err.println( usageMsg );
			return ;
		}

		final File file = new File( args[0] );
		final char type     = args.length >= 2 ? args[1].charAt(1) : 'A' ;
		final String encode = args.length >= 3 ? args[2] : null ;				// 6.2.4.2 (2015/05/29) true/false の処理が逆でした。

		switch( type ) {
			case 'A' :  if( encode == null ) {
							System.out.println( POIUtil.extractor( file ) );
						}
						else {
							System.out.println( POIUtil.extractor( file,encode ) );
						}
						break;
			// 6.3.9.0 (2015/11/06) Java 8 ﾗﾑﾀﾞ式に変更
			case 'L' : final TextConverter<String,String> conv =
								( val,cmnt ) -> {
									System.out.println( "val=" + val + " , cmnt=" + cmnt );
									return null;
								};

					//		new TextConverter<String,String>() {
					//			/**
					//			 * 入力文字列を、変換します。
					//			 *
					//			 * @param	val  入力文字列
					//			 * @param	cmnt ｺﾒﾝﾄ
					//			 * @return	変換文字列(変換されない場合は、null)
					//			 */
					//			@Override
					//			public String change( final String val , final String cmnt ) {
					//				System.out.println( "val=" + val + " , cmnt=" + cmnt );
					//				return null;
					//			}
					//		};

						if( encode == null ) {
							POIUtil.textReader( file,conv );
						}
						else {
							POIUtil.textReader( file,conv,encode );
						}
						break;
			case 'S' :  final String[] shts = POIUtil.getSheetNames( POIUtil.createWorkbook(file) );
						System.out.println( "No:\tSheetName" );
						for( int i=0; i<shts.length; i++ ) {
							System.out.println( i + "\t" + shts[i] );
						}
						break;
			case 'N' :  final String[] nms = POIUtil.getNames( POIUtil.createWorkbook(file) );
						System.out.println( "No:\tName\tFormula" );
						for( int i=0; i<nms.length; i++ ) {
							System.out.println( i + "\t" + nms[i] );
						}
						break;
			case 'C' :  final String[] sns = POIUtil.getStyleNames( POIUtil.createWorkbook(file) );
						System.out.println( "No:\tStyleName" );
						for( int i=0; i<sns.length; i++ ) {
							System.out.println( i + "\t" + sns[i] );
						}
						break;
			default :   System.err.println( usageMsg );
						break;
		}
	}
}
