/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import java.io.PrintWriter;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.TreeMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Collections;											// 6.4.3.2 (2016/02/19)
import java.util.Calendar;												// 6.9.7.0 (2018/05/14)

import jakarta.servlet.ServletContext;
import jakarta.servlet.FilterRegistration;								// 6.9.3.1 (2018/04/02)
import jakarta.servlet.ServletRegistration;								// 6.9.3.1 (2018/04/02)

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.system.BuildNumber;							// 6.4.2.0 (2016/01/29) hayabusa.common.BuildNumber → fukurou.system.BuildNumber に移動
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.system.ThrowUtil;							// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.DateSet;								// 6.9.7.0 (2018/05/14)
import org.opengion.fukurou.util.FindClassFiles;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.xml.XMLFileLoader;							// 6.0.0.0 (2014/04/11)
import org.opengion.fukurou.system.HybsConst;							// 7.2.3.1 (2020/04/17)
import static org.opengion.fukurou.system.HybsConst.CR;					// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;		// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.DB_FETCH_SIZE;		// 6.9.4.1 (2018/04/09)
import org.opengion.fukurou.fileexec.MainProcess;						// 7.2.5.3 (2020/06/16)

/**
 * ﾛｸﾞｲﾝしたｻｰﾌﾞﾚｯﾄｺﾝﾃｷｽﾄに対応したｼｽﾃﾑﾊﾟﾗﾒｰﾀﾃﾞｰﾀを取得するｸﾗｽです｡
 *
 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀﾃﾞｰﾀ(GE12)は、ﾊﾟﾗﾒｰﾀ(PARAM_ID)に対して、各種設定値(PARAM)を
 * 持っています｡
 * 従来は、resource.ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の一般設定用の ｼｽﾃﾑﾊﾟﾗﾒｰﾀ ﾌｧｲﾙと
 * ｴﾝｼﾞﾝ内部で持っている org/hsgw/hayabusa/resource/properties の
 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ ﾌｧｲﾙにより維持されていましたが、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ
 * 定義ﾃｰﾌﾞﾙ(GE12)と、commom/SystemData.java ｸﾗｽによる管理に変更されました｡
 *
 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀは、DBへのｱｸｾｽを決定するため、初期設定値を定義する必要があります｡
 * これは、ｱﾌﾟﾘｹｰｼｮﾝ・ﾃﾞｨｽｸﾘﾌﾟﾀ(WEB-INF/web.xml)に、context-param として、
 * ｷｰと値のｾｯﾄで、初期ｱｸｾｽ用の情報を渡します｡
 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ定義ﾃｰﾌﾞﾙ(GE12)には、SYSTEM_IDとして、通常のｼｽﾃﾑIDと、
 * ｴﾝｼﾞﾝﾊﾟﾗﾒｰﾀがあります｡ｴﾝｼﾞﾝﾊﾟﾗﾒｰﾀは、SYSTEM_ID='**'として、登録
 * されています｡
 *
 * <table class="plain">
 *   <caption>ｼｽﾃﾑﾊﾟﾗﾒｰﾀの説明</caption>
 *   <tr><th>種類       </th><th>SYSTEM_ID</th><th>作成区分</th><th>説明</th></tr>
 *   <tr><td>ｴﾝｼﾞﾝ共通	</td><td>**		</td><td>0:ｴﾝｼﾞﾝ</td><td>ｴﾝｼﾞﾝ共通で定義しているﾊﾟﾗﾒｰﾀ			</td></tr>
 *   <tr><td>ｴﾝｼﾞﾝ個別	</td><td>個別	</td><td>0:ｴﾝｼﾞﾝ</td><td>ｼｽﾃﾑ毎にｴﾝｼﾞﾝが登録しているﾊﾟﾗﾒｰﾀ		</td></tr>
 *   <tr><td>ｼｽﾃﾑ共通	</td><td>**		</td><td>1:ｼｽﾃﾑ	</td><td>ｼｽﾃﾑ毎にｴﾝｼﾞﾝが登録しているﾊﾟﾗﾒｰﾀ		</td></tr>
 *   <tr><td>ｼｽﾃﾑ個別	</td><td>個別	</td><td>1:ｼｽﾃﾑ	</td><td>ｼｽﾃﾑ毎に各自が独自に登録しているﾊﾟﾗﾒｰﾀ	</td></tr>
 * </table>
 *
 * <table class="plain">
 *   <caption>ｱﾌﾟﾘｹｰｼｮﾝ・ﾃﾞｨｽｸﾘﾌﾟﾀ(WEB-INF/web.xml)設定情報</caption>
 *   <tr><th>ﾊﾟﾗﾒｰﾀ		</th><th>設定値例				</th><th>解説					</th></tr>
 *   <tr><td>SYSTEM_ID	</td><td>GE						</td><td>このｱﾌﾟﾘｹｰｼｮﾝのｼｽﾃﾑID	</td></tr>
 *   <tr><td>TOMCAT_PORT</td><td>8823					</td><td>Tomcat起動時ﾎﾟｰﾄ番号	</td></tr>
 *   <tr><td>LOG_FILE	</td><td>log/log_$(yyyyMMdd).txt</td><td>JSPｱｸｾｽﾛｸﾞ出力先		</td></tr>
 * </table>
 *
 * ※ 5.6.7.0 (2013/07/27)
 *    InitFileLoader により、特定のｸﾗｽﾊﾟｽ内の 拡張XDK 形式の xml ﾌｧｲﾙ を処理します｡
 *    ｸﾗｽﾊﾟｽが、"/resource" 以下のすべての xml ﾌｧｲﾙは、DBIDが RESOURCE の接続先に対して処理されます｡
 *    ｸﾗｽﾊﾟｽが、"/xml" 以下のすべての xml ﾌｧｲﾙは、DBIDが DEFAULT の接続先に対して処理されます｡
 *    各ｸﾗｽﾊﾟｽ以下のﾌｧｲﾙは、実ﾌｫﾙﾀﾞでも、jar形式に圧縮された形式でも処理されます｡
 *
 * ※ 6.0.0.0 (2014/04/11)
 *    InitFileLoader が廃止になり、代わりに、XMLFileLoader を使用します｡処理自体は、ほぼ同様です｡
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.rev 4.0.0.0 (2007/10/26) loadDBResourceのｺﾈｸｼｮﾝをFactoryから取るように変更
 * @og.group 初期化
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public final class SystemParameter {

	/** plugin ｸﾗｽの取得用ｸﾗｽﾊﾟｽ指定	{@value}	*/
	public static final String PLUGIN = "org/opengion/plugin";

	// 4.0.0.0 (2007/10/05) CONTXT_PATH に、DEFAULT '**' NOT NULL 制約を付ける｡
	/** ｼｽﾃﾑ共通/個別ﾊﾟﾗﾒｰﾀ(SYSTEM_ID in ('**',?) and KBSAKU &gt; '0')の一括読込のｸｴﾘｰ	{@value}	*/
	public static final String QUERY = "SELECT PARAM_ID,PARAM,FGCRYPT"
									+ " FROM GE12 WHERE SYSTEM_ID IN (?,'**')"
									+ " AND CONTXT_PATH IN (?,'**')"
									+ " AND FGJ='1' AND KBSAKU > '0'"
									+ " ORDER BY SYSTEM_ID,CONTXT_PATH,FGJ,KBSAKU,SEQNO" ;

	/** ｼｽﾃﾑ共通/個別ﾊﾟﾗﾒｰﾀ(SYSTEM_ID in ('**',?) and KBSAKU &gt; '0')の一括登録のｸｴﾘｰ	{@value}	*/
	public static final String INS_SYS = "INSERT INTO GE12"
//									+ " ( SYSTEM_ID,CONTXT_PATH,PARAM_ID,SEQNO,PARAM,TITLE,CONTENTS,PARAM_LVL,FGJ,KBSAKU )"
									+ " ( SYSTEM_ID,CONTXT_PATH,PARAM_ID,SEQNO,PARAM,TITLE,CONTENTS,PARAM_LVL,FGJ,KBSAKU,DYSET )"	// 6.9.7.0 (2018/05/14)
									+ " SELECT"
//									+ " ?,?,?,?,?,TITLE,CONTENTS,PARAM_LVL,'1','0'"
									+ " ?,?,?,?,?,TITLE,CONTENTS,PARAM_LVL,'1','0',?"												// 6.9.7.0 (2018/05/14)
									+ " FROM GE12 WHERE SYSTEM_ID='**' AND FGJ='1' AND KBSAKU='0' AND PARAM_ID=?" ;

//	/** ｴﾝｼﾞﾝ個別(SYSTEM_ID='個別' KBSAKU='0' CONTXT_PATH='自身')ﾊﾟﾗﾒｰﾀの一括削除のｸｴﾘｰ	{@value}	*/
//	public static final String DEL_SYS = "DELETE FROM GE12 WHERE SYSTEM_ID=? AND KBSAKU='0' AND CONTXT_PATH=?";

	/** ｴﾝｼﾞﾝ個別(SYSTEM_ID='個別' KBSAKU='0' CONTXT_PATH='自身')ﾊﾟﾗﾒｰﾀの一括削除と、過去情報の削除ｸｴﾘｰ	{@value}	*/
	public static final String DEL_SYS = "DELETE FROM GE12 WHERE ( SYSTEM_ID=? AND KBSAKU='0' AND CONTXT_PATH=? )"					// 6.9.7.0 (2018/05/14)
									+ " OR ( SYSTEM_ID != '**' AND PARAM_LVL='9' AND ( DYSET < ? OR DYSET IS NULL ) )" ;			// 過去のﾃﾞｰﾀを考慮(DYSET is null)

	/** 6.9.7.0 (2018/05/14) DYSET ﾌｫｰﾏｯﾄ	{@value}	*/
	public static final String DYSET_FORMAT = "yyyyMMddHHmmss" ;			// 6.9.7.0 (2018/05/14)

//	/** 6.9.3.0 (2018/03/26) ﾃﾞｰﾀ検索時のﾌｪｯﾁｻｲｽﾞ	{@value}	*/
//	private static final int DB_FETCH_SIZE = 1001 ;							// HybsSystem に設定する前なので、HybsSystem.sysInt では取得できない。

	/** ｼｽﾃﾑﾊﾟﾗﾒｰﾀ構築中のｴﾗｰをｾｯﾄしていきます｡ */
	private static final List<String> ERR_MSG_LIST = new ArrayList<>();		// 6.4.1.1 (2016/01/16) errMsgList → ERR_MSG_LIST refactoring

	/**
	 *	ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰをprivateにして、
	 *	ｵﾌﾞｼﾞｪｸﾄの生成をさせないようにする｡
	 *
	 */
	private SystemParameter() {}

	/**
	 * ｺﾝﾃｷｽﾄ毎に ｼｽﾃﾑﾊﾟﾗﾒｰﾀｵﾌﾞｼﾞｪｸﾄを作成します｡
	 * このｸﾗｽのｽﾀｰﾄﾎﾟｲﾝﾄﾒｿｯﾄﾞになります｡
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) ﾛｸﾞ出力先の変更(全てLogWriter経由で出力)
	 * @og.rev 5.5.4.4 (2012/07/20) LogWriter.log 追加
	 * @og.rev 5.5.4.4 (2012/07/20) SQLException は、catch しているので、loadDBResource からは、throws されない｡
	 * @og.rev 5.7.2.0 (2014/01/10) Throwable の情報をもう少し詳細に出します｡
	 * @og.rev 6.4.3.2 (2016/02/19) engParamとsysParamを初期設定して、同期化しておく｡
	 * @og.rev 6.4.7.1 (2016/06/17) ｾｷｭﾘﾃｨの関連で、ﾛｸﾞの詳細を出していませんでしたが、出すように変更します｡
	 * @og.rev 6.8.6.0 (2018/01/19) ImageIO.write で、Java9で、NumberFormatException が出てしまう(暫定)対策｡
	 * @og.rev 7.2.5.3 (2020/06/16) MainProcessは、SystemParameterへ移動
	 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します｡
	 *
	 * @param	context	Servletｺﾝﾃｷｽﾄ
	 *
	 * @return	ｼｽﾃﾑﾊﾟﾗﾒｰﾀのﾏｯﾌﾟ
	 */
	public static Map<String,String> makeSystemParameter( final ServletContext context ) {
		ERR_MSG_LIST.clear() ;	// 呼び出し都度、ｴﾗｰﾘｽﾄをｸﾘｱします｡

		// 6.4.3.2 (2016/02/19)
		final Map<String,String> engParam = new LinkedHashMap<>();	// 最終的にﾏｰｼﾞして返す。
		final Map<String,String> sysParam = new LinkedHashMap<>();	// DB書き戻し等あるので、設定中は分けておく。

		try {
			// 6.8.6.0 (2018/01/19) ImageIO.write (暫定)対策
			// ※ ImageIO.getReaderMIMETypes(); での Java 9 has java.specification.version == 9 で、
			// Caused by: java.lang.NumberFormatException: For input string: "" が発生する対応
			// https://github.com/jai-imageio/jai-imageio-core/issues/24
			final String javaSpecVer = System.getProperty( "java.specification.version" );
			if( !javaSpecVer.startsWith( "1." ) ) {
				System.setProperty( "java.specification.version" , "1." + javaSpecVer );	// "java.specification.version" , "9" の場合、"1.9" に無理やり変更します。
			}

			final String contextName = getContextName( context );

			System.out.println( "Context Initialized [" + contextName + "] " + new Date() );
			System.out.print( "  Version [" + BuildNumber.VERSION_NO + " " );
			System.out.print( BuildNumber.BUILD_TYPE );
			System.out.println( "]" );

			// ｼｽﾃﾑﾃﾞｰﾀｸﾗｽより、ｴﾝｼﾞﾝﾊﾟﾗﾒｰﾀ情報を初期設定します｡
			// ｴﾝｼﾞﾝﾊﾟﾗﾒｰﾀは、SystemData ｸﾗｽの public static ﾒﾝﾊﾞｰです｡
			loadParameter( engParam , SystemData.class );				// 6.4.3.2 (2016/02/19)

			// ｺﾝﾃｷｽﾄより取得できるｼｽﾃﾑﾊﾟﾗﾒｰﾀ情報を取得します｡
			// web.xml で設定したﾊﾟﾗﾒｰﾀを取得します｡
			// SYSTEM_ID,DB_DRIVER,DB_URL などは、loadDBResource で使用します｡
			loadInitialParameter( sysParam , context , contextName );	// 6.4.3.2 (2016/02/19)

			// ｼｽﾃﾑ個別に設定される、ｴﾝｼﾞﾝ起動時情報を初期設定します｡
			// ｴﾝｼﾞﾝ起動時情報は、BuildNumber ｸﾗｽの public static ﾒﾝﾊﾞｰです｡
			loadParameter( sysParam , BuildNumber.class );				// 6.4.3.2 (2016/02/19)
			loadParameter( sysParam , PLUGIN );

			// GE12 ﾃﾞｰﾀﾍﾞｰｽより読み取ります｡
			// 引数のMapに読み込んだ値を追加した Map を返します｡
			// つまり、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ情報の上書きを行います｡
			// 5.5.4.4 (2012/07/20) SQLException は、catch しているので、loadDBResource からは、throws されない。
	//		try {
				loadDBResource( sysParam );								// 6.4.3.2 (2016/02/19)
	//		}
	//		catch( final SQLException ex ) {
	//		final String errMsg = "DB終了(close)処理を実行できませんでした。" + CR
	//						+ ex.getMessage() + ":" + ex.getSQLState() ;
	//			LogWriter.log( ex );					// 5.5.4.4 (2012/07/20) LogWriter.log 追加
	//			ERR_MSG_LIST.add( errMsg );
	//		}

			// 7.2.5.3 (2020/06/16) MainProcessは、SystemParameterへ移動
			// 7.2.5.0 (2020/06/01) org.opengion.fukurou.fileexec.MainProcess処理
			// InitialCallURL() の起動が一番最後なので、その前に入れます。
			if( "GE".equalsIgnoreCase( contextName ) ) {
				if( "true".equalsIgnoreCase( sysParam.get( "USE_FILEEXEC" ) ) ) {
					MainProcess.start();
				}
				else {
					MainProcess.shutdown( false );
				}
			}

			// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
			// GE12の初期値のSQLは、KBSAKU > '0' しか取得しないので、初期値を、engParam から取ってくる。
			final String JSP		= sysParam.get( "JSP" );
			final String ICON_DIR	= StringUtil.nval( sysParam.get( "ICON_DIR"  ),engParam.get( "ICON_DIR"  ) );
			final String IMAGE_DIR	= StringUtil.nval( sysParam.get( "IMAGE_DIR" ),engParam.get( "IMAGE_DIR" ) );
			sysParam.put( "JSP_ICON",JSP+"/"+ICON_DIR  );
			sysParam.put( "JSP_IMG" ,JSP+"/"+IMAGE_DIR );
		}
		catch( final Throwable th ) {					// 3.6.1.0 (2005/01/05)
			LogWriter.log( th );						// 5.5.4.4 (2012/07/20) LogWriter.log 追加
			// 5.7.2.0 (2014/01/10) Throwable の情報をもう少し詳細に出します。
			final String errMsg = "処理を実行できませんでした。"	+ CR ;
	//					+ th.getMessage()						+ CR
	//					+ StringUtil.ogStackTrace( th ) ;
	//		ERR_MSG_LIST.add( errMsg );
	//		final String errMsg = "処理を実行できませんでした。"	+ CR ;			// 6.4.7.1 (2016/06/17)
			ERR_MSG_LIST.add( ThrowUtil.ogStackTrace( errMsg,th ) );				// 6.4.2.0 (2016/01/29)
		}
		finally {
			// 初期値のｴﾝｼﾞﾝﾊﾟﾗﾒｰﾀに個別のｼｽﾃﾑﾊﾟﾗﾒｰﾀを追加設定します｡
			// つまり、ｴﾝｼﾞﾝﾊﾟﾗﾒｰﾀ情報に上書きを行います｡
			// 6.3.9.1 (2015/11/27) engParam が null になることは無い(findbugs)。
				engParam.putAll( sysParam );

			final int errCnt = ERR_MSG_LIST.size();
			if( errCnt > 0 ) {
				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
					.append( "【システムパラメータの初期化に失敗しました。】" )
					.append( CR )
					.append( "Tomcat の設定状況をご確認ください。" )
					.append( CR )
					.append( "========================================" )
					.append( CR );

				for( int i=0; i<errCnt; i++ ) {
					buf.append( ERR_MSG_LIST.get(i) )
						.append( CR )
						.append( "----------------------------------------" )
						.append( CR );
				}
				System.out.println( buf );
				engParam.put( HybsSystem.LOCAL_CONTX_ERR_KEY,buf.toString() );
			}
		}
		return Collections.synchronizedMap( engParam ) ;		// 6.4.3.2 (2016/02/19) 同期化しておく。
	}

	/**
	 * ｺﾝﾃｷｽﾄより、ｱﾌﾟﾘｹｰｼｮﾝ設定情報を取得します｡
	 * 初期値は、ｱﾌﾟﾘｹｰｼｮﾝ・ﾃﾞｨｽｸﾘﾌﾟﾀ(WEB-INF/web.xml)に
	 * context-param として、ｷｰと値のｾｯﾄで、初期ｱｸｾｽ用の情報を渡します｡
	 * ﾃﾞｰﾀﾍﾞｰｽへの接続は、WEB-INF の DBConfig.xml で設定された情報を使用します｡
	 *
	 * ここでは、各ｺﾝﾃｷｽﾄ毎の内部情報を取得します｡その中には、
	 * BuildNumber ｸﾗｽで定義されている各種ﾌｨｰﾙﾄﾞ属性も含まれます｡
	 *
	 * REAL_PATH      : ｱﾄﾞﾚｽ(/)に対する、実ﾃﾞｨﾚｸﾄﾘﾊﾟｽ
	 * CONTEXT_NAME   : ｱｸｾｽ先の仮想ﾌｫﾙﾀﾞ名(URLのｺﾝﾃｷｽﾄ名)
	 * JSP            : ｱｸｾｽ先のJSPﾌｫﾙﾀﾞ名(/URLのｺﾝﾃｷｽﾄ名/jsp)
	 * SYS_LOG_URL    : jsp以下のﾛｸﾞﾌｧｲﾙ名										// 7.1.0.0 (2020/01/20) jsp以下のﾛｸﾞﾌｧｲﾙ名
	 * SYSTEM_ID      : web.xml で指定する、SYSTEM_ID の設定値
	 * TOMCAT_PORT    : web.xml で指定する、Tomcat起動時ﾎﾟｰﾄ番号(8823)
	 * LOG_FILE       : web.xml で指定する、JSPｱｸｾｽﾛｸﾞ出力先(log/log_$(yyyyMMdd).txt)
	 * SERVLET_INFO   : ｻｰﾌﾞﾚｯﾄ情報 [例： Apache Tomcat/5.5.9 ]
	 * TOMCAT_WORK    : Tomcatﾜｰｸの位置 [例： H:\java\tomcat5.5.17\work\Catalina\localhost\ver4 ]
	 * TOMCAT_HOME    : Tomcat環境の位置 [例： H:\java\tomcat5.5.17 ]
	 * LOGS_DIR       : Tomcatのﾛｸﾞﾌｫﾙﾀﾞ										// 7.1.0.0 (2020/01/20) Tomcatのﾛｸﾞﾌｫﾙﾀﾞ
	 * JAVA_HOME      : Java実行環境の位置 [例： H:\java\jdk150\jre ]
	 * HOST_NAME      : ﾎｽﾄの名前
	 * HOST_ADRS      : ﾎｽﾄのｱﾄﾞﾚｽ
	 * SERVER_INFO    : ｻｰﾊﾞｰ情報 [例： HN50G5 ( 200.1.50.165 ) ]
	 * ENGINE_INFO    : ﾊﾞｰｼﾞｮﾝ番号 [例： 4.3.6.6 ]
	 * RESOURCE_DBID  : "RESOURCE" 固定値を設定｡WEB-INF/web.xml で指定しても無効です｡
	 * OFFICE_HOME    : ｵﾌｨｰｽのHOME												// 7.1.0.0 (2020/01/20) ｵﾌｨｰｽのHOME
	 * EXT_DIR        : 拡張jarﾌｫﾙﾀﾞ											// 7.1.0.0 (2020/01/20) 拡張jarﾌｫﾙﾀﾞ
	 * IS_SERVICE     : ｻｰﾋﾞｽ起動かどうか [ｻｰﾋﾞｽ起動時はtrue]					// 7.1.0.0 (2020/01/20) ｻｰﾋﾞｽ起動
	 *
	 * @og.rev 4.1.0.0 (2007/12/27) web.xmlからTOMCAT_PORTを読む処理を追加
	 * @og.rev 4.2.0.0 (2008/02/18) TOMCAT_PORTを環境変数から取得するよう変更
	 * @og.rev 4.2.0.0 (2008/02/20) web.xmlでSYSTEM_IDが空白の場合に大文字ｺﾝﾃｷｽﾄ名が設定されるよう変更
	 * @og.rev 4.3.6.6 (2009/05/15) ｺﾝﾃｷｽﾄ単位にｴﾝｼﾞﾝﾊﾞｰｼﾞｮﾝ情報を持つ(ﾊﾞｰｼﾞｮﾝｱｯﾌﾟ判定用)
	 * @og.rev 5.6.7.1 (2013/08/09) RESOURCE_DBID の値に、"RESOURCE" を設定しておきます｡
	 * @og.rev 5.6.7.3 (2013/08/23) TOMCAT_HOME を追加
	 * @og.rev 5.7.3.2 (2014/02/28) Tomcat8 対応｡getRealPath( "/" ) の互換性のための修正｡
	 * @og.rev 6.2.4.1 (2015/05/22) REAL_PATH 対応｡realPath は、HybsSystem経由で、取得する｡
	 * @og.rev 6.4.3.2 (2016/02/19) returnをやめて、引数のMapに直接値を設定します｡
	 * @og.rev 6.9.0.1 (2018/02/05) RequestParam で設定される値を、できるだけ構築しておきます｡
	 * @og.rev 6.9.3.1 (2018/04/02) web.xml で定義されている filter、servlet のｷｰ名称を取得し、あれば、true を設定します｡
	 * @og.rev 7.1.0.0 (2020/01/20) OFFICE_HOME,EXT_DIR,IS_SERVICE 環境変数の追加
	 * @og.rev 7.2.3.1 (2020/04/17) System.getenv → HybsConst.getenv 変更(ｻｰﾋﾞｽ化対応)
	 * @og.rev 8.4.0.0 (2022/12/23) HybsConst.getenv の 初期値使用
	 *
	 * @param	map			読み書きするMapｵﾌﾞｼﾞｪｸﾄ
	 * @param	context		Servletｺﾝﾃｷｽﾄ
	 * @param	contextName	ｺﾝﾃｷｽﾄ名
	 */
	private static void loadInitialParameter( final Map<String,String> map,final ServletContext context,final String contextName ) {
		// ｺﾝﾃｷｽﾄの初期化ﾊﾟﾗﾒｰﾀ他の情報を登録しておきます｡
		final Enumeration<?> enume = context.getInitParameterNames() ;			// 4.3.3.6 (2008/11/15) Generics警告対応
		while( enume.hasMoreElements() ) {
			final String key = (String)enume.nextElement();
			String val = context.getInitParameter( key );
			if( val != null && val.isEmpty() ) { val = null; }
			map.put( key,val );
		}

		// SYSTEM_IDがnullの場合は大文字のｺﾝﾃｷｽﾄ名を設定
		if( map.get( "SYSTEM_ID" ) == null ){ // 4.2.0.0 (2008/02/20)
			map.put( "SYSTEM_ID", contextName.toUpperCase( Locale.JAPAN ) );
		}

		// 6.2.4.1 (2015/05/22) REAL_PATH 対応。realPath は、HybsSystem経由で、取得する。
		final String REAL_PATH = context.getRealPath( "" ) ;					// 7.1.0.0 (2020/01/20)
//		HybsSystem.setRealPath( context.getRealPath( "" ) );
		HybsSystem.setRealPath( REAL_PATH );									// 7.1.0.0 (2020/01/20)

		// 各種ｼｽﾃﾑ情報を登録しておきます｡
//		map.put( "REAL_PATH"	,HybsSystem.getRealPath() );					// 6.2.4.1 (2015/05/22)
		map.put( "REAL_PATH"	,REAL_PATH );									// 6.2.4.1 (2015/05/22) 7.1.0.0 (2020/01/20)
		map.put( "CONTEXT_NAME"	,contextName );
		map.put( "JSP"			,"/" + contextName + "/jsp" );
//		map.put( "SYS_LOG_URL"	,REAL_PATH + System.getenv( "SYS_LOG_URL" ) );	// 7.1.0.0 (2020/01/20) jsp以下のﾛｸﾞﾌｧｲﾙ名

		map.put( "SERVLET_INFO"	,context.getServerInfo() );
		map.put( "TOMCAT_WORK"	,String.valueOf( context.getAttribute( "jakarta.servlet.context.tempdir" ) ) );

		// 7.1.0.0 (2020/01/20) TOMCAT_HOME を先に取得しておく。
		final String TOMCAT_HOME = System.getProperty( "catalina.home" );
//		map.put( "TOMCAT_HOME"	,System.getProperty( "catalina.home" ) );						// 5.6.7.3 (2013/08/23)
		map.put( "TOMCAT_HOME"	,TOMCAT_HOME );													// 7.1.0.0 (2020/01/20) TomcatのHOMEﾌｫﾙﾀﾞ
		map.put( "TOMCAT_LOGS"	,TOMCAT_HOME + "/logs" );										// 7.1.0.0 (2020/01/20) Tomcatのﾛｸﾞﾌｫﾙﾀﾞ
//		map.put( "LOGS_DIR"		,System.getProperty( "logsDir" ) );								// 7.1.0.0 (2020/01/20) Tomcatのﾛｸﾞﾌｫﾙﾀﾞ(ｺﾝﾃｷｽﾄ単位)
//		map.put( "SERVICE_NAME"	,StringUtil.nval( System.getenv( "SERVICE_NAME" ),"" ) );		// 7.1.0.0 (2020/01/20) ｻｰﾋﾞｽ起動時のｻｰﾋﾞｽ名
//		map.put( "SERVICE_NAME"	,StringUtil.nval( HybsConst.getenv( "SERVICE_NAME" ),"" ) );	// 7.2.3.1 (2020/04/17) ｻｰﾋﾞｽ起動時のｻｰﾋﾞｽ名
		map.put( "SERVICE_NAME"	,HybsConst.getenv( "SERVICE_NAME","" ) )				;		// 8.4.0.0 (2022/12/23) ｻｰﾋﾞｽ起動時のｻｰﾋﾞｽ名 getenvの初期値を使用

		map.put( "JAVA_HOME"	,System.getProperty( "java.home" ) );

		map.put( "HOST_NAME"	,HybsSystem.HOST_NAME );
		map.put( "HOST_ADRS"	,HybsSystem.HOST_ADRS );
		map.put( "SERVER_INFO"	,HybsSystem.HOST_NAME + " ( " + HybsSystem.HOST_ADRS + " )" );
		map.put( "ENGINE_INFO"	,BuildNumber.ENGINE_INFO );
		// 5.6.7.1 (2013/08/09) RESOURCE_DBID の値に、"RESOURCE" を設定しておきます。
		map.put( "RESOURCE_DBID", "RESOURCE" );

		// 7.1.0.0 (2020/01/20) OFFICE_HOME,EXT_DIR 環境変数の追加
//		map.put( "OFFICE_HOME"	,System.getenv( "OFFICE_HOME" ) );				// 7.1.0.0 (2020/01/20) ｵﾌｨｰｽのHOME
		map.put( "OFFICE_HOME"	,HybsConst.getenv( "OFFICE_HOME" ) );			// 7.2.3.1 (2020/04/17) ｵﾌｨｰｽのHOME
//		map.put( "EXT_DIR"		,System.getenv( "EXT_DIR" ) );					// 7.1.0.0 (2020/01/20) 拡張jarﾌｫﾙﾀﾞ
		map.put( "EXT_DIR"		,HybsConst.getenv( "EXT_DIR" ) );				// 7.2.3.1 (2020/04/17) 拡張jarﾌｫﾙﾀﾞ

		// 7.1.0.0 (2020/01/20) ﾊﾞｯﾁ起動の場合は、sun.java.command = "org.apache.catalina.startup.Bootstrap start" , ｻｰﾋﾞｽ起動時は、未設定
		final boolean IS_SERVICE = null == System.getProperty( "sun.java.command" );
		map.put( "IS_SERVICE"	, String.valueOf( IS_SERVICE ) );				// 7.1.0.0 (2020/01/20) ｻｰﾋﾞｽなら、true

		// 6.3.9.1 (2015/11/27) final化にともない、ﾛｼﾞｯｸ整理
//		final String TOMCAT_PORT = StringUtil.nval( System.getenv( "CONNECTOR_PORT" ) , map.get( "TOMCAT_PORT" ) );
//		final String TOMCAT_PORT = StringUtil.nval( HybsConst.getenv( "CONNECTOR_PORT" ) , map.get( "TOMCAT_PORT" ) );	// 7.2.3.1 (2020/04/17)
		final String TOMCAT_PORT = HybsConst.getenv( "CONNECTOR_PORT" , map.get( "TOMCAT_PORT" ) );	// 8.4.0.0 (2022/12/23) getenvの初期値を使用

		// 6.3.9.1 (2015/11/27) final化にともない、ﾛｼﾞｯｸ整理
		final String HOST_URL = TOMCAT_PORT == null || TOMCAT_PORT.isEmpty()
									? "**" : HybsSystem.HOST_NAME + ":" + TOMCAT_PORT + "/" + contextName + "/";
		map.put( "HOST_URL", HOST_URL );

		// 6.9.0.1 (2018/02/05) RequestParam で設定される値を、できるだけ構築しておきます。
		map.put( "SERVER_NAME"	, "localhost" );						// HybsSystem.HOST_NAME でも良いかも。
		map.put( "SERVER_URL"	, "http://localhost:" + TOMCAT_PORT + "/" );
		map.put( "CONTEXT_URL"	, "http://localhost:" + TOMCAT_PORT + "/" + contextName + "/" );

		// 6.9.3.1 (2018/04/02) web.xml で定義されている filter、servlet のｷｰ名称を取得し、あれば、true を設定します｡
		// キーワードは、USE_(大文字のフィルター名) です。
		final Map<String,? extends FilterRegistration> filterMap = context.getFilterRegistrations();
		for( final String fkey : filterMap.keySet() ) {
			map.put( "USE_" + fkey.toUpperCase( Locale.JAPAN ) , "true" );
		}

		// 6.9.3.1 (2018/04/02) web.xml で定義されている filter、servlet のｷｰ名称を取得し、あれば、true を設定します｡
		// ｷｰﾜｰﾄﾞは、USE_(大文字のﾌｨﾙﾀｰ名) です｡
		final Map<String,? extends ServletRegistration> servletMap = context.getServletRegistrations();
		for( final String skey : servletMap.keySet() ) {
			map.put( "USE_" + skey.toUpperCase( Locale.JAPAN ) , "true" );
		}

		System.out.println( "    Load Initial Parameter [" + map.size() + "] finished." );
	}

	/**
	 * ｱﾌﾟﾘｹｰｼｮﾝ個別に設定しているﾘｿｰｽDB(GE12)を取得します｡
	 *
	 * ﾃﾞｰﾀﾍﾞｰｽへの接続は、WEB-INF の DBConfig.xml で設定された情報を元に、
	 * org.opengion.fukurou.db.ConnectionFactory で接続先を取得します｡
	 * ここでは、web.xml で定義された各ｱﾌﾟﾘｹｰｼｮﾝ個別のﾊﾟﾗﾒ―ﾀを取得します｡
	 * SYSTEM_ID(必須) です｡
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) 接続先情報の管理見直し(ｺﾝﾃｷｽﾄ初期設定)
	 * @og.rev 4.0.0.0 (2007/10/26) ｺﾈｸｼｮﾝをﾌｧｸﾄﾘｰから取ってくるように変更
	 * @og.rev 4.3.6.5 (2009/05/08) dataﾊﾟｽ内のXMLﾌｧｲﾙも読み取るようにする
	 * @og.rev 4.3.6.6 (2009/05/15) ↑を廃止｡自動ｲﾝｽﾄｰﾙ対応｡
	 * @og.rev 5.1.2.0 (2010/01/01) connection.setAutoCommit は、ConnectionFactory で設定済みなので、ｺﾒﾝﾄｱｳﾄ
	 * @og.rev 5.1.9.0 (2010/08/01) 自動ｲﾝｽﾄｰﾙの設定見直し(画面からのｲﾝｽﾄｰﾙ対応)
	 * @og.rev 5.5.4.4 (2012/07/20) SQLException は、catch しているので、このﾒｿｯﾄﾞからは、throws されない｡
	 * @og.rev 5.5.4.5 (2012/07/27) 初期起動時のDB接続先は、RESOURCE_DBID とする｡
	 * @og.rev 5.6.6.1 (2013/07/12) xml ﾊﾟｽ内のXMLﾌｧｲﾙがあれば、DB登録します｡
	 * @og.rev 5.6.7.0 (2013/07/27) InitFileLoader で、resource以下は、DBID=RESOURCE xml以下は、DBID=DEFAULT に登録します｡
	 * @og.rev 5.6.7.3 (2013/08/23) DBID=RESOURCE 漏れ
	 * @og.rev 5.7.2.0 (2014/01/10) RuntimeException は、catch しないようにします｡
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述｡
	 * @og.rev 6.4.3.2 (2016/02/19) returnをやめて、引数のMapに直接値を設定します｡
	 * @og.rev 6.4.7.1 (2016/06/17) ｾｷｭﾘﾃｨの関連で、ﾛｸﾞの詳細を出していませんでしたが、出すように変更します｡
	 * @og.rev 6.9.3.0 (2018/03/26) ﾃﾞｰﾀ検索時のﾌｪｯﾁｻｲｽﾞを設定｡
	 * @og.rev 8.5.0.0 (2023/04/21) 対象外の resource を読込んでいた為、判定条件を変更
	 *
	 * @param	sysParam	入力ｼｽﾃﾑﾊﾟﾗﾒｰﾀﾏｯﾌﾟ
	 */
	private static void loadDBResource( final Map<String,String> sysParam ) {
		final String SYSTEM_ID		= sysParam.get( "SYSTEM_ID" );

		// 必須項目ﾁｪｯｸを行います｡SYSTEM_IDは必須です｡
		// これは、web.xml で定義が必要です。
		// 4.0.0.0 (2007/10/23)接続情報XML化につきDB_URLﾁｪｯｸを削除
		if( SYSTEM_ID == null || SYSTEM_ID.isEmpty() ) {
			final String errMsg = "システムパラメータの必須項目(SYSTEM_ID,DB_URL)が null です。" + CR
							+ "SYSTEM_ID=[" + SYSTEM_ID + "] " + CR
							+ "Versino=[" + BuildNumber.VERSION_NO + "] " + CR ;
			ERR_MSG_LIST.add( errMsg );
			return ;										// 6.4.3.2 (2016/02/19)
		}

		Connection	defConn			= null;					// 5.6.7.0 (2013/07/27) DBID=DEFAULT  のｺﾈｸｼｮﾝ
		Connection	rscConn			= null;					// 5.6.7.0 (2013/07/27) DBID=RESOURCE のｺﾈｸｼｮﾝ
		boolean		errFlag			= true;
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String CONTEXT_NAME	= sysParam.get( "CONTEXT_NAME" );	// ｺﾝﾃｷｽﾄ別ｼｽﾃﾑﾘｿｰｽ
		final String HOST_URL		= sysParam.get( "HOST_URL" );		// 4.1.0.0 (2007/12/21) TOMCATへのｱｸｾｽ用
		final String RESOURCE_DBID	= sysParam.get( "RESOURCE_DBID" );	// 5.5.4.5 (2012/07/27) 初期起動時のDB接続先
		try {

			// 4.0.0.0(2007/10/25)ConnectionをConnectionFactory経由で取得するように変更する。
			// ｺﾝﾃｷｽﾄ名で接続しにいく｡ApplicationInfoは使わないのでnull
			ConnectionFactory.init( CONTEXT_NAME, null );	// ConnectionFactoryの初期化

			defConn = ConnectionFactory.connection( null, null );			// 5.6.7.0 (2013/07/27) DBID=DEFAULT のｺﾈｸｼｮﾝ
			rscConn = ConnectionFactory.connection( RESOURCE_DBID, null );	// 5.6.7.0 (2013/07/27) DBID=RESOURCE のｺﾈｸｼｮﾝ

			// 4.3.6.6 (2009/05/15)
			// 5.1.9.0 (2010/08/01) 自動ｲﾝｽﾄｰﾙの設定見直し(画面からのｲﾝｽﾄｰﾙ対応)
			final SystemInstaller installer = new SystemInstaller( defConn , rscConn , new PrintWriter( System.out, true ) );
			installer.autoInsUpd( SYSTEM_ID, CONTEXT_NAME, HOST_URL );

			// resource ﾊﾟｽ内のXMLﾌｧｲﾙがあれば、先にDB登録します｡(DBID=RESOURCE)
			// 6.0.0.0 (2014/04/11) XMLFileLoader に変更｡
			// 7.0.1.4 (2018/11/26) XMLFileLoader をfinal宣言(どうせ、new してるので使い回しを止める｡)
//			XMLFileLoader loader = new XMLFileLoader( rscConn,true );			// ﾘｿｰｽｺﾈｸｼｮﾝとuseTimeStamp=true 指定
//			loader.loadClassPathFiles( "resource" ) ;
			final XMLFileLoader rscLoad = new XMLFileLoader( rscConn,true );	// ﾘｿｰｽｺﾈｸｼｮﾝとuseTimeStamp=true 指定
//			rscLoad.loadClassPathFiles( "resource" ) ;
			// 8.5.0.0 (2023/04/21) C:/opengionVX/apps/LibreOfficeX.X.X/program/resource を対象外にする
			rscLoad.loadClassPathFiles( "resource", "webapps" ) ;

			// 5.6.6.1 (2013/07/12) xml ﾊﾟｽ内のXMLﾌｧｲﾙがあれば、DB登録します｡
			// 6.0.0.0 (2014/04/11) XMLFileLoader に変更｡
			// 7.0.1.4 (2018/11/26) XMLFileLoader をfinal宣言(どうせ、new してるので使い回しを止める｡)
//			loader = new XMLFileLoader( defConn,true );							// ﾃﾞﾌｫﾙﾄｺﾈｸｼｮﾝとuseTimeStamp=true 指定
//			loader.loadClassPathFiles( "xml" ) ;
			final XMLFileLoader xmlLoad = new XMLFileLoader( defConn,true );	// ﾃﾞﾌｫﾙﾄｺﾈｸｼｮﾝとuseTimeStamp=true 指定
			xmlLoad.loadClassPathFiles( "xml" ) ;

			// ｺﾝﾃｷｽﾄ単位のｼｽﾃﾑﾊﾟﾗﾒｰﾀを GE12 に設定します｡
			// dbXMLResourceInsert の後に登録する必要があります｡
			dbResourceUpdate( rscConn,sysParam );			// 5.6.7.0 (2013/07/27) DBID=RESOURCE のｺﾈｸｼｮﾝ
			rscConn.commit();
	//		Closer.commit( rscConn );

			final Map<String,String> userMap = new HashMap<>(100);
			// 6.4.2.1 (2016/02/05) try-with-resources 文
			try( PreparedStatement pstmt = rscConn.prepareStatement( QUERY ) ) {
				pstmt.setFetchSize( DB_FETCH_SIZE );		// 6.9.3.0 (2018/03/26) ﾃﾞｰﾀ検索時のﾌｪｯﾁｻｲｽﾞ
				pstmt.setString( 1,SYSTEM_ID );
				pstmt.setString( 2,HOST_URL );				// 4.1.0.0 (2007/12/21)
				try( ResultSet resultSet = pstmt.executeQuery() ) {

					while( resultSet.next() ) {
						final String key = resultSet.getString(1);
						final String val = resultSet.getString(2);
						// 6.4.3.2 (2016/02/19) null値の登録はやめます。
						userMap.put( key,val == null ? "" : val );		// 6.4.3.2 (2016/02/19)
					}
				}
			}

			// DBを検索して、ﾕｰｻﾞｰ設定ﾘｿｰｽ情報を取得します｡
			System.out.println( "    Load DB Resource [" + userMap.size() + "] finished." );
			// ﾘｿｰｽをﾏｰｼﾞします｡
			sysParam.putAll( userMap );
			errFlag = false;	// ｴﾗｰでない
		}
		catch( final SQLException ex ) {
			Closer.rollback( defConn );			// 5.6.7.0 (2013/07/27) DBID=DEFAULT のｺﾈｸｼｮﾝ
			Closer.rollback( rscConn );			// 5.6.7.0 (2013/07/27) DBID=RESOURCE のｺﾈｸｼｮﾝ
			LogWriter.log( ex );
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() ;
			ERR_MSG_LIST.add( errMsg );
		}
		catch( final UnsupportedEncodingException ex ) {
			Closer.rollback( defConn );			// 5.6.7.0 (2013/07/27) DBID=DEFAULT のｺﾈｸｼｮﾝ
			Closer.rollback( rscConn );			// 5.6.7.0 (2013/07/27) DBID=RESOURCE のｺﾈｸｼｮﾝ
			LogWriter.log( ex );
			final String errMsg = "UTF-8 がサポートされていない Java VM は、正規VMではありません。"
							+ ex.getMessage();
			ERR_MSG_LIST.add( errMsg );
		}
		catch( final RuntimeException ex ) {
			Closer.rollback( defConn );			// 5.6.7.0 (2013/07/27) DBID=DEFAULT のｺﾈｸｼｮﾝ
			Closer.rollback( rscConn );			// 5.6.7.0 (2013/07/27) DBID=RESOURCE のｺﾈｸｼｮﾝ
			// 5.7.2.0 (2014/01/10) RuntimeException は、catch しないようにします。
			// 6.4.7.1 (2016/06/17)
			final String errMsg = "システムパラメータの設定で、エラーが発生しました。"
							+ ex.getMessage();
			ERR_MSG_LIST.add( errMsg );

			throw new HybsSystemException( ex );		// 6.4.3.2 (2016/02/19) 生のRuntimeExceptionをﾗｯﾌﾟしてthrowする。
		}
		finally {
			if( errFlag ) {
				ConnectionFactory.remove( defConn, null );				// 5.6.7.0 (2013/07/27) DBID=DEFAULT のｺﾈｸｼｮﾝ
				ConnectionFactory.remove( rscConn, RESOURCE_DBID );		// 5.6.7.3 (2013/08/23) DBID=RESOURCE 漏れ
			}
			else {
				ConnectionFactory.close( defConn, null );				// 5.6.7.0 (2013/07/27) DBID=DEFAULT のｺﾈｸｼｮﾝ
				ConnectionFactory.close( rscConn, RESOURCE_DBID );		// 5.6.7.3 (2013/08/23) DBID=RESOURCE 漏れ
			}
		}
	}

	/**
	 * ｴﾝｼﾞﾝ内部定義の初期ﾘｿｰｽ情報をDB(GE12)に登録します｡
	 *
	 * 初期ﾘｿｰｽ情報は、KBSAKU='0' で登録されている情報で、一旦すべて削除
	 * してから、全てのﾘｿｰｽ情報を追加するという形をとります｡
	 * ただし、属性情報(名称や概要など)を別途登録する場合は、全てを
	 * 削除せずに、UPDATE する方向で検討したいと思います｡
	 * なお、この情報をDB登録する理由は、ﾘｿｰｽの設定値を変えたい場合に、
	 * ｷｰが判らない(JavaDOCからしか読み取れない)のでは不便な為に
	 * 用意しておくだけで、内部では SystemData ｵﾌﾞｼﾞｪｸﾄとして定義
	 * されている値を使用するため、このﾃﾞｰﾀﾍﾞｰｽ値は、使用していません｡
	 *
	 * ※ 6.9.7.0 (2018/05/14)
	 *    自身のｻｰﾊﾞｰで追記した情報以外を消すため、登録日付が古いﾃﾞｰﾀは削除します｡
	 *    現状は、1ヶ月にしています｡
	 *
	 * @og.rev 6.9.7.0 (2018/05/14) DYSETの登録と、旧ﾃﾞｰﾀ（1ヶ月前）の削除を行います｡
	 *
	 * @param	conn	登録用ｺﾈｸｼｮﾝ(ﾘｿｰｽ用)
	 * @param	map		入力ｼｽﾃﾑﾊﾟﾗﾒｰﾀﾏｯﾌﾟ
	 * @throws	SQLException	ﾃﾞｰﾀﾍﾞｰｽｱｸｾｽｴﾗｰ
	 */
	private static void dbResourceUpdate( final Connection conn,final Map<String,String> map ) throws SQLException {

		final String SYSTEM_ID = map.get( "SYSTEM_ID" );
		final String HOST_URL  = map.get( "HOST_URL" );		// 4.1.0.0 (2007/12/21)

		// 6.9.7.0 (2018/05/14) 現在時刻から、1ヶ月を引いた日付と比較して、削除ﾃﾞｰﾀを決めます｡
		final Calendar now = Calendar.getInstance();
		now.add( Calendar.MONTH , -1 );
		final String OLD_DATE  = DateSet.getDate( now.getTimeInMillis() , DYSET_FORMAT );

		// 既存の設定値を全件DELETEします。
		int delCnt;
		PreparedStatement pstmt = null;
		try {
			pstmt = conn.prepareStatement( DEL_SYS );
			pstmt.setString( 1, SYSTEM_ID );
			pstmt.setString( 2, HOST_URL );						// 4.1.0.0 (2007/12/21)
			pstmt.setString( 3, OLD_DATE );						// 6.9.7.0 (2018/05/14) 古いﾃﾞｰﾀを消します。
			delCnt = pstmt.executeUpdate();
		}
		finally {
			Closer.stmtClose( pstmt );
		}

		// 新設定値を全件INSERTします。
		final Set<String> keyset = map.keySet();
		final String[] keys = keyset.toArray( new String[keyset.size()] );

		final String DYSET = DateSet.getDate( DYSET_FORMAT );	// 6.9.7.0 (2018/05/14)
		int insCnt = 0;
		try {
			pstmt = conn.prepareStatement( INS_SYS );
			for( int i=0; i<keys.length; i++ ) {
				pstmt.setString( 1,SYSTEM_ID );					// SYSTEM_ID
				pstmt.setString( 2,HOST_URL);					// CONTXT_PATH
				pstmt.setString( 3,keys[i] );					// PARAM_ID
				pstmt.setInt(    4,( i + 1 ) * 10 );			// SEQNO
				pstmt.setString( 5,map.get( keys[i] ) );		// PARAM
				pstmt.setString( 6,DYSET );						// DYSET 6.9.7.0 (2018/05/14)
				pstmt.setString( 7,keys[i] );					// PARAM_ID
				insCnt += pstmt.executeUpdate();
			}
		}
		finally {
			Closer.stmtClose( pstmt );
		}

		System.out.print( "    DB Context Resource Reconfiguration " );
		System.out.println( "DELETE=[" + delCnt + "],INSERT=[" + insCnt + "] finished." );
	}

	/**
	 * ServletContext の名称を取得します｡
	 *
	 * ｺﾝﾃｷｽﾄのｱｸｾｽされたﾊﾟｽ( /training など )の名称を、
	 * 取得します｡(ｱｸｾｽ先の仮想ﾌｫﾙﾀﾞ名)
	 * 以前は、配備記述子(WEB-INF/web.xml)の display-name 要素を見て、
	 * 無ければ、実ﾌｫﾙﾀﾞ名を返していました｡
	 * ※ warﾌｧｲﾙのﾊﾞｰｼﾞｮﾝ管理（ｾｯｼｮﾝ管理）対応として、## 以下の文字列を削除します｡
	 *    ## は、war化の際に、"%23%23" に変換されるため、それも取り除き処理の対象とします｡
	 *
	 * @og.rev 6.6.0.0 (2016/12/01) ｺﾝﾃｷｽﾄﾊﾟｽから、##ﾊﾞｰｼﾞｮﾝ番号を取り去った値を返すようにします｡
	 *
	 * @param	context	Servletｺﾝﾃｷｽﾄ
	 *
	 * @return	ｺﾝﾃｷｽﾄのｺﾝﾃｷｽﾄ名
	 */
	private static String getContextName( final ServletContext context ) {
		String name = null;
		try {
			final String path = context.getResource( "/" ).getFile();
			final int st = path.lastIndexOf( '/',path.length()-2 );		// 後ろから '/' を検索した位置(一番後ろは、'/' が付いているので注意)
			int ed = path.indexOf( "##",st+1 );							// '/' 以降で、"##" を検索した位置
			if( ed < 0 ) {
				ed = path.indexOf( "%23%23",st+1 );						// '/' 以降で、"%23%23" を検索した位置
			}
			final int ed2 = ed > 0 ? ed : path.length()-1 ;				// ## が無ければ、一番最後の文字を含まない、一つ前まで。あれば、そこまで
			name = path.substring( st+1,ed2 );
		}
		catch( final MalformedURLException ex ) {
			LogWriter.log( ex );
			final String errMsg = "このパス名は、正しいフォームではありません。 "
						+ ex.getMessage();
			ERR_MSG_LIST.add( errMsg );
		}
		return name ;
	}

	/**
	 * 指定のｸﾗｽの public static なﾌｨｰﾙﾄﾞｷｰと値のﾏｯﾌﾟを作成します｡
	 * 主に、ｴﾝｼﾞﾝ関連のｸﾗｽにﾊﾟﾗﾒｰﾀﾌｧｲﾙをｸﾗｽとして定義し、
	 * ｴﾝｼﾞﾝとともに配布します｡配布されたｸﾗｽを元に、ﾊﾟﾗﾒｰﾀを
	 * 読み取ります｡
	 * この処理は ﾘﾌﾚｸｼｮﾝを使用してｸﾗｽの public static ﾌｨｰﾙﾄﾞを
	 * 取得し、LinkedHashMap により、取得順をｷｰﾌﾟしたまま、Mapを返します｡
	 *
	 * @og.rev 5.7.2.0 (2014/01/10) ERR_MSG_LIST は、一旦文字列に変換してから追加します｡
	 * @og.rev 6.4.3.2 (2016/02/19) returnをやめて、引数のMapに直接値を設定します｡
	 *
	 * @param	map	読み書きするMapｵﾌﾞｼﾞｪｸﾄ
	 * @param	cls	ｸﾗｽｵﾌﾞｼﾞｪｸﾄ
	 */
	private static void loadParameter( final Map<String,String> map , final Class<?> cls ) {

		int cnt = 0;
		try {
		//	final Field[] field = cls.getFields(); の取得。public ﾌｨｰﾙﾄﾞのみ
			for( final Field fld : cls.getFields() ) {
				if( Modifier.isStatic( fld.getModifiers() ) ) {
					map.put( fld.getName() , (String)fld.get( null ) );
					cnt++ ;
				}
			}
		}
		catch( final IllegalAccessException ex ) {
			LogWriter.log( ex );
			// 5.7.2.0 (2014/01/10) ERR_MSG_LIST は、一旦文字列に変換してから追加します。
			final String errMsg = "クラスから、パラメータを取得できませんでした。"	+ CR
								+ "  クラス名=[" + cls.getName() + "]"				+ CR
								+ ex.getMessage();
			ERR_MSG_LIST.add( errMsg );
		}

		System.out.println( "    ClassLoad " + cls.getName() + " Parameter [" + cnt + "] finished." );
	}

	/**
	 * 指定のｷｰﾜｰﾄﾞのﾌｧｲﾙをｸﾗｽﾊﾟｽより取得し、ｷｰと値のﾏｯﾌﾟを作成します｡
	 * 主に、ｴﾝｼﾞﾝ関連のｸﾗｽにﾊﾟﾗﾒｰﾀﾌｧｲﾙをPlugInｸﾗｽとして定義し、配布します｡
	 * この処理の取得に、ｸﾗｽﾊﾟｽの順序が関係します｡最初に取得されたｷｰは、あとから
	 * 読み取られたｸﾗｽﾊﾟｽは、再ｾｯﾄしません｡
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 並び順を、ｷｰの名称順とする｡
	 * @og.rev 5.7.2.0 (2014/01/10) ERR_MSG_LIST は、一旦文字列に変換してから追加します｡
	 * @og.rev 5.7.2.0 (2014/01/10) RuntimeException は、catch しないようにします｡
	 * @og.rev 6.4.3.2 (2016/02/19) returnをやめて、引数のMapに直接値を設定します｡
	 * @og.rev 6.6.0.0 (2016/12/01) ｺﾝﾃｷｽﾄﾊﾟｽから、##ﾊﾞｰｼﾞｮﾝ番号を取り去った値を返すようにします｡
	 *
	 * @param	ioMap	読み書きするMapｵﾌﾞｼﾞｪｸﾄ
	 * @param	keyword	ｸﾗｽｵﾌﾞｼﾞｪｸﾄを検索する元
	 */
	private static void loadParameter( final Map<String,String> ioMap , final String keyword ) {
		final Map<String,String> map = new TreeMap<>();		// 5.3.6.0 (2011/06/01) 並び順を、ｷｰの名称順とする。
		try {
			final ClassLoader loader = Thread.currentThread().getContextClassLoader();
			final Enumeration<URL> enume = loader.getResources( keyword );		// 4.3.3.6 (2008/11/15) Generics警告対応
			while( enume != null && enume.hasMoreElements() ) {
				final URL url = enume.nextElement();		// 4.3.3.6 (2008/11/15) Generics警告対応
				// jar:file:/実ディレクトリ または、file:/実ディレクトリ
				final String dir = url.getFile().replaceAll( "%23%23","##" );	// 6.6.0.0 (2016/12/01)

				final FindClassFiles filenames = new FindClassFiles( dir,keyword );
				final String[] names = filenames.getFilenames();
				for( int i=0; i<names.length; i++ ) {
					final String val = names[i];
					final String key = val.substring( val.lastIndexOf( '.' )+1 );
					if( key.indexOf( '_' ) >= 0 && !map.containsKey( key ) && key.indexOf( '$' ) < 0 ) {
						map.put( key , val );
					}
				}
				System.out.println( "    FileCheck " + dir + " [" + names.length + "] find." );
			}
		}
		catch( final IOException ex ) {
			LogWriter.log( ex );
			// 5.7.2.0 (2014/01/10) ERR_MSG_LIST は、一旦文字列に変換してから追加します。
			final String errMsg = "キーワードから、パラメータを取得できませんでした。"	+ CR
						+ "  キーワード名=[" + keyword + "]"						+ CR
						+ ex.getMessage();
			ERR_MSG_LIST.add( errMsg );
		}
		// 5.7.2.0 (2014/01/10) RuntimeException は、catch しないようにします。
		System.out.println( "    FileLoad " + keyword + " Parameter [" + map.size() + "] finished." );
		ioMap.putAll( map );		// TreeMap で、ｷｰの名称順に並んだMapを、LinkedHashMapに登録順に設定する。
	}
}
