/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import java.io.UnsupportedEncodingException;
import java.io.IOException;

import jakarta.servlet.ServletException;
import jakarta.servlet.ServletConfig;
// import jakarta.servlet.RequestDispatcher;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import jakarta.servlet.annotation.WebInitParam;					// 7.3.0.0 (2021/01/06)
import jakarta.servlet.annotation.WebServlet;						// 7.3.0.0 (2021/01/06)

import org.opengion.hayabusa.common.HybsSystemException;

/**
 * FORM認証で、認証済みの場合、j_security_check に飛ばされてエラーになるので、強制的に sendRedirect する。
 * Post,Get両方に対応しています。
 *
 * @og.rev 7.3.0.0 (2021/01/06) 新規作成。
 * @version  7.3
 * @author   Kazuhiko Hasegawa
 * @since    JDK11
 *
 */
@WebServlet(
	urlPatterns = "/jsp/j_security_check" ,
	initParams  = {
		@WebInitParam(name="forwardURL", value="/jsp/index.jsp")
	}
)
public class JSecurityCheckServlet extends HttpServlet {
	private static final long serialVersionUID	= 730020210106L ;

	private String forwardURL = "/jsp/index.jsp" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 7.3.0.0 (2021/01/06) 新規作成。
	 */
	public JSecurityCheckServlet() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Postメソッドで与えられたrequestをcallClassメソッドに渡します。
	 * callClassメソッドではclassパラメータの値を利用してクラスをロードし、処理を行います。
	 * 具体的な処理はcallClassメソッドをご覧下さい。
	 *
	 * @og.rev 7.3.0.0 (2021/01/06) 新規作成。
	 *
	 * @param request HttpServletRequestリクエスト
	 * @param response HttpServletResponseレスポンス
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 * @throws IOException 入出力エラーが発生したとき
	 */
	@Override
	public void doPost( final HttpServletRequest request, final HttpServletResponse response )	throws ServletException,IOException {
		callClass( request, response );
	}

	/**
	 * Getメソッドで与えられたrequestをcallClassメソッドに渡します。
	 * callClassメソッドではclassパラメータの値を利用してクラスをロードし、処理を行います。
	 * 具体的な処理はcallClassメソッドをご覧下さい。
	 *
	 * @og.rev 7.3.0.0 (2021/01/06) 新規作成。
	 *
	 * @param request HttpServletRequestリクエスト
	 * @param response HttpServletResponseレスポンス
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 * @throws IOException 入出力エラーが発生したとき
	 */
	@Override
	public void doGet( final HttpServletRequest request, final HttpServletResponse response )	throws ServletException,IOException {
		callClass( request, response );
	}

	/**
	 * Servlet の 初期値設定を行います。
	 *
	 * @og.rev 7.3.0.0 (2021/01/06) 新規作成。
	 *
	 * WEB-INF/web.xml ファイルで、&lt;servlet&gt; タグ内で初期値設定を行います。
	 * &lt;init-param&gt;
	 *     &lt;param-name&gt;forwardURL&lt;/param-name&gt;
	 *     &lt;param-value&gt;/jsp/index.jsp&lt;/param-value&gt;
	 * &lt;/init-param&gt;
	 *
	 * @param	config	ServletConfigオブジェクト
	 */
	@Override
	public void init( final ServletConfig config ) throws ServletException {
		super.init( config );

		final String url = config.getInitParameter("forwardURL");
		if( url != null && !url.isEmpty() ) {
			forwardURL = url;
		}
	}

	/**
	 * POSTとGETに対する実際の処理です
	 * sendRedirect先の URL をforwardURL で初期設定されたアドレスに転送します。
	 *
	 * @param request リクエスト
	 * @param response レスポンス
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 * @throws IOException 入出力エラーが発生したとき
	 */
	private void callClass( final HttpServletRequest request, final HttpServletResponse response ) throws ServletException,IOException {
		try {
			request.setCharacterEncoding( "UTF-8" );
		}
		catch( final UnsupportedEncodingException ex ) {
			throw new HybsSystemException( ex );
		}

	// forward は、元のアドレスが残ってしまう。
	//	final RequestDispatcher dispatch = request.getRequestDispatcher(forwardURL);
	//	dispatch.forward(request, response);

		String url = "/" ;
		if( request.authenticate(response) ) {		// 認証済みの場合に、もう一度呼ばれると、true になっている。
			final String refe = request.getHeader( "referer" );	// 元のアドレスにリダイレクトする。
			if( refe != null && !refe.isEmpty() ) { url = refe; }
		}
		else {
			// sendRedirect で、アドレスを変えてしまう。
			// ただし、forward は、自分のｺﾝﾃｷｽﾄ基準だが、sendRedirect はｱﾌﾟﾘｹｰｼｮﾝ基準
			final String cntxPath = request.getContextPath();		// /gf などのｺﾝﾃｷｽﾄﾊﾟｽ
			url = cntxPath + forwardURL ;
		}
		response.sendRedirect(url);
	}
}
