/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;					// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.xml.XMLFileLoader;				// 6.0.0.0 (2014/04/11) XMLFileLoader を使う様に変更

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.sql.Connection;

import java.io.File;
import java.io.StringWriter;								// 6.0.0.0 (2014/04/11) XMLFileLoader に渡す Log
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;									// 6.0.0.0 (2014/04/11) keys,vals のｴﾗｰﾒｯｾｰｼﾞ作成用

/**
 * 指定の拡張XDK形式ﾌｧｲﾙを直接ﾃﾞｰﾀﾍﾞｰｽに登録するﾃﾞｰﾀ入力ﾀｸﾞです。
 *
 * このｸﾗｽは、ｵﾗｸﾙ XDKの oracle.xml.sql.dml.OracleXMLSave ｸﾗｽと
 * ほぼ同様の目的で使用できる org.opengion.fukurou.xml.XMLFileLoader のﾗｯﾊﾟｰｸﾗｽです。
 *
 * 拡張XDK形式のXMLﾌｧｲﾙを読み込み、ﾃﾞｰﾀﾍﾞｰｽに INSERT します。
 * 拡張XDK形式の元となる ｵﾗｸﾙ XDK(Oracle XML Developer's Kit)については、以下の
 * ﾘﾝｸを参照願います。
 * <a href="https://docs.oracle.com/cd/F19136_01/adxdk/introduction-to-XDK.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * このﾀｸﾞでは、keys,vals を登録することにより、<del>XMLﾌｧｲﾙに存在しないｶﾗﾑを
 * 追加したり</del>、XMLﾌｧｲﾙの情報を書き換えることが可能になります。
 * 例えば、登録日や、登録者、または、ﾃﾝﾌﾟﾚｰﾄより各ｼｽﾃﾑID毎に
 * 登録するなどです。
 *
 * 拡張XDK形式とは、ROW 以外に、SQL処理用ﾀｸﾞ(EXEC_SQL)を持つ XML ﾌｧｲﾙです。
 * また、登録するﾃｰﾌﾞﾙ(table)を ROWSETﾀｸﾞの属性情報として付与することができます。
 * (大文字小文字に注意)
 * これは、ｵﾗｸﾙXDKで処理する場合、無視されますので、同様に扱うことが出来ます。
 * この、EXEC_SQL は、それそれの XMLﾃﾞｰﾀをﾃﾞｰﾀﾍﾞｰｽに登録する際に、
 * SQL処理を自動的に流す為の、SQL文を記載します。
 * この処理は、ｲﾍﾞﾝﾄ毎に実行される為、その配置順は重要です。
 * このﾀｸﾞは、複数記述することも出来ますが、BODY部には、1つのSQL文のみ記述します。
 *
 * 6.0.0.0 (2014/04/11)
 *   指定のﾌｧｲﾙがﾌｫﾙﾀﾞの場合は、以下のﾌｧｲﾙすべて(拡張子はxml)を対象に読込-登録します。
 *   また、拡張子が、zip の場合は、zip内部の xml ﾌｧｲﾙを読込-登録します。
 *
 * ※ このﾀｸﾞは、Transaction ﾀｸﾞの対象です。
 *
 *   &lt;ROWSET tableName="XX" &gt;
 *       &lt;EXEC_SQL&gt;                    最初に記載して、初期処理(ﾃﾞｰﾀｸﾘｱ等)を実行させる。
 *           delete from GEXX where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *       &lt;MERGE_SQL&gt;                   このSQL文で UPDATEして、結果が0件ならINSERTを行います。
 *           update GEXX set AA=[AA] , BB=[BB] where CC=[CC]
 *       &lt;/MERGE_SQL&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;ｶﾗﾑ1&gt;値1&lt;/ｶﾗﾑ1&gt;
 *             ･･･
 *           &lt;ｶﾗﾑn&gt;値n&lt;/ｶﾗﾑn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *       &lt;EXEC_SQL&gt;                    最後に記載して、項目の設定(整合性登録)を行う。
 *           update GEXX set AA='XX' , BB='XX' where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *   &lt;ROWSET&gt;
 *
 * @og.formSample
 * ●形式:&lt;og:directXMLSave fileURL="[･･･]" ･･･ /&gt;
 * ●body:なし
 *
 * ●Tag定義:
 *   &lt;og:directXMLSave
 *       dbid               【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)
 *       fileURL            【TAG】読み取り元ﾃﾞｨﾚｸﾄﾘ名を指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします (初期値:null)
 *       displayMsg         【TAG】query の結果を画面上に表示するﾒｯｾｰｼﾞIDを指定します(初期値:MSG0003[ﾌｧｲﾙの登録が完了しました。])
 *       keys               【TAG】XMLﾌｧｲﾙを読み取った後で指定するｷｰをCSV形式で複数指定します
 *       vals               【TAG】XMLﾌｧｲﾙを読み取った後で指定する値をCSV形式で複数指定します
 *       useTimeStamp       【TAG】XMLﾌｧｲﾙの読み取りで、ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか[true:行う/false:行わない]指定します(初期値:false)
 *       useTimeView        【TAG】処理時間を表示する TimeView を表示するかどうかを指定します
 *										(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:directXMLSave
 *         dbid         = "ORCL"                接続ﾃﾞｰﾀﾍﾞｰｽID(初期値:DEFAULT)
 *         fileURL      = "{&#064;USER.ID}"     読み取り元ﾃﾞｨﾚｸﾄﾘ名
 *         filename     = "{&#064;filename}"    読み取り元ﾌｧｲﾙ名
 *         displayMsg   = "MSG0003"             登録完了後のﾒｯｾｰｼﾞ
 *     /&gt;
 *
 * @og.group ﾌｧｲﾙ入力
 * @og.rev 4.0.0.0 (2007/03/08) 新規追加
 * @og.rev 6.0.0.0 (2014/04/11) 単体ﾌｧｲﾙ以外(ﾌｫﾙﾀﾞ、ZIPﾌｧｲﾙ)への対応
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class DirectXMLSaveTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.1 (2016/02/05)" ;
	private static final long serialVersionUID = 642120160205L ;

	private String	dbid		;
	private String	fileURL		= HybsSystem.sys( "FILE_URL" );					// 4.0.0 (2005/01/31)
	private String	filename	;												// 6.0.0.0 (2014/04/11) 初期値:null
	private String	displayMsg	= "MSG0003";									// ﾌｧｲﾙの登録が完了しました。
	private String[]	keys	;
	private String[]	vals	;
	private boolean	useTimeStamp;												// 6.0.2.0 (2014/09/19) ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか(true:行う/false:行わない)
	private boolean	useTimeView	= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );		// 6.3.6.0 (2015/08/16)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DirectXMLSaveTag() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) ﾒｯｾｰｼﾞﾘｿｰｽ統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。例外経路で null 値を利用することが保証されています。
	 * @og.rev 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
	 * @og.rev 5.6.7.0 (2013/07/27) DDL（ﾃﾞｰﾀ定義言語：Data Definition Language）の処理件数追加
	 * @og.rev 6.0.2.0 (2014/09/19) useTimeStamp 属性追加。ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか
	 * @og.rev 6.3.5.1 (2015/08/16) doStartTag() 削除に伴う、dyStart の移動。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
		if( !useTag() ) { return EVAL_PAGE ; }
		final long dyStart = System.currentTimeMillis();						// 6.3.5.1 (2015/08/16) 時間測定用

		final StringWriter logW = new StringWriter();							// 6.0.0.0 (2014/04/11) XMLFileLoader で Logをｾｯﾄ

		// 6.0.0.0 (2014/04/11) XMLFileLoader に渡す 読み取り開始ﾌｧｲﾙｵﾌﾞｼﾞｪｸﾄ。
		final String directory = HybsSystem.url2dir( fileURL );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		final File loadFile = ( filename == null ) ? new File( directory ) : new File( directory,filename );

		final int insCnt ;														// 追加数だけ記録する。

		// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
		try( Transaction tran = getTransaction() ) {
			final Connection conn = tran.getConnection( dbid );					// 5.1.9.0 (2010/08/01) Transaction 対応

			// 6.0.0.0 (2014/04/11) ﾌｫﾙﾀﾞ一括登録ができるようにします。
			// 6.0.2.0 (2014/09/19) useTimeStamp 属性追加。ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか
			final XMLFileLoader loader = new XMLFileLoader( conn,useTimeStamp );	// ｺﾈｸｼｮﾝとuseTimeStamp 指定
			if( keys != null ) { loader.setAfterMap( getAfterMap() ); }
			loader.setLogWriter( logW );

			loader.loadXMLFiles( loadFile );

			final int[] crudCnt = loader.getCRUDCount();						// 実行結果のｶｳﾝﾄ数
			insCnt = crudCnt[XMLFileLoader.INS];

			tran.commit();														// 5.1.9.0 (2010/08/01) Transaction 対応
		}
		catch( final Throwable ex ) {											// catch は、close() されてから呼ばれます。
			// 6.2.3.0 (2015/05/01) ｴﾗｰﾒｯｾｰｼﾞの追加
			final String errMsg = "XMLファイルのLoadに失敗しました。" + CR
								+ ex.getMessage() + CR
								+ " file=" + loadFile;
			throw new HybsSystemException( errMsg,ex );
		}

		// 実行件数の表示
		// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
		if( displayMsg != null && displayMsg.length() > 0 ) {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( "<pre>" )
				.append( logW.toString() )							// 6.0.0.0 (2014/04/11) XMLFileLoader で Logをｾｯﾄ
				.append( CR )
				.append( HybsSystem.getDate() ).append( "  " )
				.append( getResource().getLabel( displayMsg ) )
				.append( CR )
				.append( "</pre>" );

			jspPrint( buf.toString() );
		}

		// 4.0.0 (2005/01/31) ｾｷｭﾘﾃｨﾁｪｯｸ(ﾃﾞｰﾀｱｸｾｽ件数登録)
		final long dyTime = System.currentTimeMillis()-dyStart;
		final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addWriteCount( insCnt,dyTime,loadFile.getPath() ); }

		if( useTimeView ) {		// 6.3.6.0 (2015/08/16)
			// 時間測定用の DIV 要素を出力
			jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		}
		return EVAL_PAGE ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 6.0.0.0 (2014/04/11) filename の初期値を、ｼｽﾃﾑﾌﾟﾛﾊﾟﾃｨｰのFILE_FILENAME から null に変更
	 * @og.rev 6.0.0.0 (2014/04/11) displayMsg の初期値を、MSG0040 から MSG0003 に変更
	 * @og.rev 6.0.2.0 (2014/09/19) useTimeStamp 属性追加。ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか
	 */
	@Override
	protected void release2() {
		super.release2();
		dbid			= null;
		fileURL			= HybsSystem.sys( "FILE_URL" );							// 4.0.0 (2005/01/31)
		filename		= null;													// 6.0.0.0 (2014/04/11) 初期値:null
		displayMsg		= "MSG0003";											// ﾌｧｲﾙの登録が完了しました。
		keys			= null;
		vals			= null;
		useTimeStamp	= false;												// 6.0.2.0 (2014/09/19) ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか
		useTimeView		= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );				// 6.3.6.0 (2015/08/16)
	}

	/**
	 * XMLﾌｧｲﾙを読み取った後で指定するｶﾗﾑと値のﾍﾟｱ(ﾏｯﾌﾟ)情報を作成します。
	 *
	 * このｶﾗﾑと値のﾍﾟｱのﾏｯﾌﾟは、ｵﾌﾞｼﾞｪｸﾄ構築後に設定される為、
	 * XMLﾌｧｲﾙのｷｰの存在に関係なく、Mapのｷｰと値が使用されます。(Map優先)
	 * key が null や ｾﾞﾛ文字列の場合は、Map に追加しません。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) key が null や ｾﾞﾛ文字列の場合は、Map に追加しません。
	 * @og.rev 6.0.0.0 (2014/04/11) keys と vals の個数ﾁｪｯｸを追加
	 *
	 * @return	ｶﾗﾑと値のﾍﾟｱ(ﾏｯﾌﾟ)情報
	 */
	private Map<String,String> getAfterMap() {
		// 6.0.2.4 (2014/10/17) NP:null 値を利用している可能性があります。
		if( keys == null || vals == null ) { return null; }		// 追加しないｹｰｽ

		// 6.0.0.0 (2014/04/11) keys と vals の個数ﾁｪｯｸを追加
		if( keys.length != vals.length ) {
			final String errMsg = "keys と vals の個数が異なります。"
								+ " keys=" + Arrays.toString( keys )
								+ " vals=" + Arrays.toString( vals ) ;
			throw new HybsSystemException( errMsg );
		}

		final Map<String,String> map = new HashMap<>();
		for( int i=0; i<keys.length; i++ ) {
			if( keys[i] != null && keys[i].length() > 0 ) {		// 5.6.6.1 (2013/07/12)
				map.put( keys[i],vals[i] );
			}
		}
		return map ;
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id	ﾃﾞｰﾀﾍﾞｰｽ接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】読み取り元ﾃﾞｨﾚｸﾄﾘ名を指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘより、ﾌｧｲﾙを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのﾃﾞｨﾚｸﾄﾘに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に、
	 * さらに、各個人ID別のﾌｫﾙﾀﾞの下より、読み取ります。
	 * (初期値:ｼｽﾃﾑ定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたﾃﾞｨﾚｸﾄﾘ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に、"/" を追加する処理を廃止。
	 *
	 * @param	url	ﾌｧｲﾙURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします(初期値:null)。
	 *
	 * @og.tag
	 * ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします。
	 * (初期値:null)。
	 *
	 * @og.rev 6.0.0.0 (2014/04/11) filename の初期値を、ｼｽﾃﾑﾌﾟﾛﾊﾟﾃｨｰのFILE_FILENAME から null に変更
	 *
	 * @param	fname	ﾌｧｲﾙ名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getRequestParameter( fname ),filename );
	}

	/**
	 * 【TAG】query の結果を画面上に表示するﾒｯｾｰｼﾞIDを指定します(初期値:MSG0003[ﾌｧｲﾙの登録が完了しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したﾒｯｾｰｼﾞをﾘｿｰｽから取得して
	 * 表示します。
	 * 表示させたくない場合は、displayMsg = "" をｾｯﾄしてください。
	 * 初期値は、検索件数を表示します。
	 * ※ この属性には、ﾘｸｴｽﾄ変数({&#064;XXXX})は使用できません。
	 *
	 * @param	id	処理結果表示ﾒｯｾｰｼﾞID
	 */
	public void setDisplayMsg( final String id ) {
		if( id != null ) { displayMsg = id; }
	}

	/**
	 * 【TAG】XMLﾌｧｲﾙを読み取った後で指定するｷｰをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * XMLﾌｧｲﾙを読み取った後で、ﾃﾞｰﾀを変更できます。
	 * 変更するｶﾗﾑ名(ｷｰ)をCSV形式で指定します。
	 * XMLﾌｧｲﾙにｷｰが存在していた場合は、vals で指定の値に書き換えます。
	 * <del>ｷｰが存在していない場合は、ここで指定するｷｰと値が、ﾃﾞｰﾀとして
	 * 追加されます。</del>
	 * 例えば、登録日や、登録者、または、ﾃﾝﾌﾟﾚｰﾄより各ｼｽﾃﾑID毎に
	 * 登録するなどの使い方を想定しています。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です。
	 *
	 * @param	key	ﾘﾝｸ先に渡すｷｰ
	 * @see		#setVals( String )
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】XMLﾌｧｲﾙを読み取った後で指定する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * XMLﾌｧｲﾙを読み取った後で、ﾃﾞｰﾀを変更できます。
	 * 変更する値をCSV形式で指定します。
	 * XMLﾌｧｲﾙにｷｰが存在していた場合は、vals で指定の値に書き換えます。
	 * <del>ｷｰが存在していない場合は、ここで指定するｷｰと値が、ﾃﾞｰﾀとして
	 * 追加されます。</del>
	 * 例えば、登録日や、登録者、または、ﾃﾝﾌﾟﾚｰﾄより各ｼｽﾃﾑID毎に
	 * 登録するなどの使い方を想定しています。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です。
	 *
	 * @param	val	keys属性に対応する値
	 * @see		#setKeys( String )
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】ﾀｲﾑｽﾀﾝﾌﾟ管理を行うかどうか[true:行う/false:行わない]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * useTimeStamp=true の場合、登録の実行有無の判断は、ﾌｧｲﾙの更新時刻より判断します。
	 * これは、読み取りﾌｧｲﾙの更新時刻が、０でない場合、読み取りを行います。
	 * 読み取りが完了した場合は、更新時刻を ０ に設定します。
	 * つまり、一度しか読み込まないように制御できます。
	 * useTimeStamp=false の場合は、無条件に読み込みます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 *
	 * @param	flag	ﾀｲﾑｽﾀﾝﾌﾟ管理 [true:行う/false:行わない]
	 */
	public void setUseTimeStamp( final String flag ) {
		useTimeStamp = nval( getRequestParameter( flag ),useTimeStamp );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します
	 *		(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するﾊﾞｰｲﾒｰｼﾞが表示されます。
	 * これは、DB検索、APｻｰﾊﾞｰ処理、画面表示の各処理時間をﾊﾞｰｲﾒｰｼﾞで
	 * 表示させる機能です。処理時間の目安になります。
	 * (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
				.println( "fileURL"			,fileURL		)
				.println( "filename"		,filename		)
				.println( "keys"			,String.join( ", " , keys ) )		// 7.2.9.5 (2020/11/28)
				.println( "vals"			,String.join( ", " , vals ) )		// 7.2.9.5 (2020/11/28)
				.println( "displayMsg"		,displayMsg		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
