/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.stream.Stream;									// 6.4.3.4 (2016/03/11)
import java.util.stream.Collectors;								// 6.4.3.4 (2016/03/11)

/**
 * ラジオボタンで表示順(Order By 句)を指定するタグです。
 *
 * name 属性値は、 "ORDER_BY" 固定です。
 * columns 属性は、CSV形式のカラム属性を設定します。
 * ロケールは、ユーザー情報の lang 属性を初期値で使用し、 セットされていない場合は、
 * リクエスト情報のロケールから取得します。
 *
 * @og.formSample
 * ●形式：&lt;og:orderBy columns="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:orderBy
 *       columns          ○【TAG】表示順対象のカラムをCSV形式で複数指定します(必須)。
 *       checked            【TAG】初めからチェックされた状態(checked)で表示します(初期値:null)
 *       lbls               【TAG】ラベルをCSV形式で複数指定します
 *       checkedValue       【TAG】初めからチェックされた状態にする値を指定します(初期値:null)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:orderBy columns="PN,YOBI" checked="checked" /&gt;
 *    &lt;og:orderBy columns="YOBI,PN"   /&gt;
 *    &lt;og:orderBy columns="MKNM,YOBI" /&gt;
 *
 * @og.rev 3.1.4.0 (2003/04/18) 継承元を、CommonTagSupport に変更。
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class OrderByTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.1.2 (2018/11/04)" ;
	private static final long serialVersionUID = 701220181104L ;

	private String columns		;
	private String labels		;
	private String checked		;
	private String checkedValue	;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public OrderByTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.4.1 (2003/04/21) HTMLTagSupport の継承を廃止した為、doEndTag() を追加。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			jspPrint( makeTag() );
		}
		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		columns      = null;
		labels       = null;
		checked      = null;
		checkedValue = null;
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) radio ボタン等で、ラベルをクリックしても値をセットできるようにする。
	 * @og.rev 7.0.1.2 (2018/11/04) StringBuilder で、若干の見直し
	 *
	 * @return  入力用のテキストフィールドタグ
	 * @og.rtnNotNull
	 */
	protected String makeTag() {
		set( "type","radio" );
		set( "name","ORDER_BY" );
		set( "value",columns );
		if( labels == null ) { setLbls( columns ); }
		final String lbls = getMsglbl();

		if( checkedValue != null && checkedValue.length() > 0 ) {
			if( checkedValue.equals( columns ) ) {
				set( "checked","checked" );
			}
		}
		else {
			set( "checked",checked );
		}

		// 7.0.1.2 (2018/11/04)
//		if( get( "id" ) == null ) { set( "id",TaglibUtil.getTagId() ); }
		final String id = nval( get( "id" ),TaglibUtil.getTagId() );

		// 7.0.1.2 (2018/11/04) StringBuilder で、若干の見直し
		return new StringBuilder( BUFFER_MIDDLE )
			.append( "<td>" ).append( XHTMLTag.input( getAttributes() ) )
			.append( "</td><td><label for=\"" ).append( id ).append( "\">" )
			.append( lbls )
			.append( "</label></td>" ).toString() ;

//		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
//			.append( "<td>" )
//			.append( XHTMLTag.input( getAttributes() ) )
//			.append( "</td>" )
//			.append( "<td>" )
//			.append( "<label for=\"" )
//			.append( get( "id" ) )
//			.append( "\">" )
//			.append( lbls )
//			.append( "</label>" )
//			.append( "</td>" );
//
//		return rtn.toString() ;
	}

	/**
	 * 【TAG】表示順対象のカラムをCSV形式で複数指定します。
	 *
	 * @og.tag 表示順対象のカラムを指定します。
	 *
	 * @param   columns カラム(CSV形式)
	 */
	public void setColumns( final String columns ) {
		this.columns = getRequestParameter( columns );
	}

	/**
	 * 【TAG】初めからチェックされた状態(checked)で表示します(初期値:null)。
	 *
	 * @og.tag
	 * check 値が "checked" の場合のみ、実行されます。
	 *
	 * @param   ch チェック指定 [checked:チェックされた状態で表示/その他:なにもしない]
	 */
	public void setChecked( final String ch ) {
		checked = getRequestParameter( ch );
		if( "checked".equals( checked )) {
			checked = "checked";
		}
		else {
			checked = null;
		}
	}

	/**
	 * 【TAG】初めからチェックされた状態にする値を指定します(初期値:null)。
	 *
	 * @og.tag
	 * チェックバリューが、自分自身のカラム名と同じであれば、チェックがつきます。
	 * 通常、checkedValue = "{&#064;ORDER_BY}" で、自動的にチェックされます。
	 *
	 * @param   ch チェックされた状態にする値
	 */
	public void setCheckedValue( final String ch ) {
		checkedValue = nval( getRequestParameter( ch ),checkedValue );
	}

	/**
	 * 【TAG】ラベルをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * ラベル(複数)を変更するときに、lbls属性を使います。
	 *
	 * シングルラベルの lbl 属性との違いは、ここではラベルを複数 CSV形式で
	 * 渡すことが可能であることです。これにより、"A,B,C" という値に対して、
	 * "Aのラベル表示,Bのラベル表示,Cのラベル表示" という具合に文字列を
	 * 再合成します。
	 * これは、SQL文のOrder By 句で、ソート順を指定する場合などに便利です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	 * @og.rev 6.4.3.4 (2016/03/11) CSV形式の文字連結を、stream 経由で行います。
	 *
	 * @param   lbls 複数ラベルID(CSV形式)
	 */
	public void setLbls( final String lbls ) {
		labels = getRequestParameter( lbls ) ;

		final String[] array = getCSVParameter( lbls );

		final String lblCSV = Stream.of( array )
								.map( lbl -> getLabel( lbl ) )
								.collect( Collectors.joining( "," ) );

		setLbl( lblCSV );		// 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "columns"			,columns		)
				.println( "labels"			,labels			)
				.println( "checked"			,checked		)
				.println( "checkedValue"	,checkedValue	)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
