/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.util.Set;											// 7.0.7.0 (2019/12/13)

import org.opengion.hayabusa.common.HybsSystem;					// 7.0.7.0 (2019/12/13)
import org.opengion.hayabusa.common.HybsSystemException;		// 7.0.7.0 (2019/12/13)
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ArraySet;						// 7.0.7.0 (2019/12/13)
import static org.opengion.fukurou.util.StringUtil.nval ;		// 7.0.7.0 (2019/12/13)

/**
 * 【入力】複数行の入力が可能なテキスト入力フィールドを作成するHTML拡張タグです。
 *
 * name 属性に ラベルリソース のキーを与えることで、 ロケールにあわせたリソースを使用して、
 * 画面に表示します。
 * ロケールは、ユーザー情報の lang 属性を初期値で使用し、 セットされていない場合は、
 * リクエスト情報のロケールから取得します。
 * ラベル部分と入力フィールド部分がテーブルタグの &lt;td&gt; により左右に分割されます。
 * HTML 表示時は、前後に&lt;tr&gt;タグで囲って、 整形できます。
 *
 * @og.formSample
 * ●形式：&lt;og:queryOption &gt;
 *             SELECT文 
 *         &lt;/og:queryOption &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:textarea
 *       name             ○【HTML】名前を指定します(必須)。
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       rows               【HTML】高さ(行数)を指定します
 *       cols               【HTML】横幅(カラム数)を指定します
 *       td                 【TAG】テーブル形式の &lt;td&gt; タグを使用するかどうか[yes/no/false]を指定します(初期値:yes)
 *       colspan            【TAG】フィールド部分の colspan を指定します
 *       rowspan            【TAG】ラベル、フィールド共に rowspan を指定します
 *       must               【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します(初期値:false)
 *       mustAny            【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します(初期値:無指定)
 *       useMustHidden      【TAG】必須の自動チェック用Hiddenを出力するかどうか[true/false]を指定します(初期値:true)
 *       aimai              【TAG】曖昧検索可能フィールドとして、曖昧検索方法を指定します(初期値:null)
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       readonly           【TAG】その部品に対して変更が出来ないように(readonly)指定します(サーバーに送信される)
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】アクセスキー(alt+キーで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       onSelect           【HTML】JavaScriptのイベント onSelect を設定します
 *       onKeydown          【HTML】JavaScriptのイベント onKeydown を設定します
 *       onKeypress         【HTML】JavaScriptのイベント onKeypress を設定します
 *       onKeyup            【HTML】JavaScriptのイベント onKeyup を設定します
 *       autofocus          【HTML5】指定した入力欄にカーソルが当たって自動的にフォーカスされます。
 *       placeholder        【HTML5】入力欄に初期表示する内容を指定します。
 *       required           【HTML5】入力必須を指定します
 *       roles              【TAG】ロールをセットします
 *       useRealTimeCheck   【TAG】(通常は使いません)リアルタイムチェックを行うかどうか(初期値:true:行う) 5.9.32.2 (2018/05/18)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:textarea&gt;
 *
 * ●使用例
 *     &lt;og:textarea name="BUN" /&gt;
 *
 *     &lt;og:textarea name="BUN"&gt;            初期値に値をセットしたいとき
 *         test
 *     &lt;/og:textarea name="BUN"&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TextareaTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.7.0 (2019/12/13)" ;
	private static final long serialVersionUID = 707020191213L ;

	private static final Set<String> TD_SET = new ArraySet<>( "yes" , "no" , "false" );		// 7.0.7.0 (2019/12/13) 

	// 7.0.7.0 (2019/12/13)
	private final String CLM = HybsSystem.sysBool( "HTML_LABEL_SEPARATOR" ) ? ":" : "" ;

	private String  tdFlag		= "yes";	// 7.0.7.0 (2019/12/13)  yes:<td> タグを使用 / false:フィールドのみ / no:ラベルとフィールド
	private String  colspan     = "";		// 7.0.7.0 (2019/12/13)  colspan 属性を追加。
	private String  rowspan		= "";		// 7.0.7.0 (2019/12/13)  rowspan 属性を追加。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TextareaTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転注意
		return useTag()
				? EVAL_BODY_BUFFERED		// Body を評価する。( extends BodyTagSupport 時)
				: SKIP_BODY ;
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.0 (2003/03/28) body 属性に、 {&#064;XXXX} パース機能を付加する。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		final String str = getBodyString();

		if( str != null && str.length() > 0 ) {
			final String label = StringUtil.htmlFilter( str );
			if( label != null && label.length() > 0 ) {
				set( "body",label );
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) td,colspan,rowspan 属性の追加。
	 */
	@Override
	protected void release2() {
		super.release2();
		tdFlag		= "yes";	// table形式の <td> タグを使用するかどうか
		colspan 	= "";
		rowspan		= "";		// 6.2.0.0 (2015/02/27) rowspan 属性を追加
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 4.3.7.1 (2009/06/08) id=labelのclass化
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性を自動化します。
	 * @og.rev 6.2.0.0 (2015/02/27) aimai 属性を追加。
	 * @og.rev 7.0.7.0 (2019/12/13) td,colspan,rowspan 属性の追加。
	 *
	 * @return  入力用のテキストフィールドタグ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		final String name = get( "name" );							// 6.2.0.0 (2015/02/27) 後で使うので変数化

		if( getMsglbl()  == null ) { setLbl( name ); }

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		// 4.3.7.1 (2009/06/08) id=labelのclass化
		// rtn.append( "<td id=\"label\">" );
//			.append( "<td class=\"label\">" )
		// 7.0.7.0 (2019/12/13) td,colspan,rowspan 属性の追加。
		if( "yes".equals( tdFlag ) ) {
			rtn.append( "<td class=\"label\" " )			// 7.0.7.0 (2019/12/13)
				.append( rowspan )							// 7.0.7.0 (2019/12/13)
				.append( '>' )								// 7.0.7.0 (2019/12/13)
				.append( getLongLabel() )					// 4.0.0 (2005/01/31)
	//			.append( "</td>" )
	//			.append( "<td>" )
				.append( "</td><td " )						// 7.0.7.0 (2019/12/13)
				.append( colspan )							// 7.0.7.0 (2019/12/13)
				.append( rowspan )							// 7.0.7.0 (2019/12/13)
				.append( '>' )								// 7.0.7.0 (2019/12/13)
				.append( XHTMLTag.textarea( getAttributes() ) )
				.append( "</td>" )
				.append( makeAimaiPicker( name ) )			// 6.2.0.0 (2015/02/27) aimai 属性を追加
				.append( makeMustHidden( name ) );			// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
		}
		else if( "no".equals( tdFlag ) ) {
			rtn.append( "<span class=\"label\">" )
				.append( getLongLabel() )					// 4.0.0 (2005/01/31)
				.append( CLM )
				.append( "</span>" )
				.append( XHTMLTag.textarea( getAttributes() ) )
				.append( makeAimaiPicker( name ) )			// 6.2.0.0 (2015/02/27) aimai 属性を追加
				.append( makeMustHidden( name ) );			// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
		// "flase".equals( tdFlag ) の条件
		}
		else {
			rtn.append( XHTMLTag.textarea( getAttributes() ) )
				.append( makeAimaiPicker( name ) )			// 6.2.0.0 (2015/02/27) aimai 属性を追加
				.append( makeMustHidden( name ) );			// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
		}

		return rtn.toString() ;
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag 名前を指定します。
	 *
	 * @param   name 名前
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】高さ(行数)を指定します。
	 *
	 * @og.tag 高さ(行数)を指定します。
	 *
	 * @param   rows 高さ(行数)
	 */
	public void setRows( final String rows ) {
		set( "rows",getRequestParameter( rows ) );
	}

	/**
	 * 【HTML】横幅(カラム数)を指定します。
	 *
	 * @og.tag 横幅を指定します。
	 *
	 * @param   cols 横幅(カラム数)
	 */
	public void setCols( final String cols ) {
		set( "cols",getRequestParameter( cols ) );
	}

	/**
	 * 【TAG】テーブル形式の &lt;td&gt; タグを使用するかどうか[yes/no/false]を指定します(初期値:yes)。
	 *
	 * @og.tag
	 * 通常(初期値:yes)は、テーブル形式の&lt;td&gt; タグを使用して、ラベルとフィールドを整列させます。
	 * no の場合は、tdタグを使用せず、ラベルとフィールドを連続して出力します。
	 * false は、ラベルを出力せず、フィールドのみ出力します。
	 *  &lt;og:input name="PN" td="false" /&gt;
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) td,colspan,rowspan 属性の追加。
	 *
	 * @param   flag tdタグ使用 [yes:tdタグを使用/no:ラベルとフィールド/false:フィールドのみ]
	 */
	public void setTd( final String flag ) {
		final String td = nval( getRequestParameter( flag ),tdFlag );

		if( check( td , TD_SET ) ) {
			tdFlag = td;
		}
		else {
			final String errMsg = "Error in SelectTag [td]: "
						 + td + "  in [" + String.join( ", " , TD_SET ) + "] only used.";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】フィールド部分の colspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ラベルとフィールドが、別のtdで囲われます。しかし、場合によっては、
	 * フィールド部の長いカラムがある場合、上下のカラムとの位置関係がおかしくなり、
	 * ラベルとフィールドが離れて表示されるケースがあります。
	 * この場合、長いフィールド部を持つカラムに、colspan 属性を指定すると、
	 * その分のカラムを割り当てることにより、上下のカラムとの位置関係を
	 * 調節することが可能になります。
	 * 通常は、3 を指定して、tdが、フィールド、ラベル、フィールドの
	 * 組み合わせ部分に、長いフィールドが適用されるように使用します。
	 *
	 *  &lt;og:input name="PN" colspan="3" /&gt;
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) td,colspan,rowspan 属性の追加。
	 *
	 * @param   sp フィールド部分のcolspan
	 */
	public void setColspan( final String sp ) {
		colspan = nval( getRequestParameter( sp ),colspan );
		if( colspan.length() > 0 ) {
			colspan = " colspan=\"" + colspan + "\" ";
		}
	}

	/**
	 * 【TAG】ラベル、フィールド共に rowspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ラベルとフィールドが、別のtdで囲われます。rowspan は、
	 * そのどちらの TDフィールドにも、設定します。
	 * 初期値は、rowspan を付けないです。
	 *
	 * &lt;og:input name="PN" rowspan="3" /&gt;
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) td,colspan,rowspan 属性の追加。
	 *
	 * @param   sp ラベル、フィールド部分のrowspan
	 */
	public void setRowspan( final String sp ) {
		rowspan = nval( getRequestParameter( sp ),rowspan );
		if( rowspan.length() > 0 ) {
			rowspan = " rowspan=\"" + rowspan + "\" ";
		}
	}

	/**
	 * 【TAG】リアルタイムチェックを行うかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * カラム単位でリアルタイムチェックを行うかどうかを設定をします。
	 * タグに独自属性としてrealTimeCheck="true/false"を追記します。
	 * falseを設定した場合にチェックが行われなくなります。
	 * 初期値はリアルタイムチェックを行う（true）です。
	 * 
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 * 引数の文字列の、true/false 判定を行っていません。そのままの文字列が設定されます。
	 * JavaScript 側では、false 判定のみ行っているので、不正な文字列の場合は、
	 * 初期値(true:リアルタイムチェックを行う)が適用されます。
	 * これは、Ver5 とﾛｼﾞｯｸを合わせておくための処置です。
	 *
	 * @og.rev 5.9.32.2 (2018/05/18) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) Ver5 とﾛｼﾞｯｸを合わせます。
	 *
	 * @param   flag リアルタイムチェックを行うかどうか [true:行う/false:行わない]
	 */
	public void setUseRealTimeCheck( final String flag ) {
		// optionAttributes扱いで登録します。
		final String rtcf = getRequestParameter( flag );

		if( StringUtil.isNotNull( rtcf ) ){
			add( "optionAttributes", "realTimeChk=\"" + rtcf + "\"" );			// 連結時のスペースは、Attributes クラスで処理
		}

//		// optionAttributes扱いで登録します。
//		final boolean useRtc = StringUtil.nval( getRequestParameter( flag ) , true );
//
//		if( !useRtc ) {		// true 以外の場合のみ、optionAttributes に属性を追加します。
//			add( "optionAttributes", "realTimeChk=\"false\"" );			// 連結時のスペースは、Attributes クラスで処理
//		}
	}
}
