/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;							// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;						// 6.1.1.0 (2015/01/17)

/**
 * ｶﾗﾑの編集ﾊﾟﾗﾒｰﾀｰのSQL文の実行結果より、datalistを作成して
 * 入力候補となるﾃﾞｰﾀﾘｽﾄを定義する編集用ｴﾃﾞｨﾀｰｸﾗｽです。
 * datalist は、HTML5 から採用されたﾀｸﾞです。
 *
 * 編集ﾊﾟﾗﾒｰﾀには、datalistを作成するための、SQL文を記述します。
 * このSQL文は、select KEY,LABEL from xx ･･･ という構文で、KEY部分とLABEL部分が
 * 選択されます。
 * datalist 自身が、HTML5からの新機能なので、現時点では、これ以上の機能はありません。
 * 将来的に、DBMENU などと同様に、第三ｶﾗﾑ以降を利用可能になると思いますので、
 * 今は使わないでください。（将来の機能追加時に互換性問題を引き起こすかもしれませんので）
 *
 * 入力ﾌｨｰﾙﾄﾞとdatalistﾀｸﾞとの関係付は、ｶﾗﾑIDに、"ｶﾗﾑID.sel" で結びつけます。
 *
 * ただし、値なしのOptionを含めても(addNoValue="true")、HTML5の仕様上、
 * 値なしのOptionは表示されません。
 *
 * &lt;input name="ｶﾗﾑID" list="ｶﾗﾑID.sel" /&gt;
 * <del>&lt;div style="display:none;"&gt;</del>
 *   &lt;datalist id="ｶﾗﾑID.sel"&gt;
 *     &lt;option value="KEY1"&gt;LABEL1&lt;/option&gt;
 *     &lt;option value="KEY2"&gt;LABEL2&lt;/option&gt;
 *     &lt;option value="KEY3"&gt;LABEL3&lt;/option&gt;
 *   &lt;/datalist&gt;
 * <del>&lt;/div&gt;</del>
 *
 * 7.0.1.4 (2018/11/26) autocomplete="off" を初期設定します。
 *
 * <del>divﾀｸﾞは、HTML5 非対応ﾌﾞﾗｳｻﾞを使用した場合、datalist の option がそのまま
 * ﾃｷｽﾄとして見えてしまうのを避けるためです。</del>
 *
 * 一覧表出力時の getValue( int ,String ) 処理では、Selection ｵﾌﾞｼﾞｪｸﾄの
 * ｷｬｯｼｭ機能を利用して、同一Selection ｵﾌﾞｼﾞｪｸﾄの間は、datalist は、
 * １度しか、出力しない様に制御しています。これにより、共有のdatalist を使用する為、
 * HTMLの出力ﾃﾞｰﾀ量を抑えることが可能になります。
 * (ｷｬｯｼｭを利用しないと100行出力すると100個のdatalistを出力する事になります。)
 * <del>(同様の機能を持つ INDBMENU では、行ごとにﾌﾟﾙﾀﾞｳﾝﾃﾞｰﾀを作成しています。)</del>
 * ただし、行単位にSQLの条件を変える機能(AAA:BBB:CCC:DDD引数)が指定された場合は、
 * 行ごとに出力します。
 *
 * 各ｶﾗﾑの値(value値)に、AAA:BBB:CCC:DDD という値を設定できます。これは、
 * $1,$2,$3,$4 に割り当てなおして、QUERYを実行します。また、$1 は、本来の値として、
 * ﾒﾆｭｰの初期値設定等に使用します。上記の例では、AAA が値で、それ以降は、
 * 引数になります。
 * 又、$Cには自分自身のｶﾗﾑ名を割り当てます。
 * この機能を使用すれば、動的ﾒﾆｭｰを行ごとに条件を変えて作成することが
 * 可能になります。
 * 例：select KEY,LABEL from xx where KUBUN='$2' and CDK='$3'
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
 * 変数は、""(ｾﾞﾛ文字列)として、扱われます。
 *
 * ｶﾗﾑの表示に必要な属性は、DBColumn ｵﾌﾞｼﾞｪｸﾄ より取り出します。
 * このｸﾗｽは、DBColumn ｵﾌﾞｼﾞｪｸﾄ毎に1つ作成されます。
 *
 * @og.rev 5.7.4.3 (2014/03/28) 新規作成
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
 * @og.group ﾃﾞｰﾀ編集(HTML5)
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class Editor_DATALIST extends AbstractEditor implements SelectionCellEditor {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.5.2.0 (2023/07/14)" ;

	// 5.7.5.0 (2014/04/04) datalist 使用時は、display:none にして、HTML5未対応のﾌﾞﾗｳｻﾞに備える。
	private static final String DIV1 = "<div style=\"display:none;\">" ;
	private static final String DIV2 = "</div>" ;
	private static final String AUTO = "autocomplete" ;		// 7.0.1.4 (2018/11/26)
	private static final String OFF  = "off" ;				// 7.0.1.4 (2018/11/26)

	private final String query	;
	private final String dbid	;
	private final String lang	;
	private final String useSLabel;							// 6.2.0.0 (2015/02/27) SLABEL 対応
	private final String addKeyLabel;						// 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式
	/** ﾃﾞｰﾀのｺｰﾄﾞ情報 */
	private final Selection selection;						// 8.5.2.0 (2023/07/14) Add
	/** ｼｰｹﾝｽﾌﾗｸﾞ */
	private final boolean seqFlag;							// 8.5.2.0 (2023/07/14) Add

	private Selection bkSel		;							// 5.7.5.0 (2014/04/04) Selection ｵﾌﾞｼﾞｪｸﾄのｷｬｯｼｭ機能

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ。
	 * このｺﾝｽﾄﾗｸﾀｰで、基本ｵﾌﾞｼﾞｪｸﾄを作成します。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 */
	public Editor_DATALIST() {
		super();						// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		query		= null;
		dbid		= null;
		lang		= null;
		useSLabel	= "auto";								// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel	= null;									// 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式
		selection	= null;									// 8.5.2.0 (2023/07/14) Add
		seqFlag		= false;								// 8.5.2.0 (2023/07/14) Add
	}

	/**
	 * DBColumnｵﾌﾞｼﾞｪｸﾄを指定したprivateｺﾝｽﾄﾗｸﾀｰ。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 7.0.1.4 (2018/11/26) autocomplete="off" を初期設定します。
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 *
	 * @param	clm	DBColumnｵﾌﾞｼﾞｪｸﾄ
	 */
	private Editor_DATALIST( final DBColumn clm ) {
		super( clm );
		// 7.0.1.4 (2018/11/26) autocomplete="off" を初期設定します。
		if( isNull( attributes.get( AUTO ) ) ) { attributes.set( AUTO,OFF ); }

		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		query		= clm.getEditorParam();
		dbid		= clm.getDbid();
		lang		= clm.getLang();						// 4.0.0.0 (2006/11/15)
		useSLabel	= clm.getUseSLabel();					// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel	= clm.getAddKeyLabel();					// 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式
		seqFlag		= "SEQ".equals( query );				// 8.5.2.0 (2023/07/14) Add

		// 編集ﾊﾟﾗﾒｰﾀが未指定の場合は、ﾃﾞｰﾀのｺｰﾄﾞ情報を取得 8.5.2.0 (2023/07/14) Add
		if( query == null || query.isEmpty() || seqFlag ) {
			selection = SelectionFactory.newSelection( "MENU",clm.getCodeData(),addKeyLabel );
		}
		else {
			selection = null;
		}

		// 3.5.5.9 (2004/06/07)
//		if( query == null || query.isEmpty() ) {
		// 編集ﾊﾟﾗﾒｰﾀ 又は ﾃﾞｰﾀのｺｰﾄﾞ が未指定の場合 8.5.2.0 (2023/07/14) Modify
		if( ( query == null || query.isEmpty() ) && selection == null ) {
			final String errMsg = "DATALIST Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 各ｵﾌﾞｼﾞｪｸﾄから自分のｲﾝｽﾀﾝｽを返します。
	 * 自分自身をｷｬｯｼｭするのか、新たに作成するのかは、各ｻﾌﾞｸﾗｽの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnｵﾌﾞｼﾞｪｸﾄ
	 * @return	CellEditorｵﾌﾞｼﾞｪｸﾄ
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DATALIST( clm );
	}

	/**
	 * ﾃﾞｰﾀの編集用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、ﾒﾆｭｰの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ｾﾞﾛ文字列)として、扱われます。
	 * 又、$Cには自分自身のｶﾗﾑ名を割り当てます。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) datalist 使用時は、display:none にして、HTML5未対応のﾌﾞﾗｳｻﾞに備える。
	 * @og.rev 5.7.6.2 (2014/05/16) list属性とid属性の結びつきを、name+".sel" に変更
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.4.5.3 (2016/05/13) value は、ｺﾛﾝ区切りの先頭だけ分離する。
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 *
	 * @param	value	入力値
	 * @return	ﾃﾞｰﾀの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		final String newValue = StringFormat.getValue( value );				// 6.4.5.3 (2016/05/13) ｺﾛﾝ区切りの先頭だけ

		// input ﾀｸﾞの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name )									// 4.3.6.0 (2009/04/01)
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
						.add( "list"	, name + ".sel" )							// datalistﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
						.add( "value"	, newValue )								// 6.4.5.3 (2016/05/13)
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

		final boolean useSlbl = "true".equalsIgnoreCase( useSLabel );			// 6.2.0.0 (2015/02/27)

		// datalist ﾀｸﾞの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
//		final String dltag = getOption(
//					new TagBuffer( "datalist" )
//						.add( "id"	, name + ".sel" )							// inputﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
//					, value
//					, false														// ｷｬｯｼｭは使用しない。(つまり、null は返らない)
//					, useSlbl													// 6.2.0.0 (2015/02/27) SLABEL 対応
//				).makeTag() ;

		// 8.5.2.0 (2023/07/14) Modify
		final String dltag;
		// ﾃﾞｰﾀのｺｰﾄﾞ情報からdatalist作成
		if( selection != null ) {
			dltag = new TagBuffer( "datalist" )
							.add( "id"		, name + ".sel" )					// inputﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
							.addBody( selection.getOption( value,seqFlag,useSlbl ) )
							.makeTag();
		}
		// 編集ﾊﾟﾗﾒｰﾀからdatalist作成
		else {
			dltag = getOption(
					new TagBuffer( "datalist" )
						.add( "id"	, name + ".sel" )							// inputﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
					, value
					, false														// ｷｬｯｼｭは使用しない。(つまり、null は返らない)
					, useSlbl													// 6.2.0.0 (2015/02/27) SLABEL 対応
				).makeTag() ;
		}

		// DIV1 の display:none は、datalist の optionのBODY部が、HTML5 以外では表示されてしまうのを防ぐため。
		return intag + CR + DIV1 + dltag + DIV2 + CR;
	}

	/**
	 * name属性を変えた、ﾃﾞｰﾀ表示/編集用のHTML文字列を作成します。
	 * ﾃｰﾌﾞﾙ上の name に 行番号を付加して、名前_行番号 で登録するｷｰを作成し、
	 * ﾘｸｴｽﾄ情報を１つ毎のﾌｨｰﾙﾄﾞで処理できます。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、ﾒﾆｭｰの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ｾﾞﾛ文字列)として、扱われます。
	 * 又、$Cには自分自身のｶﾗﾑ名を割り当てます。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) datalist 使用時は、display:none にして、HTML5未対応のﾌﾞﾗｳｻﾞに備える。
	 * @og.rev 5.7.5.0 (2014/04/04) Selection ｵﾌﾞｼﾞｪｸﾄのｷｬｯｼｭ機能
	 * @og.rev 5.7.6.2 (2014/05/16) list属性とid属性の結びつきを、name+".sel" に変更
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.4.5.3 (2016/05/13) value は、ｺﾛﾝ区切りの先頭だけ分離する。
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 *
	 * @param	row		行番号
	 * @param	value	入力値
	 * @return	ﾃﾞｰﾀ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String name2 = name + HybsSystem.JOINT_STRING + row ;

		// 5.7.5.0 (2014/04/04) Selection ｵﾌﾞｼﾞｪｸﾄのｷｬｯｼｭ機能 (true:使用可能)
		final boolean useSelCache = value != null && value.indexOf( ':' ) < 0 ;

		final String listId = useSelCache ? name : name2;						// ｷｬｯｼｭを使用する場合は、共通の name を使う。

		final String newValue = StringFormat.getValue( value );					// 6.4.5.3 (2016/05/13) ｺﾛﾝ区切りの先頭だけ

		// input ﾀｸﾞの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name2 )										// 4.3.6.0 (2009/04/01)
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
						.add( "list"	, listId + ".sel" )								// datalistﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
						.add( "value"	, newValue )									// 6.4.5.3 (2016/05/13)
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,newValue );										// 6.4.5.3 (2016/05/13)

		final boolean useSlbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );	// 6.2.0.0 (2015/02/27)

		// datalist ﾀｸﾞの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
//		final TagBuffer dltag = getOption(
//					new TagBuffer( "datalist" )
//						.add( "id"	, listId + ".sel" )									// inputﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
//					, value
//					, useSelCache
//					, useSlbl															// 6.2.0.0 (2015/02/27) SLABEL 対応
//				);

		// 8.5.2.0 (2023/07/14) Modify
		final TagBuffer dltag;
		// ﾃﾞｰﾀのｺｰﾄﾞ情報からdatalist作成
		if( selection != null ) {
			dltag = new TagBuffer( "datalist" )
							.add( "id"		, name + ".sel" )							// datalistﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
							.addBody( selection.getOption( value,seqFlag,useSlbl ) );
		}
		// 編集ﾊﾟﾗﾒｰﾀからdatalist作成
		else {
			dltag = getOption(
					new TagBuffer( "datalist" )
						.add( "id"	, listId + ".sel" )									// datalistﾀｸﾞとの関係付けるためのｷｰﾜｰﾄﾞ
					, value
					, useSelCache
					, useSlbl															// 6.2.0.0 (2015/02/27) SLABEL 対応
				);

		}

		// ｷｬｯｼｭが効くと、getOption の戻り値は、null になる。
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転注意
		return dltag == null
					? intag + CR
					: intag + CR + DIV1 + dltag.makeTag( row,newValue ) + DIV2 + CR ;		// 6.4.5.3 (2016/05/13)
	}

	/**
	 * 初期値が選択済みの 選択肢(ｵﾌﾟｼｮﾝ)をTagBuffer に反映します。
	 * このｵﾌﾟｼｮﾝは、引数の値を初期値とするｵﾌﾟｼｮﾝﾀｸﾞ作成し、TagBuffer
	 * に値を設定して返します。
	 *
	 * 第３引数は、Selection ｵﾌﾞｼﾞｪｸﾄのｷｬｯｼｭ機能を使用するかどうか指定します。
	 * true で、使用する事を前提に、ﾁｪｯｸを行います。
	 * DBMENU など、他のﾒｿｯﾄﾞでは、ﾗﾍﾞﾙ(短)の使用有無として使用しているため、異なります。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、ﾒﾆｭｰの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ｾﾞﾛ文字列)として、扱われます。
	 * 又、$Cには自分自身のｶﾗﾑ名を割り当てます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	buf			ﾀｸﾞ文字列のﾊﾞｯﾌｧｰ
	 * @param	value		選択されている値
	 * @param	useSelCache	Selection ｵﾌﾞｼﾞｪｸﾄのｷｬｯｼｭ機能を使用するかどうか。
	 * @param	useSlbl		ﾗﾍﾞﾙ(短)をﾍﾞｰｽとしたｵﾌﾟｼｮﾝ表示を行うかどうか。
	 * @return	ｵﾌﾟｼｮﾝﾀｸﾞ
	 */
	private TagBuffer getOption( final TagBuffer buf,final String value,final boolean useSelCache,final boolean useSlbl ) {

		final StringFormat format = new StringFormat( query, value, name );
		final String newQuery = format.format();

		// 6.2.0.0 (2015/02/27) ｷｰ:ﾗﾍﾞﾙ形式
		final Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang, addKeyLabel );

		if( useSelCache ) {
			if( selection == bkSel ) { return null; }		// SelectionFactory で生成したｷｬｯｼｭの確認なので、== で判定しておきます。
			bkSel = selection ;
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String newValue = format.getValue();
		return buf.addBody( selection.getOption( newValue, false, useSlbl ) );	// 6.2.0.0 (2015/02/27) SLABEL 対応
	}
}
