/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * MENU エディターは、カラムのデータをコードリソースに対応した
 * プルダウンメニューで編集する場合に使用するクラスです。
 *
 * 編集パラメータに"SEQ"と記述することで正方向にしか選べないプルダウンメニューを実現できます。
 * これにより、シーケンスにステータスを順に挙げていくような、プルダウンメニュー
 * を作成することが出来ます。(逆に戻れないメニュー)
 *
 *  カラムの表示に必要な属性は、DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_MENU extends AbstractEditor implements SelectionCellEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.5.1 (2019/09/27)" ;

	// 3.2.3.0 (2003/06/06) final を削除。サブクラスからアクセスできるように変更。
	/** セレクションオブジェクト */
	private final Selection	selection ;		// 6.0.4.0 (2014/11/28) サブクラスからの参照を廃止
	private final boolean addNoValue ;		// 3.5.5.7 (2004/05/10)
	/** シーケンスフラグ */
	private final boolean seqFlag	;		// 6.0.4.0 (2014/11/28) サブクラスからの参照を廃止
	private final String useSLabel	;		// 5.5.1.0 (2012/04/03)

	private String errMsg	;				// 6.0.4.0 (2014/11/28)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 */
	public Editor_MENU() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		// 4.3.4.4 (2009/01/01)
		selection	= null;
		addNoValue	= false;		// 3.5.5.7 (2004/05/10)
		seqFlag		= false;		// 3.6.0.6 (2004/10/22)
		useSLabel	= "auto";		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.1.1 (2003/07/03) name , attributes 属性を final にする。
	 * @og.rev 3.5.4.2 (2003/12/15) makeCodeSelection メソッドを CodeSelectionクラスに変更。
	 * @og.rev 3.5.5.7 (2004/05/10) SelectionFactory を使用して、オブジェクト作成
	 * @og.rev 3.5.5.7 (2004/05/10) addNoValue 属性を追加します。
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 SELECT_KEY を隠蔽します。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0.0 (2005/01/31) Selection_CODE の作成の引数を CodeData に変更。
	 * @og.rev 4.0.0.0 (2007/11/07) SelectionオブジェクトをDBColumnから取得
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) optionAttributes は、コンストラクタで設定します。
	 * @og.rev 6.0.4.0 (2014/11/28) サブクラスで タイプを指定するための対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式(KVMENU)で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.2.6.0 (2015/06/19) type別Selectionの場合、ラベルリソースを使用する為、言語を引数で渡す。
	 * @og.rev 6.4.4.2 (2016/04/01) nameのfinal化
	 * @og.rev 6.9.5.0 (2018/04/23) useMultiSelect 廃止
	 * @og.rev 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
	 *
	 * @param	clm		DBColumnオブジェクト
	 * @param	type	MENUのサブクラスのタイプ(FILES,KEYVAL,NUM,HM,YMD)
	 */
	protected Editor_MENU( final DBColumn clm , final String type ) {
	//	super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		super( clm );	// 6.4.4.2 (2016/04/01) nameのfinal化
	//	name		= clm.getName();
		addNoValue	= clm.isAddNoValue() ;						// 3.5.5.7 (2004/05/10)
		seqFlag		= "SEQ".equals( clm.getEditorParam() );		// 3.6.0.6 (2004/10/22)
		useSLabel	= clm.getUseSLabel() ;						// 5.5.1.0 (2012/04/03)

		final String lang		 = clm.getLang();				// 6.2.6.0 (2015/06/19)
		final String addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式
		final String disabled    = clm.isWritable() ? null : "disabled" ;

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( "disabled"	,disabled )
					.set( clm.getEditorAttributes() );				// #addAttributes( Attributes ) の代替え

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
		tagBuffer.add( XHTMLTag.selectAttri( attributes ) );
//				 .add( attributes.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)

		// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		selection = type == null
						? SelectionFactory.newSelection( "MENU",clm.getCodeData(),addKeyLabel )	// 6.2.0.0 (2015/02/27)
						: SelectionFactory.newSelection( type,clm.getEditorParam(),lang );

//		// 6.0.4.0 (2014/11/28) useMultiSelect は、selection ではなく、colomn から取得する。
//		// 6.9.5.0 (2018/04/23) useMultiSelect 廃止
//		if( clm.useMultiSelect() ) { tagBuffer.add( "onkeydown" , "setKeySelect(this);" ); }

		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			errMsg = "codeData が未設定です。"
								+ " name="  + name
								+ " label=" + clm.getLabel()
								+ " editType="  + clm.getEditor() ;
			System.out.println( errMsg );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 * @og.rev 6.0.4.0 (2014/11/28) サブクラスで タイプを指定するための対応
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_MENU( clm , null );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 3.5.5.7 (2004/05/10) addNoValue 属性を追加します。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示のみで、ツールチップ表示を行う。
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) useSLabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + "</span>";
		}

		final boolean useSlbl = "true".equalsIgnoreCase( useSLabel );		// 5.5.1.0 (2012/04/03)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "select" )
							.add( "name"	, name )										// 4.3.6.0 (2009/04/01)
							.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
							.add( tagBuffer.makeTag() )
							.addBody( Selection.NO_VALUE_OPTION , addNoValue )				// 5.5.1.0 (2012/04/03)
							.addBody( selection.getOption( value,seqFlag,useSlbl ) )
							.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 2.0.0.3 (2002/09/26) optionAttributes 属性に "$i" を使うとその行数に置き換る機能を追加。
	 * @og.rev 3.1.0.0 (2003/03/20) 名前と行番号の区切り記号を "^" から "__" に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 3.5.5.7 (2004/05/10) addNoValue 属性を追加します。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示のみで、ツールチップ表示を行う。
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) useSLabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + " row=" + row + "</span>";
		}

		final boolean useSlbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );		 // 5.5.1.0 (2012/04/03)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String newName = name + HybsSystem.JOINT_STRING + row; // 4.3.6.0 (2009/04/01)
		return new TagBuffer( "select" )
							.add( "name"	, newName )											// 4.3.6.0 (2009/04/01)
							.add( "id"		, newName , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
							.add( tagBuffer.makeTag() )
							.addBody( Selection.NO_VALUE_OPTION , addNoValue )					// 5.5.1.0 (2012/04/03)
							.addBody( selection.getOption( value,seqFlag,useSlbl ) )
							.makeTag( row,value );
	}
}
