/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * OCR2 エディターは、tesseract.js を利用した、カメラ映像で取り込んだvideoを
 * canvasにｷｬﾌﾟﾁｬしてから、文字に変換して テキストエリアに書き出すクラスです。
 *
 * 基本的な構造は、カメラ映像描画開始ボタン、静止画キャプチャボタン、映像表示領域(video)、
 * 静止画描画領域(canvas)、進捗(progressbar)、textarea で構成されます。
 * 映像表示領域(video)でｷｬﾌﾟﾁｬすると、静止画を同じ場所に上書きします。もう一度押すと再度映像に切り替わります。
 * textarea の name 以外は、固定です。よって、各ページに、1つしか設定できません。
 *
 *    &lt;button type='button' id='vidStart' onClick='videoStart()'&gt;Video Start&lt;/button&gt;
 *    &lt;button type='button' id='capStart' onClick='capture()'&gt;Capture&lt;/button&gt;&lt;br /&gt;
 *    &lt;div id='videotop' style='display:flex;' &gt;
 *        &lt;video id='player' autoplay style='background-color: black;position: absolute;z-index: 1;'&gt;&lt;/video&gt;
 *        &lt;canvas id='snapshot' style='visibility:hidden;z-index: 2; '&gt;&lt;/canvas&gt;
 *    &lt;/div&gt;
 *    &lt;progress id='progressbar' min='0' max='1' value='0' &gt; &lt;/progress&gt;&lt;br /&gt;
 *    &lt;textarea name='outdata' id='outdata' rows='10'cols='80'&gt; &lt;/textarea&gt;
 *
 * script に CDNサービス を使うと、無線環境(iPad等)ではものすごく遅くなったため、ローカルに配置することにします。
 * &lt;script src="https://unpkg.com/tesseract.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 *
 * script は、tesseract.min.js を使います。現在、1画面1つしかカメラは使えません。
 * これらは、使用する画面に、組み込んでください。
 * &lt;script src="{&#064;SYS.JSP}/option/tesseract.min.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 * &lt;script src="{&#064;SYS.JSP}/option/videocamera.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 *
 * を使用するページに設定します。
 *
 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
 * @og.group データ編集
 *
 * @version  7.4
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Editor_OCR2 extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.1 (2021/05/21)" ;

//	// 7.4.2.2 (2021/05/28) ｼｽﾃﾑ定数のJSPを使用します。(※ SYS.JSP + SYS.IMAGE_DIR)
//	private static final String JSP_OPT =  HybsSystem.sys( "JSP" ) + "/option/" ;

//	private static final String JS_SRC = "<script src='" + JSP_OPT + "tesseract.min.js' ><!-- --></script>"
//								+ CR +	 "<script src='" + JSP_OPT + "videocamera.js' ><!-- --></script>" ;

	private static final String BASE_HTML =
									"<button type='button' id='vidStart' onClick='videoStart()'>Video Start</button>"
							+ CR +	"<button type='button' id='capStart' onClick='capture()'>Capture</button><br />"
							+ CR +	"<div id='videotop' style='display:flex;' >"
							+ CR +	"  <video id='player' autoplay style='background-color: black;position: absolute;z-index: 1;'></video>"
							+ CR +	"  <canvas id='snapshot' style='visibility:hidden;z-index: 2; '></canvas>"
							+ CR +	"</div>"
							+ CR +	"<progress id='progressbar' min='0' max='1' value='0' > </progress><br />" ;

	/** 列1 */ protected  String	cols1	;
	/** 列2 */ protected  String	cols2	;
	/** 行1 */ protected  String	rows1	;
	/** 行2 */ protected  String	rows2	;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 */
	public Editor_OCR2() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * コンストラクター。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	protected Editor_OCR2( final DBColumn clm ) {
		super( clm );
		final String  disabled = clm.isWritable() ? null : "disabled" ;

		final int r1 = clm.getTotalSize()/Integer.parseInt(size1) + 1;
		rows1 = String.valueOf( r1 );

		final int r2 = clm.getTotalSize()/Integer.parseInt(size2) + 1;
		rows2 = String.valueOf( r2 );

		// size に、"rows,cols" を指定できるように変更
		final String param = StringUtil.nval( clm.getEditorParam(),clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			final int st = param.indexOf( ',' );
			if( st > 0 ) {
				rows1 = param.substring( 0,st );
				rows2 = rows1 ;
				cols1 = param.substring( st+1 );
				cols2 = cols1;
			}
		}

		// size1,size2 を使わずに、cols1,cols2 を使用。
		if( cols1 == null || cols2 == null ) {
			cols1	= size1  ;
			cols2	= size2	 ;
		}

		// Attributesの連結記述
		attributes = new Attributes()
					.set( "disabled"	, disabled )
					.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
					.add( "class"		, clm.getDbType() );

		tagBuffer.add( XHTMLTag.textareaAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_OCR2( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param   value 入力値
	 *
	 * @return  データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// TagBufferの連結記述
		return BASE_HTML
				+	new TagBuffer( "textarea" )
						.add( "name"	, name )
						.add( "id"		, "outdata" )				// ID 固定です。
						.add( "cols"	, cols2 )
						.add( "rows"	, rows2 )
						.add( tagBuffer.makeTag() )
						.addBody( value )
						.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String newName = name + HybsSystem.JOINT_STRING + row;

		// TagBufferの連結記述
		return BASE_HTML
				+	new TagBuffer( "textarea" )
						.add( "name"	, newName )
						.add( "id"		, "outdata" )				// ID 固定です。
						.add( "cols"	, cols2 )
						.add( "rows"	, rows2 )
						.add( tagBuffer.makeTag() )
						.addBody( value )
						.makeTag( row,value );
	}
}
