/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * IMAGE レンデラは、value で指定したファイルをimgタグで表示する場合に
 * 使用するクラスです。
 * imgのサイズはレンデラーパラメータに、設定したい属性を記述します。
 * 例えば、width='600px' height='450px' border='0' などです。
 * src属性はvalueを設定し、name属性はカラム名(複数行の場合は、行番号付きの名前)
 * alt属性は、そのままvalueを設定しておきます。
 * img の親には、div要素とclass="Renderer_IMAGE" を付与しておきます。
 *
 * &lt;div class="Renderer_IMAGE"&gt;
 *     &lt;img name="カラム名" src="valueの値" alt="valueの値" レンデラーパラメータ &gt;
 * &lt;/div&gt;
 *
 * カラムの表示に必要な属性は、DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 7.4.2.0 (2021/04/30) 新規作成
 * @og.group データ表示
 *
 * @version  7.4
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Renderer_IMAGE extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.0 (2021/04/30)" ;

	private static final String DIV_ST = "<div class=\"Renderer_IMAGE\">";
	private static final String DIV_ED = "</div>";

	private  String		name;
	private  String		param;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_IMAGE() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 7.4.2.0 (2021/04/30) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_IMAGE( final DBColumn clm ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		super();

		name  = clm.getName();

//		param = StringUtil.nval( clm.getRendererParam(), null );	// 空文字列はあえてnullにする。
		param = StringUtil.nvalAdd( clm.getRendererParam() ,
									clm.getRendererAttributes().get( "optionAttributes" ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_IMAGE( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.4.2.0 (2021/04/30) 新規作成
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return makeImae( name, value );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.4.2.0 (2021/04/30) 新規作成
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String newName = name + HybsSystem.JOINT_STRING + row;
		return makeImae( newName, value );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 7.4.2.0 (2021/04/30) 新規作成
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return value == null ? "" : value;
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.4.2.0 (2021/04/30) 新規作成
	 *
	 * @param	name	カラム名
	 * @param   value	入力値 表示するファイルのアドレス
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	private String makeImae( final String name,final String value ) {
		return DIV_ST
			+	new TagBuffer( "img" )
					.add( "name"	, name	)
					.add( "src"		, value	)
					.add( "alt"		, value	)
					.addOptions( param		)	// タグの属性として追加。nullの場合は、何もしない。
					.makeTag()
			+	DIV_ED;
	}
}
