/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * TEXTAREA レンデラは、カラムのデータをテキストエリアで表示する場合に
 * 使用するクラスです。
 * readonlyのテキストエリアでclass=renderer-textareaとして出力し、
 * name属性は付けません。(データは送信されません)
 * エリアの縦、横サイズはエディタのテキストエリアと同様にして算出されます。
 *
 * カラムの表示に必要な属性は、DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 4.3.5.7 (2009/03/22) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class Renderer_TEXTAREA extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.0 (2021/04/30)" ;

	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" ) ;	// 表示フィールドの大きさ
	// viewタグで表示する場合のカラムの大きさ
	private final int VIEW_COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_VIEW_COLUMNS_MAXSIZE" ) ;

//	private final TagBuffer	tagBuffer = new TagBuffer();	// 7.4.2.0 (2021/04/30)

	private  String		rows1;
	private  String		rows2;
	private  String		size1;
	private  String		size2;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_TEXTAREA() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * textareaのサイズを決めるため、sizeとrowを決定する
	 * editorの計算を移植。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_TEXTAREA( final DBColumn clm ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		super();

		// 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
		size1 = String.valueOf( clm.getFieldSize( COLUMNS_MAXSIZE      ) );
		size2 = String.valueOf( clm.getFieldSize( VIEW_COLUMNS_MAXSIZE ) );

		final int r1 = clm.getTotalSize() / Integer.parseInt( size1 ) + 1;
		if( r1 > 5 ) {
			rows1 = "5";
		}
		else {
			rows1 = String.valueOf( r1 );
		}

		final int r2 = clm.getTotalSize() / Integer.parseInt( size2 ) + 1;
		if( r2 > 5 ) {
			rows2 = "5";
		}
		else {
			rows2 = String.valueOf( r2 );
		}

		final String param = StringUtil.nval( clm.getRendererParam(), clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			final int st = param.indexOf( ',' );
			if( st > 0 ) {
				rows1 = param.substring( 0, st );
				rows2 = rows1;
				size1 = param.substring( st + 1 );
				size2 = size1;
			}
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_TEXTAREA( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getRowsColsValue( value==null ? "" : value,size1,rows1 );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		return getRowsColsValue( value==null ? "" : value,size2,rows2 );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return value == null ? "" : value;
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 * @og.rev 7.4.2.0 (2021/04/30) tagBuffer は使っていないので削除
	 *
	 * @param   value 入力値 表示文字列
	 * @param	cols	最小カラム数
	 * @param	rows	最小行数
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	private String getRowsColsValue( final String value,final String cols, final String rows ) {
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "textarea" )
					.add( "cols"	, cols )
					.add( "rows"	, rows )
					.add( "readonly", "readonly" )
					.add( "class"	, "renderer-textarea" )
//					.add( tagBuffer.makeTag() )					7.4.2.0 (2021/04/30)
					.addBody( value )
					.makeTag();
	}
}
