/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;
import java.io.File;										// 6.2.0.0 (2015/02/27)

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;

import org.opengion.fukurou.model.ExcelModel;

/**
 * ﾈｲﾃｨﾌﾞEXCELﾌｧｲﾙの書き出しｸﾗｽです。
 *
 * DefaultTableWriter を継承していますので、ﾗﾍﾞﾙ、名前、ﾃﾞｰﾀの出力部のみ
 * ｵｰﾊﾞｰﾗｲﾄﾞして、MIcrosoft Excelﾌｧｲﾙの出力機能を実現しています。
 *
 * 出力形式は、openXML形式にも対応しています。
 * 出力ﾌｧｲﾙの拡張子が、.xlsならExcel2003のﾊﾞｲﾅﾘ形式、.xlsxならExcel2007の
 * openXML形式で出力されます。
 *
 * useCellStyle      = [false/true];  ﾃﾞｰﾀを書き込んだ範囲に罫線を入れる(true)かどうか(初期値:false)
 * useAutoCellSize   = [false/true];  ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせる(true)かどうか(初期値:false)
 * useActiveWorkbook = [false/true];  ｾﾙの有効範囲を設定する(true)かどうか(初期値:false)
 * pageBreakColumn   = PBCLM;         ｼｰﾄﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)
 * fileBreakColumn   = FBCLM;         ﾌｧｲﾙﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)
 *
 * appendﾓｰﾄﾞの場合は、ｵﾘｼﾞﾅﾙが雛形ﾌｧｲﾙになります。雛形ﾌｧｲﾙを指定した場合は、
 * 同じﾌｧｲﾙ名で無ければｴﾗｰになります。
 *
 * @og.group ﾌｧｲﾙ出力
 *
 * @og.rev 4.3.4.3 (2008/12/22) 一部protected化
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * @og.rev 6.0.1.2 (2014/08/08) ﾛｼﾞｯｸの完全書き換え(保証もできない位･･･)
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class TableWriter_Excel extends TableWriter_Default {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.5.2.0 (2023/07/14)" ;

	private	int		nRowIndex		;						// 6.0.1.2 (2014/08/08) private化
	private String	filename		;						// 3.5.4.3 (2004/01/05)
	private String	directory		;						// 6.0.2.0 (2014/09/19) ﾃﾞｨﾚｸﾄﾘとﾌｧｲﾙを分けて管理
	private String	fileKey			;						// 6.0.2.0 (2014/09/19) ﾃﾞｨﾚｸﾄﾘとﾌｧｲﾙを分けて管理
	private String	sheetName		= "Sheet";				// 3.5.4.3 (2004/01/05) 書き込むSheet名
	private String	fontName		;						// 3.8.5.3 (2006/08/07)
	private short	fontPoint		= -1;					// 3.8.5.3 (2006/08/07)
	private String	refFilename		;						// 3.5.4.3 (2004/01/05)
	private String	refSheetName	;						// 3.5.4.3 (2004/01/05)
	// 5.1.4.0 (2010/03/01) 行番号情報を、出力する(true)/しない(false)を指定
	private boolean	useNumber		= true;
	private int		skipRowCount	;						// 6.0.1.2 (2014/08/08) ﾃﾞｰﾀの書き込み開始位置設定

	private ExcelModel excel		;						// 6.0.2.0 (2014/09/19) 新規作成

	// 6.0.2.0 (2014/09/19) EXCEL関係機能追加
	private boolean	useCellStyle		;					// ﾃﾞｰﾀを書き込んだ範囲に罫線を入れるかどうかを指定します
	private boolean	useAutoCellSize		;					// ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせるかどうかを指定します
	private boolean	useActiveWorkbook	;					// ｾﾙの有効範囲を設定するかどうかを指定します
	private String	fileBreakColumn		;					// ﾌｧｲﾙﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)
	private String	pageBreakColumn		;					// ｼｰﾄﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)
	private boolean noPageBreak			;					// 6.1.0.0 (2014/12/26) pageBreak,fileBreak 修正
	private String	hLinkKeyColumn		;					// Hyperlinkを作成するｷｰとなるｶﾗﾑ名を指定
	private String	hLinkValColumn		;					// Hyperlinkを作成する値となるｶﾗﾑ名を指定

	private String	addTitleSheet		;					// Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します

	private int		pageBreakClmNo		= -1;				// ｼｰﾄﾌﾞﾚｲｸするｷｰとなるｶﾗﾑNo
	private int		fileBreakClmNo		= -1;				// ﾌｧｲﾙﾌﾞﾚｲｸするｷｰとなるｶﾗﾑNo
	private int		hLinkKeyClmNo		= -1;				// Hyperlinkを作成するｷｰとなるｶﾗﾑNo
	private int		hLinkValClmNo		= -1;				// Hyperlinkを作成する値となるｶﾗﾑNo
	private int		tblRow				;					// ｼｰﾄﾌﾞﾚｲｸしても、ﾃﾞｰﾀは続きから出力します。

//	private boolean[] nvar				;					// 6.0.2.0 (2014/09/19) writeData( DBTableModel ,PrintWriter ) で使うｶﾗﾑ情報 8.5.2.0 (2023/07/14) Delete
	private boolean[] isNums			;					// 6.0.2.0 (2014/09/19) writeData( DBTableModel ,PrintWriter ) で使うｶﾗﾑ情報

	// 5.9.12.1 (2016/09/09) ｼｰﾄ上書き設定
	private boolean		sheetOverwrite		;				// 5.9.12.1 (2016/09/09)
	private String[]	recalcSheetNames	;				// 5.9.12.1 (2016/09/09)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_Excel() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBTableModel から 各形式のﾃﾞｰﾀを作成して、PrintWriter に書き出します。
	 * このﾒｿｯﾄﾞは、EXCEL 書き出し時に使用します。
	 *
	 * 雛形ｼｰﾄを使う場合は、雛形ｼｰﾄの後ろに追加していきます。
	 * 最後には、雛形ｼｰﾄとして存在しな部分は、すべて削除します。
	 * 雛形ｼｰﾄを使わない場合は、新規ｼｰﾄに追加していくことになります。
	 * appendﾓｰﾄﾞの場合は、ｵﾘｼﾞﾅﾙが雛形ﾌｧｲﾙになります。
	 * 雛形ﾌｧｲﾙの指定は、無効になります。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応 、useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) ﾛｼﾞｯｸの完全書き換え(保証もできない位･･･)
	 * @og.rev 6.0.2.0 (2014/09/19) EXCEL新機能追加
	 * @og.rev 6.1.0.0 (2014/12/26) Excel関係改善
	 * @og.rev 6.2.0.0 (2015/02/27) ﾌｧｲﾙ引数を、String → File に変更
	 * @og.rev 6.2.2.3 (2015/04/10) 雛形ｼｰﾄにそのままﾃﾞｰﾀを書き込んでいく。ｼｰﾄは削除不要
	 * @og.rev 5.9.12.1 (2016/09/09) sheetOverwrite,recalcSheetNames
	 * @og.rev 6.5.0.0 (2016/09/30) ｾﾙの計算式の再計算をさせる recalcSheetNames 属性の追加。(5.9.12.1 (2016/09/09))
	 *
	 * @see #isExcel()
	 */
	@Override
	public void writeDBTable() {
		if( ! createDBColumn() ) { return ; }		// 親ｸﾗｽを改造。2回実行されない。

		useNumber = isUseNumber();

		// 3.5.6.0 (2004/06/18) 移動
		if( filename == null ) {
			final String errMsg = "ファイルが指定されていません。";
			throw new HybsSystemException( errMsg );
		}

		// appendﾓｰﾄﾞの場合は、ｵﾘｼﾞﾅﾙが雛形ﾌｧｲﾙになります。
		// 雛形ﾌｧｲﾙの指定は、無効になります。
		if( isAppend() ) {
			// 互換性のため、雛形ﾌｧｲﾙが出力ﾌｧｲﾙと同じでない場合はｴﾗｰとします。
			if( refFilename != null && !filename.equalsIgnoreCase( refFilename ) ) {
				final String errMsg = "追加(appendモード)の時、雛形ファイルの指定は無効です。" + CR
								+ " filename=[" + filename + "] , refFilename=[" + refFilename + "]" + CR ;
				throw new HybsSystemException( errMsg );
			}

			refFilename = filename ;
		}

		// 雛形があれば、雛形ﾌｧｲﾙを読み込みます。なければ、ｵﾘｼﾞﾅﾙのﾌｧｲﾙを読み込みます。
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		final boolean isRefFile = refFilename != null && refFilename.length() > 0;
		final String nameUse = isRefFile ? refFilename : filename ;		// 6.3.9.1 (2015/11/27)

		initWriteData();						// 6.0.2.0 (2014/09/19) ｶﾗﾑ関連の初期化

		final boolean isDebug = isDebug();		// 6.1.0.0 (2014/12/26) ﾃﾞﾊﾞｯｸﾞ情報

		// 6.1.0.0 (2014/12/26) useRenderer 時に使用。できるだけﾙｰﾌﾟ内処理を避ける。
		final DBColumn fileBreakClm = fileBreakClmNo >= 0 && isUseRenderer() ? dbColumn[fileBreakClmNo] : null ;
		final DBColumn pageBreakClm = pageBreakClmNo >= 0 && isUseRenderer() ? dbColumn[pageBreakClmNo] : null ;

		// ﾌｧｲﾙﾌﾞﾚｲｸの初期値を求めます。(fileKey は、setFilename(String,String) で、初期値ｾｯﾄ済み)
		if( fileBreakClmNo >= 0 ) { fileKey = table.getValue( tblRow,fileBreakClmNo ); }
		// 処理のﾙｰﾌﾟを抜ける条件は、すべてのﾃﾞｰﾀを書出し終わった場合。fileKey = null をｾｯﾄします。
		while( fileKey != null ) {
			// fileKey は、super.writeDBTable( null ) で、writeData( DBTableModel ,PrintWriter )で、再ｾｯﾄされる。
			final String fileKey_ren = fileBreakClm == null ? fileKey : fileBreakClm.getWriteValue( fileKey );

			// ExcelModelで作成。新規の場合は、nameUseの拡張子(.xlsx/.xls)に応じて、ﾀｲﾌﾟが変わります。
			excel = new ExcelModel( new File( nameUse ) , isRefFile );		// 6.2.0.0 (2015/02/27)

			// 雛形ｼｰﾄを使わない場合のみ、ﾌｫﾝﾄ設定を行う。
			if( !isRefFile ) { excel.setFont( fontName,fontPoint ); }
			// 6.0.2.0 (2014/09/19) 新機能追加
			if( useCellStyle ) { excel.setCellStyle(); }
			excel.useAutoCellSize( useAutoCellSize );
			excel.setRecalcSheetName( recalcSheetNames );					// 6.5.0.0 (2016/09/30) ｾﾙの計算式の再計算をさせる recalcSheetNames 属性の追加。
			excel.setAddTitleSheet( addTitleSheet );

			if( isRefFile ) { excel.setRefSheetName( refSheetName ); }

			// ｼｰﾄﾌﾞﾚｲｸする場合は、sheetName名をあらかじめ取得する。
			if( pageBreakClmNo >= 0 ) { sheetName = table.getValue( tblRow,pageBreakClmNo ); }
			noPageBreak = true ;	// ﾌｧｲﾙﾌﾞﾚｲｸ、ｼｰﾄﾌﾞﾚｲｸの場合は、false をｾｯﾄ
			while( noPageBreak ) {
				// 6.1.0.0 (2014/12/26) ﾃﾞﾊﾞｯｸﾞ情報
				if( isDebug ) { System.out.println( " file=" + fileKey + " , sheet=" + sheetName + " , row=" + tblRow ) ; }

				// ｼｰﾄ名のｾｯﾄ。重複ﾁｪｯｸと新しいｼｰﾄ名の作成処理を行う。
				final String sheetName_ren = pageBreakClm == null ? sheetName : pageBreakClm.getWriteValue( sheetName );
		//		excel.createSheet( sheetName_ren );
				excel.createSheet( sheetName_ren , sheetOverwrite );

				nRowIndex = skipRowCount;		// 6.0.1.2 (2014/08/08) 開始行の指定

				// この super.writeDBTable(null) 内から、各種実態のﾒｿｯﾄﾞが呼ばれる。
				// ｼｰﾄﾌﾞﾚｲｸ,ﾌｧｲﾙﾌﾞﾚｲｸ は、writeData( DBTableModel ,PrintWriter ) で処理される。
				// ｼｰﾄﾌﾞﾚｲｸすると、新しい sheetName がｾｯﾄされ、noPageBreak=false がｾｯﾄされる。
				// ﾌｧｲﾙﾌﾞﾚｲｸすると、新しい fileKey がｾｯﾄされ、noPageBreak=false がｾｯﾄされる。
				super.writeDBTable( null );

				// 指定以降の余計な行を削除
				excel.removeRow( nRowIndex );
			}

			// 6.0.2.0 (2014/09/19) activeWorkbook は、全ｼｰﾄに対して、最後に適用する。
			if( useActiveWorkbook ) { excel.activeWorkbook( false ); }		// false は Cellの末尾削除は行わない。

			// ﾒﾓﾘ中のﾃﾞｰﾀをﾌｧｲﾙに書き込む
			excel.saveFile( new File( directory,fileKey_ren ) );			// 6.2.0.0 (2015/02/27)
		}
	}

	/**
	 * PrintWriter に DBTableModelのﾍｯﾀﾞｰ情報を書き込みます。
	 * ここでは、ﾗﾍﾞﾙ情報を同じﾃﾞｰﾀを出力します。
	 *
	 * @og.rev 7.2.6.1 (2020/07/17) H:Header(ﾍｯﾀﾞｰ)属性追加。
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeLabel2( final DBTableModel table,final PrintWriter writer ) {
		writeLabel( table,writer );
	}

	/**
	 * PrintWriter に DBTableModelのﾗﾍﾞﾙ情報を書き込みます。
	 * 第一ｶﾗﾑ目は、ﾗﾍﾞﾙ情報を示す "#Label" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 * @og.rev 6.2.4.2 (2015/05/29) StringUtil#tagCut(String) をﾗﾍﾞﾙに適用します。
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeLabel( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Label",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし
				nColIndex++;
				continue;
			}

			final String val = StringUtil.tagCut( dbColumn[clm].getLabel() );	// 6.2.4.2 (2015/05/29)
			headerCellValue( val,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのｶﾗﾑのﾗﾍﾞﾙを付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なｾﾙを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelの項目名情報を書き込みます。
	 * 第一ｶﾗﾑ目は、項目名情報を示す "#Name" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期ﾒｿｯﾄﾞ(synchronized付き)を非同期に変更する。
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeName( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Name",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし
				nColIndex++;
				continue;
			}

			final String name = table.getColumnName( clm );
			headerCellValue( name,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのｶﾗﾑの項目名を付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なｾﾙを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのｻｲｽﾞ情報を書き込みます。
	 * 第一ｶﾗﾑ目は、ｻｲｽﾞ情報を示す "#Size" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) DBColumn の size と maxlength の 意味を変更
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeSize( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Size",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし
				nColIndex++;
				continue;
			}

			// 4.0.0 (2005/01/31) ﾒｿｯﾄﾞ名変更
			final String val = String.valueOf( dbColumn[clm].getTotalSize() );
			headerCellValue( val,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのｶﾗﾑのｻｲｽﾞを付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なｾﾙを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に DBTableModelのｸﾗｽ名情報を書き込みます。
	 * 第一ｶﾗﾑ目は、ｻｲｽﾞ情報を示す "#Class" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeClass( final DBTableModel table,final PrintWriter writer ) {
		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( "Class",nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし
				nColIndex++;
				continue;
			}

			final String val = dbColumn[clm].getClassName();
			headerCellValue( val,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのｶﾗﾑのｸﾗｽ名を付けません。
			if( isKeyLbl[i] ) {
				headerCellValue( "",nColIndex++ );
			}
		}

		// 余計なｾﾙを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * PrintWriter に ｾﾊﾟﾚｰﾀｰを書き込みます。
	 * 第一ｶﾗﾑ目は、ｻｲｽﾞ情報を示す "#----" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 */
	@Override
	protected void writeSeparator( final DBTableModel table,final PrintWriter writer ) {
		final String SEP = "----" ;

		short nColIndex = 0;
		excel.createRow( nRowIndex++ );
		if( useNumber ) { headerCellValue( SEP,nColIndex++ ); }

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし
				nColIndex++;
				continue;
			}
			headerCellValue( SEP,nColIndex++ );

			// 6.2.3.0 (2015/05/01) keyLabelは、後ろのｶﾗﾑのｾﾊﾟﾚｰﾀｰを書き込みます。
			if( isKeyLbl[i] ) {
				headerCellValue( SEP,nColIndex++ );
			}
		}

		// 余計なｾﾙを削除
		excel.removeCell( nColIndex );
	}

	/**
	 * ExcelModel#setCellValue(String.int) の第一ｶﾗﾑの場合に、ｺﾒﾝﾄの # を追加する簡易ﾒｿｯﾄﾞです。
	 *
	 * colNo == 0 の場合だけ、引数の val に、"#" を追加します。
	 * これは、openGion特有のｺﾒﾝﾄ行を意味するため、ﾍｯﾀﾞｰﾚｺｰﾄﾞにのみ適用します。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 *
	 * @param	val		設定文字列
	 * @param	colNo	行番号
	 */
	private void headerCellValue( final String val,final int colNo ) {
		if( colNo == 0 ) {
			excel.setCellValue( "#" + val,colNo );
		}
		else {
			excel.setCellValue( val,colNo );
		}
	}

	/**
	 * writeData( DBTableModel ,PrintWriter ) の呼び出し前に、一度だけ実行すればよい処理をまとめました。
	 *
	 * 通常のTableWriterは、１回の DBTableModel の読み込みで、１ﾌｧｲﾙだけ出力します。
	 * そのため、writeData( DBTableModel ,PrintWriter ) の呼び出し処理の初めに、初期化している処理があります。
	 * EXCELでは、Sheet 処理と、File 処理のﾙｰﾌﾟがあり、本来１回だけ初期化すればよいのですが、
	 * writeData( DBTableModel ,PrintWriter ) が複数回呼ばれるため、無駄な処理になってしまいます。
	 * ただし、今までは、ﾛｰｶﾙ変数でしたが、ｲﾝｽﾀﾝｽ変数に変更しています。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 */
	private void initWriteData() {
		final DBTableModel table = getDBTableModel();
		pageBreakClmNo	= table.getColumnNo( pageBreakColumn,false );			// ｼｰﾄﾌﾞﾚｲｸするｷｰとなるｶﾗﾑNo
		fileBreakClmNo	= table.getColumnNo( fileBreakColumn,false );			// ﾌｧｲﾙﾌﾞﾚｲｸするｷｰとなるｶﾗﾑNo

		hLinkKeyClmNo	= table.getColumnNo( hLinkKeyColumn,false );			// Hyperlinkを作成するｷｰとなるｶﾗﾑNo
		hLinkValClmNo	= table.getColumnNo( hLinkValColumn,false );			// Hyperlinkを作成する値となるｶﾗﾑNo

//		nvar	= new boolean[numberOfColumns];									// 8.5.2.0 (2023/07/14) Delete
		isNums	= new boolean[numberOfColumns];									// 6.0.2.0 (2014/09/19) NUMBER型かどうか
		final boolean useRenderer = isUseRenderer();							// 5.2.1.0 (2010/10/01)

		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし

			// 5.7.6.3 (2014/05/23) ここでﾚﾝﾃﾞﾗ時の文字ﾀｲﾌﾟ判定を行う
			if( useRenderer && dbColumn[clm].isStringOutput() ){
				isNums[i] = false;
			}
			else {
				final NativeType nativeType = dbColumn[clm].getNativeType();
				switch( nativeType ) {
					case INT    :
					case LONG   :
					case DOUBLE :
						isNums[i] = true;
							break;
					case STRING :
					case CALENDAR :
					default :
							isNums[i] = false;
							break;
				}
			}
//			nvar[i] = "NVAR".equals( dbColumn[clm].getDbType()) ;				// 8.5.2.0 (2023/07/14) Delete
		}
	}

	/**
	 * PrintWriter に DBTableModelのﾃｰﾌﾞﾙ情報を書き込みます。
	 * このｸﾗｽでは、ﾃﾞｰﾀを ﾀﾞﾌﾞﾙｺｰﾃｰｼｮﾝ(")で囲みます。
	 * PrintWriter に DBTableModelのﾃｰﾌﾞﾙ情報を書き込みます。
	 *
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 3.8.5.3 (2006/08/07) DBType の nativeType に対応した、CELL_TYPE をｾｯﾄします。
	 * @og.rev 4.1.1.2 (2008/02/28) NativeﾀｲﾌﾟをEnum型(fukurou.model.NativeType)に変更
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput対応
	 * @og.rev 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) ﾃﾞｰﾀ出力用のﾚﾝﾃﾞﾗｰ
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	writer	PrintWriterｵﾌﾞｼﾞｪｸﾄ(未使用)
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows    = table.getRowCount();
		final boolean useRenderer = isUseRenderer();		// 5.2.1.0 (2010/10/01)

		if( useAutoCellSize ) {
			excel.setDataStartRow( nRowIndex );				// ﾃﾞｰﾀ行の書き込み開始位置の行番号を設定
		}

		int rowNo = 1;
		for( ; tblRow<numberOfRows; tblRow++ ) {			// tblRow は、ｼｰﾄﾌﾞﾚｲｸしても連続してｶｳﾝﾄｱｯﾌﾟ
			// まずはﾌｧｲﾙﾌﾞﾚｲｸ判定
			if( fileBreakClmNo >= 0 ) {
				// ﾌｧｲﾙﾌﾞﾚｲｸすると、noPageBreak=false と、noFileBreak=false がｾｯﾄされる。
				final String nowFBC = table.getValue( tblRow,fileBreakClmNo );
				if( !fileKey.equals( nowFBC ) ) {
					fileKey = nowFBC;
					noPageBreak = false;
					return;
				}
			}
			if( pageBreakClmNo >= 0 ) {
				// ｼｰﾄﾌﾞﾚｲｸすると、新しい sheetName がｾｯﾄされ、noPageBreak=false がｾｯﾄされる。
				final String nowPBC = table.getValue( tblRow,pageBreakClmNo );
				if( !sheetName.equals( nowPBC ) ) {
					sheetName = nowPBC;
					return;
				}
			}

			short nColIndex = 0;
			excel.createRow( nRowIndex++ );
			if( useNumber ) { excel.setCellValue( String.valueOf( rowNo++ ),nColIndex++,true ); }

			for( int i=0; i<numberOfColumns; i++ ) {
				final int clm = clmNo[i];
				if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) ｶﾗﾑ飛ばし
					nColIndex++;
					continue;
				}

				String val = table.getValue( tblRow,clm );

				// 6.2.3.0 (2015/05/01) keyLabelは、後ろのｶﾗﾑのﾗﾍﾞﾙを付けません。
				if( isKeyLbl[i] ) {
					excel.setCellValue( val,nColIndex++,isNums[i] );
				}

//				if( nvar[i] ) {													// 8.5.2.0 (2023/07/14) Delete
//					val = StringUtil.getReplaceEscape( val );					// 8.5.2.0 (2023/07/14) Delete
//				}																// 8.5.2.0 (2023/07/14) Delete
				// 5.2.1.0 (2010/10/01) useRenderer 対応
//				else if( useRenderer ) {										// 8.5.2.0 (2023/07/14) Modify
				if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) ﾃﾞｰﾀ出力用のﾚﾝﾃﾞﾗｰ
					val = dbColumn[clm].getWriteValue( val );
				}

				excel.setCellValue( val,nColIndex,isNums[i] );

				// ﾊｲﾊﾟｰﾘﾝｸ設定は、文字の設定の後で行います。
				if( clm == hLinkKeyClmNo && hLinkValClmNo >= 0 ) {
					final String shtNm = table.getValue( tblRow,hLinkValClmNo );
					excel.setCellLink( shtNm,nColIndex );
				}

				nColIndex++;
			}

			// 余計なｾﾙを削除
			excel.removeCell( nColIndex );
		}
		noPageBreak = false;	// 最後まで終了した
		fileKey		= null;		// 最後まで終了した
	}

	/**
	 * 出力先ﾃﾞｨﾚｸﾄﾘとﾌｧｲﾙ名をｾｯﾄします。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * 雛形ﾌｧｲﾙとの関係から、内部的に filename は合成しておきます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 * @og.rev 6.0.2.0 (2014/09/19) ﾃﾞｨﾚｸﾄﾘとﾌｧｲﾙを分けて管理します。
	 *
	 * @param	dir		出力先ﾃﾞｨﾚｸﾄﾘ
	 * @param	fname	出力ﾌｧｲﾙ名
	 */
	@Override
	public void setFilename( final String dir , final String fname ) {
		filename	= StringUtil.urlAppend( dir,fname ) ;

		directory	= dir;
		fileKey		= fname;
	}

	/**
	 * DBTableModelのﾃﾞｰﾀとして読み込むときのｼｰﾄ名を設定します。
	 * 初期値は、"Sheet1" です。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param	shtName	ｼｰﾄ名
	 */
	@Override
	public void setSheetName( final String shtName ) {
		if( shtName != null ) { this.sheetName = shtName; }
	}

	/**
	 * このｸﾗｽが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、ｼｰﾄ名のｾｯﾄ、雛型参照ﾌｧｲﾙ名のｾｯﾄ、
	 * 書き込み元ﾌｧｲﾙのFileｵﾌﾞｼﾞｪｸﾄ取得などの、特殊機能です。
	 * 本来は、ｲﾝﾀｰﾌｪｰｽを分けるべきと考えますが、taglib ｸﾗｽ等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか(常に true)
	 */
	@Override
	public boolean isExcel() {
		return true;
	}

	/**
	 * EXCEL雛型参考ﾌｧｲﾙ名をｾｯﾄします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param	rfname	EXCEL雛型参考ﾌｧｲﾙ名
	 */
	@Override
	public void setRefFilename( final String rfname ) {
		refFilename = rfname;
	}

	/**
	 * 雛形のｼｰﾄ名を、そのまま使用する(true)か、新規、または、外部指定のｼｰﾄ名を使用する(false)を指定します。(初期値:false[外部指定のｼｰﾄ名を使用])。
	 *
	 * ※ Ver5では、追記ﾓｰﾄﾞ時に、指定ｼｰﾄが存在した場合は上書きします(初期値:false[上書きしない])。5.9.12.1 (2016/09/09)
	 *    Ver6では、追記ﾓｰﾄﾞ時に、雛形を指定できないため、雛形のｼｰﾄ名を、そのまま使用する(true)か、
	 *    新規、または、外部指定のｼｰﾄ名を使用する(false)を指定する属性になります。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) sheetOverwrite で、雛形ｼｰﾄの使用時に、元のｼｰﾄ名を使用します。
	 *
	 * @param	flag	元のｼｰﾄ名を使用するかどうか[true:使用する/false:新規、または、外部指定のｼｰﾄ名を使用]
	 */
	@Override
	public void setSheetOverwrite( final boolean flag ) {
		sheetOverwrite = flag;
	}

	/**
	 * EXCELで、出力処理の最後にｾﾙの計算式の再計算をさせるｼｰﾄ名をCSV形式で指定します。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) recalcSheetName で、ｾﾙ内の計算式を再計算させるｼｰﾄ名を指定。5.9.12.1 (2016/09/09)
	 *
	 * @param	sheet	対象ｼｰﾄ名をCSV形式で指定
	 */
	@Override
	public void setRecalcSheetName( final String sheet ){
		recalcSheetNames = StringUtil.csv2Array( sheet);
	}

	/**
	 * EXCEL雛型参考ﾌｧｲﾙのｼｰﾄ名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELﾌｧｲﾙを書き出す時に、雛型として参照するｼｰﾄ名を指定します。
	 * これにより、複数の形式の異なるﾃﾞｰﾀを順次書き出したり(appendﾓｰﾄﾞを併用)する
	 * ことや、ｼｰﾄを指定して新規にEXCELを作成する場合にﾌｫｰﾑを設定する事が可能になります。
	 * 初期値は、null(第一ｼｰﾄ) です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param	rshtName	ｼｰﾄ名
	 */
	@Override
	public void setRefSheetName( final String rshtName ) {
		if( rshtName != null ) { refSheetName = rshtName; }
	}

	/**
	 * ﾃﾞｰﾀの書き込み開始位置を設定します(初期値:0)。
	 *
	 * TAB区切りﾃｷｽﾄやEXCEL等のﾃﾞｰﾀの書き込みの開始位置を指定します。
	 * 属性名は、行を飛ばす処理ということで、readTable ﾀｸﾞと同じ名称です。
	 * ﾌｧｲﾙの先頭行が、０行としてｶｳﾝﾄしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 行の読み飛ばしと、ｶﾗﾑの読み飛ばし(columns)、refFileURL、refFilename、
	 * refSheetName とともに使用すれば、ある程度のﾚｲｱｳﾄ設定が可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) 新規作成
	 *
	 * @param	skipCount	書き込み開始位置
	 */
	@Override
	public void setSkipRowCount( final int skipCount ) {
		skipRowCount = skipCount ;
	}

	/**
	 * EXCEL出力時のﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELﾌｧｲﾙを書き出す時に、ﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名を指定します。
	 * ﾌｫﾝﾄ名は、EXCELのﾌｫﾝﾄ名をそのまま使用してください。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * 雛形ﾌｧｲﾙを使用しない場合のみ、有効です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param	font	ﾃﾞﾌｫﾙﾄﾌｫﾝﾄ名
	 */
	@Override
	public void setFontName( final String font ) {
		fontName = font ;
	}

	/**
	 * EXCEL出力時のﾃﾞﾌｫﾙﾄﾌｫﾝﾄﾎﾟｲﾝﾄ数を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELﾌｧｲﾙを書き出す時に、ﾃﾞﾌｫﾙﾄﾎﾟｲﾝﾄ数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * 雛形ﾌｧｲﾙを使用しない場合のみ、有効です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param	point	ﾃﾞﾌｫﾙﾄﾌｫﾝﾄﾎﾟｲﾝﾄ数
	 */
	@Override
	public void setFontPoint( final short point ) {
		fontPoint = point;
	}

	/**
	 * EXCEL出力時に、ﾃﾞｰﾀを書き込んだ範囲に罫線を入れるかどうかを指定します。
	 *
	 * ﾃﾞｰﾀを書き込んでEXCELを作成しても、ﾉｰﾏﾙのｾﾙに値がｾｯﾄされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、ﾃﾞｰﾀのｾｯﾄされたｾﾙに罫線を入れることで、それなりのﾃﾞｰﾀが
	 * 出力された感じになります。
	 * この設定と、useAutoCellSize="true" で、ｾﾙの幅を自動調整すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useCellStyle	罫線を入れるかどうか(true:入れる/false:入れない)
	 * @see		#setUseAutoCellSize( boolean )
	 */
	@Override
	public void setUseCellStyle( final boolean useCellStyle ) {
		this.useCellStyle = useCellStyle;
	}

	/**
	 * EXCEL出力時に、ｾﾙの幅をﾃﾞｰﾀの幅に自動的に合わせるかどうかを指定します。
	 *
	 * ﾃﾞｰﾀを書き込んでEXCELを作成しても、ﾉｰﾏﾙのｾﾙに値がｾｯﾄされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、ﾃﾞｰﾀのｾｯﾄされたｾﾙの幅を自動調整することで、それなりのﾃﾞｰﾀが
	 * 出力された感じになります。
	 * この設定と、useCellStyle="true" で、ｾﾙの罫線を自動設定すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useAutoCellSize	ﾃﾞｰﾀの幅に自動的に合わせるかどうか(true:自動調整/false:何もしない)
	 * @see		#setUseCellStyle( boolean )
	 */
	@Override
	public void setUseAutoCellSize( final boolean useAutoCellSize ) {
		this.useAutoCellSize = useAutoCellSize;
	}

	/**
	 * EXCEL出力時に、ｾﾙの有効範囲を設定するかどうかを指定します。
	 *
	 * ｾﾙの有効範囲というのは、EXCELでの 空行、空列の存在しない範囲を指します。
	 * 通常、空行でも、ﾃﾞｰﾀとして残っている場合は、EXCELのｾﾙｵﾌﾞｼﾞｪｸﾄは存在します。
	 * ここで、useActiveWorkbook="true" とすると、空行、空列を削除します。
	 *
	 * 雛形を使用した場合は、ﾃﾞｰﾀより多めに設定した計算などは、この処理で
	 * 削除されますので、ﾃﾞｰﾀｻｲｽﾞにﾌｨｯﾄさせることができます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useActiveWorkbook	ｾﾙの有効範囲を設定するかどうか(true:設定する/false:そのまま)
	 */
	@Override
	public void setUseActiveWorkbook( final boolean useActiveWorkbook ) {
		this.useActiveWorkbook = useActiveWorkbook;
	}

	/**
	 * EXCEL出力時に、ｼｰﾄﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がｼｰﾄ名になります)。
	 *
	 * EXCEL帳票では、帳票雛形に、PAGE_BRAKE ｷｰを設定しましたが、TableWriterでは、
	 * ﾒﾓﾘ上のｶﾗﾑの値が変更したときに、ｼｰﾄﾌﾞﾚｲｸさせることができます。
	 * このｶﾗﾑの値がｷｰﾌﾞﾚｲｸすると、新しいｼｰﾄに書き出し始めます。
	 * ｼｰﾄ名は、このｶﾗﾑの値（ｷｰﾌﾞﾚｲｸする値）です。
	 *
	 * 雛形ﾌｧｲﾙを使用する場合、雛形ｼｰﾄもｷｰﾌﾞﾚｲｸに伴って、＋１されます。
	 * つまり、雛形ｼｰﾄとﾃﾞｰﾀｼｰﾄは同時に変更されます。
	 * ただし、雛形ｼｰﾄは、最後の雛形ｼｰﾄで止まります。
	 * これは、雛形ｼｰﾄにﾍｯﾀﾞｰ雛形とﾎﾞﾃﾞｨ雛形を用意しておき、最初のｷｰﾌﾞﾚｲｸで
	 * ﾍｯﾀﾞｰからﾎﾞﾃﾞｨの書き込みにﾁｪﾝｼﾞするｲﾒｰｼﾞで使用できます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	pageBreakColumn	ｼｰﾄﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定
	 * @see		#setFileBreakColumn( String )
	 */
	@Override
	public void setPageBreakColumn( final String pageBreakColumn ) {
		this.pageBreakColumn = pageBreakColumn;
	}

	/**
	 * EXCEL出力時に、ﾌｧｲﾙﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定します(このｶﾗﾑの値がﾌｧｲﾙ名になります)。
	 *
	 * EXCEL帳票では、ﾒﾓﾘ上のｶﾗﾑの値が変更したときに、ﾌｧｲﾙﾌﾞﾚｲｸすることができます。
	 * このｶﾗﾑの値がｷｰﾌﾞﾚｲｸすると、新しいﾌｧｲﾙに書き出し始めます。
	 * ﾌｧｲﾙ名は、このｶﾗﾑの値（ｷｰﾌﾞﾚｲｸする値）＋ 元の出力ﾌｧｲﾙ名の拡張子(.xlsなど)です。
	 * この設定を使用する場合は、出力ﾌｧｲﾙ名は無視されますが、拡張子だけは使用されます。
	 *
	 * 雛形ﾌｧｲﾙを使用する場合、雛形ﾌｧｲﾙもｷｰﾌﾞﾚｲｸに伴って、再利用されます。
	 * 例えば、pageBreakColumn と併用する場合、ｷｰﾌﾞﾚｲｸで雛形ｼｰﾄも最初から適用になります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fileBreakColumn	ﾌｧｲﾙﾌﾞﾚｲｸするｷｰとなるｶﾗﾑ名を指定
	 * @see		#setPageBreakColumn( String )
	 */
	@Override
	public void setFileBreakColumn( final String fileBreakColumn ) {
		this.fileBreakColumn = fileBreakColumn;
	}

	/**
	 * EXCEL出力時に、Hyperlinkを作成するｷｰとなるｶﾗﾑ名と値となるｶﾗﾑ名を指定します。
	 *
	 * ここで、作成するﾊｲﾊﾟｰﾘﾝｸは、EXCELのｼｰﾄに対するﾊｲﾊﾟｰﾘﾝｸです。
	 * それ以外のﾘﾝｸ(本当のURLやﾌｧｲﾙ等)のﾘﾝｸは(今は)作成できません。
	 * ﾊｲﾊﾟｰﾘﾝｸを作成するには、①作成するｶﾗﾑ と ②作成する値 が必要です。
	 * このﾒｿｯﾄﾞで設定するのは、｢①:②｣という形式でｶﾗﾑ名を指定します。
	 * ②がなければ、①と同じとします。
	 * ②の値のｼｰﾄの存在有無は、無視します。ﾊｲﾊﾟｰﾘﾝｸを作成するｼｰﾄを作成する前に
	 * ﾊｲﾊﾟｰﾘﾝｸを作成するｹｰｽが存在します。
	 * (例えば、各ｼｰﾄへのﾘﾝｸを持った一覧を作成してから、明細の各ｼｰﾄを作成する様なｹｰｽ)
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	hyperLinkColumn	Hyperlinkを作成するｷｰとなるｶﾗﾑ名と値となるｶﾗﾑ名を指定
	 */
	@Override
	public void setHyperLinkColumn( final String hyperLinkColumn ) {
		if( hyperLinkColumn != null && hyperLinkColumn.length() > 0 ) {
			final int sep = hyperLinkColumn.indexOf( ':' );						// ①:② の形式かどうかの確認
			if( sep >= 0 ) {
				hLinkKeyColumn = hyperLinkColumn.substring( 0,sep );			// ①ｷｰｶﾗﾑ
				hLinkValColumn = hyperLinkColumn.substring( sep+1 );			// ②値ｶﾗﾑ
			}
			else {
				hLinkKeyColumn = hyperLinkColumn;								// ①ｷｰｶﾗﾑ = ②値ｶﾗﾑ
				hLinkValColumn = hyperLinkColumn;
			}
		}
	}

	/**
	 * EXCEL出力時に、Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します。
	 *
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	sheetName	EXCELﾌｧｲﾙのｼｰﾄ名
	 */
	@Override
	public void setAddTitleSheet( final String sheetName ) {
		addTitleSheet = sheetName ;
	}
}
